// TTAOutput.h
// 2008/12/25

#pragma once

#include "TTADecoder.h"

namespace QAX {

/* */

#pragma pack(push, 2)

// QTTAHeader
struct QTTAHeader {

	BYTE   Signature[4];
	UINT16 Channels;
	UINT16 BitsPerSample;
	UINT32 SamplingRate;
	UINT32 SamplesPerFrame;

}; // QTTAHeader

#pragma pack(pop)

/* */

// TTAOutputBase
class TTAOutputBase : public QDecoderOutput {

	QTTAHeader m_Header;

	UINT64 m_SamplePos;
	UINT64 m_Samples;

	StreamReader* m_Reader;

	SIZE_T m_Position;
	SIZE_T m_Size;

	QTTADecoder_t* m_decoder;

	const INT16* m_Output;
	UINT32       m_OutputSample;
	UINT32       m_BytesPerSample;

	UINT32 m_SeekOffset;

	bool m_bEndOfStream;

	TTAIndexArray m_Index;

	enum {
		INIT_HEADER_SIZE = 0x100
	};

	enum {
		FLAGS_START = 0x01,
		FLAGS_END   = 0x02
	};

protected:

	TTAOutputBase() :
		m_SamplePos(0),
		m_Samples(0),
		m_Reader(0),
		m_Position(0),
		m_Size(0),
		m_decoder(0),
		m_Output(0),
		m_OutputSample(0),
		m_BytesPerSample(0),
		m_SeekOffset(0),
		m_bEndOfStream(false)
	{
		memset(&m_Header, 0, sizeof(m_Header));
	}

	void SafeRelease()
	{
		if (m_Reader != 0) {
			m_Reader->Release();
			m_Reader = 0;
		}

		if (m_decoder != 0) {
			QT_ReleaseDecoder(m_decoder);
			m_decoder = 0;
		}
	}

public:

	void SetupDecoder(
		const VOID* setup,
		SIZE_T      size,
		UINT64      samples)
	{
		if (size != sizeof(m_Header)) {
			DecoderError::Throw("SetupDecoder.Header.Size");
		}

		memcpy(&m_Header, setup, size);

		if (memcmp(m_Header.Signature, "TTA1", 4) != 0) {
			DecoderError::Throw("SetupDecoder.Header.Signature");
		}

		if (m_Header.Channels != 1 && m_Header.Channels != 2) {
			DecoderError::Throw("SetupDecoder.Header.Channels");
		}

		if (m_Header.BitsPerSample != 16) {
			DecoderError::Throw("SetupDecoder.Header.BitsPerSample");
		}

		m_SamplePos = 0;
		m_Samples   = samples;

		m_decoder = QT_CreateDecoder();
		if (m_decoder == 0) {
			DecoderError::Throw("SetupDecoder.OutOfMemory");
		}

		if (!QT_SetupDecoder(
			m_decoder,
			m_Header.Channels,
			m_Header.SamplesPerFrame)) {
			DecoderError::Throw("SetupDecoder.QT_SetupDecoder");
		}

		m_Output         = 0;
		m_OutputSample   = 0;
		m_BytesPerSample = 2 * m_Header.Channels;
		m_bEndOfStream   = false;
	}

	void SetupReader(
		StreamReader* r,
		SIZE_T        sz)
	{
		m_Reader = r;

		m_Position = 0;
		m_Size     = sz;
	}

	UINT64 GetSamples()
	{
		return m_Samples;
	}

	UINT32 GetFrameSamples()
	{
		return m_Header.SamplesPerFrame;
	}

	/* */

	~TTAOutputBase()
	{
		SafeRelease();
	}

	virtual void STDCALL Release()
	{
		delete this;
	}

	virtual QDecoderSpec STDCALL GetDecoderSpec()
	{
		QDecoderSpec spec = { 0 };

		spec.Channels     = m_Header.Channels;
		spec.SamplingRate = m_Header.SamplingRate;
		spec.Samples      = m_Samples;

		return spec;
	}

	virtual UINT32 STDCALL Decode(
		VOID*  buffer,
		UINT32 samples)
	{
		UINT32 smps = 0;

		for (; ; ) {
			if (m_OutputSample > 0) {
				UINT32 s = samples;
				if (s > m_OutputSample) {
					s = m_OutputSample;
				}

				SIZE_T cb = s * m_BytesPerSample;
				memcpy(
					buffer,
					m_Output,
					cb);

				m_Output       += s * m_Header.Channels;
				m_OutputSample -= s;

				smps += s;

				break;
			}

			if (m_bEndOfStream) {
				break;
			}

			DecodePacket();
		}

		return smps;
	}

	/* */

	virtual void STDCALL Seek(
		UINT64 sample)
	{
		if (m_Index.IsEmpty()) {
			OnBuildIndex(m_Index);
		}

		const TTAIndexEntry& e = m_Index.SeekIndex(sample);

		m_Position = SIZE_T(e.Position);

		m_Reader->Seek(m_Position);

		UINT32 offset = UINT32(sample - e.Sample);

		m_SamplePos    = sample;
		m_SeekOffset   = offset;
		m_OutputSample = 0;

		m_bEndOfStream = false;
	}

	/* */

	virtual std::string STDCALL QueryProperty(
		const char* name)
	{
		DecoderError::Throw("Not.Implemented");
	}

	/* */

	void DecodePacket()
	{
		if (m_Position >= m_Size) {
			m_bEndOfStream = true;
			return;
		}

		ReadRange r = m_Reader->Fetch(INIT_HEADER_SIZE);

		const BYTE* p   = static_cast<const BYTE*>(r.Buffer);
		const BYTE* end = p + r.Length;

		if (p >= end) {
			m_bEndOfStream = true;
			return;
		}

		BYTE flags = *(p++);

		UINT32 size;
		p += DecodeVNumber(p, end, &size);

		SIZE_T h_size = p - static_cast<const BYTE*>(r.Buffer);

		SIZE_T p_size = h_size + size + 4;

		r = m_Reader->Fetch(p_size);

		if (r.Length != p_size) {
			FormatError::Throw("Page.Size.OutOfRange");
		}

		p = static_cast<const BYTE*>(r.Buffer);

		CRC32 crc32;
		UINT32 crc = crc32.Generate(p, h_size + size);

		UINT32 val;
		CopyMemory(&val, p + h_size + size, sizeof(UINT32));

		if (val != crc) {
			FormatError::Throw("Page.CRC.Error");
		}

		QT_Output_t output = { 0 };
		if (!QT_DecodeFrame(
			m_decoder,
			p + h_size,
			size,
			&output)) {
			DecoderError::Throw("QT_DecodeFrame");
		}

		m_Reader->Retire(p_size);

		m_Position += p_size;

		UINT32 samples = output.Length;
		if ((flags & FLAGS_END) != 0) {
			samples = UINT32(m_Samples - m_SamplePos);
			if (samples > (UINT32)output.Length) {
				FormatError::Throw("Page.LastSamplesCount");
			}
		}

		m_Output  = output.Sample + m_SeekOffset * m_Header.Channels;
		samples  -= m_SeekOffset;

		m_SeekOffset = 0;

		m_OutputSample = samples;

		m_SamplePos += samples;
	}

	/* */

	virtual void OnBuildIndex(TTAIndexArray& index) = 0;

	/* */

}; // TTAOutputBase

/* */

// QAXTTAOutputSpec
struct QAXTTAOutputSpec {

	const VOID*   Setup;
	SIZE_T        SetupSize;

	ItemProperty* Property;

	UINT64        Samples;

	QFileMapping* Mapping;
	UINT64        Position;
	SIZE_T        Size;

	const VOID*   Index;
	SIZE_T        IndexSize;

}; // QAXTTAOutputSpec

// QAXTTAOutputImpl
class QAXTTAOutputImpl : public TTAOutputBase {

	ItemProperty* m_Property;

	const VOID* m_IndexChunk;
	SIZE_T      m_IndexSize;

	QAXTTAOutputImpl() :
		m_Property(0),
		m_IndexChunk(0),
		m_IndexSize(0)
	{
	}

	void SetupProperty(ItemProperty* p)
	{
		m_Property = p;
	}

	void SetupIndex(const VOID* p, SIZE_T s)
	{
		m_IndexChunk = p;
		m_IndexSize  = s;
	}

public:

	static QDecoderOutput* Create(QAXTTAOutputSpec* spec)
	{
		QAXTTAOutputImpl* impl = 0;

		try {
			impl = new QAXTTAOutputImpl();

			impl->SetupDecoder(
				spec->Setup,
				spec->SetupSize,
				spec->Samples);

			impl->SetupProperty(
				spec->Property);

			StreamReader* reader = FileMapping_Reader::Create(
				spec->Mapping,
				spec->Position,
				spec->Size);

			impl->SetupReader(
				reader,
				spec->Size);

			impl->SetupIndex(
				spec->Index,
				spec->IndexSize);

		} catch (...) {
			if (impl != 0) {
				impl->Release();
			}
			throw;
		}

		return impl;
	}

	virtual std::string STDCALL QueryProperty(
		const char* name)
	{
		ItemProperty::iterator it = m_Property->find(name);
		if (it == m_Property->end()) {
			return std::string();
		}

		return (*it).second;
	}

	virtual void OnBuildIndex(TTAIndexArray& index)
	{
		UINT32 duration = GetFrameSamples();
		UINT64 samples  = GetSamples();

		index.ParseIndex(
			m_IndexChunk,
			m_IndexSize,
			duration,
			samples);
	}

}; // QAXTTAOutputImpl

/* */

// QASTTAOutputImpl
class QASTTAOutputImpl : public TTAOutputBase {

	VOID* m_MetaData;

	const VOID* m_IndexChunk;
	SIZE_T      m_IndexSize;

public:

	QASTTAOutputImpl() :
		m_MetaData(0),
		m_IndexChunk(0),
		m_IndexSize(0)
	{
	}

	~QASTTAOutputImpl()
	{
		if (m_MetaData != 0) {
			free(m_MetaData);
			m_MetaData = 0;
		}
	}

	void SetupIndex(VOID* data, const VOID* p, SIZE_T s)
	{
		m_MetaData   = data;
		m_IndexChunk = p;
		m_IndexSize  = s;
	}

	virtual void OnBuildIndex(TTAIndexArray& index)
	{
		UINT32 duration = GetFrameSamples();
		UINT64 samples  = GetSamples();

		index.ParseIndex(
			m_IndexChunk,
			m_IndexSize,
			duration,
			samples);
	}

}; // QASTTAOutputImpl

/* */

} // namespace QAX

