/* Floor1Decoder.c */
/* 2008/11/06      */

#include "StdAfx.h"

#include "Floor1Decoder.h"

#include "HuffmanDecoder.h"

/* */

static void SetupList(
	INT32          count,
	const UINT16*  x,
	Floor1Entry_t* list);

/* */

BOOL Floor1Decoder_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	Floor1Setup_t*    setup)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	INT32 i;
	INT32 type;

	INT32 max_class = -1;

	INT32 bits;

	UINT16  X[32*8];
	UINT16* p = X;

	static const INT32 RANGE[] = { 256, 128, 86, 64 };

	type = BitDecoder_GetBits(d, 16);
	if (type != 1) {
		THROW0(ctx, "Floor1Decoder_Setup", "InvalidType")
	}

	setup->Partitions = BitDecoder_GetBits(d, 5);
	for (i = 0; i < setup->Partitions; i++) {
		INT32 p_class = BitDecoder_GetBits(d, 4);
		setup->PartitionClassList[i] = p_class;
		if (max_class < p_class) {
			max_class = p_class;
		}
	}

	for (i = 0; i <= max_class; i++) {
		INT32 j, k;
		setup->ClassDimensions[i] = BitDecoder_GetBits(d, 3) + 1;
		setup->ClassSubClasses[i] = BitDecoder_GetBits(d, 2);
		if (setup->ClassSubClasses[i] != 0) {
			setup->ClassMasterBooks[i] = BitDecoder_GetBits(d, 8);
		}
		k = 1 << setup->ClassSubClasses[i];
		for (j = 0; j < k; j++) {
			setup->SubclassBooks[i][j] = BitDecoder_GetBits(d, 8) - 1;
		}
	}

	setup->Multiplier = BitDecoder_GetBits(d, 2) + 1;
	setup->Range      = RANGE[setup->Multiplier - 1];
	setup->RangeBits  = ilog(setup->Range - 1);

	bits = BitDecoder_GetBits(d, 4);

	*(p++) = 0;
	*(p++) = 1 << bits;

	for (i = 0; i < setup->Partitions; i++) {
		INT32 j;
		INT32 k = setup->PartitionClassList[i];
		for (j = 0; j < setup->ClassDimensions[k]; j++) {
			*(p++) = BitDecoder_GetBits(d, bits);
		}
	}

	setup->Values = p - X;

	setup->List = (Floor1Entry_t*)MemoryPool_Allocate(pool, sizeof(Floor1Entry_t) * setup->Values);
	if (setup->List == NULL) {
		THROW0(ctx, "Floor1Decoder_Setup", "OutOfMemory")
	}

	memset(setup->List, 0, sizeof(Floor1Entry_t) * setup->Values);

	SetupList(
		setup->Values,
		X,
		setup->List);

	result = TRUE;

ErrorEnd:

	return result;
}

void SetupList(
	INT32          count,
	const UINT16*  x,
	Floor1Entry_t* list)
{
	INT32 i, j;

	for (i = 0; i < count; i++) {
		list[i].X = x[i];
	}

	list[0].V = 0;
	list[1].V = 1;

	for (i = 2; i < count; i++) {
		list[i].Low  = 0;
		list[i].High = 1;
		list[i].V    = i;

		for (j = 2; j < i; j++) {
			UINT16 t = list[j].X;
			if (t < list[i].X) {
				if (t > list[list[i].Low].X) {
					list[i].Low = j;
				}
			} else {
				if (t < list[list[i].High].X) {
					list[i].High = j;
				}
			}
		}
	}

	for (i = 0; i < count - 1; i++) {
		for (j = i + 1; j < count; j++) {
			if (list[list[i].V].X > list[list[j].V].X) {
				UINT16 t = list[i].V;
				list[i].V = list[j].V;
				list[j].V = t;
			}
		}
	}
}

/* */

BOOL Floor1Decoder_Parse(
	DecoderContext_t*       ctx,
	BitDecoder_t*           d,
	const Floor1Setup_t*    setup,
	const HuffmanDecoder_t* huff,
	INT16*                  y)
{
	BOOL result = FALSE;

	INT32 i, j;

	INT16* p = y;

	if (BitDecoder_GetBits(d, 1) == 0) {
		memset(y, 0, sizeof(INT16) * setup->Values);
		return TRUE;
	}

	*(p++) = BitDecoder_GetBits(d, setup->RangeBits);
	*(p++) = BitDecoder_GetBits(d, setup->RangeBits);

	for (i = 0; i < setup->Partitions; i++) {
		INT32 pc = setup->PartitionClassList[i];
		INT32 cd = setup->ClassDimensions[pc];
		INT32 cs = setup->ClassSubClasses[pc];

		INT32 sub = (1 << cs) - 1;
		INT32 val = 0;

		if (cs > 0) {
			INT32 book = setup->ClassMasterBooks[pc];
			val = HuffmanDecoder_Lookup(huff + book, d);
			if (val < 0) {
				THROW0(ctx, "Floor1Decoder_Parse", "InvalidMasterCode")
			}
		}

		for (j = 0; j < cd; j++) {
			INT32 book = setup->SubclassBooks[pc][val & sub];
			val >>= cs;
			if (book >= 0) {
				INT32 v = HuffmanDecoder_Lookup(huff + book, d);
				if (v < 0) {
					THROW0(ctx, "Floor1Decoder_Parse", "InvalidSubclassCode")
				}
				*(p++) = v;

			} else {
				*(p++) = 0;
			}
		}
	}

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

static const FLOAT DB_TABLE[256] = {
	1.0649863e-07F, 1.1341951e-07F, 1.2079015e-07F, 1.2863978e-07F,
	1.3699951e-07F, 1.4590251e-07F, 1.5538408e-07F, 1.6548181e-07F,
	1.7623575e-07F, 1.8768855e-07F, 1.9988561e-07F, 2.1287530e-07F,
	2.2670913e-07F, 2.4144197e-07F, 2.5713223e-07F, 2.7384213e-07F,
	2.9163793e-07F, 3.1059021e-07F, 3.3077411e-07F, 3.5226968e-07F,
	3.7516214e-07F, 3.9954229e-07F, 4.2550680e-07F, 4.5315863e-07F,
	4.8260743e-07F, 5.1396998e-07F, 5.4737065e-07F, 5.8294187e-07F,
	6.2082472e-07F, 6.6116941e-07F, 7.0413592e-07F, 7.4989464e-07F,
	7.9862701e-07F, 8.5052630e-07F, 9.0579828e-07F, 9.6466216e-07F,
	1.0273513e-06F, 1.0941144e-06F, 1.1652161e-06F, 1.2409384e-06F,
	1.3215816e-06F, 1.4074654e-06F, 1.4989305e-06F, 1.5963394e-06F,
	1.7000785e-06F, 1.8105592e-06F, 1.9282195e-06F, 2.0535261e-06F,
	2.1869758e-06F, 2.3290978e-06F, 2.4804557e-06F, 2.6416497e-06F,
	2.8133190e-06F, 2.9961443e-06F, 3.1908506e-06F, 3.3982101e-06F,
	3.6190449e-06F, 3.8542308e-06F, 4.1047004e-06F, 4.3714470e-06F,
	4.6555282e-06F, 4.9580707e-06F, 5.2802740e-06F, 5.6234160e-06F,
	5.9888572e-06F, 6.3780469e-06F, 6.7925283e-06F, 7.2339451e-06F,
	7.7040476e-06F, 8.2047000e-06F, 8.7378876e-06F, 9.3057248e-06F,
	9.9104632e-06F, 1.0554501e-05F, 1.1240392e-05F, 1.1970856e-05F,
	1.2748789e-05F, 1.3577278e-05F, 1.4459606e-05F, 1.5399272e-05F,
	1.6400004e-05F, 1.7465768e-05F, 1.8600792e-05F, 1.9809576e-05F,
	2.1096914e-05F, 2.2467911e-05F, 2.3928002e-05F, 2.5482978e-05F,
	2.7139006e-05F, 2.8902651e-05F, 3.0780908e-05F, 3.2781225e-05F,
	3.4911534e-05F, 3.7180282e-05F, 3.9596466e-05F, 4.2169667e-05F,
	4.4910090e-05F, 4.7828601e-05F, 5.0936773e-05F, 5.4246931e-05F,
	5.7772202e-05F, 6.1526565e-05F, 6.5524908e-05F, 6.9783085e-05F,
	7.4317983e-05F, 7.9147585e-05F, 8.4291040e-05F, 8.9768747e-05F,
	9.5602426e-05F, 0.00010181521F, 0.00010843174F, 0.00011547824F,
	0.00012298267F, 0.00013097477F, 0.00013948625F, 0.00014855085F,
	0.00015820453F, 0.00016848555F, 0.00017943469F, 0.00019109536F,
	0.00020351382F, 0.00021673929F, 0.00023082423F, 0.00024582449F,
	0.00026179955F, 0.00027881276F, 0.00029693158F, 0.00031622787F,
	0.00033677814F, 0.00035866388F, 0.00038197188F, 0.00040679456F,
	0.00043323036F, 0.00046138411F, 0.00049136745F, 0.00052329927F,
	0.00055730621F, 0.00059352311F, 0.00063209358F, 0.00067317058F,
	0.00071691700F, 0.00076350630F, 0.00081312324F, 0.00086596457F,
	0.00092223983F, 0.00098217216F, 0.0010459992F, 0.0011139742F,
	0.0011863665F, 0.0012634633F, 0.0013455702F, 0.0014330129F,
	0.0015261382F, 0.0016253153F, 0.0017309374F, 0.0018434235F,
	0.0019632195F, 0.0020908006F, 0.0022266726F, 0.0023713743F,
	0.0025254795F, 0.0026895994F, 0.0028643847F, 0.0030505286F,
	0.0032487691F, 0.0034598925F, 0.0036847358F, 0.0039241906F,
	0.0041792066F, 0.0044507950F, 0.0047400328F, 0.0050480668F,
	0.0053761186F, 0.0057254891F, 0.0060975636F, 0.0064938176F,
	0.0069158225F, 0.0073652516F, 0.0078438871F, 0.0083536271F,
	0.0088964928F, 0.009474637F, 0.010090352F, 0.010746080F,
	0.011444421F, 0.012188144F, 0.012980198F, 0.013823725F,
	0.014722068F, 0.015678791F, 0.016697687F, 0.017782797F,
	0.018938423F, 0.020169149F, 0.021479854F, 0.022875735F,
	0.024362330F, 0.025945531F, 0.027631618F, 0.029427276F,
	0.031339626F, 0.033376252F, 0.035545228F, 0.037855157F,
	0.040315199F, 0.042935108F, 0.045725273F, 0.048696758F,
	0.051861348F, 0.055231591F, 0.058820850F, 0.062643361F,
	0.066714279F, 0.071049749F, 0.075666962F, 0.080584227F,
	0.085821044F, 0.091398179F, 0.097337747F, 0.10366330F,
	0.11039993F, 0.11757434F, 0.12521498F, 0.13335215F,
	0.14201813F, 0.15124727F, 0.16107617F, 0.17154380F,
	0.18269168F, 0.19456402F, 0.20720788F, 0.22067342F,
	0.23501402F, 0.25028656F, 0.26655159F, 0.28387361F,
	0.30232132F, 0.32196786F, 0.34289114F, 0.36517414F,
	0.38890521F, 0.41417847F, 0.44109412F, 0.46975890F,
	0.50028648F, 0.53279791F, 0.56742212F, 0.60429640F,
	0.64356699F, 0.68538959F, 0.72993007F, 0.77736504F,
	0.82788260F, 0.88168307F, 0.9389798F, 1.0F,
};

/* */

static INT32 RenderPoint(INT32 x0, INT32 x1, INT32 yy0, INT32 yy1, INT32 x)
{
	INT32 y0 = yy0 & 0x7fff;
	INT32 y1 = yy1 & 0x7fff;

	INT32 dy = y1 - y0;

	INT32 ady = (dy >= 0) ? dy : - dy;
	INT32 adx = x1 - x0;

	INT32 error  = ady * (x - x0);
	INT32 offset = error / adx;

	return (dy >= 0) ? y0 + offset : y0 - offset;
}

static void RenderLine(INT32 x0, INT32 x1, INT32 y0, INT32 y1, FLOAT* z)
{
	INT32 dy = y1 - y0;

	INT32 adx = x1 - x0;
	INT32 ady = (dy >= 0) ? dy : - dy;

	INT32 base = dy / adx;
	INT32 ab   = (base >= 0) ? base : -base;

	INT32 x = x0;
	INT32 y = y0;

	INT32 e = 0;
	INT32 sy = (dy < 0) ? -1 : 1;

	ady -= ab * adx;

	z[x++] *= DB_TABLE[y];

	while (x < x1) {
		e += ady;
		if (e >= adx) {
			e -= adx;
			y += sy;
		}

		y += base;

		z[x++] *= DB_TABLE[y];
	}
}

/* */

void Floor1Decoder_Decode(
	const Floor1Setup_t* setup,
	INT16*               y,
	FLOAT*               z,
	INT32                count)
{
	INT32 i;
	INT32 lx, ly;

	for (i = 2; i < setup->Values; i++) {
		INT32 low  = setup->List[i].Low;
		INT32 high = setup->List[i].High;

		INT32 x0 = setup->List[low ].X;
		INT32 x1 = setup->List[high].X;

		INT32 y0 = y[low ];
		INT32 y1 = y[high];

		INT32 x = setup->List[i].X;

		INT32 pred = RenderPoint(x0, x1, y0, y1, x);

		INT32 hi_room = setup->Range - pred;
		INT32 lo_room = pred;

		INT32 room = (hi_room < lo_room) ? hi_room * 2 : lo_room * 2;

		INT32 value = y[i];

		if (value != 0) {
			if (value >= room) {
				if (hi_room > lo_room) {
					value = value - lo_room;
				} else {
					value = -1 - (value - hi_room);
				}
			} else {
				if ((value & 1) != 0) {
					value = -(value + 1) / 2;
				} else {
					value /= 2;
				}
			}

			y[low ] &= 0x7fff;
			y[high] &= 0x7fff;

			y[i] = pred + value;

		} else {
			y[i] = pred | 0x8000;
		}
	}

	/* */

	lx = 0;
	ly = (y[0] & 0x7fff) * setup->Multiplier;

	for (i = 1; i < setup->Values; i++) {
		INT32 pos = setup->List[i].V;
		INT32 yy = y[pos];

		if ((yy & 0x8000) == 0) {
			INT32 x1 = setup->List[pos].X;
			INT32 y1 = (yy & 0x7fff) * setup->Multiplier;

			if (lx < count) {
				INT32 xx = (x1 < count) ? x1 : count;
				RenderLine(lx, xx, ly, y1, z);
			}

			lx = x1;
			ly = y1;
		}

		if (lx >= count) {
			break;
		}
	}

	if (lx < count) {
		RenderLine(lx, count, ly, ly, z);
	}
}

/* */

