// WuQVorbis.h
// 2008/12/10

#pragma once

#include "kr2/tvpsnd.h"

#ifndef STDCALL
#define STDCALL __stdcall
#endif

/* */

// QModule
class QModule : public ITSSModule {

	ULONG m_RefCount;

	ITSSStorageProvider* m_Provider;

	QAX::QDecoderFactory* m_Factory;

	struct DecoderBundle {
		QAX::QFileMapping* Mapping;
		QAX::QDecoder*     Decoder;
	};

	typedef std::map<std::wstring, DecoderBundle> DecoderMap;

	DecoderMap m_Decoders;

public:

	QModule(ITSSStorageProvider* p) : m_Provider(p), m_RefCount(1), m_Factory(0)
	{
		Initialize();
	}

	~QModule()
	{
		Uninitialize();
	}

private:

	void Initialize()
	{
		try {
			m_Factory = QAX::QDecoderFactoryCreate();

		} catch (QAX::Exception&) {

		}
	}

	void Uninitialize()
	{
		for (DecoderMap::iterator it = m_Decoders.begin(); it != m_Decoders.end(); ++it) {
			const DecoderBundle& b = (*it).second;

			if (b.Decoder != 0) {
				b.Decoder->Release();
			}

			if (b.Mapping != 0) {
				b.Mapping->Release();
			}
		}

		m_Decoders.clear();
	}

	QAX::QDecoder* SetupDecoder(const std::wstring& key)
	{
		QAX::QDecoder* p = 0;

		try {
			DecoderMap::iterator it = m_Decoders.find(key);
			if (it != m_Decoders.end()) {
				p = (*it).second.Decoder;

			} else {
				QAX::QFileMapping* mapping = QAX::QFileMappingCreate(key.c_str());

				QAX::QDecoder* decoder = QAX::QDecoderCreate(m_Factory, mapping);

				DecoderBundle bundle;
				bundle.Mapping = mapping;
				bundle.Decoder = decoder;

				m_Decoders.insert(
					DecoderMap::value_type(key, bundle));

				p = decoder;
			}

		} catch (QAX::Exception&) {

		}

		return p;
	}

public:

	/* IUnknown */

	STDMETHOD(QueryInterface)(
		REFIID iid,
		void** ppvObject)
	{
		if (ppvObject == 0) {
			return E_INVALIDARG;
		}

		void* pv = 0;

		if (iid == IID_IUnknown) {
			pv = static_cast<IUnknown*>(this);
		} else if (iid == IID_ITSSModule) {
			pv = static_cast<ITSSModule*>(this);
		}

		*ppvObject = pv;

		if (pv == 0) {
			return E_NOINTERFACE;
		}

		AddRef();

		return S_OK;
	}

	STDMETHOD_(ULONG, AddRef)()
	{
		m_RefCount += 1;
		return m_RefCount;
	}

	STDMETHOD_(ULONG, Release)()
	{
		if (m_RefCount > 1) {
			m_RefCount -= 1;
			return m_RefCount;
		}

		delete this;
		return 0;
	}

	/* ITSSModule */

	STDMETHOD(GetModuleCopyright)(LPWSTR buffer, unsigned long buflen)
	{
		lstrcpynW(buffer, L"WuQVorbis QAS/QAX Decoder Plug-in (C) 2008 Porject Quilla", buflen);
		return S_OK;
	}

	STDMETHOD(GetModuleDescription)(LPWSTR buffer, unsigned long buflen)
	{
		lstrcpynW(buffer, L"QAS/QAX (*.qas, *.qax) Decoder", buflen);
		return S_OK;
	}

	STDMETHOD(GetSupportExts)(unsigned long index, LPWSTR mediashortname, LPWSTR buf, unsigned long buflen)
	{
		switch (index) {
		case 0:
			lstrcpyW(mediashortname, L"QAS Audio File");
			lstrcpynW(buf, L".qas", buflen);
			break;

		case 1:
			lstrcpyW(mediashortname, L"QAX Audio File");
			lstrcpynW(buf, L".qax", buflen);
			break;

		default:
			return S_FALSE;
		}

		return S_OK;
	}

	/* */

	STDMETHOD(GetMediaInfo)(LPWSTR url, ITSSMediaBaseInfo** info)
	{
		return E_NOTIMPL;
	}

	STDMETHOD(GetMediaSupport)(LPWSTR url)
	{
		return E_NOTIMPL;
	}

	/* */

	STDMETHOD(GetMediaInstance)(
		LPWSTR     url,
		IUnknown** instance);

	/* */

private:

	/* */

	HRESULT CreateQAXOutput(
		LPWSTR     url,
		IUnknown** instance);

	HRESULT CreateQASOutput(
		LPWSTR     url,
		IUnknown** instance);

	/* */

}; // QModule

/* */

// QDecoder
class QDecoder : public ITSSWaveDecoder {

	ULONG m_RefCount;

	QAX::QDecoderOutput* m_Decoder;

	UINT32 m_Channels;

public:

	QDecoder(QAX::QDecoderOutput* p) :
		m_RefCount(1),
		m_Decoder(p),
		m_Channels(0)
	{
		QAX::QDecoderSpec spec = m_Decoder->GetDecoderSpec();

		m_Channels = spec.Channels;
	}

	~QDecoder()
	{
		if (m_Decoder != 0) {
			m_Decoder->Release();
			m_Decoder = 0;
		}
	}

	/* IUnknown */

	STDMETHOD(QueryInterface)(
		REFIID iid,
		void** ppvObject)
	{
		if (ppvObject == 0) {
			return E_INVALIDARG;
		}

		void* pv = 0;

		if (iid == IID_IUnknown) {
			pv = static_cast<IUnknown*>(this);
		} else if (iid == IID_ITSSWaveDecoder) {
			pv = static_cast<ITSSWaveDecoder*>(this);
		}

		*ppvObject = pv;

		if (pv == 0) {
			return E_NOINTERFACE;
		}

		AddRef();

		return S_OK;
	}

	STDMETHOD_(ULONG, AddRef)()
	{
		m_RefCount += 1;
		return m_RefCount;
	}

	STDMETHOD_(ULONG, Release)()
	{
		if (m_RefCount > 1) {
			m_RefCount -= 1;
			return m_RefCount;
		}

		delete this;
		return 0;
	}

	/* ITSSWaveDecoder */

	STDMETHOD(GetFormat)(TSSWaveFormat* format)
	{
		if (format == 0) {
			return E_INVALIDARG;
		}

		ZeroMemory(format, sizeof(TSSWaveFormat));

		QAX::QDecoderSpec spec = m_Decoder->GetDecoderSpec();

		format->dwSamplesPerSec  = spec.SamplingRate;
		format->dwChannels       = spec.Channels;
		format->dwBitsPerSample  = 16;
		format->dwSeekable       = 2;
		format->ui64TotalSamples = spec.Samples;
		format->dwTotalTime      = UINT32(spec.Samples * 1000 / spec.SamplingRate);

		return S_OK;
	}

	STDMETHOD(Render)(
		void*          buf,
		unsigned long  bufsamplelen,
		unsigned long* rendered,
		unsigned long* status)
	{
		if (buf == 0) {
			return E_INVALIDARG;
		}

		if (rendered != 0) {
			*rendered = 0;
		}

		bool end_of_stream = false;

		INT16* p = static_cast<INT16*>(buf);

		UINT32 rest = bufsamplelen;
		while (rest > 0) {
			try {
				UINT32 count = m_Decoder->Decode(
					p,
					rest);
				if (count == 0) {
					end_of_stream = true;
					break;
				}

				rest -= count;

				p += count * m_Channels;

			} catch (QAX::Exception&) {
				return E_FAIL;
			}
		}

		if (rendered != 0) {
			*rendered = bufsamplelen - rest;
		}

		if (status != 0) {
			*status = end_of_stream ? 0 : 1;
		}

		return S_OK;
	}

	STDMETHOD(SetPosition)(unsigned __int64 samplepos)
	{
		try {
			m_Decoder->Seek(samplepos);

		} catch (QAX::Exception&) {
			return E_FAIL;
		}

		return S_OK;
	}

	/* */

}; // QDecoder

/* */

