// QWaveOutputer.h
// 2009/08/21

#pragma once

// QWaveOutputer
class QWaveOutputer {

	INT32 m_Channels;
	INT32 m_SamplingRate;

	ATL::CComPtr<IMediaSample> m_sample;

	BYTE* m_pbSample;
	DWORD m_cbSample;

	DWORD m_cbFilled;

	bool m_bDiscontinue;

public:

	QWaveOutputer() :
		m_Channels(2), m_SamplingRate(44100),
		m_pbSample(0), m_cbSample(0), m_cbFilled(0),
		m_bDiscontinue(true)
	{
	}

	~QWaveOutputer()
	{
	}

	void Setup(INT32 c, INT32 s)
	{
		m_Channels     = c;
		m_SamplingRate = s;
	}

	INT32 GetTime(INT32 s)
	{
		return s / m_SamplingRate;
	}

	INT32 GetSamples(INT32 c)
	{
		return c / m_Channels;
	}

	bool Output(
		IMemAllocator* alloc,
		IMemInputPin*  pin,
		REFERENCE_TIME rt,
		const void*    pv,
		INT32          count)
	{
		if (alloc == 0 || pin == 0) {
			return true;
		}

		REFERENCE_TIME ts = rt;

		const BYTE* pb = static_cast<const BYTE*>(pv);

		INT32 s = 0;

		INT32 samples = count / m_Channels;

		for (; ; ) {
			if (m_sample != 0) {
				INT32 cb = m_cbSample - m_cbFilled;
				INT32 ss = cb / (2 * m_Channels);

				INT32 s0 = samples - s;
				if (s0 > ss) {
					s0 = ss;
				}

				INT32 b0 = s0 * (2 * m_Channels);

				memcpy(
					m_pbSample + m_cbFilled,
					pb,
					b0);

				m_cbFilled += b0;

				pb += b0;
				s  += s0;

				INT64 t = INT64(s0) * 10*1000*1000;
				ts += t / m_SamplingRate;
			}

			INT32 s0 = samples - s;
			if (s0 > 0 || m_cbFilled >= m_cbSample) {
				if (!Deliver(pin)) {
					return false;
				}
			}

			if (s0 == 0) {
				break;
			}

			if (!Allocate(
				alloc,
				ts)) {
				return false;
			}
		}

		return true;
	}

	bool EndOfStream(IMemInputPin* pin)
	{
		if (pin == 0) {
			return true;
		}

		if (!Deliver(pin)) {
			return false;
		}

		return true;
	}

	bool Flush()
	{
		if (m_sample != 0) {
			m_sample.Release();

			m_pbSample = 0;
			m_cbSample = 0;
			m_cbFilled = 0;
		}

		m_bDiscontinue = true;

		return true;
	}

private:

	bool Deliver(IMemInputPin* pin)
	{
		if (m_sample != 0) {
			m_sample->SetActualDataLength(m_cbFilled);

			HRESULT hRslt = pin->Receive(m_sample);
			if (FAILED(hRslt)) {
				return false;
			}

			m_sample.Release();

			m_pbSample = 0;
			m_cbSample = 0;
			m_cbFilled = 0;
		}

		return true;
	}

	bool Allocate(
		IMemAllocator* alloc,
		REFERENCE_TIME rt)
	{
		HRESULT hRslt = alloc->GetBuffer(
			&m_sample,
			0,
			0,
			0);
		if (FAILED(hRslt)) {
			return false;
		}

		BYTE* pb = 0;
		m_sample->GetPointer(&pb);

		INT32 cb = m_sample->GetSize();

		m_pbSample = pb;
		m_cbSample = cb;
		m_cbFilled = 0;

		if (m_bDiscontinue) {
			m_bDiscontinue = false;

			m_sample->SetTime(&rt, 0);

			m_sample->SetDiscontinuity(TRUE);
		}

		return true;
	}

}; // QWaveOutputer

