package libsvm.wrapper;

import java.util.ArrayList;
import java.util.List;

import libsvm.svm_node;

/**
 * 特徴ベクトル
 *
 * @author hirainaoki
 */
public class SvmFeatureVector {

	/**
	 * クラスID
	 */
	private int classId;

	/**
	 * すべての特徴
	 */
	private List<SvmFeature> features;

	/**
	 * コンストラクタ
	 */
	public SvmFeatureVector() {
		this.features = new ArrayList<SvmFeature>();
		this.classId = -1;
	}

	/**
	 * コンストラクタ
	 *
	 * @param classId クラスID
	 */
	public SvmFeatureVector(int classId) {
		this();
		this.classId = classId;
	}

	/**
	 * <pre>
	 * 特徴を追加します．
	 * 特徴IDが昇順になるように追加します．
	 * </pre>
	 *
	 * @param feature 特徴
	 */
	public void addFeature(SvmFeature feature) {

		if(!features.contains(feature)) {

			int featureId = feature.getFeatureId();

			if(features.size() == 0) {
				features.add(feature);

			}else if(features.size() == 1) {

				int nextFeatureId = features.get(0).getFeatureId();
				if(featureId >= nextFeatureId) {
					features.add(feature);

				}else{
					features.add(0, feature);
				}

			}else{

				for(int i = 1; i < features.size(); i++) {

					int priorFeatureId = features.get(i - 1).getFeatureId();
					int nextFeatureId = features.get(i).getFeatureId();

					if(priorFeatureId <= featureId && featureId <= nextFeatureId) {
						features.add(i, feature);
						break;

					// 最後尾に追加
					}else if(i == features.size() - 1) {
						features.add(feature);
						break;
					}
				}
			}
		}
	}

	/**
	 * 指定した特徴を除去する．
	 *
	 * @param featureId 特徴ID
	 */
	public void remove(int featureId) {

		for(int i = 0; i < features.size(); i++) {

			SvmFeature feature = features.get(i);

			if(feature.getFeatureId() == featureId) {
				features.remove(i);
				break;
			}
		}
	}

	/**
	 * 利用可能な状態に変換します．
	 *
	 * @return
	 */
	public svm_node[] toUsable() {

		svm_node[] res = new svm_node[features.size()];

		for(int i = 0; i < features.size(); i++) {
			res[i] = features.get(i).toUsable();
		}

		return res;
	}

	/**
	 * <pre>
	 * クラスIDを取得します。
	 *
	 * クラスIDがセットされていない場合は-1を返します。
	 * </pre>
	 *
	 * @return クラスID
	 */
	public int getClassId() {
		return this.classId;
	}

	/**
	 * @param classId クラスID
	 */
	public void setClassId(int classId) {
		this.classId = classId;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((features == null) ? 0 : features.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SvmFeatureVector other = (SvmFeatureVector) obj;
		if (features == null) {
			if (other.features != null)
				return false;
		} else if (!features.equals(other.features))
			return false;
		return true;
	}
}
