/*
 * Decompiled with CFR 0.152.
 */
package firesimulator.simulator;

import firesimulator.simulator.EnergyHistory;
import firesimulator.simulator.ExtinguishRequest;
import firesimulator.simulator.Monitor;
import firesimulator.simulator.WindShift;
import firesimulator.util.Configuration;
import firesimulator.util.Rnd;
import firesimulator.world.Building;
import firesimulator.world.FireBrigade;
import firesimulator.world.World;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Random;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.uncommons.maths.random.GaussianGenerator;

public class Simulator {
    private static final Log LOG = LogFactory.getLog(Simulator.class);
    private World world;
    private WindShift windShift;
    public static float GAMMA = 0.5f;
    public static float AIR_TO_AIR_COEFFICIENT = 0.5f;
    public static float AIR_TO_BUILDING_COEFFICIENT = 45.0f;
    public static float WATER_COEFFICIENT = 0.5f;
    public static float ENERGY_LOSS = 0.9f;
    public static float WIND_DIRECTION = 0.9f;
    public static float WIND_RANDOM = 0.0f;
    public static int WIND_SPEED = 0;
    public static float RADIATION_COEFFICENT = 1.0f;
    public static float TIME_STEP_LENGTH = 1.0f;
    public static float WEIGHT_GRID = 0.2f;
    public static float AIR_CELL_HEAT_CAPACITY = 1.0f;
    public Set monitors;
    public static boolean verbose;
    private static Simulator me;
    private EnergyHistory energyHistory;

    public Simulator(World world) {
        me = this;
        this.monitors = new HashSet();
        verbose = true;
        this.world = world;
        this.windShift = null;
    }

    public static Simulator getSimulator() {
        return me;
    }

    public void addMonitor(Monitor monitor) {
        this.monitors.add(monitor);
    }

    public void removeMonitor(Monitor monitor) {
        this.monitors.remove(monitor);
    }

    private void informStep() {
        Iterator i = this.monitors.iterator();
        while (i.hasNext()) {
            ((Monitor)i.next()).step(this.world);
        }
    }

    private void informDone() {
        Iterator i = this.monitors.iterator();
        while (i.hasNext()) {
            ((Monitor)i.next()).done(this.world);
        }
    }

    private void informReset() {
        Iterator i = this.monitors.iterator();
        while (i.hasNext()) {
            ((Monitor)i.next()).reset(this.world);
        }
    }

    public void step(int timestep) {
        this.energyHistory = new EnergyHistory(this.world, timestep);
        this.refill();
        this.executeExtinguishRequests();
        this.burn();
        this.cool();
        this.updateGrid();
        this.exchangeBuilding();
        this.cool();
        this.energyHistory.registerFinalEnergy(this.world);
        this.energyHistory.logSummary();
    }

    private void cool() {
        for (Building b : this.world.getBuildings()) {
            this.waterCooling(b);
        }
    }

    private void refill() {
        for (FireBrigade fb : this.world.getFirebrigades()) {
            if (!fb.refill()) continue;
            LOG.debug((Object)("refilling fire brigade " + fb.getID()));
        }
    }

    private void executeExtinguishRequests() {
        Iterator i = this.world.getExtinguishIterator();
        while (i.hasNext()) {
            ExtinguishRequest er = (ExtinguishRequest)i.next();
            er.execute();
        }
        this.world.clearExtinguishRequests();
    }

    private void burn() {
        for (Building b : this.world.getBuildings()) {
            if (b.getTemperature() >= (double)b.getIgnitionPoint() && b.fuel > 0.0f && b.isInflameable()) {
                float consumed = b.getConsum();
                if (consumed > b.fuel) {
                    consumed = b.fuel;
                }
                double oldFuel = b.fuel;
                double oldEnergy = b.getEnergy();
                double oldTemp = b.getTemperature();
                b.setEnergy(b.getEnergy() + (double)consumed);
                this.energyHistory.registerBurn(b, consumed);
                b.fuel -= consumed;
                b.setPrevBurned(consumed);
                continue;
            }
            b.setPrevBurned(0.0f);
        }
    }

    private void waterCooling(Building b) {
        double lWATER_COEFFICIENT = b.getFieryness() > 0 && b.getFieryness() < 4 ? WATER_COEFFICIENT : WATER_COEFFICIENT * GAMMA;
        boolean cond = false;
        if (b.getWaterQuantity() > 0) {
            double oldEnergy = b.getEnergy();
            double oldTemp = b.getTemperature();
            double oldWater = b.getWaterQuantity();
            double dE = b.getTemperature() * (double)b.getCapacity();
            if (dE <= 0.0) {
                return;
            }
            double effect = (double)b.getWaterQuantity() * lWATER_COEFFICIENT;
            int consumed = b.getWaterQuantity();
            if (effect > dE) {
                cond = true;
                double pc = 1.0 - (effect - dE) / effect;
                effect *= pc;
                consumed = (int)((double)consumed * pc);
            }
            b.setWaterQuantity(b.getWaterQuantity() - consumed);
            b.setEnergy(b.getEnergy() - effect);
            this.energyHistory.registerCool(b, effect);
            LOG.debug((Object)("Building " + b.getID() + " water cooling"));
            LOG.debug((Object)("Old energy: " + oldEnergy + ", old temperature: " + oldTemp + ", old water: " + oldWater));
            LOG.debug((Object)("Consumed " + consumed + " water: effect = " + effect));
            LOG.debug((Object)("New energy: " + b.getEnergy() + ", new temperature: " + b.getTemperature() + ", new water: " + b.getWaterQuantity()));
        }
    }

    private void exchangeBuilding() {
        double radEn;
        for (Building b : this.world.getBuildings()) {
            this.exchangeWithAir(b);
        }
        double sumdt = 0.0;
        HashMap<Building, Double> radiation = new HashMap<Building, Double>();
        for (Building b : this.world.getBuildings()) {
            radEn = b.getRadiationEnergy();
            radiation.put(b, radEn);
        }
        for (Building b : this.world.getBuildings()) {
            radEn = (Double)radiation.get(b);
            Building[] bs = b.connectedBuilding;
            float[] vs = b.connectedValues;
            for (int c = 0; c < vs.length; ++c) {
                double oldEnergy = bs[c].getEnergy();
                double connectionValue = vs[c];
                double a = radEn * connectionValue;
                double sum = oldEnergy + a;
                bs[c].setEnergy(sum);
                this.energyHistory.registerRadiationGain(bs[c], a);
            }
            b.setEnergy(b.getEnergy() - radEn);
            this.energyHistory.registerRadiationLoss(b, -radEn);
        }
    }

    private void exchangeWithAir(Building b) {
        double oldTemperature = b.getTemperature();
        double oldEnergy = b.getEnergy();
        double energyDelta = 0.0;
        for (int[] nextCell : b.cells) {
            int cellX = nextCell[0];
            int cellY = nextCell[1];
            double cellCover = (double)nextCell[2] / 100.0;
            double cellTemp = this.world.getAirCellTemp(cellX, cellY);
            double dT = cellTemp - b.getTemperature();
            double energyTransferToBuilding = dT * (double)AIR_TO_BUILDING_COEFFICIENT * (double)TIME_STEP_LENGTH * cellCover * (double)this.world.SAMPLE_SIZE;
            energyDelta += energyTransferToBuilding;
            double newCellTemp = cellTemp - (double)(AIR_CELL_HEAT_CAPACITY * TIME_STEP_LENGTH * (float)this.world.SAMPLE_SIZE) * energyTransferToBuilding;
            this.world.setAirCellTemp(cellX, cellY, newCellTemp);
        }
        b.setEnergy(oldEnergy + energyDelta);
        this.energyHistory.registerAir(b, energyDelta);
    }

    private void updateGrid() {
        LOG.debug((Object)"Updating air grid");
        double[][] airtemp = this.world.getAirTemp();
        double[][] newairtemp = new double[airtemp.length][airtemp[0].length];
        for (int x = 0; x < airtemp.length; ++x) {
            for (int y = 0; y < airtemp[0].length; ++y) {
                double dt = this.averageTemp(x, y) - airtemp[x][y];
                double change = dt * (double)AIR_TO_AIR_COEFFICIENT * (double)TIME_STEP_LENGTH;
                newairtemp[x][y] = this.relTemp(airtemp[x][y] + change);
                if (!(newairtemp[x][y] > -1.7976931348623157E308) || !(newairtemp[x][y] < Double.MAX_VALUE)) {
                    LOG.warn((Object)("Value is not sensible: " + newairtemp[x][y]));
                    newairtemp[x][y] = 1.3482698511467367E308;
                }
                if (newairtemp[x][y] != Double.NEGATIVE_INFINITY && newairtemp[x][y] != Double.POSITIVE_INFINITY) continue;
                LOG.warn((Object)"aha");
            }
        }
        this.world.setAirTemp(newairtemp);
        this.world.setAirTemp(this.getWindShift().shift(this.world.getAirTemp(), this));
    }

    private double relTemp(double deltaT) {
        return Math.max(0.0, deltaT * (double)ENERGY_LOSS * (double)TIME_STEP_LENGTH);
    }

    private double averageTemp(int x, int y) {
        double rv = this.neighbourCellAverage(x, y) / (double)this.weightSummCells(x, y);
        return rv;
    }

    private double neighbourCellAverage(int x, int y) {
        double total = this.getTempAt(x + 1, y - 1);
        total += this.getTempAt(x + 1, y);
        total += this.getTempAt(x + 1, y + 1);
        total += this.getTempAt(x, y - 1);
        total += this.getTempAt(x, y + 1);
        total += this.getTempAt(x - 1, y - 1);
        total += this.getTempAt(x - 1, y);
        return (total += this.getTempAt(x - 1, y + 1)) * (double)WEIGHT_GRID;
    }

    private float weightSummCells(int x, int y) {
        return 8.0f * WEIGHT_GRID;
    }

    protected double getTempAt(int x, int y) {
        if (x < 0 || y < 0 || x >= this.world.getAirTemp().length || y >= this.world.getAirTemp()[0].length) {
            return 0.0;
        }
        return this.world.getAirTemp()[x][y];
    }

    public void setWind(float direction, float speed) {
        this.windShift = new WindShift(direction, speed, this.world.SAMPLE_SIZE);
    }

    public void setWindSpeed(float speed) {
        this.windShift = this.getWindShift();
        this.setWind(this.windShift.getDirection(), speed);
    }

    public void setWindDirection(float direction) {
        this.windShift = this.getWindShift();
        this.setWind(direction, this.windShift.speed);
    }

    public WindShift getWindShift() {
        if (WIND_RANDOM > 0.0f && this.windShift != null) {
            float nd = (float)((double)this.windShift.direction + (double)(this.windShift.direction * WIND_RANDOM) * Rnd.get01());
            float ns = (float)((double)this.windShift.speed + (double)(this.windShift.speed * WIND_RANDOM) * Rnd.get01());
            this.setWind(nd, ns);
        }
        if (this.windShift == null || this.windShift.direction != WIND_DIRECTION || this.windShift.speed != (float)WIND_SPEED) {
            this.setWind(WIND_DIRECTION, WIND_SPEED);
        }
        return this.windShift;
    }

    private void loadVars() {
        AIR_TO_BUILDING_COEFFICIENT = new Float(Configuration.getValue("resq-fire.air_to_building_flow")).floatValue();
        AIR_TO_AIR_COEFFICIENT = new Float(Configuration.getValue("resq-fire.air_to_air_flow")).floatValue();
        ENERGY_LOSS = new Float(Configuration.getValue("resq-fire.energy_loss")).floatValue();
        WATER_COEFFICIENT = new Float(Configuration.getValue("resq-fire.water_thermal_capacity")).floatValue();
        WIND_SPEED = new Integer(Configuration.getValue("resq-fire.wind_speed"));
        WIND_DIRECTION = new Float(Configuration.getValue("resq-fire.wind_direction")).floatValue();
        WIND_RANDOM = new Float(Configuration.getValue("resq-fire.wind_random")).floatValue();
        RADIATION_COEFFICENT = new Float(Configuration.getValue("resq-fire.radiation_coefficient")).floatValue();
        AIR_CELL_HEAT_CAPACITY = new Float(Configuration.getValue("resq-fire.air_cell_heat_capacity")).floatValue();
        ExtinguishRequest.MAX_WATER_PER_CYCLE = new Integer(Configuration.getValue("resq-fire.max_extinguish_power_sum"));
        ExtinguishRequest.MAX_DISTANCE = new Integer(Configuration.getValue("resq-fire.water_distance"));
        GAMMA = new Float(Configuration.getValue("resq-fire.gamma")).floatValue();
        Rnd.setSeed(new Long(Configuration.getValue("resq-fire.randomseed")));
        Random random = new Random(new Long(Configuration.getValue("resq-fire.randomseed")));
        Building.burnRate = new GaussianGenerator(new Double(Configuration.getValue("resq-fire.burn-rate-average")).doubleValue(), new Double(Configuration.getValue("resq-fire.burn-rate-variance")).doubleValue(), random);
    }

    public void initialize() {
        try {
            this.loadVars();
        }
        catch (Exception e) {
            LOG.fatal((Object)"invalid configuration, aborting", (Throwable)e);
            System.exit(-1);
        }
        this.world.initialize();
    }

    public void reset() {
        this.loadVars();
        this.world.reset();
        this.informReset();
    }
}

