// -*-c++-*-

/*!
  \file basic_actions.cpp
  \brief basic player actions Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "basic_actions.h"

#include "bhv_scan_field.h"
#include "neck_scan_field.h"

#include <rcsc/player/logger.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/param/server_param.h>

namespace rcsc {


/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_TurnToAngle::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s%d: Body_TurnToAngle"
                  ,__FILE__, __LINE__ );

    const SelfObject & self = agent->world().self();

    if ( ! self.faceValid() )
    {
        agent->doTurn( 0.0 );
        return false;
    }

    agent->doTurn( M_angle - self.body() );
    return true;
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_TurnToPoint::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s%d: Body_TurnToPoint"
                  ,__FILE__, __LINE__ );
    const SelfObject & self = agent->world().self();

    if ( ! self.posValid() )
    {
        return agent->doTurn( 60.0 );
    }

    // relative angle from my predicted pos & body angle to point
    Vector2D my_point
        = self.playerType().inertiaPoint( self.pos(),
                                          self.vel(),
                                          M_cycle );
    AngleDeg target_rel_angle
        = ( M_point - my_point ).th() - self.body();


    // not consider about max effective turn (inertia moment)
    // try to face to point greedy

    agent->doTurn( target_rel_angle );

    if ( target_rel_angle.abs() < 1.0 )
    {
        return false;
    }
    return true;
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_TurnToBall::execute( PlayerAgent * agent )
{
    if ( ! agent->world().ball().posValid() )
    {
        return false;
    }

    Vector2D ball_point = agent->world().ball().inertiaPoint( M_cycle );

    return Body_TurnToPoint( ball_point, M_cycle ).execute( agent );
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Neck_TurnToRelative::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s:%d: Neck_TurnToRelative"
                  ,__FILE__, __LINE__ );

    return agent->doTurnNeck( M_angle_rel_to_body
                              - agent->world().self().neck() );
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Neck_TurnToPoint::execute( PlayerAgent * agent )
{
    AngleDeg target_rel_angle
        = agent->effector().queuedNextAngleFromBody( M_point );

    //if ( ( agent->world().self().neck() - target_rel_angle ).abs() < 1.0 )
    //{
    //    return false;
    //}

    target_rel_angle
        = ServerParam::i().normalizeNeckAngle( target_rel_angle.degree() );

    return agent->doTurnNeck( target_rel_angle
                              - agent->world().self().neck() );
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Neck_TurnToBall::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s:%d: Neck_TurnToBall"
                  ,__FILE__, __LINE__ );
    if ( ! agent->world().ball().posValid() )
    {
        Neck_ScanField().execute( agent );
        return true;
    }

    const Vector2D ball_next = agent->effector().queuedNextBallPos();
    const AngleDeg ball_rel_angle_next
        = agent->effector().queuedNextAngleFromBody( ball_next );
#if 0
    const ViewWidth next_vwidth = agent->effector().queuedNextViewWidth();
    const double next_vwidth_value = next_vwidth.getWidth();

    dlog.addText( Logger::ACTION,
                  "%s:%d: ball_next=(%.2f, %.2f), ball_rel_angle=%.1f, next_view=%.1f"
                  ,__FILE__, __LINE__,
                  ball_next.x, ball_next.y,
                  ball_rel_angle_next.degree(),
                  next_vwidth_value );
    // check if it is necessary to face neck to ball
    if ( agent->world().ball().distFromSelf() > 30.0
         || next_vwidth.type() == ViewWidth::WIDE )
    {
        double dir_diff = ( ball_rel_angle_next
                            - agent->world().self().neck() ).abs();
        // need to turn neck to ball
        if ( dir_diff > next_vwidth_value * 0.5 - 5.0 )
        {
            agent->doTurnNeck( ball_rel_angle_next
                               - agent->world().self().neck() );
        }
    }
    else
#endif
    {
        agent->doTurnNeck( ball_rel_angle_next
                           - agent->world().self().neck() );
    }

    return true;
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_BodyNeckToPoint::execute( PlayerAgent * agent )
{
    if ( ! agent->world().self().posValid() )
    {
        return Bhv_ScanField().execute( agent );
    }

    // relative angle from my predicted pos & body angle to point
    Vector2D mynext
        = agent->world().self().pos()
        + agent->world().self().vel();

    AngleDeg target_rel_angle
        = (M_point - mynext).th()
        - agent->world().self().body();

    // body & neck is already face to point.
    //if ( (agent->world().self().neck() - target_rel_angle).abs() < 1.0
    //     && target_rel_angle.abs() < 1.0 )
    //{
    //    return false;
    //}

    // calc the max turn angle with current my speed.
    double max_turn
        = effective_turn( ServerParam::i().maxMoment(),
                          agent->world().self().vel().r(),
                          agent->world().self().playerType().inertiaMoment());

    // can face only turn
    if ( target_rel_angle.abs() < max_turn )
    {
        Body_TurnToPoint( M_point ).execute( agent );
        agent->setNeckAction( new Neck_TurnToRelative( 0.0 ) );
        return true;
    }
    // cannot face only turn
    // turn_neck is required.
    agent->doTurn( target_rel_angle );

    if ( target_rel_angle.degree() > 0.0 )
    {
        target_rel_angle -= max_turn;
    }
    else
    {
        target_rel_angle += max_turn;
    }

    if ( target_rel_angle.degree() < ServerParam::i().minNeckAngle() )
    {
        target_rel_angle = ServerParam::i().minNeckAngle();
    }
    else if ( target_rel_angle.degree() > ServerParam::i().maxNeckAngle() )
    {
        target_rel_angle = ServerParam::i().maxNeckAngle();
    }

    agent->setNeckAction( new Neck_TurnToRelative( target_rel_angle ) );
    return true;
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_BodyNeckToBall::execute( PlayerAgent * agent )
{
    if ( agent->world().ball().posValid() )
    {
        Vector2D ballnext
            = agent->world().ball().pos()
            + agent->world().ball().vel();
        return Bhv_BodyNeckToPoint( ballnext ).execute( agent );
    }
    else
    {
        return Bhv_ScanField().execute( agent );
    }
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_NeckBodyToPoint::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s:%d: Bhv_NeckBodyToPoint. to(%f, %f)"
                  ,__FILE__, __LINE__,
                  M_point.x, M_point.y );

    // relative angle from my predicted pos & body angle to point
    Vector2D mynext = agent->world().self().pos();
    if ( agent->world().self().velValid() )
    {
        mynext += agent->world().self().vel();
    }

    AngleDeg target_rel_angle
        = ( M_point - mynext ).th()
        - agent->world().self().body();

    //if ( ( agent->world().self().neck() - target_rel_angle ).abs() < 1.0 )
    //{
    //    dlog.addText( Logger::ACTION,
    //                  "%s:%d: alerady facing"
    //                  ,__FILE__, __LINE__ );
    //    return false;
    //}

    // I can face to point only turn_neck.
    if ( ServerParam::i().minNeckAngle() + 5.0 < target_rel_angle.degree()
         && target_rel_angle.degree() < ServerParam::i().maxNeckAngle() - 5.0 )
    {
        agent->doTurn( 0.0 );
        agent->setNeckAction( new Neck_TurnToRelative( target_rel_angle ) );
        return true;
    }
    // calc the max turn angle with current my speed
    double max_turn
        = effective_turn( ServerParam::i().maxMoment(),
                          agent->world().self().vel().r(),
                          agent->world().self().playerType().inertiaMoment() );

    // can face only turn
    if ( target_rel_angle.abs() < max_turn )
    {
        dlog.addText( Logger::ACTION,
                      "%s:%d: can face only turn"
                      ,__FILE__, __LINE__ );
        agent->doTurn( target_rel_angle );
        agent->setNeckAction( new Neck_TurnToRelative( 0.0 ) );
        return true;
    }
    // cannot face only turn
    // turn_neck is required.
    agent->doTurn( target_rel_angle );

    if ( target_rel_angle.degree() > 0.0 )
    {
        target_rel_angle -= max_turn;
    }
    else
    {
        target_rel_angle += max_turn;
    }

    dlog.addText( Logger::ACTION,
                  "%s:%d: turn & turn_neck"
                  ,__FILE__, __LINE__ );
    // moment is justified automatically.
    agent->setNeckAction( new Neck_TurnToRelative( target_rel_angle ) );
    return true;
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_NeckBodyToBall::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s:%d: Bhv_NeckBodyToBall"
                  ,__FILE__, __LINE__ );
    if ( agent->world().ball().posValid() )
    {
        Vector2D ball_next
            = agent->world().ball().pos()
            + agent->world().ball().vel();

        return Bhv_NeckBodyToPoint( ball_next ).execute( agent );
    }
    else
    {
        return Bhv_ScanField().execute( agent );
    }
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
View_Wide::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s:%d: View_Wide"
                  ,__FILE__, __LINE__ );

    return agent->doChangeView( ViewWidth::WIDE );
}

/////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
View_Normal::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ACTION,
                  "%s:%d: View_Normal"
                  ,__FILE__, __LINE__ );

    return agent->doChangeView( ViewWidth::NORMAL );
}

/////////////////////////////////////////////////////////////////////

}
