// -*-c++-*-

/*!
  \file shoot_table.cpp
  \brief shoot plan search and holder class
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "shoot_table.h"

#include "body_kick_one_step.h"

#include <rcsc/player/player_agent.h>
#include <rcsc/player/logger.h>
#include <rcsc/player/interception.h>
#include <rcsc/param/server_param.h>
#include <rcsc/math_util.h>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
void
ShootTable::search( const PlayerAgent * agent )
{
    /////////////////////////////////////////////////////////////////////
    static const Rect2D
        shootable_area( ServerParam::i().theirPenaltyAreaLine() - 5.0, // left
                        -ServerParam::i().penaltyAreaHalfWidth(), // top
                        ServerParam::i().penaltyAreaLength() + 5.0, // length
                        ServerParam::i().penaltyAreaWidth() ); // width
    static const Vector2D goal_l( ServerParam::i().pitchHalfLength(),
                                  -ServerParam::i().goalHalfWidth() + 1.0 );
    static const Vector2D goal_r( ServerParam::i().pitchHalfLength(),
                                  ServerParam::i().goalHalfWidth() - 1.0 );
    static const Line2D goal_line( goal_l, goal_r );

    /////////////////////////////////////////////////////////////////////
    const WorldModel & wm = agent->world();

    if ( M_time == wm.time() )
    {
        return;
    }

    M_time = wm.time();
    M_pathes.clear();

    if ( ! wm.self().isKickable()
         || ! shootable_area.contains( wm.ball().pos() ) )
    {
        return;
    }

    // TODO: change the angle devision policy.
    // search step should be defeined by the reach points.
    // so, we should devide the segment on goal line.

    const AngleDeg angle_l = ( goal_l - wm.ball().pos() ).th();
    const AngleDeg angle_r = ( goal_r - wm.ball().pos() ).th();
    const double angle_range = ( angle_r - angle_l ).abs();

    AngleDeg search_angle;
    int max_loop = 0;
    double angle_inc = 0.0;

    if ( angle_range < 2.0 )
    {
        max_loop = 1;
        angle_inc = 0.0;
        search_angle = AngleDeg::bisect( angle_l, angle_r );
    }
    else if ( angle_range < 16.0 )
    {
        angle_inc = 2.0;
        max_loop = static_cast< int >( angle_range / angle_inc );
        search_angle
            = angle_l
            + ( angle_range - std::floor( angle_range ) ) * 0.5;
    }
    else
    {
        angle_inc = angle_range / 8.0;
        max_loop = 8;
        search_angle = angle_l;
    }

    int goalie_conf = 1000;
    const PlayerObject * opp_goalie = agent->world().getOpponentGoalie();
    if ( opp_goalie )
    {
        goalie_conf = opp_goalie->posCount();
    }


    // search loop for each angle
    for ( int i = 0; i < max_loop; ++i )
    {
        if ( goalie_conf > 5
             && agent->world().getDirCount( search_angle ) > 3 )
        {
            search_angle += angle_inc;
            continue;
        }

        Line2D ball_line( wm.ball().pos(), search_angle );
        Vector2D reach_point = ball_line.intersection( goal_line );
        if ( ! reach_point.valid() )
        {
            std::cerr << agent->world().time()
                      << "Why shoot cource does not intersect with goal line??"
                      << std::endl;
            search_angle += angle_inc;
            continue;
        }
#ifdef DEBUG
        dlog.addText( Logger::SHOOT,
                      "__Shoot to(%.1f %.1f) angle=%.1f",
                      reach_point.x, reach_point.y, search_angle.degree() );
#endif
        double dist2goal = wm.ball().pos().dist( reach_point );

        Vector2D max_one_step_vel
            = Body_KickOneStep::get_max_possible_vel( search_angle,
                                                      wm.self().kickRate(),
                                                      wm.ball().vel() );

        double shoot_first_speed
            = ( dist2goal + 5.0 ) * ( 1.0 - ServerParam::i().ballDecay() );
        shoot_first_speed = std::max( max_one_step_vel.r(), shoot_first_speed );
        shoot_first_speed = std::max( 1.5, shoot_first_speed );

        bool over_max = false;
        while ( ! over_max )
        {
            if ( shoot_first_speed > ServerParam::i().ballSpeedMax() )
            {
                over_max = true;
                shoot_first_speed = ServerParam::i().ballSpeedMax();
            }

            int min_cycle = calcOpponentCycle( agent,
                                               search_angle,
                                               shoot_first_speed,
                                               ball_line,
                                               dist2goal );

            if ( min_cycle > 0 )
            {
#ifdef DEBUG
                dlog.addText( Logger::SHOOT,
                              "____ Success! to(%.1f %.1f).speed=%.2f  opp_min_cycle= %d",
                              reach_point.x, reach_point.y,
                              shoot_first_speed, min_cycle );
#endif
                M_pathes.push_back( Path( reach_point,
                                          shoot_first_speed,
                                          search_angle,
                                          min_cycle ) );
            }

            shoot_first_speed += 0.5; // Magic Number: speed inc
        }

        search_angle += angle_inc;
    }

#ifdef DEBUG
    dlog.addText( Logger::SHOOT,
                  "__total shoot course is %d",
                  M_pathes.size() );

#endif

    // sort by first speed
    std::sort( M_pathes.begin(),
               M_pathes.end(),
               SpeedCmp() );

    // merge sort by opponent reach cycle
    //std::stable_sort( S_cached_shoot_route.begin(),
    //                  S_cached_shoot_route.end(),
    //                  ShootRouteCycleComp() );

    const PlayerObject * goalie = agent->world().getOpponentGoalie();
    if ( goalie )
    {
        AngleDeg angle = ( agent->world().ball().pos()
                           - goalie->pos() ).th();
        std::stable_sort( M_pathes.begin(),
                          M_pathes.end(),
                          AngleCmp( angle ) );
    }

}


/*-------------------------------------------------------------------*/
/*!

*/
int
ShootTable::calcOpponentCycle( const PlayerAgent * agent,
                               const AngleDeg & shoot_angle,
                               const double & first_speed,
                               const Line2D & shoot_line,
                               const double & dist_to_goal )
{
    static const double opp_x_thr = ServerParam::i().theirPenaltyAreaLine() - 5.0;
    static const double opp_y_thr = ServerParam::i().penaltyAreaHalfWidth();
    static const double goalie_max_speed
        = ServerParam::i().defaultRealSpeedMax() * 0.9;
    static const double player_max_speed
        = ServerParam::i().defaultPlayerSpeedMax() * 0.8;
    static const double player_control_area
        = ServerParam::i().defaultKickableArea() - 0.3;

    // estimate required ball travel step
    const double ball_reach_step
        = calc_length_geom_series( first_speed,
                                   dist_to_goal,
                                   ServerParam::i().ballDecay() )
        - 1.0;

    // cannot calculate the ball reach step
    if ( ball_reach_step < 0.0 )
    {
        return -1;
    }

    if ( ball_reach_step < 1.0 )
    {
        dlog.addText( Logger::SHOOT,
                      "____ Success! not need to check opponent."
                      " speed = %.2f dist_to_goal = %.2f ball reach step = %.1f",
                      first_speed, dist_to_goal, ball_reach_step );
        return 1000;
    }

#ifdef DEBUG
    dlog.addText( Logger::SHOOT,
                  "____ speed = %.2f  ball reach step = %.1f",
                  first_speed, ball_reach_step );
#endif
    // estimate opponent interception
    const Interception util( agent->world().ball().pos()
                             + Vector2D::polar2vector( first_speed, shoot_angle ),
                             first_speed * ServerParam::i().ballDecay(),
                             shoot_angle );

    int min_cycle = 1000;
    const PlayerPtrCont::const_iterator end = agent->world().opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator it = agent->world().opponentsFromSelf().begin();
          it != end;
          ++it )
    {
        // outside of penalty
        if ( (*it)->pos().x < opp_x_thr ) continue;
        if ( (*it)->pos().absY() > opp_y_thr ) continue;
        // behind of shoot course
        if ( ( shoot_angle - (*it)->angleFromSelf() ).abs() > 90.0 )
        {
            continue;
        }

        int cycle = 1000;
        int conf_decay = 0;
        if ( (*it)->goalie() )
        {
            // goalie already exist on shoot line
            /*
            if ( (*it)->distFromSelf() < dist_to_goal + 1.0
                 && ( shoot_angle - (*it)->angleFromSelf() ).abs() < 90.0
                 && ( shoot_line.dist( (*it)->pos() )
                      < ServerParam::i().catchAreaLength() )
                 )
            {
                return -1;
            }
            */
            conf_decay = std::min( 5, (*it)->posCount() );

            cycle = static_cast< int >
                ( std::ceil( util.getReachCycle( (*it)->pos(),
                                                 &((*it)->vel()),
                                                 NULL,
                                                 (*it)->posCount(),
                                                 ServerParam::i().catchAreaLength(),
                                                 goalie_max_speed ) ) );
#ifdef DEBUG
            dlog.addText( Logger::SHOOT,
                          "______  goalie_cycle = %d.",
                          cycle );
#endif
        }
        else
        {
            cycle = static_cast< int >
                ( std::ceil( util.getReachCycle( (*it)->pos(),
                                                 &((*it)->vel()),
                                                 NULL,
                                                 0,
                                                 player_control_area,
                                                 player_max_speed ) ) );
            cycle += 2;
        }

        if ( cycle < min_cycle )
        {
            min_cycle = cycle;
        }

        // opponent can reach the ball
        if ( cycle - conf_decay < ball_reach_step )
        {
#ifdef DEBUG
            dlog.addText( Logger::SHOOT,
                          "______ opponent can reach. cycle = %d",
                          cycle );
#endif
            return -1;
        }
    }

    return min_cycle;
}

}
