// -*-c++-*-

/*!
  \file shoot_table.h
  \brief shoot plan search and holder class
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_ACTION_SHOOT_TABLE_H
#define RCSC_ACTION_SHOOT_TABLE_H

#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/vector_2d.h>
#include <rcsc/game_time.h>

#include <functional>
#include <vector>

namespace rcsc {

class PlayerAgent;

/*!
  \class ShootTable
  \brief shoot plan search and holder table
 */
class ShootTable {
public:
    struct Path {
        Vector2D point_;
        double speed_;
        AngleDeg angle_;
        int opp_cycle_;
        Path( const Vector2D & point,
              const double & speed,
              const AngleDeg & angle,
              const int min_cycle )
            : point_( point )
            , speed_( speed )
            , angle_( angle )
            , opp_cycle_( min_cycle )
          { }
    };

    typedef std::vector< Path > PathCont;

    ////////////////////////////////////////////////////////////
    /*!
      \class CycleCmp
      \brief function object to compare the cycles that the ball reach the
      goal.
    */
    class CycleCmp
        : public std::binary_function< Path,
                                       Path,
                                       bool > {
    public:
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              // !!! CAUTION: operator is greater<>.
              return lhs.opp_cycle_ > rhs.opp_cycle_;
          }
    };

    ////////////////////////////////////////////////////////////
    /*!
      \class SpeedCmp
      \brief function object to compare the shoot first speed.
    */
    class SpeedCmp
        : public std::binary_function< Path,
                                       Path,
                                       bool > {
    public:
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.speed_ < rhs.speed_;
          }
    };

    ////////////////////////////////////////////////////////////
    class AngleCmp
        : public std::binary_function< Path,
                                       Path,
                                       bool > {
    private:
        const AngleDeg M_goalie_angle;
    public:
        AngleCmp( const AngleDeg & goalie_angle )
            : M_goalie_angle( goalie_angle )
          { }

        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return ( lhs.angle_ - M_goalie_angle ).abs()
                  <  ( rhs.angle_ - M_goalie_angle ).abs();
          }
    };

private:
    //! last calculated time
    GameTime M_time;
    //! cached calculated shoot pathes
    PathCont M_pathes;

public:
    /*!
      \brief accessible from global.
     */
    ShootTable()
      { }

    const
    PathCont & getPathes( const PlayerAgent * agent )
      {
          search( agent );
          return M_pathes;
      }

private:

    /*!
      \brief search shoot route patterns. goal mouth is divided to several
      segment and several ball speed for each segment are calculated.
      \param agent const pointer to the agent itself
     */
    void search( const PlayerAgent * agent );

    /*!
      \brief calculate ball reach step for all opponent.
      \param agent const  pointer to the agent itself
      \param shoot_angle shoot route angle from current ball position
      \param first_speed shoot first speed
      \param shoot_line shoot route trace line
      \param dist_to_goal distance from the ball position to goal line
      \return minimal reach cycle value. if opponent *can* get the ball,
      negative value is returned.
     */
    int calcOpponentCycle( const PlayerAgent * agent,
                           const AngleDeg & shoot_angle,
                           const double & first_speed,
                           const Line2D & shoot_line,
                           const double & dist_to_goal );

};

}

#endif
