// -*-c++-*-

/*!
	\file formation_bpn.cpp
	\brief BPN formation data classes Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "formation_bpn.h"

#include <rcsc/math_util.h>

#include <boost/random.hpp>
#include <sstream>
#include <algorithm>

namespace rcsc {

const double FormationBPN::Param::PITCH_LENGTH = 105.0 + 10.0;
const double FormationBPN::Param::PITCH_WIDTH = 68.0 + 10.0;


/*-------------------------------------------------------------------*/
/*!

*/
FormationBPN::Param::Param()
    : M_param( 0.3, 0.9 )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FormationBPN::Param::randomize()
{
    static boost::mt19937 gen( std::time( 0 ) );
    boost::uniform_real<> dst( -0.5, 0.5 );
    boost::variate_generator< boost::mt19937 &, boost::uniform_real<> >
        rng( gen, dst );

    M_param.randomize( rng );
}

/*-------------------------------------------------------------------*/
/*!

*/
Vector2D
FormationBPN::Param::getPosition( const Vector2D & ball_pos,
                                  const Formation::SideType type ) const
{
    double msign = 1.0;
    if ( type == Formation::MIRROR ) msign =  -1.0;
    if ( type == Formation::CENTER && ball_pos.y > 0.0 ) msign = -1.0;

    PosNet::input_array input;

    input[0] = std::max( 0.0,
                         std::min( ball_pos.x / PITCH_LENGTH + 0.5,
                                   1.0 ) );
    input[1] = std::max( 0.0,
                         std::min( (ball_pos.y * msign) / PITCH_WIDTH + 0.5,
                                   1.0 ) );

    PosNet::output_array output;

    M_param.propagate( input, output );
    //std::cerr << "getPosition. raw output = "
    //<< output[0] << " ,  " << output[1]
    //<< std::endl;
    return Vector2D( ( output[0] - 0.5 ) * PITCH_LENGTH,
                     ( output[1] - 0.5 ) * PITCH_WIDTH * msign );
}

/*-------------------------------------------------------------------*/
/*!
  Format:  Role <RoleNameStr>
*/
bool
FormationBPN::Param::readRoleName( std::istream & is )
{
    std::string line_buf;
    if ( ! std::getline( is, line_buf ) )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    std::istringstream istr( line_buf );
    if ( ! istr.good() )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    std::string tag;
    istr >> tag;
    if ( tag != "Role" )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    istr >> M_role_name;
    if ( M_role_name.empty() )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read role name" << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationBPN::Param::readParam( std::istream & is )
{
    std::string line_buf;
    if ( ! std::getline( is, line_buf ) )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    std::istringstream istr( line_buf );
    if ( ! istr.good() )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    if ( ! M_param.read( istr ) )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read position param" << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationBPN::Param::read( std::istream & is )
{
    // read role name
    if ( ! readRoleName( is ) )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Failed to read role name" << std::endl;
        return false;
    }

    if ( ! readParam( is ) )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Failed to read parameters" << std::endl;
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
FormationBPN::Param::printRoleName( std::ostream & os ) const
{
    if ( M_role_name.empty() )
    {
        os << "Role Default\n";
    }
    else
    {
        os << "Role " << M_role_name << '\n';
    }
    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
FormationBPN::Param::printParam( std::ostream & os ) const
{
    M_param.print( os ) << '\n';
    return os;
}

/*-------------------------------------------------------------------*/
/*!
  Role <role name>
  <bko.x> <bkoy>
  <offense playon>
  <defense playon>
  ...

*/
std::ostream &
FormationBPN::Param::print( std::ostream & os ) const
{
    printRoleName( os );
    printParam( os );

    return os << std::flush;
}

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
FormationBPN::FormationBPN()
    : Formation()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
Formation::Snapshot
FormationBPN::createDefaultParam()
{
#if 1
    // 1: goalie
    // 2: left center back
    // 3(2): right center back
    // 4: left side back
    // 5(4): right side back
    // 6: defensive half
    // 7: left offensive half
    // 8(7): left side half
    // 9(8): right side half
    // 10: left forward
    // 11(10): right forward
    createNewRole( 1, "Goalie", Formation::CENTER );
    createNewRole( 2, "CenterBack", Formation::SIDE );
    setMirrorType( 3, 2 );
    createNewRole( 4, "SideBack", Formation::SIDE );
    setMirrorType( 5, 4 );
    createNewRole( 6, "DefensiveHalf", Formation::CENTER );
    createNewRole( 7, "OffensiveHalf", Formation::SIDE );
    setMirrorType( 8, 7 );
    createNewRole( 9, "SideForward", Formation::SIDE );
    setMirrorType( 10, 9 );
    createNewRole( 11, "CenterForward", Formation::CENTER );

    Snapshot snap;

    snap.ball_.assign( 0.0, 0.0 );
    snap.players_.push_back( Vector2D( -50.0, 0.0 ) );
    snap.players_.push_back( Vector2D( -20.0, -8.0 ) );
    snap.players_.push_back( Vector2D( -20.0, 8.0 ) );
    snap.players_.push_back( Vector2D( -18.0, -18.0 ) );
    snap.players_.push_back( Vector2D( -18.0, 18.0 ) );
    snap.players_.push_back( Vector2D( -15.0, 0.0 ) );
    snap.players_.push_back( Vector2D( 0.0, -12.0 ) );
    snap.players_.push_back( Vector2D( 0.0, 12.0 ) );
    snap.players_.push_back( Vector2D( 10.0, -22.0 ) );
    snap.players_.push_back( Vector2D( 10.0, 22.0 ) );
    snap.players_.push_back( Vector2D( 10.0, 0.0 ) );

    return snap;
#else
    // 1: goalie
    // 2: center back
    // 3: left side back
    // 4(3): right side back
    // 5: left defensive half
    // 6(5): right defensive half
    // 7: offensive half
    // 8: left side half
    // 9(8): right side half
    // 10: left forward
    // 11(10): right forward
    createNewRole( 1, "Goalie", Formation::CENTER );
    createNewRole( 2, "CenterBack", Formation::CENTER );
    createNewRole( 3, "SideBack", Formation::SIDE );
    setMirrorType( 4, 3 );
    createNewRole( 5, "DefensiveHalf", Formation::SIDE );
    setMirrorType( 6, 5 );
    createNewRole( 7, "SideHalf", Formation::SIDE );
    setMirrorType( 8, 7 );
    createNewRole( 9, "OffensiveHalf", Formation::CENTER );
    createNewRole( 10, "Forward", Formation::SIDE );
    setMirrorType( 11, 10 );
#endif
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FormationBPN::setRoleName( const int unum,
                           const std::string & name )
{
    boost::shared_ptr< FormationBPN::Param > p = getParam( unum );

    if ( ! p )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " You cannot set the role name of player "
                  << unum
                  << std::endl;
        return;
    }

    p->setRoleName( name );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
FormationBPN::getRoleName( const int unum ) const
{
    const boost::shared_ptr< const FormationBPN::Param > p = param( unum );
    if ( ! p )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " You cannot get the role name of player "
                  << unum
                  << std::endl;
        return std::string( "" );;
    }

    return p->roleName();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FormationBPN::createNewRole( const int unum,
                             const std::string & role_name,
                             const Formation::SideType type )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return;
    }

    if ( type == Formation::CENTER )
    {
        setCenterType( unum );
    }
    else if ( type == Formation::SIDE )
    {
        setSideType( unum );
    }
    else
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** You cannot create a new parameter for mirror type."
                  << std::endl;
        return;
    }

    // erase old parameter, if exist
    std::map< int, boost::shared_ptr< FormationBPN::Param > >::iterator it
        = M_param_map.find( unum );
    if ( it != M_param_map.end() )
    {
        M_param_map.erase( it );
    }

    boost::shared_ptr< FormationBPN::Param > param( new FormationBPN::Param );
    param->setRoleName( role_name );
    param->randomize();

    M_param_map.insert( std::make_pair( unum, param ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
Vector2D
FormationBPN::getPosition( const int unum,
                           const Vector2D & ball_pos ) const
{
    const boost::shared_ptr< const FormationBPN::Param > ptr = param( unum );
    if ( ! ptr )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " *** ERROR *** FormationBPN::Param not found. unum = "
                  << unum
                  << std::endl;
        return Vector2D( 0.0, 0.0 );
    }
    Formation::SideType type = Formation::SIDE;
    if ( M_mirror_number[unum - 1] > 0 )  type = Formation::MIRROR;
    if ( M_mirror_number[unum - 1] == 0 ) type = Formation::CENTER;

    return ptr->getPosition( ball_pos, type );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FormationBPN::train( const std::list< Snapshot > & train_data )
{
    if ( train_data.empty() )
    {
        return;
    }

    const double PITCH_LENGTH = FormationBPN::Param::PITCH_LENGTH;
    const double PITCH_WIDTH = FormationBPN::Param::PITCH_WIDTH;

    std::cerr << "FormationBPN::train. Started!!" << std::endl;

    for ( int unum = 1; unum <= 11; ++unum )
    {
        int number = unum;
        Formation::SideType type = Formation::SIDE;
        if ( isMirrorType( unum ) )
        {
            std::cerr << "  Training. player " << unum << " >>> mirror type "
                      << std::endl;
            continue;
        }

        if ( isCenterType( unum ) )
        {
            type = Formation::CENTER;
            std::cerr << "  Training. player " << unum << " >>> center type "
                      << std::endl;
        }
        else
        {
            std::cerr << "  Training. player " << unum << " >>> side type "
                      << std::endl;
        }

        boost::shared_ptr< FormationBPN::Param > param = getParam( number );
        if ( ! param )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " *** ERROR ***  No formation parameter for player " << unum
                      << std::endl;
            break;
        }

        FormationBPN::Param::PosNet & net = param->getParam();

        FormationBPN::Param::PosNet::input_array input;
        FormationBPN::Param::PosNet::output_array teacher;

        const std::list< Formation::Snapshot >::const_iterator snap_end
            = train_data.end();
        int loop = 0;
        double ave_err = 0.0;
        double max_err = 0.0;
        bool success = false;
        while ( ++loop <= 5000 )
        {
            ave_err = 0.0;
            max_err = 0.0;
            double snap_count = 1.0;
            for ( std::list< Formation::Snapshot >::const_iterator snap
                      = train_data.begin();
                  snap != snap_end;
                  ++snap, snap_count += 1.0 )
            {
                double by = snap->ball_.y;
                double py = snap->players_[unum - 1].y;

                if ( type == Formation::CENTER
                     && by > 0.0 )
                {
                    if ( loop == 2 )
                    {
                        std::cerr << "      unum " << unum
                                  << "  training data Y is reversed"
                                  << std::endl;
                    }
                    by *= -1.0;
                    py *= -1.0;
                }

                input[0] = min_max( 0.0,
                                    snap->ball_.x / PITCH_LENGTH + 0.5,
                                    1.0 );
                input[1] = min_max( 0.0,
                                    by / PITCH_WIDTH + 0.5,
                                    1.0 );
                teacher[0] = min_max( 0.0,
                                      snap->players_[unum - 1].x / PITCH_LENGTH + 0.5,
                                      1.0 );
                teacher[1] = std::max( 0.0,
                                       std::min( py / PITCH_WIDTH + 0.5, 1.0 ) );

                double err = net.train( input, teacher );
                if ( max_err < err )
                {
                    max_err = err;
                }
                ave_err
                    = ave_err * ( ( snap_count - 1.0 ) / snap_count )
                    + err / snap_count;
            }
#if 0
            if ( loop % 500 == 0 )
            {
                std::cerr << "      Training. player " << unum
                          << "  counter " << loop << std::endl;
            }
#endif
            if ( max_err < 0.003 )
            {
                std::cerr << "  ----> converged. average err=" << ave_err
                          << "  last max err=" << max_err
                          << std::endl;
                success = true;
                //printMessageWithTime( "train. converged. loop=%d", loop );
                break;
            }
        }
        if ( ! success )
        {
            std::cerr << "  *** Failed to converge *** " << std::endl;
            //printMessageWithTime( "train. Failed to converge. player %d", unum );
        }
        std::cerr << "  ----> " << loop
                  << " loop. last average err=" << ave_err
                  << "  last max err=" << max_err
                  << std::endl;
    }
    std::cerr << "FormationBPN::train. Ended!!" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::shared_ptr< FormationBPN::Param >
FormationBPN::getParam( const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return boost::shared_ptr< FormationBPN::Param >
            ( static_cast< FormationBPN::Param * >( 0 ) );
    }

    if ( M_mirror_number[unum - 1] > 0 )
    {
        return boost::shared_ptr< FormationBPN::Param >
            ( static_cast< FormationBPN::Param * >( 0 ) );
    }

    std::map< int, boost::shared_ptr< FormationBPN::Param > >::const_iterator
        it = M_param_map.find( unum );

    if ( it == M_param_map.end() )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** Parameter not found! unum = "
                  << unum << std::endl;
        return boost::shared_ptr< FormationBPN::Param >
            ( static_cast< FormationBPN::Param * >( 0 ) );
    }

    return it->second;
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::shared_ptr< const FormationBPN::Param >
FormationBPN::param( const int unum ) const
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return boost::shared_ptr< const FormationBPN::Param >
            ( static_cast< FormationBPN::Param * >( 0 ) );
    }

    const int player_number = ( M_mirror_number[unum - 1] <= 0 // center or original
                                ? unum
                                : M_mirror_number[unum - 1] );

    std::map< int, boost::shared_ptr< FormationBPN::Param > >::const_iterator
        it = M_param_map.find( player_number );

    if ( it == M_param_map.end() )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** Parameter not found! unum = "
                  << unum << std::endl;
        return boost::shared_ptr< const FormationBPN::Param >
            ( static_cast< FormationBPN::Param * >( 0 ) );
    }

    return it->second;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationBPN::read( std::istream & is )
{
    int n_line = 0;

    n_line += readName( is );
    if ( n_line < 0 )
    {
        return false;
    }

    //---------------------------------------------------
    // read each player's parameter set
    if ( ! readPlayers( is ) )
    {
        return false;
    }

    //---------------------------------------------------
    // check mirror number circuration reference
    for ( int i = 0; i < 11; ++i )
    {
        int refered_unum = M_mirror_number[i];
        if ( refered_unum <= 0 ) continue;
        if ( M_mirror_number[refered_unum - 1] > 0 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** failed to read formation."
                      << " Bad mirroring. player "
                      << i + 1
                      << " mirro = " << refered_unum
                      << " is already mirroring player"
                      << std::endl;
            return false;
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationBPN::readPlayers( std::istream & is )
{
    int player_read = 0;
    std::string line_buf;

    //---------------------------------------------------
    // read each player's parameter set
    for ( int i = 0; i < 11; ++i )
    {
        if ( ! std::getline( is, line_buf ) )
        {
            break;
        }

        // check id
        int unum = 0;
        int mirror = 0;
        int n_read = 0;
        if ( std::sscanf( line_buf.c_str(),
                          " player %d %d %n ",
                          &unum, &mirror,
                          &n_read ) != 2
             || n_read == 0 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** failed to read formation at number "
                      << i + 1
                      << " [" << line_buf << "]"
                      << std::endl;
            return false;
        }
        if ( unum != i + 1 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** failed to read formation "
                      << " Invalid player number.  Expected " << i + 1
                      << "  but read number = " << unum
                      << std::endl;
            return false;
        }
        if ( mirror == unum )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** failed to read formation."
                      << " Invalid mirror number. at "
                      << i
                      << " mirroing player itself. unum = " << unum
                      << "  mirror = " << mirror
                      << std::endl;
            return false;
        }
        if ( 11 < mirror )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** failed to read formation."
                      << " Invalid mirror number. at "
                      << i
                      << "  Mirror number is out of range. unum = " << unum
                      << "  mirror = " << mirror
                      << std::endl;
            return false;
        }

        M_mirror_number[i] = mirror;

        // this player is mirror type
        if ( mirror > 0 )
        {
            ++player_read;
            continue;
        }

        // read parameters
        boost::shared_ptr< FormationBPN::Param > param( new FormationBPN::Param );
        if ( ! param->read( is ) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** failed to read formation. at number "
                      << i + 1
                      << " [" << line_buf << "]"
                      << std::endl;
            return false;
        }

        ++player_read;
        M_param_map.insert( std::make_pair( unum, param ) );
    }

    if ( player_read != 11 )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** Invalid formation format."
                  << " The number of read player is " << player_read
                  << std::endl;
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  condition <playmode> <>
  player <unum> 0 <Role> <Param1> <Param2> ...
  player <unum> <mirro> <Role>
  ...
  important <x> <y>
  important <x> <y>
  ...
*/
std::ostream &
FormationBPN::print( std::ostream & os ) const
{
    printName( os );

    for ( int i = 0; i < 11; ++i )
    {
        const int unum = i + 1;
        os << "player " << unum << " " << M_mirror_number[i] << " ";
        if ( M_mirror_number[i] > 0 )
        {
            os << '\n';
            continue;
        }
        os << '\n';

        std::map< int, boost::shared_ptr< FormationBPN::Param > >::const_iterator
            it = M_param_map.find( unum );
        if ( it == M_param_map.end() )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** Invalid player Id at number "
                      << i + 1
                      << ".  No formation param!"
                      << std::endl;
        }
        else
        {
            it->second->print( os );
        }
    }

    os << "End" << std::endl;
    return os;
}

}
