// -*-c++-*-

/*!
  \file player_object.h
  \brief player object class Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_PLAYER_OBJECT_H
#define RCSC_PLAYER_PLAYER_OBJECT_H

#include <rcsc/player/localization.h>
#include <rcsc/player/fullstate_sensor.h>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/angle_deg.h>
#include <rcsc/types.h>

#include <functional>

namespace rcsc {

/*!
  \class PlayerObject
  \brief observed player object class
*/
class PlayerObject {
private:
    //! validation count threshold value for M_pos and M_rpos
    static int S_pos_count_thr;
    //! validation count threshold value for M_vel
    static int S_vel_count_thr;
    //! validation count threshold value for M_body and M_face
    static int S_face_count_thr;

    int  M_unum; //!< uniform number
    bool M_goalie; //!< goalie flag

    double M_dist_from_self; //!< distance from self
    AngleDeg M_angle_from_self; //!< angle from self global
    double M_dist_from_ball; //!< distance from ball

    Vector2D M_pos; //!< global coordinate
    Vector2D M_rpos; //!< relative coodinate
    int M_pos_count; //!< main accuracy counter
    int M_rpos_count; //!< relative coodinate accuracy counter

    Vector2D M_vel; //!< global velocity
    int M_vel_count; //!< accuracy count

    AngleDeg M_body; //!< global body angle
    int M_body_count; //!< body angle accuracy
    AngleDeg M_face; //!< global neck angle
    int M_face_count; //!< face angle accuracy

    AngleDeg M_pointto_angle; //!< global pointing angle
    int M_pointto_count; //!< pointing angle accuracy

    int M_tackle_count; //!< tackle info accuracy
public:

    /*!
      \brief initialize member variables.
    */
    PlayerObject();

    /*!
      \brief initialize member variables using observed info
      \param p analyzed seen player info
    */
    explicit
    PlayerObject( const Localization::PlayerT & p );

    /*!
      \brief destructor. nothing to do
    */
    ~PlayerObject()
      { }

    /*!
      \brief set accuracy count threshold values.
      \param pos_thr threshold value for M_pos
      \param vel_thr threshold value for M_vel
      \param face_thr threshold value for M_body and M_face
    */
    static
    void set_count_thr( const int pos_thr,
                        const int vel_thr,
                        const int face_thr );

    // ------------------------------------------
    /*!
      \brief get player's uniform number
      \return uniform number. if unknown player, returned -1
    */
    int unum() const
      {
          return M_unum;
      }

    /*!
      \brief get goalie flag
      \return true if this player is goalie
    */
    bool goalie() const
      {
          return M_goalie;
      }

    /*!
      \brief get distance from self
      \return distance value from self
    */
    const
    double & distFromSelf() const
      {
          return M_dist_from_self;
      }

    /*!
      \brief get global angle from self position
      \return angle value from self position
    */
    const
    AngleDeg & angleFromSelf() const
      {
          return M_angle_from_self;
      }

    /*!
      \brief get distance from ball
      \return distance value from ball
    */
    const
    double & distFromBall() const
      {
          return M_dist_from_ball;
      }

    /*!
      \brief get global position
      \return const reference to the point object
    */
    const
    Vector2D & pos() const
      {
          return M_pos;
      }

    /*!
      \brief get position relative to self position
      \return const reference to the point object
    */
    const
    Vector2D & rpos() const
      {
          return M_rpos;
      }

    /*!
      \brief get global position accuracy
      \return count from last observation
    */
    int posCount() const
      {
          return M_pos_count;
      }

    /*!
      \brief get velocity
      \return const reference to the vector object
    */
    const
    Vector2D & vel() const
      {
          return M_vel;
      }

    /*!
      \brief get velocity accuracy
      \return count from last observation
    */
    int velCount() const
      {
          return M_vel_count;
      }

    /*!
      \brief get global body angle
      \return const reference to the angle object
    */
    const
    AngleDeg & body() const
      {
          return M_body; // global body angle
      }

    /*!
      \brief get global body angle accuracy
      \return count from last observation
    */
    int bodyCount() const
      {
          return M_body_count;
      }

    /*!
      \brief get global neck angle
      \return const reference to the angle object
    */
    const
    AngleDeg & face() const
      {
          return M_face; // global neck angle
      }

    /*!
      \brief get global neck angle accuracy
      \return count from last observation
    */
    int faceCount() const
      {
          return M_face_count;
      }

    /*!
      \brief get global pointing angle
      \return const reference to the angle object
    */
    const
    AngleDeg & pointtoAngle() const
      {
          return M_pointto_angle; // global pointing angle
      }

    /*!
      \brief get global pointing angle accuracy
      \return count from last observation
    */
    int pointtoCount() const
      {
          return M_pointto_count;
      }

    /*!
      \brief get tackling status accuracy
      \return count from last observation
    */
    int tackleCount() const
      {
          return M_tackle_count;
      }

    /*!
      \brief check if player is tackling
      \return true if tackle accuracy is less than tackle cycles in ServerParam
    */
    bool isTackling() const;


    /*!
      \brief velify global position accuracy
      \return true if position has enough accuracy
    */
    bool posValid() const
      {
          return M_pos_count < S_pos_count_thr;
      }

    /*!
      \brief verify relative position accuracy
      \return true if relative position has enough accuracy
    */
    bool rposValid() const
      {
          return M_rpos_count < S_pos_count_thr;
      }

    /*!
      \brief verify velocity accuracy
      \return true if accuracy  has enough accuracy
    */
    bool velValid() const
      {
          return M_vel_count < S_vel_count_thr;
      }

    /*!
      \brief verify angle accuracy
      \return true if angle has enoubh accuracy
    */
    bool faceValid() const
      {
          return M_face_count < S_face_count_thr;
      }

    /*!
      \brief check if player can kick the ball or not
      \return true if player can kick the ball
     */
    bool isKickable() const;

    // ------------------------------------------
    /*!
      \brief template method. check if player is in the region
      \param region template resion. REGION must have method contains()
      \return true if region contains player position
    */
    template < typename REGION >
    bool isWithin( const REGION & region ) const
      {
          return region.contains( this->pos() );
      }

    // ------------------------------------------
    /*!
      \brief update status only with intenal info
    */
    void update();

    /*!
      \brief update status using localized player info
      \param p localized player info
    */
    void updateBySee( const Localization::PlayerT & p );

    /*!
      \brief update status using fullstate info
      \param p fullstate player info
      \param self_pos global self position
      \param ball_pos global ball position
     */
    void updateByFullstate( const FullstateSensor::PlayerT & p,
                            const Vector2D & self_pos,
                            const Vector2D & ball_pos );

    /*!
      \brief update staus using heard info
      \param heard_unum heard uniform number
      \param goalie update goalie info, only if this value is true.
      \param heard_pos heard global position
    */
    void updateByHear( const int heard_unum,
                       const bool goalie,
                       const Vector2D & heard_pos );

    /*!
      \brief update status related to other objects
      \param self self position
      \param ball ball position

      This method is called just before decide action
    */
    void updateSelfBallRelated( const Vector2D & self,
                                const Vector2D & ball );

    /*!
      \brief reset accuracy info
    */
    void forget();

    ///////////////////////////////////////////////////////////////
    /*!
      \brief functor to update
    */
    class UpdateOp
        : public std::unary_function< PlayerObject, void > {
    public:
        result_type operator()( argument_type & player )
          {
              player.update();
          }
    };

    /*!
      \brief functor to check if player has enough accuracy
    */
    class IsInvalidOp
        : public std::unary_function< PlayerObject, bool > {
    public:
        result_type operator()( const argument_type & player ) const
          {
              return ( ! player.posValid() );
          }
    };

    ///////////////////////////////////////////////////////////////
    /*!
      \brief predicate functor to compare player's accuracy. reference version
    */
    class ConfCmp
        : public std::binary_function< PlayerObject, PlayerObject, bool > {
    public:
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.posCount() < rhs.posCount();
          }
    };

    /*!
      \brief predicate functor to compare player's accuracy. pointer version
    */
    class PtrConfCmp
        : public std::binary_function< PlayerObject *, PlayerObject *, bool > {
    public:
        result_type operator()( const first_argument_type lhs,
                                const second_argument_type rhs ) const
          {
              return lhs->posCount() < rhs->posCount();
          }
    };

    ///////////////////////////////////////////////////////////////
    /*!
      \brief predicate functor to compare player's distance from self
    */
    class PtrSelfDistCmp
        : public std::binary_function< PlayerObject *, PlayerObject *, bool > {
    public:
        result_type operator()( const first_argument_type lhs,
                                const second_argument_type rhs ) const
          {
              return lhs->distFromSelf() < rhs->distFromSelf();
          }
    };

    /*!
      \brief predicate functor to compare player's distance from ball
    */
    class PtrBallDistCmp
        : public std::binary_function< PlayerObject *, PlayerObject *, bool > {
    public:
        result_type operator()( const first_argument_type lhs,
                                const second_argument_type rhs ) const
          {
              return lhs->distFromBall() < rhs->distFromBall();
          }
    };

};

typedef std::list< PlayerObject > PlayerCont;
typedef std::vector< PlayerObject * > PlayerPtrCont;

}

#endif
