// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sample_player.h"

#include "soccer_role.h"

#include "bhv_goalie_free_kick.h"
#include "bhv_penalty_kick.h"
#include "bhv_pre_process.h"
#include "bhv_set_play.h"
#include "bhv_set_play_kick_in.h"

#include <rcsc/formation/formation.h>
#include <rcsc/player/logger.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/basic_client.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_param.h>
#include <rcsc/player/audio_sensor.h>
#include <rcsc/param/param_map.h>
#include <rcsc/param/cmd_line_parser.h>

#include <boost/shared_ptr.hpp>

#include <sstream>

/*-------------------------------------------------------------------*/
/*!

*/
SamplePlayer::SamplePlayer()
    : PlayerAgent()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
SamplePlayer::~SamplePlayer()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
bool
SamplePlayer::initImpl( rcsc::CmdLineParser & cmd_parser )
{
    rcsc::ParamMap my_params;
#if 0
    std::string formation_conf;
    my_params.add()
        ( &formation_conf, "fconf" )
        ;
#endif

    if ( ! rcsc::PlayerAgent::initImpl( cmd_parser ) )
    {
        my_params.printHelp( std::cout );
        M_client->setServerAlive( false );
        return false;
    }

    if ( cmd_parser.failed() )
    {
        std::cerr << "player: ***WARNING*** detected invalid options: ";
        cmd_parser.print( std::cerr );
        std::cerr << std::endl;
    }

    if ( ! M_strategy.read( config().configDir() ) )
    {
        std::cerr << "***ERROR*** Failed to read team strategy." << std::endl;
        M_client->setServerAlive( false );
        return false;
    }

    //////////////////////////////////////////////////////////////////
    // Add your code here.
    //////////////////////////////////////////////////////////////////


    return true;
}

/*-------------------------------------------------------------------*/
/*!
  main decision
  virtual method in super class
*/
void
SamplePlayer::actionImpl()
{
    if ( audioSensor().trainerMessage().time_ == world().time() )
    {
        std::cerr << world().teamName() << ' '
                  << world().self().unum() << ": "
                  << world().time()
                  << " hear trainer message ["
                  << audioSensor().trainerMessage().message_
                  << "]"
                  << std::endl;
    }

    if ( audioSensor().freeformMessage().time_ == world().time() )
    {
        std::cerr << world().teamName() << ' '
                  << world().self().unum() << ": "
                  << world().time()
                  << " hear freeform message ["
                  << audioSensor().freeformMessage().message_
                  << "]"
                  << std::endl;
    }

    //////////////////////////////////////////////////////////////
    // check tackle expires
    // check self position accuracy
    // ball search
    // check queued intention
    // check simultaneous kick
    if ( Bhv_PreProcess( M_strategy ).execute( this ) )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: preprocess done"
                            ,__FILE__, __LINE__ );
        return;
    }

    //////////////////////////////////////////////////////////////
    // create current role
    boost::shared_ptr< SoccerRole >
        role_ptr = M_strategy.createRole( config().playerNumber(),
                                          world() );

    if ( ! role_ptr )
    {
        std::cerr << config().teamName() << ": "
                  << world().self().unum()
                  << " Error. Role is not registerd.\nExit ..."
                  << std::endl;
        M_client->setServerAlive( false );
        return;
    }
    if  ( ! role_ptr->hasFormation() )
    {
        std::cerr << config().teamName() << ": "
                  << world().self().unum()
                  << " Error. does not have formation.\nExit ..."
                  << std::endl;
        M_client->setServerAlive( false );
        return;
    }

    //////////////////////////////////////////////////////////////
    // play_on mode
    if ( world().gameMode().type() == rcsc::GameMode::PlayOn  )
    {
        role_ptr->execute( this );
        return;
    }

    //////////////////////////////////////////////////////////////
    // kick_in or corner_kick
    if ( ( world().gameMode().type() == rcsc::GameMode::KickIn_
           || world().gameMode().type() == rcsc::GameMode::CornerKick_ )
         && world().ourSide() == world().gameMode().side() )
    {
        if ( world().self().goalie() )
        {
            Bhv_GoalieFreeKick().execute( this );
        }
        else
        {
            const rcsc::Vector2D base_pos = world().ball().pos();
            rcsc::Vector2D home_pos
                = role_ptr->formation().getPosition( config().playerNumber(),
                                                     base_pos );
            Bhv_SetPlayKickIn( home_pos ).execute( this );
        }
        return;
    }


    //////////////////////////////////////////////////////////////
    // penalty kick mode
    if ( world().gameMode().isPenaltyKickMode() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: penalty kick"
                            ,__FILE__, __LINE__ );
        Bhv_PenaltyKick().execute( this );
        return;
    }

    //////////////////////////////////////////////////////////////
    // goalie free kick mode
    if ( world().self().goalie() )
    {
        Bhv_GoalieFreeKick().execute( this );
        return;
    }

    //////////////////////////////////////////////////////////////
    // other set play mode

    rcsc::Vector2D move_pos
        = M_strategy.getSetPlayPosition( config().playerNumber(),
                                         role_ptr->formation(),
                                         world() );

    //////////////////////////////////////////////////////////////

    Bhv_SetPlay( move_pos ).execute( this );
}

/*-------------------------------------------------------------------*/
/*!
  communication decision.
  virtual method in super clas
*/
void
SamplePlayer::communicationImpl()
{
    if ( ! config().useCommunication()
         || world().gameMode().type() == rcsc::GameMode::BeforeKickOff
         || world().gameMode().type() == rcsc::GameMode::AfterGoal_
         || world().gameMode().isPenaltyKickMode() )
    {
        return;
    }

    const rcsc::PlayerPtrCont & ball_near_mates = world().teammatesFromBall();

    std::string say_msg = effector().getSayMessage();
    say_msg.reserve( rcsc::ServerParam::i().playerSayMsgSize() );

    bool say_ball = false;

    if ( ! say_msg.empty() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "say reserved message" );
    }

    // ball Info: seen at current
    if ( (int)say_msg.length() + 7 <= rcsc::ServerParam::i().playerSayMsgSize()
         && world().ball().posCount() == 0
         && world().ball().velCount() == 0 )
    {
        if ( effector().queuedNextMyPos().dist( effector().queuedNextBallPos() )
             < world().self().kickableArea() - 0.1 )
        {
            // if I am kickable in next cycle, ball info should not be said.
        }
        else if ( world().self().isKickable()  // I am kickable
                  || ( ball_near_mates.empty()
                       || ( ball_near_mates[0]->distFromBall() >
                            world().ball().distFromSelf() - 3.0 ) ) // I am nearest to ball
                  || ( ball_near_mates.size() > 1
                       && ( ball_near_mates[0]->distFromBall()  // nearest to ball teammate is
                            > rcsc::ServerParam::i().visibleDistance() - 0.5 ) // over vis dist
                       && ( ball_near_mates[1]->distFromBall()
                            > world().ball().distFromSelf() - 5.0 ) ) ) // I am second
        {
            // say
            std::string msg; msg.reserve( 7 );
            audioSensor().encodeBall( effector().queuedNextBallPos(),
                                      effector().queuedNextBallVel(),
                                      msg );
            if ( ! msg.empty() )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "say ball status" );
                debugClient().addMessage( "SayB" );
                say_msg += msg;
                say_ball = true;
            }
            else
            {
                std::cerr << config().teamName() << ": "
                          << world().self().unum()
                          << " Failed to encode ball info"
                          << std::endl;
            }
        }
    }

    // goalie info: ball is in chance area
    if ( (int)say_msg.length() + 6 <= rcsc::ServerParam::i().playerSayMsgSize()
         && world().ball().pos().x > 34.0
         && world().ball().pos().absY() < 25.0 )
    {
        // goalie is seen
        const rcsc::PlayerObject * opp_goalie = world().getOpponentGoalie();
        if ( opp_goalie
             && opp_goalie->posCount() == 0
             && opp_goalie->unum() > 0
             && opp_goalie->distFromSelf() < 25.0 )
        {
            std::string msg; msg.reserve( 6 );
            audioSensor().encodeGoalie( opp_goalie->unum(),
                                        opp_goalie->pos(),
                                        msg );
            if ( ! msg.empty() )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "say goalie info to %d (%.1f %.1f)",
                                    opp_goalie->unum(),
                                    opp_goalie->pos().x,
                                    opp_goalie->pos().y );
                debugClient().addMessage( "SayG" );
                say_msg += msg;
            }
            else
            {
                std::cerr << config().teamName() << ": "
                          << world().self().unum()
                          << " Failed to encode goalie info"
                          << std::endl;
            }
        }
    }

    // intercept info
    if ( world().ball().posCount() == 0
         && world().ball().velCount() == 0
         && (int)say_msg.length() + 3 <= rcsc::ServerParam::i().playerSayMsgSize() )
    {
        int self_min = world().interceptTable()->selfReachCycle();
        int mate_min = world().interceptTable()->teammateReachCycle();
        int opp_min = world().interceptTable()->opponentReachCycle();

        if ( world().self().isKickable()
             && ( effector().queuedNextMyPos().dist( effector().queuedNextBallPos() )
                  > world().self().kickableArea() - 0.1 )
             )
        {
            self_min = 100;
        }

        if ( self_min <= mate_min
             && self_min <= opp_min
             && self_min <= 8 )
        {
            std::string msg; msg.reserve( 3 );
            audioSensor().encodeInterceptInfo( world().self().unum(),
                                               self_min,
                                               msg );
            if ( ! msg.empty() )
            {
              rcsc::dlog.addText( rcsc::Logger::TEAM,
                                  "say self intercept info %d",
                                  self_min );
              debugClient().addMessage( "SayI" );
              say_msg += msg;
            }
        }

    }

    // offside line info
    if ( (int)say_msg.length() + 3 <= rcsc::ServerParam::i().playerSayMsgSize()
         && world().offsideLineCount() <= 3
         && std::fabs( world().self().pos().x - world().offsideLineX() ) < 40.0 )
    {
        std::string msg; msg.reserve( 3 );
        audioSensor().encodeOffsideLineX( world().offsideLineX(), msg );
        if ( ! msg.empty() )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "say offside line %.1f",
                                world().offsideLineX() );
            debugClient().addMessage( "SayO" );
            say_msg += msg;
        }
    }

    if ( ! say_msg.empty() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "say message [%s]",
                            say_msg.c_str() );
        debugClient().addMessage( "Say[%s]", say_msg.c_str() );
        doSay( say_msg );
    }

    // attention

    if ( ball_near_mates.empty() )
    {
        if ( world().self().attentiontoUnum() > 0 )
        {
            doAttentiontoOff();
        }
    }
    else if ( ball_near_mates.front()->distFromBall() >
              world().ball().distFromSelf() ) // I am nearest
    {
        // if I am nearest to ball, set attention to self nearest teammate
        if ( ! world().teammatesFromSelf().empty() )
        {
            doAttentionto( world().ourSide(),
                           world().teammatesFromSelf().front()->unum() );
        }
    }
    else
    {
        // set attention to ball nearest teammate
        doAttentionto( world().ourSide(),
                       ball_near_mates.front()->unum() );
    }
}
