// -*-c++-*-

/*!
	\file fedit_canvas.h
	\brief formation editor canvas class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef FEDIT_QT3_FEDIT_CANVAS_H
#define FEDIT_QT3_FEDIT_CANVAS_H

#include <rcsc/geom/vector_2d.h>

#include <qbrush.h>
#include <qpen.h>
#include <qpixmap.h>
#include <qpoint.h>
#include <qwidget.h>

#include <boost/weak_ptr.hpp>

#include "mouse_state.h"

class MainWindow;
class FEditData;

///////////////////////////////////////////////////////////////////////
//! formation editor canvas class
class FEditCanvas
		: public QWidget {

    Q_OBJECT

private:

    //! parent frame
    MainWindow * M_parent;

    //! weak reference pointer
    boost::weak_ptr< FEditData > M_data_ptr;

    //! scaling magnitude
    double M_field_scale;

    //! zoom flag
    bool M_zoomed;

    //! field center point on the screen
    QPoint M_field_center;

    //! coordinate value of focus point
    rcsc::Vector2D M_focus_point;

    //! object enlarge status flag
    bool M_enlarged;

    //! 0: left, 1: middle, 2: right
    MouseState M_mouse_state[3];

    // pixmap memory for double buffering
    QPixmap M_canvas_pixmap;

    //---------------------------------------------------
    // GDI objects
    QBrush M_field_brush;
    QBrush M_field_dark_brush;
    QPen M_line_pen;
    QPen M_triangle_pen;
    QFont M_triangle_font;
    QPen M_area_pen;
    QPen M_ball_pen;
    QBrush M_ball_brush;
    QPen M_player_pen;
    QPen M_select_pen;
    QBrush M_left_team_brush;
    QBrush M_right_team_brush;
    QBrush M_mirror_brush;
    QFont M_player_font;


    bool M_ball_draggable;
    bool M_player_auto_move;

public:
    explicit
		FEditCanvas( MainWindow * parent );

		~FEditCanvas();

    void setData( boost::shared_ptr< FEditData > ptr )
      {
          M_data_ptr = ptr;
      }

protected:

    void paintEvent( QPaintEvent * event );

    void mouseDoubleClickEvent( QMouseEvent * event );
    void mousePressEvent( QMouseEvent * event );
    void mouseReleaseEvent( QMouseEvent * event );
    void mouseMoveEvent( QMouseEvent * event );

private:

    void updateSize();

    void drawField( QPainter & painter );
    void drawContainedArea( QPainter & painter );
    void drawTrainingData( QPainter & painter );
		void drawPlayers( QPainter & painter );
		void drawBall( QPainter & painter );


    const
    rcsc::Vector2D & focusPoint() const
      {
          return M_focus_point;
      }

    int scale( const double & len ) const
      {
          return static_cast< int >( ::rint( len * M_field_scale ) );
      }

    int screenX( const double & x ) const
      {
          return M_field_center.x() + scale( x );
      }

    int screenY( const double & y ) const
      {
          return M_field_center.y() + scale( y );
      }

    double fieldX( const int x ) const
      {
          return ( x - M_field_center.x() ) / M_field_scale;
      }

    double fieldY( const int y ) const
      {
          return ( y - M_field_center.y() ) / M_field_scale;
      }

public slots:

    void setFocusPoint( const QPoint & pos );

    void zoomIn();
    void zoomOut();
    void fitSize();

    void toggleEnlarge( bool on );

signals:

    void objectMoved();
    void pointerMoved( const QString & str );

};

#endif
