// -*-c++-*-

/*!
  \file view_config.cpp
  \brief field canvas configuration class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "view_config.h"
#include "app_config.h"

#include <rcsc/common/server_param.h>
#include <rcsc/math_util.h>

const double ViewConfig::MIN_FIELD_SCALE = 1.0;
const double ViewConfig::MAX_FIELD_SCALE = 400.0;

const double ViewConfig::ZOOM_RATIO = 1.5;

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfig::ViewConfig()
    : M_canvas_width( 640 )
    , M_canvas_height( 480 )
    , M_reverse_side( false )
    , M_grass_type( GRASS_NORMAL )
{
    const AppConfig & conf = AppConfig::instance();

    if ( conf.fieldGrassType() == "mono" )
    {
        M_grass_type = GRASS_NORMAL;
    }
    else if ( conf.fieldGrassType() == "lines" )
    {
        M_grass_type = GRASS_LINES;
    }
    else if ( conf.fieldGrassType() == "checker" )
    {
        M_grass_type = GRASS_CHECKER;
    }
    else
    {
        std::cerr << "Invalid grass type name ["
                  << conf.fieldGrassType()
                  << "]"
                  << std::endl;
    }

    M_keepaway_mode = conf.keepaway();

    reset();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::reset()
{
    M_focus_type = FOCUS_POINT;
    M_focus_point.assign( 0.0, 0.0 );

    M_field_center.x = 0;
    M_field_center.y = 0;
    M_score_board_font_size = 11;
    M_reverse_side = AppConfig::instance().reverseSide();
    M_player_reverse_draw = AppConfig::instance().playerReverseDraw();
    M_zoomed = false;
    M_enlarge = true;
    M_show_score_board = true;
    M_show_ball = true;
    M_show_players = true;
    //M_show_player_number = true;
    M_show_player_number = AppConfig::instance().showPlayerNumber();
    //M_show_stamina = true;
    M_show_stamina = AppConfig::instance().showStamina();
    //M_show_hetero_number = false;
    M_show_hetero_number = AppConfig::instance().showPlayerType();
    //M_show_view_cone = true;
    M_show_view_cone = AppConfig::instance().showViewCone();
    M_show_control_area = false;
    M_show_flags = false;
    M_show_offside_line = false;
    M_show_voronoi_diagram = false;
    M_show_delaunay_trianglation = false;
    M_voronoi_target = rcsc::NEUTRAL;
    M_selected_number = 0;
    M_player_select_type = SELECT_UNSELECT;

    M_ball_auto_trace = false;
    M_player_auto_trace = false;

    M_auto_trace_start = -10;
    M_auto_trace_period = 20;

    M_line_trace = true;
    M_ball_trace_start = 0;
    M_ball_trace_end = 0;
    M_player_trace_start = 0;
    M_player_trace_end = 0;

    M_ball_future_cycle = 0;
    M_player_future_cycle = 0;

    M_show_debug_view = true;
    M_show_debug_view_ball = true;
    M_show_debug_view_self = true;
    M_show_debug_view_players = true;
    M_show_debug_view_comment = true;
    M_show_debug_view_figure = true;
    M_show_debug_view_target = true;
    M_show_debug_view_message = true;

    M_show_debug_log_view = true;

    int temp_width = M_canvas_width;
    M_canvas_width += 1;
    updateFieldSize( temp_width, M_canvas_height );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::updateFieldSize( const int canvas_width,
                             const int canvas_height )
{
    static bool s_score_board = true;

    if ( M_canvas_width != canvas_width
         || M_canvas_height != canvas_height
         || isShownScoreBoard() != s_score_board )
    {
        M_canvas_width = canvas_width;
        M_canvas_height = canvas_height;

        s_score_board = isShownScoreBoard();
        // adjust score board size
        updateScoreBoardSize( canvas_width, canvas_height );

        // draw area height
        int field_height = canvas_height - scoreBoardHeight();

        // adjust field scale to window size.
        if ( ! isZoomed() )
        {
            double total_pitch_l = ( rcsc::ServerParam::i().pitchLength()
                                     + rcsc::ServerParam::i().pitchMargin() * 2.0
                                     + 1.0 );
            double total_pitch_w = ( rcsc::ServerParam::i().pitchWidth()
                                     + rcsc::ServerParam::i().pitchMargin() * 2.0
                                     + 1.0 );

            M_field_scale = static_cast< double >( canvas_width ) / total_pitch_l;
            // automatically adjust a field scale
            if ( total_pitch_w * fieldScale() > field_height )
            {
                M_field_scale = static_cast< double >( field_height ) / total_pitch_w;
            }
            // check the scale threshold
            if ( fieldScale() < MIN_FIELD_SCALE )
            {
                M_field_scale = MIN_FIELD_SCALE;
            }
        }
    }

    updateFieldCenter( canvas_width, canvas_height );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::updateFieldCenter( const int canvas_width,
                               const int canvas_height )
{
    int field_height = canvas_height - scoreBoardHeight();

    M_field_center.x = canvas_width/2 - scale( focusPoint().x );

    M_field_center.y
        = scoreBoardHeight()
        + field_height/2
        - scale( focusPoint().y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::updateScoreBoardSize( const int canvas_width,
                                  const int canvas_height )
{
    if ( isShownScoreBoard() )
    {
        M_score_board_height = rcsc::bound( AppConfig::MIN_SCORE_BOARD_HEIGHT,
                                            canvas_height / 16,
                                            AppConfig::MAX_SCORE_BOARD_HEIGHT );
        M_score_board_font_size = rcsc::bound( 7,
                                               canvas_width / (16 * 3),
                                               16 );
    }
    else
    {
        M_score_board_height = 0;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::zoomIn()
{
    M_zoomed = true;
    M_field_scale *= ZOOM_RATIO;
    if ( M_field_scale > MAX_FIELD_SCALE )
    {
        M_field_scale = MAX_FIELD_SCALE;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::zoomOut()
{
    M_zoomed = true;
    M_field_scale /= ZOOM_RATIO;
    if ( M_field_scale < MIN_FIELD_SCALE )
    {
        M_field_scale = MIN_FIELD_SCALE;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::unzoom()
{
    if ( M_zoomed
         || focusType() != ViewConfig::FOCUS_POINT
         || focusPoint().x != 0.0
         || focusPoint().y != 0.0 )
    {
        M_focus_type = ViewConfig::FOCUS_POINT;
        M_focus_point.assign( 0.0, 0.0 );
        M_zoomed = false;

        int temp_width = M_canvas_width;
        M_canvas_width += 1;
        updateFieldSize( temp_width, M_canvas_height );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setFieldScale( const double & scale )
{
    if ( std::fabs( M_field_scale - scale ) > 0.01 )
    {
        M_field_scale = scale;
        if ( M_field_scale > MAX_FIELD_SCALE )
        {
            M_field_scale = MAX_FIELD_SCALE;
        }
        if ( M_field_scale < MIN_FIELD_SCALE )
        {
            M_field_scale = MIN_FIELD_SCALE;
        }
        M_zoomed = true;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setFocusPoint( const int screen_x,
                           const int screen_y )
{
    M_focus_type = ViewConfig::FOCUS_POINT;

    M_focus_point.x = fieldX( screen_x );
    M_focus_point.y = fieldY( screen_y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setFocusPointReal( const double & field_x,
                               const double & field_y )
{
    M_focus_type = ViewConfig::FOCUS_POINT;

    M_focus_point.x = field_x;
    M_focus_point.y = field_y;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::unselectPlayer()
{
    M_player_select_type = ViewConfig::SELECT_UNSELECT;
    setSelectedNumber( rcsc::NEUTRAL, 0 );
    M_focus_type = ViewConfig::FOCUS_POINT;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setPlayerSelectType( const ViewConfig::PlayerSelectType type )
{
    if ( type != ViewConfig::SELECT_FIX
         && type == M_player_select_type )
    {
        unselectPlayer();
    }
    else
    {
        M_player_select_type = type;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setBallTraceStart( const long & cycle )
{
    if ( 0 <= cycle )
    {
        M_ball_trace_start = cycle;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setBallTraceEnd( const long & cycle )
{
    if ( 0 <= cycle )
    {
        M_ball_trace_end = std::max( cycle, ballTraceStart() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setPlayerTraceStart( const long & cycle )
{
    if ( 0 <= cycle )
    {
        M_player_trace_start = cycle;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setPlayerTraceEnd( const long & cycle )
{
    if ( 0 <= cycle )
    {
        M_player_trace_end = std::max( cycle, playerTraceStart() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setBallFutureCycle( const int cycle )
{
    M_ball_future_cycle = std::min( std::max( 0, cycle ), 100 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfig::setPlayerFutureCycle( const int cycle )
{
    M_player_future_cycle = std::min( std::max( 0, cycle ), 100 );
}
