// -*-c++-*-

/*!
  \file view_holder.cpp
  \brief view data repository Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include "view_holder.h"

#include "debug_client_parser.h"
#include "app_config.h"

#include <rcsc/common/player_param.h>
#include <rcsc/common/server_param.h>
#include <rcsc/rcg/serializer.h>
#include <rcsc/rcg/factory.h>
#include <rcsc/types.h>

#include <iterator>
#include <sstream>
#include <fstream>
#include <iostream>
#include <cassert>

const std::size_t RCG_SIZE_LIMIT = 6000 * 8;

const std::string DEFAULT_DEBUG_VIEW_EXTENSION = ".log";

/*-------------------------------------------------------------------*/
/*!
  default constructor.
*/
ViewHolder::ViewHolder()
    : M_last_playmode( rcsc::PM_Null )
{
    M_player_types.assign( 1, rcsc::PlayerType( rcsc::ServerParam::i() ) );

    M_monitor_view_cont.reserve( 1024 * 8 );
}

/*-------------------------------------------------------------------*/
/*!

*/
ViewHolder::~ViewHolder()
{
    //std::cerr << "ViewHolder Left Debug View Size " << M_left_debug_view.size() << std::endl;
    //std::cerr << "ViewHolder Right Debug View Size " << M_right_debug_view.size() << std::endl;
}

/*-------------------------------------------------------------------*/
/*!
  clear all stored data
*/
void
ViewHolder::clear()
{
    M_last_playmode = rcsc::PM_Null;

    std::memset( M_last_team_left.name, '\0', 16 );
    M_last_team_left.score = 0;
    std::memset( M_last_team_right.name, '\0', 16 );
    M_last_team_right.score = 0;

    M_score_change_indexes.clear();
    M_penalty_scores_left.clear();
    M_penalty_scores_right.clear();

    M_latest_view_data.reset();
    M_monitor_view_cont.clear();

    M_left_debug_view.clear();
    M_right_debug_view.clear();

    M_player_types.assign( 1, rcsc::PlayerType( rcsc::ServerParam::i() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
ViewHolder::saveRCG( std::ostream & os ) const
{
    rcsc::rcg::SerializerPtr serializer
        = rcsc::rcg::make_serializer( logVersion() );

    if ( ! serializer )
    {
        return os;
    }

    // header
    serializer->serializeHeader( os );

    rcsc::rcg::dispinfo_t2 disp2;

    // server param
    rcsc::ServerParam::i().convertTo( disp2.body.sparams );
    serializer->serialize( os, disp2.body.sparams );

    // player param
    rcsc::PlayerParam::i().convertTo( disp2.body.pparams );
    serializer->serialize( os, disp2.body.pparams );

    // player types
    for ( std::vector< rcsc::PlayerType >::const_iterator it
              = playerTypeCont().begin();
          it != playerTypeCont().end();
          ++it )
    {
        it->convertTo( disp2.body.ptinfo );
        serializer->serialize( os, disp2.body.ptinfo );
    }

    // view data, team data, playmode ...
    const std::vector< MonitorViewPtr >::const_iterator end
        = monitorViewCont().end();
    for ( std::vector< MonitorViewPtr >::const_iterator it
              = monitorViewCont().begin();
          it != end;
          ++it )
    {
        (*it)->convertTo( disp2.body.show );

        serializer->serialize( os, disp2.body.show );
    }

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewHolder::saveDebugView( const std::string & dir_path ) const
{
    if ( dir_path.empty() )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " Dir path string is empty!"
                  << std::endl;
        return;
    }

    if ( ! M_latest_view_data )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " No view data!!"
                  << std::endl;
        return;
    }

    std::string left_team = M_latest_view_data->leftTeam().name();

    if ( ! left_team.empty()
         && ! M_left_debug_view.empty() )
    {
        for ( int unum = 1; unum <= 11; ++unum )
        {
            if ( saveDebugView( dir_path, left_team, unum,
                                M_left_debug_view ) )
            {
                std::cerr << "saved debug view data : "
                          << left_team << "-" << unum
                          << std::endl;
            }
        }
    }

    std::string right_team = M_latest_view_data->rightTeam().name();

    if ( ! right_team.empty()
         && ! M_right_debug_view.empty() )
    {
        for ( int unum = 1; unum <= 11; ++unum )
        {
            if ( saveDebugView( dir_path, right_team, unum,
                                M_right_debug_view ) )
            {
                std::cerr << "saved debug view data : "
                          << right_team << "-" << unum
                          << std::endl;
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ViewHolder::saveDebugView( const std::string & dir_path,
                           const std::string & team_name,
                           const int unum,
                           const std::map< long, DebugViewCont > & data ) const
{
    std::string filepath = dir_path;
    {
        if ( filepath.empty()
             || *filepath.rbegin() != '/' )
        {
            filepath += '/';
        }

        std::ostringstream ostr;
        ostr << team_name << '-' << unum
             << DEFAULT_DEBUG_VIEW_EXTENSION;
        ostr << std::flush;
        filepath += ostr.str();
    }

    std::ofstream fout( filepath.c_str() );

    if ( ! fout )
    {
        return false;
    }

    const long last_cycle = M_latest_view_data->cycle();

    for ( long i = 0; i < last_cycle; ++i )
    {
        std::map< long, DebugViewCont >::const_iterator it
            = data.find( i );
        if ( it != data.end()
             && it->second[unum - 1] )
        {
            it->second[unum - 1]->print( fout, i );
        }
    }

    fout.close();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewHolder::loadDebugView( const std::string & dir_path )
{
    if ( dir_path.empty() )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " Dir path string is empty!"
                  << std::endl;
        return;
    }

    if ( ! M_latest_view_data )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " No view data!!"
                  << std::endl;
        return;
    }

    std::string left_team = M_latest_view_data->leftTeam().name();

    if ( ! left_team.empty() )
    {
        for ( int unum = 1; unum <= 11; ++unum )
        {
            if ( loadDebugView( dir_path, left_team, unum ) )
            {
                std::cerr << "loaded debug view data : "
                          << left_team << "-" << unum
                          << std::endl;
            }
        }
    }

    std::string right_team = M_latest_view_data->rightTeam().name();

    if ( ! right_team.empty() )
    {
        for ( int unum = 1; unum <= 11; ++unum )
        {
            if ( loadDebugView( dir_path, right_team, unum ) )
            {
                std::cerr << "loaded debug view data : "
                          << right_team << "-" << unum
                          << std::endl;
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ViewHolder::loadDebugView( const std::string & dir_path,
                           const std::string & team_name,
                           const int unum )
{
    std::string filepath = dir_path;
    {
        if ( filepath.empty()
             || *filepath.rbegin() != '/' )
        {
            filepath += '/';
        }

        std::ostringstream ostr;
        ostr << team_name << '-' << unum
             << DEFAULT_DEBUG_VIEW_EXTENSION;
        ostr << std::flush;
        filepath += ostr.str();
    }

    std::ifstream fin( filepath.c_str() );

    if ( ! fin )
    {
        return false;
    }

    DebugClientParser parser;

    std::string line_buf;
    while ( std::getline( fin, line_buf ) )
    {
        if ( line_buf.compare( 0, 10, "%% debug [" ) )
        {
            // other debug log data
            continue;
        }

        parser.parse( line_buf.c_str() + 10, *this );
    }

    fin.close();
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addShowInfo( const rcsc::rcg::showinfo_t & show )
{
    if ( M_monitor_view_cont.size() > RCG_SIZE_LIMIT )
    {
        return false;
    }

    MonitorViewPtr ptr( new MonitorViewData( show ) );

    addPlayMode( show.pmode );
    addTeamInfo( show.team[0], show.team[1] );

    addMonitorViewData( ptr );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addShowInfo2( const rcsc::rcg::showinfo_t2 & show2 )
{
    if ( M_monitor_view_cont.size() > RCG_SIZE_LIMIT )
    {
        return false;
    }

    MonitorViewPtr ptr( new MonitorViewData( show2 ) );

    addPlayMode( show2.pmode );
    addTeamInfo( show2.team[0], show2.team[1] );

    addMonitorViewData( ptr );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addShortShowInfo2( const rcsc::rcg::short_showinfo_t2 & show2 )
{
    if ( M_monitor_view_cont.size() > RCG_SIZE_LIMIT )
    {
        return false;
    }

    MonitorViewPtr ptr( new MonitorViewData( show2,
                                             M_last_playmode,
                                             M_last_team_left,
                                             M_last_team_right ) );
    addMonitorViewData( ptr );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewHolder::addMonitorViewData( MonitorViewPtr ptr )
{
    if ( M_latest_view_data )
    {
        if ( ! M_monitor_view_cont.empty() )
        {
            // before adding the latest view data,
            // we should remove old BeforeKickOffMode or TimeOver data
            // from data container if exist.

            if ( ptr->playmode().mode() == rcsc::PM_BeforeKickOff
                 || ptr->playmode().mode() == rcsc::PM_TimeOver )
            {
                rcsc::PlayMode back_pmode
                    = M_monitor_view_cont.back()->playmode().mode();
                if ( back_pmode == rcsc::PM_BeforeKickOff
                     || back_pmode == rcsc::PM_TimeOver )
                {
                    // last data must be swapped.
                    M_monitor_view_cont.pop_back();
                }
            }
            else if ( AppConfig::instance().monitorClientMode()
                      && ! AppConfig::instance().timeShiftReplay() )
            {
                M_monitor_view_cont.pop_back();
            }
        }

        M_monitor_view_cont.push_back( M_latest_view_data );
    }

    M_latest_view_data = ptr;
}

/*-------------------------------------------------------------------*/
/*!
  this methos should be called just after finish to load rcg file.
*/
void
ViewHolder::pushBackLatestViewData()
{
    if ( M_latest_view_data )
    {
        M_monitor_view_cont.push_back( M_latest_view_data );
    }
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addMsgInfo( const rcsc::rcg::Int16 board,
                        const std::string & msg )
{
    //std::cerr << "ViewHolder::addMsgInfo" << std::endl;
    std::cout << "recv msginfo_t: message = [" << msg << ']'
              << std::endl;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addDrawInfo( const rcsc::rcg::drawinfo_t & msg )
{
    std::cout << "recv drawinfo_t: mode = " << msg.mode
              << std::endl;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addPlayMode( const char pmode )
{
    rcsc::PlayMode playmode = static_cast< rcsc::PlayMode >( pmode );

    if ( M_last_playmode != playmode )
    {
        if ( playmode == rcsc::PM_PenaltyScore_Left
             || playmode == rcsc::PM_PenaltyMiss_Left )
        {
            long cycle = ( M_latest_view_data
                           ? M_latest_view_data->cycle()
                           : 0 );
            M_penalty_scores_left.push_back( std::make_pair( cycle, playmode ) );
        }
        else if(  playmode == rcsc::PM_PenaltyScore_Right
                  || playmode == rcsc::PM_PenaltyMiss_Right )
        {
            long cycle = ( M_latest_view_data
                           ? M_latest_view_data->cycle()
                           : 0 );
            M_penalty_scores_right.push_back( std::make_pair( cycle, playmode ) );
        }
    }

    M_last_playmode = playmode;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addTeamInfo( const rcsc::rcg::team_t & team_l,
                         const rcsc::rcg::team_t & team_r )
{
    if ( M_last_team_left.score != team_l.score
         || M_last_team_right.score != team_r.score )
    {
        M_score_change_indexes.push_back( M_monitor_view_cont.size() );
    }

    M_last_team_left = team_l;
    M_last_team_right = team_r;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  add new hetero player param
*/
bool
ViewHolder::addPlayerType( const rcsc::rcg::player_type_t & param )
{
    //std::cerr << "ViewHolder::addPlayerType" << std::endl;

    int id = static_cast< int >( htons( param.id ) );
    if ( id < 0 )
    {
        std::cerr << "addPlayerType. over the array size" << std::endl;
        return false;
    }

    if ( id + 1 >= static_cast< int >( M_player_types.size() ) )
    {
        M_player_types.resize( id + 1, rcsc::PlayerType( rcsc::ServerParam::i() ) );
    }

    M_player_types[id] = rcsc::PlayerType( rcsc::ServerParam::i(),
                                           param );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  set server param
*/
bool
ViewHolder::addServerParam( const rcsc::rcg::server_params_t & param )
{
    //std::cerr << "ViewHolder::addServerParam" << std::endl;
    rcsc::ServerParam::instance().convertFrom( param );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  set player param
*/
bool
ViewHolder::addPlayerParam( const rcsc::rcg::player_params_t & param )
{
    //std::cerr << "ViewHolder::addPlayerParam" << std::endl;
    rcsc::PlayerParam::instance().convertFrom( param );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewHolder::addDebugView( const long & cycle,
                          char side,
                          const int unum,
                          DebugViewPtr debug_view )
{
    typedef std::map< long, DebugViewCont >::iterator IterType;

    if ( ! debug_view )
    {
        return;
    }

    if ( unum < 1 || 11 < unum )
    {
        std::cerr << "ViewHolder::addDebugView. Error unum over range"
                  << std::endl;
        return;
    }

    std::map< long, DebugViewCont > * view_map = ( side == 'l'
                                                   ? &M_left_debug_view
                                                   : &M_right_debug_view );

    IterType it = view_map->find( cycle );

    // already exist this cycle data
    if ( it != view_map->end() )
    {
        it->second[unum - 1] = debug_view;
        return;
    }

    // insert new cycle
    std::pair< IterType, bool > p
        = view_map->insert ( std::make_pair( cycle, DebugViewCont( 11 ) ) );

    if ( ! p.second )
    {
        std::cerr << "ViewHolder::addDebugView. Failed to add DebugView"
                  << std::endl;
        return;
    }

    p.first->second[unum - 1] = debug_view;
}

/*-------------------------------------------------------------------*/
/*!

*/
MonitorViewConstPtr
ViewHolder::getViewData( const std::size_t idx ) const
{
    if ( M_monitor_view_cont.size() <= idx )
    {
        return MonitorViewConstPtr(); // null pointer
    }

    return M_monitor_view_cont[idx];
}

/*-------------------------------------------------------------------*/
/*!

*/
std::size_t
ViewHolder::getIndexOf( const int cycle ) const
{
    MonitorViewCont::size_type len = M_monitor_view_cont.size();
    MonitorViewCont::size_type half;

    MonitorViewCont::const_iterator first = M_monitor_view_cont.begin();
    MonitorViewCont::const_iterator middle;

    // lower bound algorithm
    while ( len > 0 )
    {
        half = len >> 1;
        middle = first;
        middle += half;
        if ( (*middle)->cycle() < cycle )
        {
            first = middle;
            ++first;
            len = len - half - 1;
        }
        else
        {
            len = half;
        }
    }

    if ( first == M_monitor_view_cont.end() )
    {
        if ( cycle > 0
             && ! M_monitor_view_cont.empty() )
        {
            return M_monitor_view_cont.size() - 1;
        }
        return 0;
    }

    return std::distance( M_monitor_view_cont.begin(), first );
}

/*-------------------------------------------------------------------*/
/*!

*/
const
rcsc::PlayerType &
ViewHolder::playerType( const std::size_t id ) const
{
    try
    {
        return M_player_types.at( id );
    }
    catch ( std::exception & e )
    {
        std::cerr << "Exception caught! " << e.what()
                  << "invalid player type index " << id
                  << std::endl;
        return M_player_types.front();
    }
}
