// -*-c++-*-

/*!
  \file detail_dialog.cpp
  \brief Detail info Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "detail_dialog.h"

#include "main_data.h"

#include <rcsc/geom/vector_2d.h>

#include <iostream>
#include <cstdio>

/*-------------------------------------------------------------------*/
/*!

*/
DetailDialog::DetailDialog( QWidget * parent,
                            const MainData & main_data )
    : QDialog( parent )
    , M_main_data( main_data )
{
    this->setWindowTitle( tr( "Detail Dialog" ) );

    QFont font = this->font();
    font.setPointSize( 8 );
    this->setFont( font );

    createWidgets();
}

/*-------------------------------------------------------------------*/
/*!

*/
DetailDialog::~DetailDialog()
{
    std::cerr << "delete DetailDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DetailDialog::createWidgets()
{
    QVBoxLayout * layout = new QVBoxLayout();
    layout->setMargin( 2 );
    layout->setSpacing( 2 );


    layout->addWidget( createBallLabels(),
                       1, Qt::AlignHCenter );

    layout->addWidget( createPlayerLabels(),
                       1, Qt::AlignHCenter );

    this->setLayout( layout );
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
DetailDialog::createBallLabels()
{
    QGroupBox * group_box = new QGroupBox( tr( "Ball" ) );
    group_box->setMinimumWidth( 160 );

    QGridLayout * layout = new QGridLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );
    layout->setColumnMinimumWidth( 1, this->fontMetrics().width( tr( " -12.345 -12.345" ) ) );

    int row = 0;

    layout->addWidget( new QLabel( tr( "Pos " ) ), row, 0, Qt::AlignRight );
    M_ball_pos = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_ball_pos, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Vel " ) ), row, 0, Qt::AlignRight );
    M_ball_vel = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_ball_vel, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "PolarVel " ) ), row, 0, Qt::AlignRight );
    M_ball_polar_vel = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_ball_polar_vel, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "LastMove " ) ), row, 0, Qt::AlignRight );
    M_ball_last_move = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_ball_last_move, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "LastPolar " ) ), row, 0, Qt::AlignRight );
    M_ball_last_polar_move = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_ball_last_polar_move, row, 1, Qt::AlignRight );
    ++row;

    group_box->setLayout( layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
DetailDialog::createPlayerLabels()
{
    QGroupBox * group_box = new QGroupBox( tr( "Player" ) );
    group_box->setMinimumWidth( 160 );

    QGridLayout * layout = new QGridLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );
    layout->setColumnMinimumWidth( 1, this->fontMetrics().width( tr( " -12.345 -12.345" ) ) );

    int row = 0;

    layout->addWidget( new QLabel( tr( "Unum " ) ), row, 0, Qt::AlignRight );
    M_player_number = new QLabel( tr( " Neutral  0" ) );
    layout->addWidget( M_player_number, row, 1, Qt::AlignHCenter );
    ++row;

    layout->addWidget( new QLabel( tr( "Type " ) ), row, 0, Qt::AlignRight );
    M_player_type = new QLabel( tr( "0" ) );
    layout->addWidget( M_player_type, row, 1, Qt::AlignHCenter );
    ++row;

    layout->addWidget( new QLabel( tr( "Pos " ) ), row, 0, Qt::AlignRight );
    M_player_pos = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_pos, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Vel " ) ), row, 0, Qt::AlignRight );
    M_player_vel= new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_vel, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Polar " ) ), row, 0, Qt::AlignRight );
    M_player_polar_vel = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_polar_vel, row, 1, Qt::AlignRight );
    ++row;


    layout->addWidget( new QLabel( tr( "LastMove " ) ), row, 0, Qt::AlignRight );
    M_player_last_move = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_last_move, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Polar " ) ), row, 0, Qt::AlignRight );
    M_player_last_polar_move = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_last_polar_move, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Body " ) ), row, 0, Qt::AlignRight );
    M_player_body = new QLabel( tr( " 000.0" ) );
    layout->addWidget( M_player_body, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Head " ) ), row, 0, Qt::AlignRight );
    M_player_head = new QLabel( tr( " 000.0" ) );
    layout->addWidget( M_player_head, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Stamina " ) ), row, 0, Qt::AlignRight );
    M_stamina = new QLabel( tr( " 0000.0" ) );
    layout->addWidget( M_stamina, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Effort " ) ), row, 0, Qt::AlignRight );
    M_effort = new QLabel( tr( " 0.000 / 0.000" ) );
    layout->addWidget( M_effort, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Recovery " ) ), row, 0, Qt::AlignRight );
    M_recovery = new QLabel( tr( " 0.000 / 1.0" ) );
    layout->addWidget( M_recovery, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Kick " ) ), row, 0, Qt::AlignRight );
    M_kick_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_kick_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Dash " ) ), row, 0, Qt::AlignRight );
    M_dash_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_dash_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Turn " ) ), row, 0, Qt::AlignRight );
    M_turn_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_turn_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Say " ) ), row, 0, Qt::AlignRight );
    M_say_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_say_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "TurnNeck " ) ), row, 0, Qt::AlignRight );
    M_turn_neck_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_turn_neck_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Catch " ) ), row, 0, Qt::AlignRight );
    M_catch_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_catch_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Move " ) ), row, 0, Qt::AlignRight );
    M_move_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_move_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "ChbView " ) ), row, 0, Qt::AlignRight );
    M_change_view_count = new QLabel( tr( " 00000" ) );
    layout->addWidget( M_change_view_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "BallRelPos " ) ), row, 0, Qt::AlignRight );
    M_player_ball_rel_pos = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_ball_rel_pos, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Polar " ) ), row, 0, Qt::AlignRight );
    M_player_ball_rel_polar = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_ball_rel_polar, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "BallRelVel " ) ), row, 0, Qt::AlignRight );
    M_player_ball_rel_vel = new QLabel( tr( " 000.00, 000.00" ) );
    layout->addWidget( M_player_ball_rel_vel, row, 1, Qt::AlignRight );
    ++row;


    group_box->setLayout( layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DetailDialog::showEvent( QShowEvent * event )
{
    updateLabels();
    std::cerr << "show DetailDialog" << std::endl;
    event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DetailDialog::updateLabels()
{
    if ( ! this->isVisible() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );

    if ( ! view )
    {
        return;
    }

    MonitorViewConstPtr last_view
        = ( M_main_data.viewIndex() <= 0
            ? MonitorViewConstPtr()
            : M_main_data.getViewData( M_main_data.viewIndex() - 1 ) );

    //std::cerr << "DetailDialog::updteLabels start" << std::endl;

    char buf[32];

    // update ball

    const Ball & ball = view->ball();

    std::snprintf( buf, 32, " %6.2f, %6.2f",
                   ball.x(), ball.y() );
    M_ball_pos->setText( QString::fromAscii( buf ) );

    if ( ball.hasDelta() )
    {
        rcsc::Vector2D vel( ball.deltaX(), ball.deltaY() );
        std::snprintf( buf, 32, " %6.2f, %6.2f",
                       vel.x, vel.y );
				M_ball_vel->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, " %6.3f, %6.1f",
                       vel.r(), vel.th().degree() );
				M_ball_polar_vel->setText( QString::fromAscii( buf ) );
		}

    if ( ! last_view )
    {
        M_ball_last_move->setText( tr( "      0,      0" ) );
        M_ball_last_polar_move->setText( tr( "      0,      0" ) );
    }
    else
    {
        rcsc::Vector2D move( ball.x() - last_view->ball().x(),
                             ball.y() - last_view->ball().y() );

        std::snprintf( buf, 32, " %6.2f, %6.2f",
                       move.x, move.y );
        M_ball_last_move->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, " %6.3f, %6.1f",
                       move.r(), move.th().degree() );
        M_ball_last_polar_move->setText( QString::fromAscii( buf ) );
    }

    // update player

    int number = M_main_data.viewConfig().selectedNumber();
    if ( number == 0 || std::abs( number ) > 11 )
    {
        return;
    }

    const int player_index = std::abs( number ) - 1 + ( number > 0 ? 0 : 11 );
    const Player & player = view->players()[player_index];
    const rcsc::PlayerType & player_type
        = M_main_data.viewHolder().playerType( player.type() );

    // side, number
    std::snprintf( buf, 32, "%s     %2d",
                   number > 0 ? "   Left" : "   Right",
                   number > 0 ? number : -number );
    M_player_number->setText( QString::fromAscii( buf ) );

    // player type
    std::snprintf( buf, 32, " %d" ,
                   player.type() );
    M_player_type->setText( QString::fromAscii( buf ) );

    // pos
    std::snprintf( buf, 32, " %6.2f, %6.2f",
                   player.x(), player.y() );
		M_player_pos->setText( QString::fromAscii( buf ) );

    // velocity, speed, speed dir
		if ( player.hasDelta() )
		{
				rcsc::Vector2D vel( player.deltaX(), player.deltaY() );

				std::snprintf( buf, 32, " %6.2f, %6.2f",
                       vel.x, vel.y );
				M_player_vel->setText( QString::fromAscii( buf ) );

				std::snprintf( buf, 32, " %6.3f, %6.1f",
                       vel.r(), vel.th().degree() );
				M_player_polar_vel->setText( QString::fromAscii( buf ) );

        // ball relative velocity
        std::snprintf( buf, 32, " %6.2f, %6.2f",
                       ball.deltaX() - vel.x, ball.deltaY() - vel.y );
        M_player_ball_rel_vel->setText( QString::fromAscii( buf ) );
		}

    if ( ! last_view )
    {
        M_player_last_move->setText( tr( "      0,      0" ) );
        M_player_last_polar_move->setText( tr( "      0,      0" ) );
    }
    else
    {
        const Player & last_player = last_view->players()[player_index];
        rcsc::Vector2D last_move( player.x() - last_player.x(),
                                  player.y() - last_player.y() );
				std::snprintf( buf, 32, " %6.2f, %6.2f",
                       last_move.x, last_move.y );
				M_player_last_move->setText( QString::fromAscii( buf ) );

				std::snprintf( buf, 32, " %6.3f, %6.1f",
                       last_move.r(), last_move.th().degree() );
				M_player_last_polar_move->setText( QString::fromAscii( buf ) );
    }

    // body angle
    std::snprintf( buf, 32," %6.1f",
                   player.body() );
    M_player_body->setText( QString::fromAscii( buf ) );

    // neck, head angle
    if ( player.hasView() )
    {
        rcsc::AngleDeg neck = player.head() - player.body();

        std::snprintf( buf, 32, " %6.1f(%6.1f)",
                       player.head(), neck.degree() );
        M_player_head->setText( QString::fromAscii( buf ) );
    }

    // stamina, effort, recovery
    if ( player.hasStamina() )
    {
        std::snprintf( buf, 32, " %6.1f",
                       player.stamina() );
        M_stamina->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, " %5.3f / %5.3f",
                       player.effort(), player_type.effortMax() );
        M_effort->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, " %5.3f / 1.0",
                       player.recovery() );
        M_recovery->setText( QString::fromAscii( buf ) );
    }

    // command count
    if ( player.hasCommandCount() )
    {
        std::snprintf( buf, 32, "  %5d",
                       player.kickCount() );
        M_kick_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.dashCount() );
        M_dash_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.turnCount() );
        M_turn_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.sayCount() );
        M_say_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.turnNeckCount() );
        M_turn_neck_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.catchCount() );
        M_catch_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.moveCount() );
        M_move_count->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "  %5d",
                       player.changeViewCount() );
        M_change_view_count->setText( QString::fromAscii( buf ) );
    }

    // ball related info
    {
        rcsc::Vector2D player_to_ball( ball.x() - player.x(), ball.y() - player.y() );

        std::snprintf( buf, 32, " %6.2f, %6.2f",
                       player_to_ball.x, player_to_ball.y );
        M_player_ball_rel_pos->setText( QString::fromAscii( buf ) );

        std::snprintf( buf, 32, "%7.3f, %6.1f",
                       player_to_ball.r(), player_to_ball.th().degree() );
        M_player_ball_rel_polar->setText( QString::fromAscii( buf ) );
    }
}
