// -*-c++-*-

/*!
  \file player_control_painter.cpp
  \brief player control painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "player_control_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"
#include "monitor_view_data.h"

#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerControlPainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownPlayers()
         || ! vconf.isShownControlArea() )
    {
        return;
    }

    int number = vconf.selectedNumber();
    if ( number == 0 )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );

    if ( ! view )
    {
        return;
    }

    const Ball & ball = view->ball();

    if ( ! ball.hasDelta() )
    {
        return;
    }

    const DrawConfig & dconf = DrawConfig::instance();

    const Player & player = view->players()[ ( number > 0
                                               ? number - 1
                                               : -number - 1 + 11 ) ];
    const rcsc::ServerParam & sparam = rcsc::ServerParam::i();
    const rcsc::PlayerType & ptype
        = M_main_data.viewHolder().playerType( player.type() );

    rcsc::Vector2D ppos( player.x(), player.y() );
    rcsc::Vector2D bpos( ball.x(), ball.y() );
    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - player.body() );

    rcsc::Vector2D bnext( bpos.x + ball.deltaX(),
                          bpos.y + ball.deltaY() );
    double ball_dist = player_to_ball.r();

    if ( ball_dist < ptype.kickableArea() )
    {
        double max_kick_accel
            = sparam.maxPower()
            * sparam.kickPowerRate()
            * ( 1.0 - 0.25 * player_to_ball.th().abs() / 180.0
                - 0.25
                * ( ball_dist - ptype.playerSize() - sparam.ballSize() )
                / ptype.kickableMargin() );

        if ( max_kick_accel > sparam.ballAccelMax() )
        {
            max_kick_accel = sparam.ballAccelMax();
        }

        int max_kick_accel_screen = vconf.scale( max_kick_accel );
        QPoint bnext_screen( vconf.screenX( bnext.x ),
                             vconf.screenY( bnext.y ) );

        painter.setPen( dconf.kickAccelPen() );
        painter.setBrush( dconf.transparentBrush() );
        // draw no noise next ball move only by inertia
        painter.drawLine( vconf.screenX( bpos.x ),
                          vconf.screenY( bpos.y ),
                          bnext_screen.x(),
                          bnext_screen.y() );
        // draw next ball reachable area
        painter.drawEllipse( bnext_screen.x()- max_kick_accel_screen,
                             bnext_screen.y() - max_kick_accel_screen,
                             max_kick_accel_screen * 2,
                             max_kick_accel_screen * 2 );
        // draw kick info text
        painter.setFont( dconf.playerFont() );
        painter.setPen( dconf.kickAccelPen() );

        char buf[32];
        std::snprintf( buf, 32, "MaxAccel=%.3f", max_kick_accel );
        painter.drawText( bnext_screen.x() + 10,
                          bnext_screen.y() + painter.fontMetrics().ascent(),
                          QString::fromAscii( buf ) );
    }

}
