// -*-c++-*-

/*!
  \file color_setting_dialog.cpp
  \brief color setting dialog for the field canvas Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "color_setting_dialog.h"
#include "draw_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
ColorButton::ColorButton( const QString & name,
                          QPen * pen,
                          QBrush * brush,
                          ColorSettingDialog * parent )
    : QPushButton( parent )
    , M_name( name )
    , M_pen( pen )
    , M_brush( brush )
{
    //btn->setFlat( true );
    //btn->setMinimumSize( 16, 16 );
    //this->setMaximumSize( 32, 32 );

    if ( M_pen )
    {
        M_color = M_pen->color();
        M_default_color = M_pen->color();
        updateIcon();
    }
    else if ( M_brush )
    {
        M_color = M_brush->color();
        M_default_color = M_brush->color();
        updateIcon();
    }

    this->setText( name );

    connect( this, SIGNAL( clicked() ),
             this, SLOT( colorDialog() ) );
    connect( this, SIGNAL( colorChanged() ),
             parent, SIGNAL( colorChanged() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorButton::setColor( const QColor & color )
{
    if ( M_color != color )
    {
        M_color = color;

        updateIcon();

        if ( M_pen ) M_pen->setColor( color );
        if ( M_brush ) M_brush->setColor( color );

        emit colorChanged();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorButton::colorDialog()
{
    QColor col = QColorDialog::getColor( color() );
    if ( col.isValid() )
    {
        setColor( col );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorButton::updateIcon()
{
    QPixmap pixmap( 16, 16 );
    pixmap.fill( M_color );
    this->setIconSet( pixmap );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorButton::checkColor()
{
    if ( M_pen && M_pen->color() != M_color )
    {
        M_color = M_pen->color();
        updateIcon();
    }
    else if ( M_brush && M_brush->color() != M_color )
    {
        M_color = M_brush->color();
        updateIcon();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorButton::revert()
{
    setColor( defaultColor() );
}

/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/

/*-------------------------------------------------------------------*/
/*!

*/
ColorSettingDialog::ColorSettingDialog( QWidget * parent )
    : QDialog( parent )
{

    this->setCaption( tr( "Color Settings" ) );

    createControls();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::createControls()
{
    QVBoxLayout * main_layout = new QVBoxLayout( this );

    main_layout->addLayout( createColorButtons() );

    main_layout->addLayout( createAnswerButtons() );
}

/*-------------------------------------------------------------------*/
/*!

*/
ColorSettingDialog::~ColorSettingDialog()
{
    //std::cerr << "delete ColorSettingDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
ColorSettingDialog::createColorButtons()
{
    QGridLayout * layout = new QGridLayout();
    layout->setMargin( 4 );
    layout->setSpacing( 2 );

    int col = 0;
    int row = 0;
    {
        QString name = tr( "Score Board" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().scoreBoardBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Score Board Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().scoreBoardFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Field 1" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().fieldBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Field 2" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().fieldDarkBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Line" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().linePenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Ball" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().ballBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Measure" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().measurePenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Measure Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().measureFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }


    row = 0;
    ++col;

    {
        QString name = tr( "Left Team" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().leftTeamPenPtr(),
                                             DrawConfig::instance().leftTeamBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Left Goalie" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().leftGoaliePenPtr(),
                                             DrawConfig::instance().leftGoalieBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Right Team" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().rightTeamPenPtr(),
                                             DrawConfig::instance().rightTeamBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Right Goalie" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().rightGoaliePenPtr(),
                                             DrawConfig::instance().rightGoalieBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Player Number Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().playerNumberFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Stamina Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().playerStaminaFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Player Detail Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().playerDetailFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Selected Player Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().selectedPlayerFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Effort Decayed" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().effortDecayedPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Recovery Decayed" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().recoveryDecayedPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "View Cone" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().viewConePenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "View Area" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().viewAreaPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }

    row = 0;
    ++col;

    {
        QString name = tr( "Kick Pen" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().kickPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Kick Fault" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().kickFaultPenPtr(),
                                             DrawConfig::instance().kickFaultBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Catch Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().catchBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Catch Fault Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().catchFaultBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Collide Ball Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().collideBallBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Collide Player Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().collidePlayerBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Tackle" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().tacklePenPtr(),
                                             DrawConfig::instance().tackleBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Tackle Fault Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().tackleFaultBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Kick Accel" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().kickAccelPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Tackle Area" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().tackleAreaPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }

    row = 0;
    ++col;

    {
        QString name = tr( "Debug Ball Pen" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugBallPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Ball Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugBallBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Ball Trace Pen" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugBallTracePenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Ball Trace Brush" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugBallTraceBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Player Pen" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugPlayerPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Player Pen" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugPlayerPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Self" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugSelfBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Teammate" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugTeammateBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Opponent" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugOpponentBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Unknown Teammate" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugUnknownTeammateBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Unknown Opponent" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugUnknownOpponentBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Unknown Player" );
        ColorButton * btn = new ColorButton( name,
                                             NULL,
                                             DrawConfig::instance().debugUnknownPlayerBrushPtr(),
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Target" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugTargetPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Comment" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugCommentFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Message" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugMessageFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Log Font" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugLogMessageFontPenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;
    {
        QString name = tr( "Debug Figure" );
        ColorButton * btn = new ColorButton( name,
                                             DrawConfig::instance().debugFigurePenPtr(),
                                             NULL,
                                             this );
        M_color_buttons.insert( std::pair< QString, ColorButton * >( name, btn ) );
        layout->addWidget( btn, row, col, Qt::AlignLeft );
    }
    ++row;

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
ColorSettingDialog::createAnswerButtons()
{
    QHBoxLayout * layout = new QHBoxLayout();

    QFontMetrics fm = this->fontMetrics();
    int text_width = fm.width( tr( "Cancel" ) ) + 8;

    QPushButton * def = new QPushButton( tr( "Default" ), this );
    //cancel->resize( text_width, fm.height() );
    connect( def, SIGNAL( clicked() ),
             this, SLOT( setDefaultColors() ) );

    QPushButton * ok = new QPushButton( tr( "OK" ), this );
    ok->resize( text_width, fm.height() );
    connect( ok, SIGNAL( clicked() ),
             this, SLOT( accept() ) );

    QPushButton * cancel = new QPushButton( tr( "Cancel" ), this );
    cancel->resize( text_width, fm.height() );
    connect( cancel, SIGNAL( clicked() ),
             this, SLOT( reject() ) );


    layout->addWidget( def, 0, Qt::AlignLeft );
    layout->addStretch( 1 );
    layout->addWidget( ok, 0, Qt::AlignRight );
    layout->addWidget( cancel, 0, Qt::AlignRight );

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::revert()
{
    for ( std::map< QString, ColorButton * >::iterator it = M_color_buttons.begin();
          it != M_color_buttons.end();
          ++it )
    {
        it->second->revert();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::setDefaultColors()
{
    DrawConfig::instance().setDefaultColors();

    for ( std::map< QString, ColorButton * >::iterator it = M_color_buttons.begin();
          it != M_color_buttons.end();
          ++it )
    {
        it->second->checkColor();
    }

    emit colorChanged();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::reject()
{
    // reset all changes

    revert();

    QDialog::reject();
}
