// -*-c++-*-

/*!
  \file main_window.cpp
  \brief main application window class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "main_window.h"

#include "color_setting_dialog.h"
#include "detail_dialog.h"
#include "font_setting_dialog.h"
#include "image_save_dialog.h"
#include "launcher_dialog.h"
#include "player_type_dialog.h"
#include "view_config_dialog.h"
#include "debug_message_window.h"
#include "field_canvas.h"
#include "debug_server.h"
#include "monitor_client.h"
#include "log_player.h"
#include "log_player_tool_bar.h"
//#include "dir_selector.h"
#include "app_config.h"

#include <string>
#include <iostream>
#include <cstring>

#include <csignal> // ::kill()
#include <cstdlib> // system()
#include <sys/types.h> // SIGINT


#include "xpm/soccerwindow2.xpm"
#include "xpm/soccerwindow2-nostr.xpm"

#include "xpm/debug_server_switch.xpm"
#include "xpm/logplayer_live_mode.xpm"
#include "xpm/open_rcg.xpm"
#include "xpm/save.xpm"

#ifndef PACKAGENAME
#define PACKAGENAME "soccerwindow2"
#endif

/*-------------------------------------------------------------------*/
/*!

*/
MainWindow::MainWindow()
    : M_log_player( new LogPlayer( M_main_data, this ) )
    , M_detail_dialog( static_cast< DetailDialog * >( 0 ) )
    , M_player_type_dialog( static_cast< PlayerTypeDialog * >( 0 ) )
    , M_view_config_dialog( static_cast< ViewConfigDialog * >( 0 ) )
    , M_launcher_dialog( static_cast< LauncherDialog * >( 0 ) )
    , M_debug_message_window( static_cast< DebugMessageWindow * >( 0 ) )
    , M_monitor_client( static_cast< MonitorClient * >( 0 ) )
    , M_last_connected_host( "localhost" )
{
    loadSettings();

    createActions();
    createMenus();
    createToolBars();
    createStatusBar();
    createFieldCanvas();
    createViewConfigDialog();

    connect( M_log_player, SIGNAL( updated() ),
             this, SIGNAL( viewUpdated() ) );

    this->setIcon( QPixmap( soccerwindow2_xpm ) );
    this->setCaption( tr( PACKAGENAME ) );
    /*
    this->setDockEnabled( Qt::DockBottom, true );
    this->setDockEnabled( Qt::DockLeft, false );
    this->setDockEnabled( Qt::DockRight, false );
    */
    this->setMinimumSize( 280, 220 );
    this->resize( AppConfig::instance().frameWidth() > 0
                  ? AppConfig::instance().frameWidth()
                  : 640,
                  AppConfig::instance().frameHeight() > 0
                  ? AppConfig::instance().frameHeight()
                  : 480 );
    this->move( AppConfig::instance().framePosX() > 0
                ? AppConfig::instance().framePosX() > 0
                : this->x(),
                AppConfig::instance().framePosY() > 0
                ? AppConfig::instance().framePosY()
                : this->y() );

    // this->setWindowOpacity( 0.5 ); // window transparency

    if ( AppConfig::instance().hideToolBar() )
    {
        M_log_player_tool_bar->hide();
    }

    if ( AppConfig::instance().hideStatusBar() )
    {
        this->statusBar()->hide();
    }

    if ( AppConfig::instance().maximize() )
    {
        this->showMaximized();
    }

    if ( AppConfig::instance().fullScreen() )
    {
        this->showFullScreen();
    }

    if ( AppConfig::instance().hideMenuBar() )
    {
        this->menuBar()->hide();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
MainWindow::~MainWindow()
{
    //std::cerr << "delete MainWindow" << std::endl;

    if ( AppConfig::instance().killServer() )
    {
        killServer();
    }
    saveSettings();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::init()
{
    if ( ! AppConfig::instance().gameLogFilePath().empty() )
    {
        //loadRCG( QString::fromAscii( AppConfig::instance().gameLogFilePath().c_str() ) );
        loadRCG( AppConfig::instance().gameLogFilePath() );
        return;
    }

    if ( AppConfig::instance().connect() )
    {
        std::string host = AppConfig::instance().host();
        if ( host.empty() )
        {
            host = "localhost";
        }

        connectMonitorTo( host.c_str() );

        if ( AppConfig::instance().debugServerMode() )
        {
            startDebugServer();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::loadSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./AppConfig" );

    QString val;

    val = settings.readEntry( "gameLogDir" );
    if ( ! val.isNull()
         && AppConfig::instance().gameLogDir().empty() )
    {
        AppConfig::instance().setGameLogDir( val.ascii() );
    }

    val = settings.readEntry( "debugLogDir" );
    if ( ! val.isNull()
         && AppConfig::instance().debugLogDir().empty() )
    {
        AppConfig::instance().setDebugLogDir( val.ascii() );
    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./AppConfig" );

    settings.writeEntry( "gameLogDir",
                         QString( AppConfig::instance().gameLogDir() ) );
    settings.writeEntry( "debugLogDir",
                         QString( AppConfig::instance().debugLogDir() ) );

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActions()
{
    createActionsFile();
    createActionsMonitor();
    createActionsView();
    createActionsTool();
    createActionsHelp();

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsFile()
{
    M_open_rcg_act = new QAction( QIconSet( QPixmap( open_rcg_xpm ) ),
                                  tr( "&Open rcg file..." ),
                                  QKeySequence(),
                                  this );
    M_open_rcg_act->setAccel( tr( "Ctrl+O" ) );
    M_open_rcg_act->setStatusTip( tr( "Open RoboCup Game Log file" ) );
    connect( M_open_rcg_act, SIGNAL( activated() ), this, SLOT( openRCG() ) );

    M_save_rcg_act = new QAction( QIconSet( QPixmap( save_xpm ) ),
                                  tr( "Save rcg file as..." ),
                                  QKeySequence(),
                                  this );
    M_save_rcg_act->setStatusTip( tr( "Save RoboCup Game Log file" ) );
    connect( M_save_rcg_act, SIGNAL( activated() ), this, SLOT( saveRCG() ) );

    M_open_debug_view_act = new QAction( QIconSet( QPixmap( open_rcg_xpm ) ),
                                         tr( "Open debug view" ),
                                         QKeySequence(),
                                         this );
    M_open_debug_view_act
        ->setStatusTip( tr( "Open the directory where debug view logs exist" ) );
    connect( M_open_debug_view_act, SIGNAL( activated() ),
             this, SLOT( openDebugView() ) );

    M_save_debug_view_act = new QAction( QIconSet( QPixmap( save_xpm ) ),
                                         tr( "Save debug view" ),
                                         QKeySequence(),
                                         this );
    M_save_debug_view_act
        ->setStatusTip( tr( "Save debug view logs to the directory..." ) );
    connect( M_save_debug_view_act, SIGNAL( activated() ),
             this, SLOT( saveDebugView() ) );

    M_exit_act = new QAction( tr( "&Quit" ),
                              QKeySequence(),
                              this );
    M_exit_act->setAccel( tr( "Ctrl+Q" ) );
    M_exit_act->setStatusTip( tr( "Exit the application." ) );
    connect( M_exit_act, SIGNAL( activated() ), this, SLOT( close() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsMonitor()
{
    M_kick_off_act = new QAction( tr( "&KickOff" ),
                                  QKeySequence(),
                                  this );
    M_kick_off_act->setAccel( tr( "Ctrl+K" ) );
    M_kick_off_act->setStatusTip( tr( "Start the game" ) );
    M_kick_off_act->setEnabled( false );
    connect( M_kick_off_act, SIGNAL( activated() ), this, SLOT( kickOff() ) );

    M_set_live_mode_act = new QAction( QIconSet( QPixmap( logplayer_live_mode_xpm ) ),
                                       tr( "&Live Mode" ),
                                       QKeySequence(),
                                       this );
    M_set_live_mode_act->setAccel( tr( "Ctrl+L" ) );
    M_set_live_mode_act->setStatusTip( tr( "set monitor to live mode" ) );
    M_set_live_mode_act->setEnabled( false );
    connect( M_set_live_mode_act, SIGNAL( activated() ),
             this, SLOT( setLiveMode() ) );

    M_connect_monitor_act = new QAction( tr( "&Connect" ),
                                         QKeySequence(),
                                         this );
    M_connect_monitor_act->setAccel( tr( "Ctrl+C" ) );
    M_connect_monitor_act
        ->setStatusTip( "Connect to the rcssserver on localhost" );
    M_connect_monitor_act->setEnabled( true );
    connect( M_connect_monitor_act, SIGNAL( activated() ),
             this, SLOT( connectMonitor() ) );

    M_connect_monitor_to_act = new QAction( tr( "Connect &to ..." ),
                                            QKeySequence(),
                                            this );
    M_connect_monitor_to_act
        ->setStatusTip( tr( "Connect to the rcssserver on other host" ) );
    M_connect_monitor_to_act->setEnabled( true );
    connect( M_connect_monitor_to_act, SIGNAL( activated() ),
             this, SLOT( connectMonitorTo() ) );

    M_disconnect_monitor_act = new QAction( tr( "&Disconnect" ),
                                            QKeySequence(),
                                            this );
    M_disconnect_monitor_act->setStatusTip( tr( "Disonnect from rcssserver" ) );
    M_disconnect_monitor_act->setEnabled( false );
    connect( M_disconnect_monitor_act, SIGNAL( activated() ),
             this, SLOT( disconnectMonitor() ) );

    M_kill_server_act = new QAction( tr( "&Kill server" ),
                                     QKeySequence(),
                                     this );
    M_kill_server_act->setStatusTip( tr( "Kill the rcssserver process" ) );
    M_kill_server_act->setEnabled( false );
    connect( M_kill_server_act, SIGNAL( activated() ),
             this, SLOT( killServer() ) );

    M_restart_server_act = new QAction( tr( "(Re)&start server" ),
                                        QKeySequence(),
                                        this );
    M_restart_server_act->setStatusTip( tr( "(Re)start rcssserver" ) );
    connect( M_restart_server_act, SIGNAL( activated() ),
             this, SLOT( restartServer() ) );

    M_show_launcher_dialog_act = new QAction( tr( "Launcher Dialog" ),
                                              QKeySequence(),
                                              this );
    M_show_launcher_dialog_act->setAccel( tr( "Ctrl+X" ) );
    M_show_launcher_dialog_act->setStatusTip( tr( "Show launcher dialog" ) );
    connect( M_show_launcher_dialog_act, SIGNAL( activated() ),
             this, SLOT( showLauncherDialog() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsView()
{
     M_toggle_menu_bar_act = new QAction( tr( "&Menu Bar" ),
                                          QKeySequence(),
                                          this );
     M_toggle_menu_bar_act->setStatusTip( tr( "Show/Hide Menu Bar" ) );
     M_toggle_menu_bar_act->setAccel( tr( "Ctrl+M" ) );
     connect( M_toggle_menu_bar_act, SIGNAL( activated() ),
              this, SLOT( toggleMenuBar() ) );

    M_toggle_tool_bar_act = new QAction( tr( "&Tool Bar" ),
                                         QKeySequence(),
                                         this );
    M_toggle_tool_bar_act->setStatusTip( tr( "Show/Hide Tool Bar" ) );
    connect( M_toggle_tool_bar_act, SIGNAL( activated() ),
             this, SLOT( toggleToolBar() ) );

    M_toggle_status_bar_act = new QAction( tr( "&Status Bar" ),
                                           QKeySequence(),
                                           this );
    M_toggle_status_bar_act->setStatusTip( tr( "Show/Hide Status Bar" ) );
    connect( M_toggle_status_bar_act, SIGNAL( activated() ),
             this, SLOT( toggleStatusBar() ) );

    M_full_screen_act = new QAction( tr( "&Full Screen" ),
                                     QKeySequence(),
                                     this );
    M_full_screen_act->setAccel( Qt::Key_F11 ); //tr( "Alt+Enter" ) );
    //M_full_screen_act->setAccel( tr( "Alt+Enter" ) );
    //M_full_screen_act->setAccel( Qt::ALT + Qt::Key_Return );
    //M_full_screen_act->setAccel( Qt::ALT + Qt::Key_Enter );
    M_full_screen_act->setStatusTip( tr( "Toggle Full Screen" ) );
    connect( M_full_screen_act, SIGNAL( activated() ),
             this, SLOT( toggleFullScreen() ) );
    // qt4
    //(void) new QShortcut( Qt::ALT + Qt::Key_Enter, this, SLOT( toggleFullScreen() ) );
    //(void) new QShortcut( Qt::ALT + Qt::Key_Return,
    //                      this, SLOT( toggleFullScreen() ) );
    // qt3
    {
        QAccel * accel = new QAccel( this );
        accel->connectItem( accel->insertItem( Qt::ALT + Qt::Key_Return ),
                            this, SLOT( toggleFullScreen() ) );
        accel->connectItem( accel->insertItem( Qt::ALT + Qt::Key_Enter ),
                            this, SLOT( toggleFullScreen() ) );
        //accel->connectItem( accel->insertItem( Qt::Key_F11 ),
        //                    this, SLOT( toggleFullScreen() ) );
    }

    M_show_player_type_dialog_act = new QAction( tr( "&Player Type List" ),
                                                 QKeySequence(),
                                                 this );
    M_show_player_type_dialog_act->setAccel( tr( "Ctrl+T" ) );
    M_show_player_type_dialog_act
        ->setStatusTip( tr( "Show player type parameters dialog" ) );
    connect( M_show_player_type_dialog_act, SIGNAL( activated() ),
             this, SLOT( showPlayerTypeDialog() ) );

    M_show_detail_dialog_act = new QAction( tr( "&Object Detail" ),
                                            QKeySequence(),
                                            this );
    M_show_detail_dialog_act->setAccel( tr( "Ctrl+I" ) );
    M_show_detail_dialog_act
        ->setStatusTip( tr( "Show detail information dialog" ) );
    connect( M_show_detail_dialog_act, SIGNAL( activated() ),
             this, SLOT( showDetailDialog() ) );

    // qt style menu group
    QSignalMapper * style_mapper = new QSignalMapper( this );
    connect( style_mapper, SIGNAL( mapped( const QString & ) ),
             this, SLOT( changeStyle( const QString & ) ) );;
    M_style_act_group = new QActionGroup( this );
    QStringList style_list = QStyleFactory::keys();
    for ( QStringList::iterator it = style_list.begin();
          it != style_list.end();
          ++it )
    {
        QAction * subaction = new QAction( M_style_act_group );
        subaction->setText( tr( "%1 Style" ).arg( *it ) );
        subaction->setToggleAction( true );
        if ( (*it).lower()
             == QString( QApplication::style().name() ).lower() )
        {
            subaction->setOn( true );
        }
        connect( subaction, SIGNAL( activated() ),
                 style_mapper, SLOT( map() ) );
        style_mapper->setMapping( subaction, *it );
    }

    M_show_color_setting_dialog_act = new QAction( tr( "&Color Settings" ),
                                                   QKeySequence(),
                                                   this );
    M_show_color_setting_dialog_act
        ->setStatusTip( tr( "Show color setting dialog" ) );
    connect( M_show_color_setting_dialog_act, SIGNAL( activated() ),
             this, SLOT( showColorSettingDialog() ) );

    M_show_font_setting_dialog_act = new QAction( tr( "&Font Settings" ),
                                                  QKeySequence(),
                                                  this );
    M_show_color_setting_dialog_act
        ->setStatusTip( tr( "Show font setting dialog" ) );
    connect( M_show_font_setting_dialog_act, SIGNAL( activated() ),
             this, SLOT( showFontSettingDialog() ) );

    M_show_view_config_dialog_act = new QAction( tr( "&View Preference" ),
                                                 QKeySequence(),
                                                 this );
    M_show_view_config_dialog_act->setAccel( tr( "Ctrl+V" ) );
    M_show_view_config_dialog_act
        ->setStatusTip( tr( "Show view preference dialog" ) );
    connect( M_show_view_config_dialog_act, SIGNAL( activated() ),
             this, SLOT( showViewConfigDialog() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsTool()
{
    M_show_debug_message_window_act = new QAction( tr( "Debug &Message" ),
                                                   QKeySequence(),
                                                   this );
    M_show_debug_message_window_act->setAccel( tr( "Ctrl+D" ) );
    M_show_debug_message_window_act
        ->setStatusTip( tr( "Show debug message window" ) );
    connect( M_show_debug_message_window_act, SIGNAL( activated() ),
             this, SLOT( showDebugMessageWindow() ) );

    M_toggle_debug_server_mode_act
        = new QAction( QIconSet( QPixmap( debug_server_switch_xpm ) ),
                       tr( "Debug Server Mode" ),
                       QKeySequence(),
                       this );
    M_toggle_debug_server_mode_act->setAccel( tr( "Ctrl+S" ) );
    M_toggle_debug_server_mode_act
        ->setStatusTip( tr( "Start/Stop the debug server." ) );
    M_toggle_debug_server_mode_act->setEnabled( false );
    M_toggle_debug_server_mode_act->setToggleAction( true );
    connect( M_toggle_debug_server_mode_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugServer( bool ) ) );
    M_show_image_save_dialog_act = new QAction( tr( "Save &Image" ),
                                                QKeySequence(),
                                                this );
    M_show_image_save_dialog_act
        ->setStatusTip( tr( "Save game log data as image files" ) );
    connect( M_show_image_save_dialog_act, SIGNAL( activated() ),
             this, SLOT( showImageSaveDialog() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsHelp()
{
    M_about_act = new QAction( QIconSet( QPixmap( soccerwindow2_nostr_xpm ) ),
                               tr( "&About" ),
                               QKeySequence(),
                               this );
    M_about_act->setStatusTip( tr( "Show the about dialog." ) );
    connect( M_about_act, SIGNAL( activated() ), this, SLOT( about() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenus()
{
    createMenuFile();
    createMenuMonitor();
    createMenuView();
    createMenuTool();
    createMenuHelp();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuFile()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();
    M_open_rcg_act->addTo( menu );
    M_save_rcg_act->addTo( menu );
    menu->insertSeparator();
    M_open_debug_view_act->addTo( menu );
    M_save_debug_view_act->addTo( menu );
    menu->insertSeparator();
    M_exit_act->addTo( menu );

    menuBar()->insertItem( tr( "&File" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuMonitor()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();
    M_kick_off_act->addTo( menu );
    M_set_live_mode_act->addTo( menu );
    menu->insertSeparator();
    M_connect_monitor_act->addTo( menu );
    M_connect_monitor_to_act->addTo( menu );
    M_disconnect_monitor_act->addTo( menu );
    menu->insertSeparator();
    M_kill_server_act->addTo( menu );
    M_restart_server_act->addTo( menu );
    menu->insertSeparator();
    M_show_launcher_dialog_act->addTo( menu );

    menuBar()->insertItem( tr( "&Monitor" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuView()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();
    M_toggle_menu_bar_act->addTo( menu );
    M_toggle_tool_bar_act->addTo( menu );
    M_toggle_status_bar_act->addTo( menu );
    menu->insertSeparator();
    M_full_screen_act->addTo( menu );
    menu->insertSeparator();
    M_show_player_type_dialog_act->addTo( menu );
    M_show_detail_dialog_act->addTo( menu );
    menu->insertSeparator();
    {
        QPopupMenu * submenu = new QPopupMenu( this );
        M_style_act_group->addTo( submenu );
        menu->insertItem( tr( "Qt &Style" ), submenu );
    }
    M_show_color_setting_dialog_act->addTo( menu );
    M_show_font_setting_dialog_act->addTo( menu );
    M_show_view_config_dialog_act->addTo( menu );

    menuBar()->insertItem( tr( "&View" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuTool()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();
    M_show_debug_message_window_act->addTo( menu );
    menu->insertSeparator();
    M_toggle_debug_server_mode_act->addTo( menu );
    menu->insertSeparator();
    M_show_image_save_dialog_act->addTo( menu );

    menuBar()->insertItem( tr( "&Tool" ), menu );
}
/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuHelp()
{
    QPopupMenu * menu = new QPopupMenu( this );
    M_about_act->addTo( menu );

    menuBar()->insertItem( tr( "&Help" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createToolBars()
{
    M_log_player_tool_bar = new LogPlayerToolBar( M_log_player,
                                                  M_main_data,
                                                  this );

    M_log_player_tool_bar->addSeparator();

    M_set_live_mode_act->addTo( M_log_player_tool_bar );
    M_toggle_debug_server_mode_act->addTo( M_log_player_tool_bar );

    connect( this, SIGNAL( viewUpdated() ),
             M_log_player_tool_bar, SLOT( updateSlider() ) );


    //this->addToolBar( M_log_player_tool_bar ); // qt4
    //this->addDockWindow( M_log_player_tool_bar, Qt::DockTop ); // qt3
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createStatusBar()
{
    this->statusBar()->message( tr( "Ready" ) );

    M_position_label = new QLabel( tr( "(0.0, 0.0)" ), this->statusBar() );

    int min_width
        = M_position_label->fontMetrics().width(  tr( "(-60.0, -30.0)" ) )
        + 16;
    M_position_label->setMinimumWidth( min_width );
    M_position_label->setAlignment( Qt::AlignRight );

    //this->statusBar()->addPermanentWidget( M_position_label ); // qt4
    this->statusBar()->addWidget( M_position_label, 0, true ); // permanent

    //QSlider * slider = new QSlider( Qt::Horizontal );
    //this->statusBar()->addWidget( slider );

    //QLineEdit * edit = new QLineEdit( tr( "0" ) );
    //edit->setMaximumSize( 36, 20 );
    //this->statusBar()->addWidget( edit );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createFieldCanvas()
{
    M_field_canvas = new FieldCanvas( M_main_data, this );
    //M_field_canvas->setParent( this );
    this->setCentralWidget( M_field_canvas );
    M_field_canvas->setFocus();

    connect( this, SIGNAL( viewUpdated() ),
             M_field_canvas, SLOT( update() ) );
    //       M_field_canvas, SLOT( repaint() ) );

    connect( M_field_canvas, SIGNAL( mouseMoved( const QPoint & ) ),
             this, SLOT( updatePositionLabel( const QPoint & ) ) );

    connect( M_field_canvas, SIGNAL( dropBall( const QPoint & ) ),
             this, SLOT( dropBall( const QPoint & ) ) );
    connect( M_field_canvas, SIGNAL( freeKickLeft( const QPoint & ) ),
             this, SLOT( freeKickLeft( const QPoint & ) ) );
    connect( M_field_canvas, SIGNAL( freeKickRight( const QPoint & ) ),
             this, SLOT( freeKickRight( const QPoint & ) ) );

    // create & set context menus
    {
        QPopupMenu * menu = new QPopupMenu( M_field_canvas );
        M_open_rcg_act->addTo( menu );
        M_connect_monitor_act->addTo( menu );
        M_restart_server_act->addTo( menu );

        M_field_canvas->setNormalMenu( menu );
    }
    {
        QPopupMenu * menu = new QPopupMenu( M_field_canvas );
        M_open_rcg_act->addTo( menu );
        M_connect_monitor_act->addTo( menu );
        menu->insertSeparator();
        M_kill_server_act->addTo( menu );
        M_restart_server_act->addTo( menu );

        M_field_canvas->setSystemMenu( menu );
    }
    {
        QPopupMenu * menu = new QPopupMenu( M_field_canvas );
        M_kick_off_act->addTo( menu );
        menu->insertSeparator();
        menu->insertItem( tr( "Drop Ball" ),
                          M_field_canvas, SLOT( dropBall() ) );
        menu->insertItem( tr( "Free Kick Left" ),
                          M_field_canvas, SLOT( freeKickLeft() ) );
        menu->insertItem( tr( "Free Kick Right" ),
                          M_field_canvas, SLOT( freeKickRight() ) );
        menu->insertSeparator();

        QAction * drop_ball_there = new QAction( tr( "Drop Ball There" ),
                                                 QKeySequence(),
                                                 this );
        drop_ball_there->setStatusTip( tr( "Drop ball at the current ball position." ) );
        connect( drop_ball_there, SIGNAL( activated() ),
                 this, SLOT( dropBallThere() ) );
        drop_ball_there->addTo( menu );

        M_field_canvas->setMonitorMenu( menu );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createViewConfigDialog()
{
    if ( M_view_config_dialog )
    {
        return;
    }

    M_view_config_dialog
        = new ViewConfigDialog( this,
                                M_main_data,
                                M_main_data.getViewConfig() );
    M_view_config_dialog->hide();

    connect( M_view_config_dialog, SIGNAL( configured() ),
             this, SIGNAL( viewUpdated() ) );

    connect( M_view_config_dialog, SIGNAL( canvasResized( const QSize & ) ),
             this, SLOT( resizeCanvas( const QSize & ) ) );

    connect( M_field_canvas, SIGNAL( focusChanged( const QPoint & ) ),
             M_view_config_dialog, SLOT( setFocusPoint( const QPoint & ) ) );

    {
        // register short cut keys
        QAccel * accel = new QAccel( this );

        // zoom
        // z
        accel->connectItem( accel->insertItem( Qt::Key_Z ),
                            M_view_config_dialog, SLOT( zoomIn() ) );
        // +
        accel->connectItem( accel->insertItem( Qt::Key_Plus ),
                            M_view_config_dialog, SLOT( zoomIn() ) );
        // x
        accel->connectItem( accel->insertItem( Qt::Key_X ),
                            M_view_config_dialog, SLOT( zoomOut() ) );
        // Ctrl + +
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_Plus ),
                            M_view_config_dialog, SLOT( zoomOut() ) );
        // Ctrl + z
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_Z ),
                            M_view_config_dialog, SLOT( zoomOut() ) );
        // i
        accel->connectItem( accel->insertItem( Qt::Key_I ),
                            M_view_config_dialog, SLOT( unzoom() ) );
        // e
        accel->connectItem( accel->insertItem( Qt::Key_E ),
                            M_view_config_dialog, SLOT( toggleEnlarge() ) );

        // reverse mode
        // Ctrl + r
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_R ),
                            M_view_config_dialog, SLOT( toggleReverseSide() ) );
        // Ctrl + Shift + r
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::SHIFT + Qt::Key_R ),
                            M_view_config_dialog, SLOT( togglePlayerReverseDraw() ) );

        // player detail
        // n
        accel->connectItem( accel->insertItem( Qt::Key_N ),
                            M_view_config_dialog, SLOT( toggleShowPlayerNumber() ) );
        // h
        accel->connectItem( accel->insertItem( Qt::Key_H ),
                            M_view_config_dialog, SLOT( toggleShowHeteroNumber() ) );
        // s
        accel->connectItem( accel->insertItem( Qt::Key_S ),
                            M_view_config_dialog, SLOT( toggleShowStamina() ) );
        // v
        accel->connectItem( accel->insertItem( Qt::Key_V ),
                            M_view_config_dialog, SLOT( toggleShowViewCone() ) );
        // c
        accel->connectItem( accel->insertItem( Qt::Key_C ),
                            M_view_config_dialog, SLOT( toggleShowControlArea() ) );

        // show/hide
        // t
        accel->connectItem( accel->insertItem( Qt::Key_T ),
                            M_view_config_dialog, SLOT( toggleShowScoreBoard() ) );
        // Ctrl + b
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_B ),
                            M_view_config_dialog, SLOT( toggleShowBall() ) );
        // Ctrl + p
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_P ),
                            M_view_config_dialog, SLOT( toggleShowPlayers() ) );
        // f
        accel->connectItem( accel->insertItem( Qt::Key_F ),
                            M_view_config_dialog, SLOT( toggleShowFlags() ) );
        // o
        accel->connectItem( accel->insertItem( Qt::Key_O ),
                            M_view_config_dialog, SLOT( toggleShowOffsideLine() ) );

        // grass type
        // k
        accel->connectItem( accel->insertItem( Qt::Key_K ),
                            M_view_config_dialog, SLOT( toggleKeepawayMode() ) );

        // focus
        // b
        accel->connectItem( accel->insertItem( Qt::Key_B ),
                            M_view_config_dialog, SLOT( toggleFocusBall() ) );
        // p
        accel->connectItem( accel->insertItem( Qt::Key_P ),
                            M_view_config_dialog, SLOT( toggleFocusPlayer() ) );

        // player selection
        // a
        accel->connectItem( accel->insertItem( Qt::Key_A ),
                            M_view_config_dialog, SLOT( toggleSelectAutoAll() ) );
        // l
        accel->connectItem( accel->insertItem( Qt::Key_L ),
                            M_view_config_dialog, SLOT( toggleSelectAutoLeft() ) );
        // r
        accel->connectItem( accel->insertItem( Qt::Key_R ),
                            M_view_config_dialog, SLOT( toggleSelectAutoRight() ) );
        // u
        accel->connectItem( accel->insertItem( Qt::Key_U ),
                            M_view_config_dialog, SLOT( setUnselect() ) );
    }

    // number keys & minus
    {
        QAccel * select_accel = new QAccel( this );
        // number 1-9
        for ( int i = 1; i <= 9; ++i )
        {
            select_accel->insertItem( Qt::Key_0 + i, i );
            select_accel->insertItem( Qt::CTRL + Qt::Key_0 + i, 11 + i );
        }
        // number 10
        select_accel->insertItem( Qt::Key_0, 10 );
        select_accel->insertItem( Qt::CTRL + Qt::Key_0, 11 + 10 );
        // number 11
        select_accel->insertItem( Qt::Key_Minus, 11 );
        select_accel->insertItem( Qt::CTRL + Qt::Key_Minus, 11 + 11 );

        connect( select_accel, SIGNAL( activated( int ) ),
                 M_view_config_dialog, SLOT( selectPlayer( int ) ) );

    }
}

/*-------------------------------------------------------------------*/
/*!

*/
// void
// MainWindow::closeEvent( QCloseEvent * event )
// {
//     event->ignore();

//     qApp->quit();
// }

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::resizeEvent( QResizeEvent * event )
{
    //event->accept(); qt4

    if ( M_view_config_dialog
         && M_view_config_dialog->isVisible() )
    {
        M_view_config_dialog->updateFieldScale();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::wheelEvent( QWheelEvent * event )
{
    if ( event->delta() < 0 )
    {
        M_log_player->stepForward();
    }
    else
    {
        M_log_player->stepBack();
    }

    event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::openRCG()
{
    //std::cerr << "MainWindow::openRCG()" << std::endl;

#ifdef HAVE_LIBRCSC_GZ
    QString filter( tr( "Game Log files (*.rcg *.rcg.gz);;"
                        "All files (*)" ) );
#else
    QString filter( tr( "Game Log files (*.rcg);;"
                        "All files (*)" ) );
#endif

    QString default_dir( AppConfig::instance().gameLogDir().c_str() );

    QString file_path = QFileDialog::getOpenFileName( default_dir,
                                                      filter,
                                                      this,
                                                      tr( "open file dialog" ),
                                                      tr( "Choose a game log file to open" ) );

    if ( file_path.isEmpty() )
    {
        //std::cerr << "MainWindow::opneRCG() empty file path" << std::endl;
        return;
    }

    std::cerr << "selected file = [" << file_path.ascii() << std::endl;

    loadRCG( file_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::loadRCG( const QString & file_path )
{
    disconnectMonitor();
    M_log_player->stop();

    if ( ! M_main_data.loadRCG( std::string( file_path.ascii() ) ) )
    {
        QString err_msg = tr( "Failed to read [" );
        err_msg += file_path;
        err_msg += tr( "]" );
        QMessageBox::critical( this,
                               tr( "Error" ),
                               err_msg,
                               QMessageBox::Ok, QMessageBox::NoButton );
        return;
    }

    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QString err_msg = tr( "Empty log file [" );
        err_msg += file_path;
        err_msg += tr( "]" );
        QMessageBox::critical( this,
                               tr( "Error" ),
                               err_msg,
                               QMessageBox::Ok, QMessageBox::NoButton );
        return;
    }

    QFileInfo file_info( file_path );

    AppConfig::instance().setGameLogDir( file_info.absFilePath().ascii() );

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->updateData();
    }

    if ( M_debug_message_window )
    {
        M_debug_message_window->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->unzoom();
    }

    emit viewUpdated();

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveRCG()
{
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "No Monitor View Data!" ) );
        return;
    }

    QString default_file_name;
    {
        const MonitorViewConstPtr latest = M_main_data.viewHolder().latestViewData();

        if ( latest )
        {
            default_file_name
                = QDateTime::currentDateTime().toString( "yyyyMMddhhmm-" );

            QString left_team( latest->leftTeam().name() );
            QString left_score = QString::number( latest->leftTeam().score() );

            QString right_team( latest->rightTeam().name() );
            QString right_score = QString::number( latest->rightTeam().score() );

            default_file_name += left_team;
            default_file_name += tr( "_" );
            default_file_name += left_score;
            default_file_name += tr( "-" );
            default_file_name += right_team;
            default_file_name += tr( "_" );
            default_file_name += right_score;

            default_file_name += tr( ".rcg" );
        }
    }

#ifdef HAVE_LIBRCSC_GZ
    QString filter( tr( "Game Log files (*.rcg *.rcg.gz);;"
                        "All files (*)" ) );
#else
    QString filter( tr( "Game Log files (*.rcg);;"
                        "All files (*)" ) );
#endif

    QString default_dir( AppConfig::instance().gameLogDir() );
    if ( ! default_file_name.isEmpty() )
    {
        default_dir += tr( "/" );
        default_dir += default_file_name;
    }

    QString file_path = QFileDialog::getSaveFileName( default_dir,
                                                      filter,
                                                      this,
                                                      tr( "open file dialog" ),
                                                      tr( "Save a game log file as" ) );

    if ( file_path.isEmpty() )
    {
        //std::cerr << "MainWindow::saveRCG() empty file path" << std::endl;
        return;
    }

    std::string file_path_std = file_path.ascii();

    std::cerr << "save game log data to the file = [" << file_path_std
              << std::endl;

    // update game log dir
    QFileInfo file_info( file_path );
    AppConfig::instance().setGameLogDir( file_info.absFilePath().ascii() );

    // check gzip usability
    bool is_gzip = false;
    if ( file_path_std.length() > 3
         && file_path_std.compare( file_path_std.length() - 3, 3, ".gz" ) == 0 )
    {
#ifdef HAVE_LIBRCSC_GZ
        if ( file_path_std.length() <= 7
             || file_path_std.compare( file_path_std.length() - 4, 4, ".rcg.gz" ) != 0 )
        {
            file_path_std == ".rcg.gz";
        }
        is_gzip = true;
#else
        // erase '.gz'
        file_path_std.erase( file_path_std.length() - 3 );
#endif
    }

    // check the extention string
    if ( ! is_gzip )
    {
        if ( file_path_std.length() <= 4
             || file_path_std.compare( file_path_std.length() - 4, 4, ".rcg" ) != 0 )
        {
            file_path_std += ".rcg";
        }
    }

    M_main_data.saveRCG( file_path_std );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::openDebugView()
{
    //std::cerr << "MainWindow::openDebugView()" << std::endl;

    QString default_dir( AppConfig::instance().debugLogDir() );
#if 1
    QString dir_path
        = QFileDialog::getExistingDirectory( default_dir,
                                             this,
                                             "get existing directory",
                                             tr( "Choose a debug view log directory" ),
                                             true ); // dir only

#else
    DirSelector selector( this,
                          tr( "Choose a debug view log directory" ),
                          default_dir );

    if ( ! selector.exec() )
    {
        //std::cerr << "MainWindow::openDebugView() canceled" << std::endl;
        return;
    }

    QString dir_path = selector.dirPath();
#endif
    if ( dir_path.isEmpty() )
    {
        //std::cerr << "MainWindow::openDebugView() empty dir path" << std::endl;
        return;
    }

    std::cerr << "open debug view. dir = [" << dir_path.ascii() << std::endl;

    M_main_data.getViewHolder().loadDebugView( dir_path.ascii() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveDebugView()
{
    //std::cerr << "MainWindow::saveDebugView()" << std::endl;
#if 1
    QString dir_path
        = QFileDialog::getExistingDirectory( QString::null, // means current path
                                             this,
                                             "get existing directory",
                                             tr( "Choose a directory to save a debug view logs" ),
                                             true ); // dir only

#else
    DirSelector selector( this,
                          tr( "Choose a debug view log directory" ),
                          QDir::current().absPath() );

    if ( ! selector.exec() )
    {
        //std::cerr << "MainWindow::openDebugView() canceled" << std::endl;
        return;
    }

    QString dir_path = selector.dirPath();
#endif
    if ( dir_path.isEmpty() )
    {
        //std::cerr << "MainWindow::openDebugView() empty dir path" << std::endl;
        return;
    }

    std::cerr << "save debug view. dir = [" << dir_path.ascii() << std::endl;


    M_main_data.viewHolder().saveDebugView( dir_path.ascii() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::kickOff()
{
    //std::cerr << "MainWindow::kickOff()" << std::endl;
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_monitor_client->sendKickOff();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::setLiveMode()
{
    //std::cerr << "MainWindow::setLiveMode()" << std::endl;

    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_log_player->setLiveMode();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::connectMonitor()
{
    //std::cerr << "MainWindow::connectMonitor()" << std::endl;
    connectMonitorTo( "localhost" );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::connectMonitorTo()
{
    //std::cerr << "MainWindow::connectMonitorTo()" << std::endl;

    bool ok = true;
    QString text = QInputDialog::getText( tr( "Input sserver host name" ),
                                          tr( "Host name: "),
                                          QLineEdit::Normal,
                                          M_last_connected_host,
                                          & ok,
                                          this );
    if ( ok
         && ! text.isEmpty() )
    {
        connectMonitorTo( text.ascii() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::connectMonitorTo( const char * hostname )
{
    if ( std::strlen( hostname ) == 0 )
    {
        std::cerr << "Empty host name! Connection failed!" << std::endl;
        return;
    }

    std::cerr << "Connect to rcssserver on [" << hostname << "]" << std::endl;

    M_monitor_client
        = boost::shared_ptr< MonitorClient >
        ( new MonitorClient( M_main_data.getViewHolder(),
                             hostname,
                             AppConfig::instance().port(),
                             AppConfig::instance().clientVersion() ) );

    if ( ! M_monitor_client->isConnected() )
    {
        std::cerr << "Conenction failed." << std::endl;
        M_monitor_client.reset();
        return;
    }

    M_last_connected_host = hostname;

    // reset all data
    M_main_data.clear();

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->hide();
    }

    if ( M_debug_message_window )
    {
        M_debug_message_window->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->unzoom();
    }

    if ( M_toggle_debug_server_mode_act->isOn() )
    {
        stopDebugServer();
        startDebugServer();
    }

    AppConfig::instance().setMonitorClientMode( true );

    M_save_rcg_act->setEnabled( false );

    M_kick_off_act->setEnabled( true );
    M_set_live_mode_act->setEnabled( true );
    M_connect_monitor_act->setEnabled( false );
    M_connect_monitor_to_act->setEnabled( false );
    M_disconnect_monitor_act->setEnabled( true );
    M_kill_server_act->setEnabled( true );

    M_toggle_debug_server_mode_act->setEnabled( true );
    M_show_image_save_dialog_act->setEnabled( false );

    connect( M_monitor_client.get(), SIGNAL( received() ),
             this, SLOT( receiveMonitorPacket() ) );
    connect( M_monitor_client.get(), SIGNAL( timeout() ),
             this, SLOT( disconnectMonitor() ) );

    M_log_player->setLiveMode();

    // this method will be called automatically.
    //M_monitor_client->sendDispInit();

    if ( QApplication::overrideCursor() )
    {
        QApplication::restoreOverrideCursor();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::disconnectMonitor()
{
    //std::cerr << "MainWindow::disconnectMonitor()" << std::endl;

    if ( M_monitor_client )
    {
        M_monitor_client->disconnect();

        disconnect( M_monitor_client.get(), SIGNAL( received() ),
                    this, SLOT( receiveMonitorPacket() ) );

        disconnect( M_monitor_client.get(), SIGNAL( timeout() ),
                    this, SLOT( disconnectMonitor() ) );

        M_monitor_client.reset();
    }

    if ( M_debug_server )
    {
        M_debug_server.reset();
    }

    AppConfig::instance().setMonitorClientMode( false );

    M_save_rcg_act->setEnabled( true );

    M_kick_off_act->setEnabled( false );
    M_set_live_mode_act->setEnabled( false );
    M_connect_monitor_act->setEnabled( true );
    M_connect_monitor_to_act->setEnabled( true );
    M_disconnect_monitor_act->setEnabled( false );
    M_kill_server_act->setEnabled( false );

    M_toggle_debug_server_mode_act->setEnabled( false );
    M_show_image_save_dialog_act->setEnabled( true );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::killServer()
{
    //std::cerr << "MainWindow::killServer()" << std::endl;

    disconnectMonitor();

    AppConfig::instance().setKillServer( false );

    if ( AppConfig::instance().serverPID() != 0 )
    {
        AppConfig::instance().setServerPID( 0 );
        ::kill( pid_t( AppConfig::instance().serverPID() ), SIGINT );
    }
    else
    {
#if 1
        std::system( "killall -s INT rcssserver" );
#else
        // qt4
        //QString command( "killall -s INT rcssserver" );
        //QProcess::execute( command );
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::startServer()
{
    AppConfig::instance().setServerPID( 0 );
    AppConfig::instance().setKillServer( true );

    std::string server_command;
    if ( M_server_command.isEmpty() )
    {
        server_command = AppConfig::instance().serverPath();
    }
    else
    {
        server_command = M_server_command.ascii();
        M_server_command.setLength( 0 );
    }

#if 1
    //command += " > /dev/null 2>&1 &";
    server_command += " &";
    std::system( server_command.c_str() );
#else
    QProcess * process = new QProcess( server_command, this );
    process->start();
#endif
    if ( ! QApplication::overrideCursor() )
    {
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
    }

    QTimer::singleShot( 1000,
                        this, SLOT( connectMonitor() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::restartServer()
{
    restartServer( AppConfig::instance().serverPath() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::restartServer( const QString & command )
{
    static bool s_last_auto_start = false;

    M_server_command = command;

    bool auto_start = false;
    if ( command.find( "server::team_l_start" ) != -1 )
    {
        auto_start = true;
    }

    if ( M_monitor_client )
    {
        killServer();

        if ( ! QApplication::overrideCursor() )
        {
            QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
        }

        int timer_msec = 1000;
        if ( s_last_auto_start )
        {
            timer_msec = 3000;
        }

        QTimer::singleShot( timer_msec,
                            this, SLOT( startServer() ) );
    }
    else
    {
        startServer();
    }

    s_last_auto_start = auto_start;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showLauncherDialog()
{
    if ( M_launcher_dialog )
    {
        if ( M_launcher_dialog->isVisible() )
        {
            M_launcher_dialog->hide();
        }
        else
        {
            M_launcher_dialog->show();
        }
    }
    else
    {
        M_launcher_dialog = new LauncherDialog( this );
        M_launcher_dialog->show();

        //QSize size = M_launcher_dialog->size();
        //M_launcher_dialog->setMinimumSize( size );
        //M_launcher_dialog->setMaximumSize( size );
        QSize size = M_launcher_dialog->size();
        M_launcher_dialog->setMinimumSize( size );
        M_launcher_dialog->setMaximumSize( 1024, size.height() );

        connect( M_launcher_dialog, SIGNAL( launchServer( const QString & ) ),
                 this, SLOT( restartServer( const QString & ) ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleMenuBar()
{
    if ( this->menuBar()->isVisible() )
    {
        this->menuBar()->hide();
    }
    else
    {
        this->menuBar()->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleToolBar()
{
    if ( M_log_player_tool_bar->isVisible() )
    {
        M_log_player_tool_bar->hide();
    }
    else
    {
        M_log_player_tool_bar->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleStatusBar()
{
    if ( this->statusBar()->isVisible() )
    {
        this->statusBar()->hide();
    }
    else
    {
        this->statusBar()->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleFullScreen()
{
    if ( this->isFullScreen() )
    {
        this->showNormal();
    }
    else
    {
        this->showFullScreen();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showPlayerTypeDialog()
{
    if ( M_player_type_dialog )
    {
        if ( M_player_type_dialog->isVisible() )
        {
            M_player_type_dialog->hide();
        }
        else
        {
            M_player_type_dialog->show();
        }
    }
    else
    {
        M_player_type_dialog = new PlayerTypeDialog( this, M_main_data );
        M_player_type_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showDetailDialog()
{
    //std::cerr << "MainWindow::showDetailDialog()" << std::endl;

    if ( M_detail_dialog )
    {
        if ( M_detail_dialog->isVisible() )
        {
            M_detail_dialog->hide();
        }
        else
        {
            M_detail_dialog->show();
        }
    }
    else
    {
        M_detail_dialog = new DetailDialog( this, M_main_data );
        connect( this,  SIGNAL( viewUpdated() ),
                 M_detail_dialog, SLOT( updateLabels() ) );

        M_detail_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::changeStyle( const QString & style )
{
    QApplication::setStyle( style );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showColorSettingDialog()
{
    ColorSettingDialog dlg( this );

    connect( &dlg, SIGNAL( colorChanged() ),
             M_field_canvas, SLOT( update() ) );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showFontSettingDialog()
{
    FontSettingDialog dlg( this );

    connect( &dlg, SIGNAL( fontChanged() ),
             M_field_canvas, SLOT( update() ) );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showViewConfigDialog()
{
    if ( M_view_config_dialog->isVisible() )
    {
        M_view_config_dialog->hide();
    }
    else
    {
        M_view_config_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showDebugMessageWindow()
{
    //std::cerr << "MainWindow::showDebugMessageWindow()" << std::endl;

    if ( M_debug_message_window )
    {
        if ( M_debug_message_window->isVisible() )
        {
            M_debug_message_window->hide();
        }
        else
        {
            M_debug_message_window->show();
        }
    }
    else
    {
        M_debug_message_window = new DebugMessageWindow( this,
                                                         M_main_data );
        connect( M_debug_message_window, SIGNAL( configured() ),
                 this, SIGNAL( viewUpdated() ) );

        M_debug_message_window->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleDebugServer( bool on )
{
    if ( on )
    {
        startDebugServer();
    }
    else
    {
        stopDebugServer();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::startDebugServer()
{
    //std::cerr << "MainWindow::startDebugServer()" << std::endl;

    if ( M_debug_server )
    {
        // the server instance has already existed.
        return;
    }

    std::cerr << "Start Debug Server" << std::endl;

    int port = static_cast< int >( AppConfig::instance().debugServerPort() );

    M_debug_server
        = boost::shared_ptr< DebugServer >
        ( new DebugServer( M_main_data.getViewHolder(), port ) );

    if ( ! M_debug_server->isConnected() )
    {
        std::cerr << "failed to create Debug Server" << std::endl;
        M_debug_server.reset();
        return;
    }

    M_toggle_debug_server_mode_act->setOn( true );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::stopDebugServer()
{
    //std::cerr << "MainWindow::stopDebugServer()" << std::endl;

    if ( M_debug_server )
    {
        //std::cerr << "Stop Debug Server" << std::endl;
        M_debug_server.reset();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showImageSaveDialog()
{
    //std::cerr << "MainWindow::showImageSaveDialog()" << std::endl;

    M_log_player->stop();

    ImageSaveDialog dlg( this,
                         M_field_canvas,
                         M_main_data );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::about()
{
    //QString msg( "soccerwindow2" ); //QString msg( tr( PACKAGE ) );
    //msg += tr( "-" );
    //msg += tr( "2.0.0" ); //msg += tr( VERSION );
    QString msg( tr( PACKAGE "-" VERSION ) );
    msg += tr( "\n\n" );
    msg += tr( "soccerwindow2 is a viewer applicaton for\n"
               "the RoboCup Soccer Simulator.\n"
               "  http://sserver.sourceforge.net/\n"
               "\n"
               "soccerwindow2 Development Site:\n"
               "  http://rctools.sourceforge.jp/\n"
               "Author:\n"
               "  Hidehisa AKIYAMA <akky@users.sourceforge.jp>" );

    QMessageBox::about( this,
                        tr( "About soccerwindow2" ),
                        msg );

    // from Qt 4.1 documents
    /*
      about() looks for a suitable icon in four locations:

      1. It prefers parent->icon() if that exists.
      2. If not, it tries the top-level widget containing parent.
      3. If that fails, it tries the active window.
      4. As a last resort it uses the Information icon.

      The about box has a single button labelled "OK".
    */
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::resizeCanvas( const QSize & size )
{
    if ( size.width() < this->minimumWidth()
         || size.height() < this->minimumHeight() )
    {
        std::cerr << "Too small canvas size ("
                  << size.width() << " "
                  << size.height() << ")"
                  << std::endl;
        return;
    }

    if ( centralWidget() )
    {
        if ( this->isMaximized()
             || this->isFullScreen() )
        {
            this->showNormal();
        }

        QRect rect = this->geometry();

        int width_diff = rect.width() - centralWidget()->width();
        int height_diff = rect.height() - centralWidget()->height();

        rect.setWidth( size.width() + width_diff );
        rect.setHeight( size.height() + height_diff );

        this->setGeometry( rect );

        //std::cerr << "centralWidget width = " << centralWidget()->width()
        //          << " height = " << centralWidget()->height()
        //          << std::endl;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::receiveMonitorPacket()
{
    //std::cerr << "receive monitor packet" << std::endl;

    if ( M_log_player->isLiveMode() )
    {
        M_log_player->showLive();
    }
    else
    {
        //std::cerr << "receive monitor packet  no live" << std::endl;
        M_log_player_tool_bar->updateSlider();
    }


    if ( AppConfig::instance().autoQuitMode() )
    {
        if ( M_main_data.viewHolder().lastPlayMode() == rcsc::PM_TimeOver )
        {
            static QDateTime s_game_end_time;

            if ( ! s_game_end_time.isValid() )
            {
                s_game_end_time = QDateTime::currentDateTime();
            }
            else
            {
                if ( s_game_end_time.secsTo( QDateTime::currentDateTime() )
                     >= AppConfig::instance().waitSeconds() )
                {
                    std::cerr << "Elapsed " << AppConfig::instance().waitSeconds()
                              << " seconds after game end\n"
                              << "Exit..."
                              << std::endl;
                    qApp->quit();
                }
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::updatePositionLabel( const QPoint & point )
{
    if ( M_position_label
         && statusBar()
         && statusBar()->isVisible()
         )
    {
        double x = M_main_data.viewConfig().fieldX( point.x() );
        double y = M_main_data.viewConfig().fieldY( point.y() );

        char buf[32];
        std::snprintf( buf, 32,
                       "(%.2f, %.2f)",
                       x, y );

        M_position_label->setText( QString::fromAscii( buf ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::dropBallThere()
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        MonitorViewConstPtr view = M_main_data.viewHolder().latestViewData();
        if ( view )
        {
            //std::cerr << "drop ball at current position "
            //          << std::endl;
            M_monitor_client->sendDropBall( view->ball().x(),
                                            view->ball().y() );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::dropBall( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().fieldX( point.x() );
        double y = M_main_data.viewConfig().fieldY( point.y() );

        std::cerr << "drop ball to ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendDropBall( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::freeKickLeft( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().fieldX( point.x() );
        double y = M_main_data.viewConfig().fieldY( point.y() );

        std::cerr << "free kick left at ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendFreeKickLeft( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::freeKickRight( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().fieldX( point.x() );
        double y = M_main_data.viewConfig().fieldY( point.y() );

        std::cerr << "free kick right at ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendFreeKickRight( x, y );
    }
}
