// -*-c++-*-

/*!
  \file udp_socket.h
  \brief UDP connection socket class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_NET_UDP_SOCKET_H
#define RCSC_NET_UDP_SOCKET_H

#include <boost/scoped_ptr.hpp>

namespace rcsc {

struct AddrImpl;

/*!
  \class UDPSocket
  \brief UDP/IP connection socket class
*/
class UDPSocket {
private:
    //! socket file descriptor
    int M_fd;

    //! destination address
    boost::scoped_ptr< AddrImpl > M_dest;

    //! not used
    UDPSocket();
public:
    /*!
      \brief constructor for server socket
      \param port port number to receive packet.
    */
    explicit
    UDPSocket( const int port );

    /*!
      \brief constructor for client socket
      \param hostname remote host name (or IP address)
      \param port port number to send packet
     */
    UDPSocket( const char * hostname,
               const int port );

    /*!
      \brief destructor. close socket automatically
     */
    ~UDPSocket();

    /*!
      \brief returns socket file descriptor
    */
    int fd() const
      {
          return M_fd;
      }

private:

    /*!
      \brief open socket
      \return value of close(fd) if socket is opened..
     */
    bool open();

    /*!
      \brief bind the socket to local address
      \param port port number to be binded.
      \return true if successfully binded.
    */
    bool bind( const int port = 0 );

    /*!
      \brief set the address info of the specified remote host.
      \param hostname the name of remote host.(or IP address)
      \param port port number of remote host.
      \return true if generated binary address of remote host.
    */
    bool setAddr( const char * hostname,
                  const int port );

    /*
      void connect()
      {

      if ( ::connect( fd(),
      reinterpret_cast< const struct sockaddr *>( &M_dest ),
      sizeof( M_dest ) ) < 0 )
      {
      std::cerr << "connect error" << std::endl;
      }
      }
    */

    /*!
      \brief set non blocking mode.
      \return returned value of fcntl()
    */
    int setNonBlocking();

public:

    /*!
      \brief check if socket is opened or not.
      \return true if socket has the valid file descripter.
     */
    bool isOpen() const
      {
          return fd() != -1;
      }

    /*!
      \brief close socket
      \return value that close(fd) returns if socket is opened, otherwise 0.
     */
    int close();

    /*!
      \brief send message to the connected host.
      \param msg the pointer to the message to be sent.
      \param len the length of message.
      \return the length of sent message if successfuly sent, otherwise -1.
     */
    int send( const char * msg,
              const int len );

    /*!
      \brief receive message from the connected remote host.
      \param buf message buffer
      \param len maximal length of buffer buf
      \retval 0 error occured and errno is EWOULDBLOCK
      \retval -1 error occured
      \return the length of received message.
     */
    int receive( char * buf,
                 std::size_t len );

};

} // end namespace

#endif
