#pragma once

// CStepDebugManager
//
//   Save or Load the world model of one loop to enable step debug. Singleton class.
//	 Created by Huqian 2007-09-02   

#include <memory>
#include <iostream>
#include <map>
#include "rcsc/player/world_model.h"
#include <rcsc/player/player_agent.h>
#include "StepDebug_Define.h"

using namespace std;


class CStepDebugManager
{
public:
	~CStepDebugManager(void);
	static CStepDebugManager* Instance();		// Get Instance

	enum TYPE{ 
		ONLINE_WRITE	= 0,
		OFFLINE_READ	= 1,
		DISABLE			= 2
	};

	bool Initialize( rcsc::PlayerAgent*	pAgent, TYPE t = ONLINE_WRITE );	// This should be done in main() 
																				// after agent get the num
	TYPE GetType();
	void SetType( TYPE );


	// ONLINE_WRITE functions

	bool Online_Write();								// write world model to file
	bool Write2StepDebugFile();							// this will be called in sub thread
	void CloseThread();									// close thread
	bool IsCloseThread();								// checked by thread
	inline bool Lock(){return _cs.Lock() == TRUE;};		// enter CCriticalSection
	inline bool Unlock(){return _cs.Unlock() == TRUE;};	// leave CCriticalSection


	// OFFLINE_READ functions
	
	const char* Load_By_Index( int iIndex );										// Load one loop data at index
	const STEPDEBUG_INDEX* GetIndexBlock( int iIndex );								// Load one index block at index
	bool GetIndexLoopMap( std::map< int, STEPDEBUG_INDEX >  **ppIndexLoopMap );		// Get all index blocks
	void agent_action( const char* data );											// load one loop data and run agent to make decision
	void ShowDecision( const std::string& str );									// show or save decision, called by agent
	void OutputAllDecision();														// run all loops and save decision to file
	inline const char* GetDecision(){return _strDecision.c_str();};					// get current decision
	int		_iCommandList_PlayerIndex;												// the player index that links to Commandlist

protected:

    CStepDebugManager(void);
    friend class auto_ptr<CStepDebugManager>;
    static auto_ptr<CStepDebugManager>			_instance;

	TYPE										_type;
	rcsc::PlayerAgent*							_pAgent;

	FILE*										_pIndexFile;	// record the position of data file, loop and index
	FILE*										_pDataFile;

	// OFFLINE_READ variables

	std::map< int, STEPDEBUG_INDEX >			_IndexLoopMap;	// index -> loop
	char*										_pData;
	int											_iCurrentIndex;
	std::string									_logfilepath;
	std::string									_strDecision;
	bool										_bOutputing;	// output all decisions
	FILE*										_pDecisionFile;	// used to output all decisions

	// ONLINE_WRITE variables

	bool										_bCloseThread;
	CCriticalSection							_cs;
	static void Thread_Online_Write( void *p );



};



#define HTRACE_TIME( str )								CTimeWatch hqtw( __FILE__, __LINE__, 0,0, str );
#define HTRACE_TIME_EX( lParam, wParam, str )			CTimeWatch hqtw( __FILE__, __LINE__, lParam, wParam, str );

// Trace time
class CTimeWatch  
{
public:
	CTimeWatch( const char* pszFile, int iLine, unsigned long lParam, unsigned long wParam, const char* lpszStr );
	virtual ~CTimeWatch();

	void RecordCurrentTime( const char* pszDescription = NULL );

protected:
	LARGE_INTEGER	m_liStart;
	LARGE_INTEGER	m_liStop;
	LONGLONG		m_llFrequency;

	const char*		m_pszFile;
	const char*		m_pszStr;
	int				m_iLine;
	unsigned long	m_lParam;
	unsigned long	m_wParam;


};


inline CTimeWatch::CTimeWatch( const char* pszFile, int iLine, unsigned long lParam, unsigned long wParam, const char* lpszStr )
	: m_pszFile(pszFile), m_pszStr(lpszStr), m_lParam(lParam), m_wParam(wParam), m_iLine(iLine)
{
	LARGE_INTEGER liFrequency;

	::QueryPerformanceFrequency(&liFrequency);
	m_llFrequency = liFrequency.QuadPart;

	m_liStart.QuadPart = 0;
	m_liStop.QuadPart = 0;

	::QueryPerformanceCounter(&m_liStart);
	m_liStop = m_liStart;
}

inline CTimeWatch::~CTimeWatch()
{
	RecordCurrentTime();
}

inline void CTimeWatch::RecordCurrentTime( const char* pszDescription )
{
	::QueryPerformanceCounter(&m_liStop);

	char szTime[512]={0};
	if( pszDescription == NULL )
		sprintf( szTime, "%s(cost: %8.3fms)",m_pszStr,( (m_liStop.QuadPart - m_liStart.QuadPart) * 1000.0)/ m_llFrequency );
	else
		sprintf( szTime, "%s(cost: %8.3fms) -> %s",m_pszStr,( (m_liStop.QuadPart - m_liStart.QuadPart) * 1000.0)/ m_llFrequency, pszDescription );


//	HTrace( m_pszFile, m_iLine, m_lParam, m_wParam, szTime );
	std::cout<< m_pszFile << m_iLine << szTime << std::endl;

	::QueryPerformanceCounter(&m_liStart);
}