// -*-c++-*-

/*!
  \file debug_log_holder.cpp
  \brief debug data repository Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "debug_log_holder.h"

#include "debug_log_data.h"

#include <sstream>
#include <iostream>
#include <cstdio>

namespace {
const int MAX_SPAN = 2;
}

const std::string DebugLogHolder::DEFAULT_EXTENSION = ".log";

/*-------------------------------------------------------------------*/
/*!

*/
DebugLogHolder::~DebugLogHolder()
{
    clear();
    //std::cerr << "delete DebugLogHolder" << std::endl;
}


/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogHolder::clear()
{
    for ( int i = 1; i <= 11; ++i )
    {
        M_player_data[i-1].clear();
    }
}

/*-------------------------------------------------------------------*/
/*!
  \return if available file is found more then one, returun true.
*/
bool
DebugLogHolder::setDir( const std::string & team_name,
                        const rcsc::SideID side,
                        const int unum,
                        const std::string & dirpath )
{
    if ( team_name.empty() )
    {
        std::cerr << "***ERROR*** DebugLogHolder::setDir()"
                  << " No team name!" << std::endl;
        return false;
    }

    if ( unum < 1 || 11 < unum )
    {
        std::cerr << "***ERROR*** DebugLogHolder::setDir()"
                  << " Invalid unum! " << unum
                  << std::endl;
        return false;
    }


    std::string base_path = dirpath;
    if ( ! base_path.empty()
         && base_path[base_path.size() - 1] != '/' )
    {
        base_path += '/';
    }

    //for ( int i = 1; i <= 11; ++i )
    {
        int idx = unum - 1;
        M_player_data[idx].clear();

        M_player_data[idx].side_ = side;

        std::ostringstream ostr;
        ostr << team_name
             << '-'
             << unum
             << DEFAULT_EXTENSION;
        ostr << std::flush;

        // open file stream
        std::string filepath = base_path + ostr.str();
        std::cerr << "open debug file [" << filepath << "]" << std::endl;
        M_player_data[idx].fin_.open( filepath.c_str() );

        if ( ! M_player_data[idx].fin_.is_open()
             || ! M_player_data[idx].fin_.good() )
        {
            std::cerr << "***ERROR*** DebugLogHolder::setDir()"
                      << " Failed to open [" << filepath << "]"
                      << std::endl;
            M_player_data[idx].clear();
            return false;
        }

        // if file open is succeeded,
        // try to record cycle changed stream positions

        std::streampos pos = 0;
        std::string buf;
        int prev_cycle = -1;
        int cycle;

        int n_line = 0;
        std::ifstream & fin = M_player_data[idx].fin_;
        while ( std::getline( fin, buf ) )
        {
            ++n_line;
            // first element should be cycle value.
            if ( std::sscanf( buf.c_str(), " %d ", &cycle ) == 1 )
            {
                if ( cycle > prev_cycle )
                {
                    M_player_data[idx].cycle_map_.insert( std::make_pair( cycle, pos ) );
                    prev_cycle = cycle;
                }

                if ( cycle < prev_cycle )
                {
                    std::cerr << "***ERROR*** DebugLogHolder::setDir()"
                              << " Invalid cycle! path= "<< filepath
                              << "  line= " << n_line
                              << std::endl;
                }
            }
            pos = fin.tellg();
        }

        // reset get pointer
        M_player_data[idx].fin_.clear();
        M_player_data[idx].fin_.seekg( 0 );

        std::cerr << unum << " : read " << n_line << " lines"
                  << " cycle size = " << M_player_data[idx].cycle_map_.size()
                  << std::endl;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogHolder::getCycleData( const int unum,
                              const int cycle,
                              boost::shared_ptr< DebugLogData > data_ptr )
{
    if ( unum < 1 || 11 < unum )
    {
        return false;
    }

    const int idx = unum - 1;

    // get the start stream position of 'cycle'
    std::map< int, std::streampos >::iterator it
        = M_player_data[idx].cycle_map_.find( cycle );

    if ( it == M_player_data[idx].cycle_map_.end() )
    {
        // no data!
        return false;
    }

    ///////////////////////////////////////////////////
    // seek & analyze data

    // seek stream position
    std::ifstream & fin = M_player_data[idx].fin_;
    fin.clear();
    fin.seekg( it->second );

    // read loop
    char buf[8192];
    int n_line = 0;
    while ( fin.getline( buf, 8192 ) )
    {
        if ( ++n_line > 8192 * 4 )
        {
            std::cerr << "***ERROR*** DebugLogHolder::getCycleData()"
                      << " Too many lines! cycle ="
                      << cycle
                      << std::endl;
            break;
        }

        int read_cycle = 0;
        int n_read = 0;
        if ( std::sscanf( buf, " %d %n", &read_cycle, &n_read ) != 1
             || n_read == 0 )
        {
            // invalide cycle line.
            continue;
        }

        // reach new cycle
        if ( read_cycle > cycle )
        {
            break;
        }

        if ( ! data_ptr->parse( buf + n_read ) )
        {
            std::cerr << "***ERROR*** DebugLogHolder::getCycleData()"
                      << " Parse Error! unum = " << idx + 1
                      << "  cycle = " << cycle
                      << "  line = " << n_line
                      << std::endl;
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogHolder::seekData( const int unum,
                          const int cycle )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << "DebugLogHolder::seekData. Invalid number"
                  << std::endl;
        return false;
    }

    /*
    if ( s_last_seek_number == unum
         && M_cycle == cycle )
    {
        return false;
    }
    */

    const int idx = unum - 1;

    if ( ! M_player_data[idx].fin_.is_open() )
    {
        return false;
    }

    ///////////////////////////////////////////////////
    // get recent span stream position

    bool found = false;
    std::streampos first_pos = 0;
    int first_cycle = std::max( 1, cycle - MAX_SPAN + 1 );
    for ( ; first_cycle <= cycle; ++first_cycle )
    {
        std::map< int, std::streampos >::iterator it
            = M_player_data[idx].cycle_map_.find( first_cycle );
        if ( it != M_player_data[idx].cycle_map_.end() )
        {
            found = true;
            first_pos = it->second;
            break;
        }
    }

    if ( ! found  )
    {
        std::cerr << "***ERROR*** DebugLogHolder::seekData()"
                  << " could not find stream position."
                  << " unum = " << unum
                  << " cycle = " << cycle
                  << std::endl;
        return false;
    }

    ///////////////////////////////////////////////////
    // seek & analyze data

    // reset old data
    M_player_data[idx].data_list_.clear();

    const int last_cycle = cycle + MAX_SPAN;
    for ( int cyc = first_cycle; cyc < last_cycle; ++cyc )
    {
        boost::shared_ptr< DebugLogData > data( new DebugLogData( cyc ) );
        if ( getCycleData( unum, cyc, data ) )
        {
            M_player_data[idx].data_list_.push_back( data );
        }
    }

    M_cycle = cycle;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogHolder::incDataCycle( const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << "***ERROR*** DebugLogHolder::incDataCycle()"
                  << " Invalid unum " << unum
                  << std::endl;
        return false;
    }

    const int idx = unum - 1;

    if ( ! M_player_data[idx].fin_.is_open() )
    {
        //std::cerr << "***ERROR*** DebugLogHolder::incDataCycle()"
        //          << " File is not opened for unum = " << unum
        //          << std::endl;
        return false;
    }

    if ( M_player_data[idx].data_list_.empty() )
    {
        std::cerr << "***ERROR*** DebugLogHolder::incDataCycle()"
                  << " No current data! unum = " << unum
                  << std::endl;
        return false;
    }

    bool next_found = false;
    const DataList::iterator end = M_player_data[idx].data_list_.end();
    for ( DataList::iterator it = M_player_data[idx].data_list_.begin();
          it != end;
          ++it )
    {
        if ( (*it)->cycle() == M_cycle )
        {
            // cycle info in data equals to current focused cycle
            ++it;
            if ( it != end )
            {
                M_cycle = (*it)->cycle();
                next_found = true;
            }
            break;
        }
    }

    if ( ! next_found )
    {
        std::cerr << "***ERROR*** DebugLogHolder::incDataCycle()"
                  << " Not found next data" << std::endl;
        return false;
    }

    int new_cycle = M_player_data[idx].data_list_.back()->cycle() + 1;
    boost::shared_ptr< DebugLogData > next_data_ptr( new DebugLogData( new_cycle ) );

    // max trial is 10 cycles
    // if next possible cycle found, return immediately
    for ( int i = 0; i < 10; ++i )
    {
        if ( getCycleData( unum, new_cycle, next_data_ptr ) )
        {
            M_player_data[idx].data_list_.pop_front();
            M_player_data[idx].data_list_.push_back( next_data_ptr );
            return true;
        }
        ++new_cycle;
        next_data_ptr->setCycle( new_cycle );
    }

    std::cerr << "***ERROR*** DebugLogHolder::incDataCycle()"
              << " Not found data" << std::endl;
    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogHolder::decDataCycle( const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << "***ERROR*** DebugLogHolder::incDataCycle()"
                  << " Invalid unum " << unum
                  << std::endl;
        return false;
    }

    const int idx = unum - 1;

    if ( ! M_player_data[idx].fin_.is_open() )
    {
        //std::cerr << "***ERROR*** DebugLogHolder::decDataCycle()"
        //          << " File is not opened for unum = " << unum
        //          << std::endl;
        return false;
    }

    if ( M_player_data[idx].data_list_.empty() )
    {
        std::cerr << "***ERROR*** DebugLogHolder::decDataCycle()"
                  << " No current data! unum = " << unum
                  << std::endl;
        return false;
    }

    bool next_found = false;
    const DataList::reverse_iterator rend = M_player_data[idx].data_list_.rend();
    for ( DataList::reverse_iterator it = M_player_data[idx].data_list_.rbegin();
          it != rend;
          ++it )
    {
        if ( (*it)->cycle() == M_cycle )
        {
            ++it;
            if ( it != rend )
            {
                M_cycle = (*it)->cycle();
                next_found = true;
            }
            break;
        }
    }

    if ( ! next_found )
    {
        return false;
    }

    int new_cycle = M_player_data[idx].data_list_.front()->cycle() - 1;
    if ( new_cycle <= 0 )
    {
        return false;
    }

    boost::shared_ptr< DebugLogData > next_data_ptr( new DebugLogData( new_cycle ) );
    for ( int i = 0; i < 10; ++i )
    {
        if ( getCycleData( unum, new_cycle, next_data_ptr ) )
        {
            M_player_data[idx].data_list_.pop_back();
            M_player_data[idx].data_list_.push_front( next_data_ptr );
            return true;
        }
        --new_cycle;
        if ( new_cycle <= 0 )
        {
            return false;
        }
    }

    return false;
}
