// -*-c++-*-

/*!
  \file color_setting_dialog.cpp
  \brief color setting dialog for the field canvas Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "color_setting_dialog.h"
#include "draw_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
ColorEntry::ColorEntry( QPen * pen,
                        QBrush * brush )
    : pen_( pen )
    , brush_( brush )
{
    if ( pen_ )
    {
        default_color_ = pen_->color();
        color_ = pen_->color();
    }
    else if ( brush_ )
    {
        default_color_ = brush_->color();
        color_ = brush_->color();
    }

    if ( pen_
         && brush_
         && pen_->color() != brush_->color() )
    {
        brush_->setColor( pen_->color() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ColorEntry::setColor( const QColor & color )
{
    if ( color_ != color )
    {
        color_ = color;

        if ( pen_ ) pen_->setColor( color );
        if ( brush_ ) brush_->setColor( color );

        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ColorEntry::checkColor()
{
    if ( pen_ && pen_->color() != color_ )
    {
        color_ = pen_->color();
        return true;
    }

    if ( brush_ && brush_->color() != color_ )
    {
        color_ = brush_->color();
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ColorEntry::revert()
{
    return setColor( default_color_ );
}

/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/

/*-------------------------------------------------------------------*/
/*!

*/
ColorSettingDialog::ColorSettingDialog( QWidget * parent )
    : QDialog( parent )
{

    this->setCaption( tr( "Color Settings" ) );

    createControls();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::createControls()
{
    QVBoxLayout * main_layout = new QVBoxLayout( this );

    main_layout->addLayout( createColorButtons() );

    main_layout->addSpacing( 5 );

    main_layout->addLayout( createAnswerButtons() );
}

/*-------------------------------------------------------------------*/
/*!

*/
ColorSettingDialog::~ColorSettingDialog()
{
    //std::cerr << "delete ColorSettingDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
ColorSettingDialog::createColorButtons()
{
    QHBoxLayout * layout = new QHBoxLayout();
    M_list_box = new QListBox( this );


    addEntry( tr( "Score Board" ),
              NULL,
              DrawConfig::instance().scoreBoardBrushPtr() );
    addEntry( tr( "Score Board Font" ),
              DrawConfig::instance().scoreBoardFontPenPtr(),
              NULL );
    addEntry( tr( "Field Light" ),
              NULL,
              DrawConfig::instance().fieldBrushPtr() );
    addEntry( tr( "Field Dark" ),
              NULL,
              DrawConfig::instance().fieldDarkBrushPtr() );
    addEntry( tr( "Line" ),
              DrawConfig::instance().linePenPtr(),
              NULL );
    addEntry( tr( "Ball" ),
              NULL,
              DrawConfig::instance().ballBrushPtr() );
    addEntry( tr( "Measure" ),
              DrawConfig::instance().measurePenPtr(),
              NULL );
    addEntry( tr( "Measure Font" ),
              DrawConfig::instance().measureFontPenPtr(),
              NULL );
    //
    addEntry( tr( "Left Team" ),
              DrawConfig::instance().leftTeamPenPtr(),
              DrawConfig::instance().leftTeamBrushPtr() );
    addEntry( tr( "Left Goalie" ),
              DrawConfig::instance().leftGoaliePenPtr(),
              DrawConfig::instance().leftGoalieBrushPtr() );
    addEntry( tr( "Right Team" ),
              DrawConfig::instance().rightTeamPenPtr(),
              DrawConfig::instance().rightTeamBrushPtr() );
    addEntry( tr( "Right Goalie" ),
              DrawConfig::instance().rightGoaliePenPtr(),
              DrawConfig::instance().rightGoalieBrushPtr() );
    addEntry( tr( "Player Number Font" ),
              DrawConfig::instance().playerNumberFontPenPtr(),
              NULL );
    addEntry( tr( "Stamina Font" ),
              DrawConfig::instance().playerStaminaFontPenPtr(),
              NULL );
    addEntry( tr( "Player Detail Font" ),
              DrawConfig::instance().playerDetailFontPenPtr(),
              NULL );
    addEntry( tr( "Selected Player Font" ),
              DrawConfig::instance().selectedPlayerFontPenPtr(),
              NULL );
    addEntry( tr( "Effort Decayed" ),
              DrawConfig::instance().effortDecayedPenPtr(),
              NULL );
    addEntry( tr( "Recovery Decayed" ),
              DrawConfig::instance().recoveryDecayedPenPtr(),
              NULL );
    addEntry( tr( "View Cone" ),
              DrawConfig::instance().viewConePenPtr(),
              NULL );
    addEntry( tr( "View Area" ),
              DrawConfig::instance().viewAreaPenPtr(),
              NULL );
    //
    addEntry( tr( "Kick Pen" ),
              DrawConfig::instance().kickPenPtr(),
              NULL );
    addEntry( tr( "Kick Fault" ),
              DrawConfig::instance().kickFaultPenPtr(),
              DrawConfig::instance().kickFaultBrushPtr() );
    addEntry( tr( "Catch Brush" ),
              NULL,
              DrawConfig::instance().catchBrushPtr() );
    addEntry( tr( "Catch Fault Brush" ),
              NULL,
              DrawConfig::instance().catchFaultBrushPtr() );
    addEntry( tr( "Collide Ball Brush" ),
              NULL,
              DrawConfig::instance().collideBallBrushPtr() );
    addEntry( tr( "Collide Player Brush" ),
              NULL,
              DrawConfig::instance().collidePlayerBrushPtr() );
    addEntry( tr( "Tackle" ),
              DrawConfig::instance().tacklePenPtr(),
              DrawConfig::instance().tackleBrushPtr() );
    addEntry( tr( "Tackle Fault Brush" ),
              NULL,
              DrawConfig::instance().tackleFaultBrushPtr() );
    addEntry( tr( "PointTo" ),
              DrawConfig::instance().pointtoPenPtr(),
              NULL );
    //    addEntry( tr( "AttentionTo" ),
    //              DrawConfig::instance().attentiontoPenPtr(),
    //              NULL );
    addEntry( tr( "Kick Accel" ),
              DrawConfig::instance().kickAccelPenPtr(),
              NULL );
    addEntry( tr( "Tackle Area" ),
              DrawConfig::instance().tackleAreaPenPtr(),
              NULL );
    //
    addEntry( tr( "Debug Ball Pen" ),
              DrawConfig::instance().debugBallPenPtr(),
              NULL );
    addEntry( tr( "Debug Ball Brush" ),
              NULL,
              DrawConfig::instance().debugBallBrushPtr() );
    addEntry( tr( "Debug Ball Trace Pen" ),
              DrawConfig::instance().debugBallTracePenPtr(),
              NULL );
    addEntry( tr( "Debug Ball Trace Brush" ),
              NULL,
              DrawConfig::instance().debugBallTraceBrushPtr() );
    addEntry( tr( "Debug Player Pen" ),
              DrawConfig::instance().debugPlayerPenPtr(),
              NULL );
    addEntry( tr( "Debug Player Pen" ),
              DrawConfig::instance().debugPlayerPenPtr(),
              NULL );
    addEntry( tr( "Debug Self" ),
              NULL,
              DrawConfig::instance().debugSelfBrushPtr() );
    addEntry( tr( "Debug Teammate" ),
              NULL,
              DrawConfig::instance().debugTeammateBrushPtr() );
    addEntry( tr( "Debug Opponent" ),
              NULL,
              DrawConfig::instance().debugOpponentBrushPtr() );
    addEntry( tr( "Debug Unknown Teammate" ),
              NULL,
              DrawConfig::instance().debugUnknownTeammateBrushPtr() );
    addEntry( tr( "Debug Unknown Opponent" ),
              NULL,
              DrawConfig::instance().debugUnknownOpponentBrushPtr() );
    addEntry( tr( "Debug Unknown Player" ),
              NULL,
              DrawConfig::instance().debugUnknownPlayerBrushPtr() );
    addEntry( tr( "Debug Target" ),
              DrawConfig::instance().debugTargetPenPtr(),
              NULL );
    addEntry( tr( "Debug Comment" ),
              DrawConfig::instance().debugCommentFontPenPtr(),
              NULL );
    addEntry( tr( "Debug Message" ),
              DrawConfig::instance().debugMessageFontPenPtr(),
              NULL );
    addEntry( tr( "Debug Log Font" ),
              DrawConfig::instance().debugLogMessageFontPenPtr(),
              NULL );
    addEntry( tr( "Debug Shapes" ),
              DrawConfig::instance().debugShapePenPtr(),
              NULL );

    M_list_box->setMinimumWidth( M_list_box->maxItemWidth() + 16 + 10 );
    M_list_box->setMinimumHeight( 300 );

    layout->addWidget( M_list_box );

    QPushButton * select_btn = new QPushButton( tr( "Select" ), this );
    connect( select_btn, SIGNAL( clicked() ),
             this, SLOT( selectEntry() ) );
    layout->addWidget( select_btn );

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::addEntry( const QString & name,
                              QPen * pen,
                              QBrush * brush )
{
    QColor color;
    if ( pen )
    {
        color = pen->color();
    }
    else if ( brush )
    {
        color = brush->color();
    }
    else
    {
        return;
    }

    QPixmap pixmap( 16, 16 );
    pixmap.fill( color );

    M_list_box->insertItem( pixmap, name );

    M_color_entries.push_back( ColorEntry( pen, brush ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
ColorSettingDialog::createAnswerButtons()
{
    QHBoxLayout * layout = new QHBoxLayout();

    QFontMetrics fm = this->fontMetrics();
    int text_width = fm.width( tr( "Cancel" ) ) + 8;

    QPushButton * def = new QPushButton( tr( "Default" ), this );
    //cancel->resize( text_width, fm.height() );
    connect( def, SIGNAL( clicked() ),
             this, SLOT( setDefaultColors() ) );

    QPushButton * ok = new QPushButton( tr( "OK" ), this );
    ok->resize( text_width, fm.height() );
    connect( ok, SIGNAL( clicked() ),
             this, SLOT( accept() ) );

    QPushButton * cancel = new QPushButton( tr( "Cancel" ), this );
    cancel->resize( text_width, fm.height() );
    connect( cancel, SIGNAL( clicked() ),
             this, SLOT( reject() ) );


    layout->addWidget( def, 0, Qt::AlignLeft );
    layout->addStretch( 1 );
    layout->addWidget( ok, 0, Qt::AlignRight );
    layout->addWidget( cancel, 0, Qt::AlignRight );

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::revert()
{
    bool changed = false;

    for ( std::size_t i = 0; i < M_list_box->count(); ++i )
    {
        if ( M_color_entries[i].revert() )
        {
            QString text = M_list_box->text( i );
            QPixmap pixmap( 16, 16 );
            pixmap.fill( M_color_entries[i].color_ );
            M_list_box->changeItem( pixmap, text, i );
            changed = true;
        }
    }

    if ( changed )
    {
        emit colorChanged();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::setDefaultColors()
{
    DrawConfig::instance().setDefaultColors();

    bool changed = false;
    for ( std::size_t i = 0; i < M_list_box->count(); ++i )
    {
        if ( M_color_entries[i].checkColor() )
        {
            QString text = M_list_box->text( i );
            QPixmap pixmap( 16, 16 );
            pixmap.fill( M_color_entries[i].color_ );
            M_list_box->changeItem( pixmap, text, i );
            changed = true;
        }
    }

    if ( changed )
    {
        emit colorChanged();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::selectEntry()
{
    int index = M_list_box->currentItem();

    if ( index < 0 )
    {
        return;
    }

    if ( index >= static_cast< int >( M_color_entries.size() ) )
    {
        std::cerr << "Illegal index " << index << std::endl;
        return;
    }

    QColor col = QColorDialog::getColor( M_color_entries[index].color_ );
    if ( ! col.isValid() )
    {
        return;
    }

    if ( M_color_entries[index].setColor( col ) )
    {
        QString text = M_list_box->text( index );
        QPixmap pixmap( 16, 16 );
        pixmap.fill( col );
        M_list_box->changeItem( pixmap, text, index );

        emit colorChanged();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ColorSettingDialog::reject()
{
    // reset all changes

    revert();

    QDialog::reject();
}
