// -*-c++-*-

/*!
  \file debug_painter.cpp
  \brief debug info painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "debug_painter.h"

#include "draw_config.h"

#include "options.h"
#include "main_data.h"
#include "debug_view_data.h"

#include <rcsc/common/server_param.h>

#include <iostream>
#include <cmath>

namespace {

/*-------------------------------------------------------------------*/
/*!

*/
void
draw_communication( QPainter & painter,
                    const rcsc::SideID self_side,
                    const MonitorViewData & monitor_view,
                    const DebugViewCont & debug_view_cont,
                    const DebugViewData & debug_view )
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::RIGHT
                             ? -1.0
                             : 1.0 );

    const int say_r = std::max( 3, opt.scale( 3.5 ) );
    const int hear_r = std::max( 2, opt.scale( 3.0 ) );

    //
    // draw say message sender marks
    //

    painter.setPen( DrawConfig::instance().debugCommentFontPen() );
    painter.setBrush( DrawConfig::instance().transparentBrush() );

    for ( size_t i = 0; i < debug_view_cont.size(); ++i )
    {
        const DebugViewPtr v = debug_view_cont[i];
        if ( ! v ) continue;
        if ( v->sayMessage().empty() ) continue;

        const boost::shared_ptr< DebugViewData::SelfT > self = v->self();
        if ( ! self ) continue;

        const int ix = opt.screenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int iy = opt.screenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

        painter.drawEllipse( ix - say_r, iy - say_r, say_r * 2, say_r * 2 );
    }

    //
    // draw heard message sender marks
    //
    if ( ! debug_view.hearMessages().empty() )
    {
        const boost::shared_ptr< DebugViewData::SelfT > self = debug_view.self();
        if ( ! self )
        {
            return;
        }

        const int self_x = opt.screenX( self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int self_y = opt.screenY( self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

        const std::vector< Player > & players = monitor_view.players();
        const std::vector< Player >::const_iterator p_end = players.end();

        //painter.setPen( dconf.debugShapePen() );
        painter.setPen( Qt::darkMagenta );
        painter.setBrush( dconf.transparentBrush() );

        for ( std::map< int, std::string >::const_iterator it = debug_view.hearMessages().begin();
              it != debug_view.hearMessages().end();
              ++it )
        {
            for ( std::vector< Player >::const_iterator p = players.begin();
                  p != p_end;
                  ++p )
            {
                if ( p->side() == self_side
                     && p->unum() == it->first )
                {
                    int px = opt.screenX( p->x() );
                    int py = opt.screenY( p->y() );

                    painter.drawEllipse( px - hear_r, py - hear_r,
                                         hear_r * 2, hear_r * 2 );
                    painter.drawLine( px, py, self_x, self_y );
                    break;
                }
            }
        }
    }
}

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::draw( QPainter & painter )
{
    const Options & opt = Options::instance();

    if ( ! opt.showDebugView() )
    {
        return;
    }

    int number = opt.selectedNumber();
    if ( number == 0 )
    {
        return;
    }

    const rcsc::SideID self_side = ( number < 0 ? rcsc::RIGHT : rcsc::LEFT );

    const std::map< int, DebugViewCont > & view_map
        = ( self_side == rcsc::RIGHT
            ? M_main_data.viewHolder().rightDebugView()
            : M_main_data.viewHolder().leftDebugView() );

    MonitorViewConstPtr monitor_view = M_main_data.getCurrentViewData();
    if ( ! monitor_view )
    {
        return;
    }

    std::map< int, DebugViewCont >::const_iterator it = view_map.find( monitor_view->cycle() );

    if ( it == view_map.end() )
    {
        return;
    }

#if 1
    //
    // draw say message sender marks
    //

    for ( size_t i = 0; i < it->second.size(); ++i )
    {
        const DebugViewPtr v = it->second[i];
        if ( ! v ) continue;
        if ( v->sayMessage().empty() ) continue;

        const boost::shared_ptr< DebugViewData::SelfT > self = v->self();
        if ( ! self ) continue;

        const double reverse = ( self_side == rcsc::RIGHT ? -1.0 : 1.0 );
        const int ix = opt.screenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int iy = opt.screenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int r = std::max( 3, opt.scale( 3.5 ) );

        painter.setPen( DrawConfig::instance().debugCommentFontPen() );
        painter.setBrush( DrawConfig::instance().transparentBrush() );
        painter.drawEllipse( ix - r, iy - r, r * 2, r * 2 );
    }
#endif

    number = std::abs( number );

    if ( static_cast< int >( it->second.size() ) < number )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error debug view container size is small."
                  << std::endl;
        return;
    }

    const DebugViewPtr view = it->second[ number - 1 ];
    if ( ! view )
    {
        // no data!
        //std::cerr << __FILE__ << ":" << __LINE__
        //          << " Error. No debug view data of " << number
        //          << std::endl;
        return;
    }

    const DrawConfig & dconf = DrawConfig::instance();

    // draw self
    if ( opt.showDebugViewSelf() )
    {
        drawSelf( painter, self_side, *view );
    }

    // draw players
    if ( opt.showDebugViewTeammates() )
    {
        drawPlayers( painter,
                     self_side,
                     *view,
                     view->teammates(),
                     ( opt.showDebugViewTarget()
                       ? view->targetTeammate()
                       : -1 ),
                     dconf.debugTeammateBrush() );
        drawPlayers( painter,
                     self_side,
                     *view,
                     view->unknownTeammates(),
                     -1,
                     dconf.debugUnknownTeammateBrush() );
    }

    if ( opt.showDebugViewOpponents() )
    {
        drawPlayers( painter,
                     self_side,
                     *view,
                     view->opponents(),
                     -1,
                     dconf.debugOpponentBrush() );
        drawPlayers( painter,
                     self_side,
                     *view,
                     view->unknownOpponents(),
                     -1,
                     dconf.debugUnknownOpponentBrush() );
        drawPlayers( painter,
                     self_side,
                     *view,
                     view->unknownPlayers(),
                     -1,
                     dconf.debugUnknownPlayerBrush() );
    }

    // draw ball
    if ( opt.showDebugViewBall()
         && view->ball() )
    {
        drawBall( painter, self_side, *view );
    }

    // draw lines
    if ( opt.showDebugViewShape() )
    {
        drawLines( painter, self_side, *view );
        drawTriangles( painter, self_side, *view );
        drawRectangles( painter, self_side, *view );
        drawCircles( painter, self_side, *view );
    }

    if ( opt.showDebugViewMessage() )
    {
        drawMessage( painter, *view );
    }
}

/*-------------------------------------------------------------------*/
/*!
  pos, vel, body, neck, comment
*/
void
DebugPainter::drawSelf( QPainter & painter,
                        const rcsc::SideID self_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::SelfT > self = view.self();
    if ( ! self )
    {
        std::cerr << "DebugPainter::drawSelf. No self data!" << std::endl;
        return;
    }

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    const int ix = opt.screenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int iy = opt.screenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int r = std::max( 2,
                            ( opt.enlargeMode()
                              ? opt.scale( 1.0 )
                              : opt.scale( 0.3 ) ) );

    // draw targets
    if ( opt.showDebugViewTarget()
         &&  view.targetPoint() )
    {
        int tx = opt.screenX( view.targetPoint()->x * reverse );
        int ty = opt.screenY( view.targetPoint()->y * reverse );

        painter.setPen( dconf.debugTargetPen() );
        painter.setBrush( dconf.transparentBrush() );

        painter.drawLine( tx - r, ty - r,
                          tx + r, ty + r );
        painter.drawLine( tx - r, ty + r,
                          tx + r, ty - r );
    }

    // draw body half circle
    double body_start_dir = - ( (double)self->body_ + 90.0 );
    if ( opt.reverseSide() ) body_start_dir += 180.0;
    if ( self_side == rcsc::RIGHT ) body_start_dir -= 180.0;

    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.debugSelfBrush() );

    painter.drawPie( ix - r,
                     iy - r,
                     r * 2,
                     r * 2,
                     static_cast< int >( rint( body_start_dir * 16 ) ),
                     180 * 16 );

    // draw edge

    painter.setPen( dconf.debugPlayerPen() );
    painter.setBrush( dconf.transparentBrush() );

    painter.drawEllipse( ix - r,
                         iy - r,
                         r * 2,
                         r * 2 );

    // draw comment
    if ( opt.showDebugViewComment()
         && ! self->comment_.empty() )
    {
        painter.setFont( dconf.debugCommentFont() );
        painter.setPen( dconf.debugCommentFontPen() );

        painter.drawText( ix - r,
                          iy + r + painter.fontMetrics().ascent(),
                          self->comment_ );
                          //QString::fromAscii( self->comment_.c_str() ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawBall( QPainter & painter,
                        const rcsc::SideID self_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::BallT > & ball = view.ball();

    if ( ! ball )
    {
        return;
    }

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    rcsc::Vector2D bpos( (double)ball->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    rcsc::Vector2D bvel( (double)ball->vx_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->vy_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const int ix = opt.screenX( bpos.x );
    const int iy = opt.screenY( bpos.y );
    const int r = std::max( 2,
                            ( opt.enlargeMode()
                              ? opt.scale( 0.25 )
                              : opt.scale( 0.07 ) ) );

    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.debugBallBrush() );

    painter.drawEllipse( ix - r,
                         iy - r,
                         r * 2,
                         r * 2 );

    {
        painter.setPen( dconf.debugBallTracePen() );
        painter.setBrush( dconf.debugBallTraceBrush() );

        int bx = ix;
        int by = iy;
        for ( int i = 0; i < 10; ++i )
        {
            bpos += bvel;
            bvel *= ball_decay;
            bx = opt.screenX( bpos.x );
            by = opt.screenY( bpos.y );
            painter.drawEllipse( bx - 2, by - 2,
                                 4, 4 );
        }
        painter.drawLine( ix, iy,
                          bx, by );
    }

    // draw comment
    if ( opt.showDebugViewComment()
         && ! ball->comment_.empty() )
    {
        painter.setFont( dconf.debugCommentFont() );
        painter.setPen( dconf.debugCommentFontPen() );

        painter.drawText( ix - r,
                          iy + r + painter.fontMetrics().ascent(),
                          ball->comment_ );
                          //QString::fromAscii( ball->comment_.c_str() ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawPlayers( QPainter & painter,
                           const rcsc::SideID self_side,
                           const DebugViewData & view,
                           const DebugViewData::PlayerCont & players,
                           const int target_number,
                           const QBrush & body_brush ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::RIGHT
                             ? -1.0
                             : 1.0 );

    const bool comment = opt.showDebugViewComment();
    const int r = std::max( 2,
                            ( opt.enlargeMode()
                              ? opt.scale( 1.0 )
                              : opt.scale( 0.3 ) ) );

    const DebugViewData::PlayerCont::const_iterator end = players.end();
    for ( DebugViewData::PlayerCont::const_iterator it = players.begin();
          it != end;
          ++it )
    {
        const int ix = opt.screenX( (double)(*it)->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int iy = opt.screenY( (double)(*it)->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

        if ( (*it)->body_ != -360 )
        {
            double body_start_dir = - ( (double)(*it)->body_ + 90.0 );
            if ( opt.reverseSide() ) body_start_dir += 180.0;
            if ( self_side == rcsc::RIGHT ) body_start_dir -= 180.0;
            //double body_end_dir = body_start_dir + 180.0;
            // draw body half circle
            painter.setPen( dconf.transparentPen() );
            painter.setBrush( body_brush );
            painter.drawPie( ix - r, iy - r,
                             r * 2, r * 2,
                             static_cast< int >( rint( body_start_dir * 16 ) ),
                             180 * 16 );
            // draw edge
            painter.setPen( dconf.debugPlayerPen() );
            painter.setBrush( dconf.transparentBrush() );
            painter.drawEllipse( ix - r, iy - r, r * 2, r * 2 );
        }
        else
        {
            // draw simple circle
            painter.setPen( dconf.transparentPen() );
            painter.setBrush( body_brush );
            painter.drawEllipse( ix - r, iy - r, r * 2, r * 2 );
        }

        if ( (*it)->unum_ > 0 )
        {
            painter.setFont( dconf.debugCommentFont() );
            painter.setPen( dconf.debugCommentFontPen() );

            painter.drawText( ix + r, iy + 4,
                              QString::number( (*it)->unum_ ) );
        }

        if ( target_number == (*it)->unum_ )
        {
            painter.setPen( dconf.debugTargetPen() );
            painter.setBrush( dconf.transparentBrush() );

            const boost::shared_ptr< DebugViewData::SelfT > self = view.self();
            if ( self )
            {
                const int sx = opt.screenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
                const int sy = opt.screenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
                painter.drawLine( sx, sy, ix, iy );
            }

            int rr = r + 2;
            painter.drawEllipse( ix - rr, iy - rr,
                                 rr * 2, rr * 2 );
        }

        if ( comment
             && ! (*it)->comment_.empty() )
        {
            painter.setFont( dconf.debugCommentFont() );
            painter.setPen( dconf.debugCommentFontPen() );

            painter.drawText( ix - r,
                              iy + r + painter.fontMetrics().ascent(),
                              (*it)->comment_ );
                              //QString::fromAscii( (*it)->comment_.c_str() ) );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawLines( QPainter & painter,
                         const rcsc::SideID self_side,
                         const DebugViewData & view ) const
{
    if ( view.lines().empty() )
    {
        return;
    }

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::RIGHT
                             ? -1.0
                             : 1.0 );

    painter.setPen( dconf.debugShapePen() );
    painter.setBrush( dconf.transparentBrush() );

    const std::list< DebugViewData::LineT >::const_iterator end = view.lines().end();
    for ( std::list< DebugViewData::LineT >::const_iterator it = view.lines().begin();
          it != end;
          ++it )
    {
        painter.drawLine( opt.screenX( it->x1_ * reverse ),
                          opt.screenY( it->y1_ * reverse ),
                          opt.screenX( it->x2_ * reverse ),
                          opt.screenY( it->y2_ * reverse ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawTriangles( QPainter & painter,
                             const rcsc::SideID self_side,
                             const DebugViewData & view ) const
{
    if ( view.triangles().empty() )
    {
        return;
    }

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::RIGHT
                             ? -1.0
                             : 1.0 );

    painter.setPen( dconf.debugShapePen() );
    painter.setBrush( dconf.transparentBrush() );

    const std::list< DebugViewData::TriangleT >::const_iterator end = view.triangles().end();
    for ( std::list< DebugViewData::TriangleT >::const_iterator it = view.triangles().begin();
          it != end;
          ++it )
    {
        int x1 = opt.screenX( it->x1_ * reverse );
        int y1 = opt.screenY( it->y1_ * reverse );
        int x2 = opt.screenX( it->x2_ * reverse );
        int y2 = opt.screenY( it->y2_ * reverse );
        int x3 = opt.screenX( it->x3_ * reverse );
        int y3 = opt.screenY( it->y3_ * reverse );

        painter.drawLine( x1, y1,
                          x2, y2 );
        painter.drawLine( x2, y2,
                          x3, y3 );
        painter.drawLine( x3, y3,
                          x1, y1 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawRectangles( QPainter & painter,
                              const rcsc::SideID self_side,
                              const DebugViewData & view ) const
{
    if ( view.rectangles().empty() )
    {
        return;
    }

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::RIGHT
                             ? -1.0
                             : 1.0 );

    painter.setPen( dconf.debugShapePen() );
    painter.setBrush( dconf.transparentBrush() );

    const std::list< DebugViewData::RectT >::const_iterator end = view.rectangles().end();
    for ( std::list< DebugViewData::RectT >::const_iterator it = view.rectangles().begin();
          it != end;
          ++it )
    {
        int left_x = opt.screenX( it->left_x_ * reverse );
        int top_y = opt.screenY( it->top_y_ * reverse );
        int right_x = opt.screenX( it->right_x_ * reverse );
        int bottom_y = opt.screenY( it->bottom_y_ * reverse );

        painter.drawRect( left_x,
                          top_y,
                          ( right_x - left_x ),
                          ( bottom_y - top_y ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawCircles( QPainter & painter,
                           const rcsc::SideID self_side,
                           const DebugViewData & view ) const
{
    if ( view.circles().empty() )
    {
        return;
    }

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( self_side == rcsc::RIGHT
                             ? -1.0
                             : 1.0 );

    painter.setPen( dconf.debugShapePen() );
    painter.setBrush( dconf.transparentBrush() );

    const std::list< DebugViewData::CircleT >::const_iterator end = view.circles().end();
    for ( std::list< DebugViewData::CircleT >::const_iterator it = view.circles().begin();
          it != end;
          ++it )
    {
        int r = opt.scale( it->radius_ );
        painter.drawEllipse( opt.screenX( it->center_x_ * reverse ) - r,
                             opt.screenY( it->center_y_ * reverse ) - r,
                             r * 2,
                             r * 2 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawMessage( QPainter & painter,
                           const DebugViewData & view ) const
{
    painter.setFont( DrawConfig::instance().debugMessageFont() );
    painter.setPen( DrawConfig::instance().debugMessageFontPen() );

    int height = painter.fontMetrics().ascent() + 2;
    int y = Options::instance().scoreBoardHeight() + height;

    if ( ! view.message().empty() )
    {
        painter.drawText( 10, y,
                          view.message() );
        y += height;
    }


    if ( ! view.sayMessage().empty() )
    {
        QString text = QObject::tr( "Say: " );
        text += view.sayMessage();
        //text += QObject::tr( "\"" );

        painter.drawText( 10, y, text );
        y+= height;
    }

    if ( ! view.hearMessages().empty() )
    {
        QString text = QObject::tr( "Hear: " );

        for ( std::map< int, std::string >::const_iterator it = view.hearMessages().begin();
              it != view.hearMessages().end();
              ++it )
        {
            text += QString( "(%1" ).arg( it->first );
            text += it->second;
            text += QObject::tr( ")" );
        }

        painter.drawText( 10, y, text );
    }
}
