// -*-c++-*-

/*!
  \file detail_dialog.cpp
  \brief Detail info Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "detail_dialog.h"

#include "options.h"
#include "main_data.h"

#include <rcsc/geom/vector_2d.h>

#include <iostream>
#include <cstdio>

/*-------------------------------------------------------------------*/
/*!

 */
DetailDialog::DetailDialog( QWidget * parent,
                            const MainData & main_data )
    : QDialog( parent )
    , M_main_data( main_data )
{
    //this->setWindowTitle( tr( "Detail Dialog" ) ); // qt4
    this->setCaption( tr( "Detail Dialog" ) );

    //QFont font = this->font();
    //font.setPointSize( 8 );
    //this->setFont( font );

    createWidgets();
}

/*-------------------------------------------------------------------*/
/*!

 */
DetailDialog::~DetailDialog()
{
    //std::cerr << "delete DetailDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
DetailDialog::createWidgets()
{
    QVBoxLayout * layout = new QVBoxLayout( this );
    layout->setMargin( 2 );
    layout->setSpacing( 2 );
    layout->setResizeMode( QLayout::Fixed );


    layout->addWidget( createBallLabels(),
                       1, Qt::AlignHCenter );

    layout->addWidget( createPlayerLabels(),
                       1, Qt::AlignHCenter );

    //this->setLayout( layout ); // qt4
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
DetailDialog::createBallLabels()
{
    QGroupBox * group_box = new QGroupBox( tr( "Ball" ), this );

    QVBoxLayout * top_layout = new QVBoxLayout( group_box );
    top_layout->addSpacing( 10 );

    QGridLayout * layout = new QGridLayout( top_layout );
    layout->setMargin( 5 );
    layout->setSpacing( 0 );
    layout->setColSpacing( 0, this->fontMetrics().width( tr( "AttentionTo " ) ) );
    layout->setColSpacing( 1, this->fontMetrics().width( tr( " -12.34, -12.34" ) ) );
    layout->setColSpacing( 2, this->fontMetrics().width( tr( " (123.456, -123.4)" ) ) );

    int row = 0;

    layout->addWidget( new QLabel( tr( "Pos " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_ball_pos = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_ball_pos, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Vel " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_ball_vel = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_ball_vel, row, 1, Qt::AlignRight );
    M_ball_polar_vel = new QLabel( tr( " (  0.000,    0.0)" ), group_box );
    layout->addWidget( M_ball_polar_vel, row, 2, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "LastMove " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_ball_last_move = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_ball_last_move, row, 1, Qt::AlignRight );
    M_ball_last_polar_move = new QLabel( tr( " (  0.000,    0.0)" ), group_box );
    layout->addWidget( M_ball_last_polar_move, row, 2, Qt::AlignRight );
    ++row;

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
DetailDialog::createPlayerLabels()
{
    QGroupBox * group_box = new QGroupBox( tr( "Player" ), this );
    group_box->setMinimumWidth( 160 );

    QVBoxLayout * top_layout = new QVBoxLayout( group_box );
    top_layout->addSpacing( 10 );

    QGridLayout * layout = new QGridLayout( top_layout );
    layout->setMargin( 5 );
    layout->setSpacing( 0 );
    layout->setColSpacing( 0, this->fontMetrics().width( tr( "AttentionTo " ) ) );
    layout->setColSpacing( 1, this->fontMetrics().width( tr( " -12.34, -12.34" ) ) );
    layout->setColSpacing( 2, this->fontMetrics().width( tr( " (123.456, -123.4)" ) ) );

    int row = 0;

    layout->addWidget( new QLabel( tr( "Unum " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_number = new QLabel( tr( " Neutral  0" ), group_box );
    layout->addWidget( M_player_number, row, 1, Qt::AlignHCenter );
    ++row;

    layout->addWidget( new QLabel( tr( "Type " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_type = new QLabel( tr( "0" ), group_box );
    layout->addWidget( M_player_type, row, 1, Qt::AlignHCenter );
    ++row;

    layout->addWidget( new QLabel( tr( "Pos " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_pos = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_player_pos, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Vel " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_vel= new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_player_vel, row, 1, Qt::AlignRight );
    M_player_polar_vel = new QLabel( tr( " (  0.000,    0.0)" ), group_box );
    layout->addWidget( M_player_polar_vel, row, 2, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "MaxTurn " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_max_turn_moment = new QLabel( tr( "   0.00" ), group_box );
    layout->addWidget( M_max_turn_moment, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "LastMove " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_last_move = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_player_last_move, row, 1, Qt::AlignRight );
    M_player_last_polar_move = new QLabel( tr( " (  0.000,    0.0)" ), group_box );
    layout->addWidget( M_player_last_polar_move, row, 2, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Body " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_body = new QLabel( tr( "   0.0" ), group_box );
    layout->addWidget( M_player_body, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Head " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_head = new QLabel( tr( "   0.0" ), group_box );
    layout->addWidget( M_player_head, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "PointToPos " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_pointto_pos = new QLabel( tr( "      -,      -" ), group_box );
    layout->addWidget( M_player_pointto_pos, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Focus " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_focus_target = new QLabel( tr( "      -,      -" ), group_box );
    layout->addWidget( M_player_focus_target, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Stamina " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_stamina = new QLabel( tr( "    0.0" ), group_box );
    layout->addWidget( M_stamina, row, 1, Qt::AlignRight );
    //    ++row;
    //    layout->addWidget( new QLabel( tr( "Capacity " ), group_box ),
    //                       row, 0, Qt::AlignRight );
    M_stamina_capacity = new QLabel( tr( "   -1.0" ), group_box );
    layout->addWidget( M_stamina_capacity, row, 2, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Effort " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_effort = new QLabel( tr( " 0.000 / 0.000" ), group_box );
    layout->addWidget( M_effort, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Recovery " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_recovery = new QLabel( tr( " 0.000 / 1.0" ), group_box );
    layout->addWidget( M_recovery, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Kick " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_kick_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_kick_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Dash " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_dash_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_dash_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Turn " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_turn_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_turn_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Say " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_say_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_say_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "TurnNeck " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_turn_neck_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_turn_neck_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Catch " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_catch_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_catch_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Move " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_move_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_move_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "ChgView " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_change_view_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_change_view_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "Tackle " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_tackle_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_tackle_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "PointTo " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_pointto_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_pointto_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "AttentionTo " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_attentionto_count = new QLabel( tr( "     0" ), group_box );
    layout->addWidget( M_attentionto_count, row, 1, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "BallRelPos " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_ball_rel_pos = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_player_ball_rel_pos, row, 1, Qt::AlignRight );
    M_player_ball_rel_polar = new QLabel( tr( " (  0.000,    0.0)" ), group_box );
    layout->addWidget( M_player_ball_rel_polar, row, 2, Qt::AlignRight );
    ++row;

    layout->addWidget( new QLabel( tr( "BallRelVel " ), group_box ),
                       row, 0, Qt::AlignRight );
    M_player_ball_rel_vel = new QLabel( tr( "   0.00,   0.00" ), group_box );
    layout->addWidget( M_player_ball_rel_vel, row, 1, Qt::AlignRight );
    ++row;

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
DetailDialog::showEvent( QShowEvent * )
{
    updateLabels();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
DetailDialog::updateLabels()
{
    if ( ! this->isVisible() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getCurrentViewData();

    if ( ! view )
    {
        return;
    }

    const Options & opt = Options::instance();

    MonitorViewConstPtr last_view
        = ( M_main_data.viewIndex() <= 0
            ? MonitorViewConstPtr()
            : M_main_data.getViewData( M_main_data.viewIndex() - 1 ) );

    //std::cerr << "DetailDialog::updteLabels start" << std::endl;

    const double reverse = opt.reverseValue();

    char buf[64];

    // update ball

    const Ball & ball = view->ball();

    snprintf( buf, 64, " % 6.2f, % 6.2f",
              ball.x() * reverse,
              ball.y() * reverse );
    M_ball_pos->setText( QString::fromAscii( buf ) );

    if ( ball.hasDelta() )
    {
        rcsc::Vector2D vel( ball.deltaX(), ball.deltaY() );
        vel *= reverse;
        snprintf( buf, 64, " %6.2f, %6.2f",
                  vel.x, vel.y );
        M_ball_vel->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " (%7.3f, %6.1f)",
                  vel.r(), vel.th().degree() );
        M_ball_polar_vel->setText( QString::fromAscii( buf ) );
    }

    if ( ! last_view )
    {
        M_ball_last_move->setText( tr( "   0.00,   0.00" ) );
        M_ball_last_polar_move->setText( tr( " (  0.000,    0.0)" ) );
    }
    else
    {
        rcsc::Vector2D move( ball.x() - last_view->ball().x(),
                             ball.y() - last_view->ball().y() );
        move *= reverse;

        snprintf( buf, 64, " %6.2f, %6.2f",
                  move.x, move.y );
        M_ball_last_move->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " (%7.3f, %6.1f)",
                  move.r(), move.th().degree() );
        M_ball_last_polar_move->setText( QString::fromAscii( buf ) );
    }

    // update player

    int number = opt.selectedNumber();
    if ( number == 0 || std::abs( number ) > 11 )
    {
        return;
    }

    const int player_index = std::abs( number ) - 1 + ( number > 0 ? 0 : 11 );
    const Player & player = view->players()[player_index];
    const rcsc::PlayerType & player_type
        = M_main_data.viewHolder().playerType( player.type() );

    // side, number
    snprintf( buf, 64, "%8s  %2d",
              ( opt.reverseSide()
                ? ( number > 0 ? "   Right" : "   Left" )
                : ( number > 0 ? "   Left" : "   Right" ) ),
              number > 0 ? number : -number );
    M_player_number->setText( QString::fromAscii( buf ) );

    // player type
    snprintf( buf, 64, " %d" ,
              player.type() );
    M_player_type->setText( QString::fromAscii( buf ) );

    // pos
    snprintf( buf, 64, " %6.2f, %6.2f",
              player.x() * reverse,
              player.y() * reverse );
    M_player_pos->setText( QString::fromAscii( buf ) );

    // velocity, speed, speed dir
    if ( player.hasDelta() )
    {
        rcsc::Vector2D vel( player.deltaX(), player.deltaY() );
        vel *= reverse;

        snprintf( buf, 64, " %6.2f, %6.2f",
                  vel.x, vel.y );
        M_player_vel->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " (%7.3f, %6.1f)",
                  vel.r(), vel.th().degree() );
        M_player_polar_vel->setText( QString::fromAscii( buf ) );

        // ball relative velocity
        snprintf( buf, 64, " %6.2f, %6.2f",
                  ball.deltaX() - vel.x, ball.deltaY() - vel.y );
        M_player_ball_rel_vel->setText( QString::fromAscii( buf ) );

        // max turn moment
        double max_turn = player_type.effectiveTurn( 180.0, vel.r() );
        snprintf( buf, 64, " %3.2f", max_turn );
        M_max_turn_moment->setText( QString::fromAscii( buf ) );
    }

    if ( ! last_view )
    {
        M_player_last_move->setText( tr( "   0.00,   0.00" ) );
        M_player_last_polar_move->setText( tr( "  (  0.000,    0.0)" ) );
    }
    else
    {
        const Player & last_player = last_view->players()[player_index];
        rcsc::Vector2D last_move( player.x() - last_player.x(),
                                  player.y() - last_player.y() );
        last_move *= reverse;

        snprintf( buf, 64, " %6.2f, %6.2f",
                  last_move.x, last_move.y );
        M_player_last_move->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " (%7.3f, %6.1f)",
                  last_move.r(), last_move.th().degree() );
        M_player_last_polar_move->setText( QString::fromAscii( buf ) );
    }

    // body angle
    rcsc::AngleDeg body = player.body();
    if ( opt.reverseSide() ) body += 180.0;

    snprintf( buf, 64," %6.1f",
              body.degree() );
    M_player_body->setText( QString::fromAscii( buf ) );

    // neck, head angle
    if ( player.hasView() )
    {
        rcsc::AngleDeg neck = player.head() - player.body();
        rcsc::AngleDeg head = player.head();
        if ( opt.reverseSide() ) head += 180.0;

        snprintf( buf, 64, " %6.1f(%6.1f)",
                  head.degree(), neck.degree() );
        M_player_head->setText( QString::fromAscii( buf ) );
    }

    // pointto point
    if ( player.isPointing() )
    {
        snprintf( buf, 64, " %6.2f, %6.2f",
                  player.data().point_x_ * reverse,
                  player.data().point_y_ * reverse );
        M_player_pointto_pos->setText( QString::fromAscii( buf ) );
    }
    else
    {
        M_player_pointto_pos->setText( tr( "      -,      -") );
    }

    // focus target
    if ( player.isFocusing() )
    {
        snprintf( buf, 64, " %5s %2d",
                  ( opt.reverseSide()
                    ? player.data().focus_side_ == 'l' ? "Right" : "Left"
                    : player.data().focus_side_ == 'l' ? "Left" : "Right" ),
                  player.data().focus_unum_ );
        M_player_focus_target->setText( QString::fromAscii( buf ) );
    }
    else
    {
        M_player_focus_target->setText( tr( " --- ") );
    }

    // stamina, effort, recovery
    if ( player.hasStamina() )
    {
        snprintf( buf, 64, " %6.1f",
                  player.stamina() );
        M_stamina->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " %.0f",
                  player.staminaCapacity() );
        M_stamina_capacity->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " %5.3f / %5.3f",
                  player.effort(), player_type.effortMax() );
        M_effort->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " %5.3f / 1.0",
                  player.recovery() );
        M_recovery->setText( QString::fromAscii( buf ) );
    }

    // command count
    if ( player.hasCommandCount() )
    {
        snprintf( buf, 64, "  %5d",
                  player.kickCount() );
        M_kick_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.dashCount() );
        M_dash_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.turnCount() );
        M_turn_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.sayCount() );
        M_say_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.turnNeckCount() );
        M_turn_neck_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.catchCount() );
        M_catch_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.moveCount() );
        M_move_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.changeViewCount() );
        M_change_view_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.tackleCount() );
        M_tackle_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.pointtoCount() );
        M_pointto_count->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, "  %5d",
                  player.attentiontoCount() );
        M_attentionto_count->setText( QString::fromAscii( buf ) );
    }

    // ball related info
    {
        rcsc::Vector2D player_to_ball( ball.x() - player.x(),
                                       ball.y() - player.y() );
        player_to_ball *= reverse;

        snprintf( buf, 64, " %6.2f, %6.2f",
                  player_to_ball.x, player_to_ball.y );
        M_player_ball_rel_pos->setText( QString::fromAscii( buf ) );

        snprintf( buf, 64, " (%7.3f, %6.1f)",
                  player_to_ball.r(), player_to_ball.th().degree() );
        M_player_ball_rel_polar->setText( QString::fromAscii( buf ) );
    }
}
