// -*-c++-*-

/*!
  \file plaeyr_painter.cpp
  \brief player painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "player_painter.h"

#include "draw_config.h"
// model
#include "options.h"
#include "main_data.h"

#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/angle_deg.h>
#include <rcsc/geom/circle_2d.h>

#include <cstring>
#include <cstdio>
#include <cmath>

/*-------------------------------------------------------------------*/
/*

 */
inline
PlayerPainter::Param::Param( const Player & player,
                             const Ball & ball,
                             const rcsc::PlayerType & ptype )
    : x_( 0 )
    , y_( 0 )
    , body_( player.body() )
    , head_( player.head() )
    , body_radius_( 1 )
    , kick_radius_( 5 )
    , have_full_effort_( player.hasFullEffort( ptype.effortMax() ) )
    , player_( player )
    , ball_( ball )
    , player_type_( ptype )
{
    const Options & opt = Options::instance();

    x_ = opt.screenX( player.x() );
    y_ = opt.screenY( player.y() );

    if ( opt.reverseSide() )
    {
        body_ += 180.0;
        head_ += 180.0;
    }

    body_radius_ = opt.scale( ptype.playerSize() );
    kick_radius_ = opt.scale( ptype.kickableArea() );

    if ( body_radius_ < 1 ) body_radius_ = 1;
    if ( kick_radius_ < 5 ) kick_radius_ = 5;

    if ( opt.enlargeMode() )
    {
        if ( opt.playerSize() > 0.0 )
        {
            draw_radius_ = opt.scale( Options::instance().playerSize() );
            if ( draw_radius_ < 1 ) draw_radius_ = 1;
        }
        else
        {
            draw_radius_ = kick_radius_;
        }
    }
    else
    {
        draw_radius_ = body_radius_;
    }
}

/*-------------------------------------------------------------------*/
/*

 */
PlayerPainter::PlayerPainter( const MainData & main_data )
    : M_main_data( main_data )
{

}

/*-------------------------------------------------------------------*/
/*

 */
void
PlayerPainter::draw( QPainter & painter )
{
    if ( ! Options::instance().showPlayers() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getCurrentViewData();

    if ( ! view )
    {
        return;
    }

    const Ball & ball = view->ball();

    if ( Options::instance().playerReverseDraw() )
    {
        const std::vector< Player >::const_reverse_iterator end = view->players().rend();
        for ( std::vector< Player >::const_reverse_iterator it = view->players().rbegin();
              it != end;
              ++it )
        {
            drawAll( painter, *it, ball );
        }
    }
    else
    {
        const std::vector< Player >::const_iterator end = view->players().end();
        for ( std::vector< Player >::const_iterator it = view->players().begin();
              it != end;
              ++it )
        {
            drawAll( painter, *it, ball );
        }
    }
}

/*-------------------------------------------------------------------*/
/*

 */
void
PlayerPainter::drawAll( QPainter & painter,
                        const Player & player,
                        const Ball & ball ) const
{
    const Options & opt = Options::instance();
    const bool selected = Options::instance().isSelectedPlayer( player.side(), player.unum() );
    const Param param( player,
                       ball,
                       M_main_data.viewHolder().playerType( player.type() ) );

    drawBody( painter, param );
    if ( opt.showBodyShadow() )
    {
        drawShadow( painter, param );
    }
    drawEdge( painter, param );

    if ( selected
         && opt.playerFutureCycle() > 0
         && player.hasDelta() )
    {
        drawFuture( painter, param );
    }

    if ( player.hasView() )
    {
        if ( opt.showViewArea() )
        {
            drawViewArea( painter, param );
        }
        else
        {
            drawViewDir( painter, param );
        }
    }

    if ( opt.showCatchableArea() )
    {
        drawCatchableArea( painter, param );
    }

    if ( opt.showTackleArea() )
    {
        drawTackleArea( painter, param );
    }

    if ( selected
         && opt.showKickAccelArea() )
    {
        drawKickAccelArea( painter, param );
    }

    if ( opt.showPointto() )
    {
        drawPointto( painter, param );
    }

    if ( opt.showAttentionto() )
    {
        drawAttentionto( painter, param );
    }

    drawText( painter, param );
}

/*-------------------------------------------------------------------*/
/*

 */
void
PlayerPainter::drawBody( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // decide base color
    painter.setPen( dconf.playerPen() );

    rcsc::SideID side = param.player_.side();
    if ( Options::instance().reverseSide() )
    {
        side = static_cast< rcsc::SideID >( -1 * side );
    }

    switch ( side ) {
    case rcsc::LEFT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( dconf.leftGoalieBrush() );
        }
        else
        {
            painter.setBrush( dconf.leftTeamBrush() );
        }
        break;
    case rcsc::RIGHT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( dconf.rightGoalieBrush() );
        }
        else
        {
            painter.setBrush( dconf.rightTeamBrush() );
        }
        break;
    case rcsc::NEUTRAL:
        painter.setBrush( dconf.shadowBrush() );
        break;
    default:
        painter.setBrush( dconf.shadowBrush() );
        break;
    }


    // decide status color
    if ( ! param.player_.isAlive() )
    {
        painter.setBrush( dconf.shadowBrush() );
    }
    if ( param.player_.isKicking() )
    {
        painter.setPen( dconf.kickPen() );
    }
    if ( param.player_.isKickingFault() )
    {
        painter.setPen( dconf.kickFaultPen() );
        painter.setBrush( dconf.kickFaultBrush() );
    }
    if ( param.player_.isCatching() )
    {
        painter.setBrush( dconf.catchBrush() );
    }
    if ( param.player_.isCatchingFault() )
    {
        painter.setBrush( dconf.catchFaultBrush() );
    }
    if ( param.player_.isTackling() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleBrush() );
    }
    if ( param.player_.isTacklingFault() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleFaultBrush() );
    }
    if ( param.player_.isCollidedBall() )
    {
        painter.setBrush( dconf.collideBallBrush() );
    }
    if ( param.player_.isCollidedPlayer() )
    {
        painter.setBrush( dconf.collidePlayerBrush() );
    }

    painter.drawEllipse( param.x_ - param.draw_radius_ ,
                         param.y_ - param.draw_radius_ ,
                         param.draw_radius_ * 2,
                         param.draw_radius_ * 2 );

}

/*-------------------------------------------------------------------*/
/*

 */
void
PlayerPainter::drawShadow( QPainter & painter,
                           const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // set size
    int shadow_radius = static_cast< int >( param.draw_radius_ * 0.9 );
    // set angle
    int shadow_start_dir
        = static_cast< int >( rint( ( - param.body_ + 90.0 ) * 16 ) );
    //double shadow_end_dir = shadow_start_dir + 180.0;

    // decide shadow color
#if 0
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / param.server_param_.staminaMax() )
                            : 1.0 );
    const QColour & base_col = dconf.playerShadowBrush().color();
    int r = 255 - static_cast< int >( (255 - base_col.red()) * stamina_rate );
    int g = 255 - static_cast< int >( (255 - base_col.green()) * stamina_rate );
    int b = 255 - static_cast< int >( (255 - base_col.blue()) * stamina_rate );
    QBrush tmp_brush( QColour( r, g, b ), Qt::SolidPattern );
    painter.setPen( dconf.transparentPen() );
    painter.setBrush( tmp_brush );
#elif 0
    // create temporary brush
    int col = 255
        - static_cast< int >( 255
                              * ( param.player_.hasStamina()
                                  ? ( param.player_.stamina()
                                      / rcsc::ServerParam::i().staminaMax() )
                                  : 1.0 ) );
    //QBrush tmp_brush( QColor( col, col, col ), Qt::SolidPattern );
    painter.setPen( dconf.transparentPen() );
    //painter.setBrush( tmp_brush );
    painter.setBrush( dconf.shadowBrush( col ) );
#else
    painter.setPen( dconf.transparentPen() );
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / rcsc::ServerParam::i().staminaMax() )
                            : 1.0 );
    //int level = 255 - (int)rint( 255 * rint( stamina_rate * 16.0 ) / 16.0 );
    int level = 255 - (int)rint( 255 * rint( stamina_rate * 8.0 ) / 8.0 );
    painter.setBrush( dconf.shadowBrush( level ) );
#endif
    // draw half circle of shadow
    painter.drawPie( param.x_ - shadow_radius, // left x
                     param.y_ - shadow_radius, // top y
                     shadow_radius * 2, // width
                     shadow_radius * 2, // height
                     shadow_start_dir,
                     180 * 16 ); // span angle
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawEdge( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    int edge_radius = 1;
    if ( opt.enlargeMode() )
    {
        if ( opt.showBodyShadow() )
        {
            painter.setPen( dconf.realBodyPen() );
        }
        else
        {
            painter.setPen( dconf.playerPen() );
        }

        edge_radius = param.body_radius_;
    }
    else
    {
        // draw kickable area edge
        rcsc::SideID side = param.player_.side();
        if ( opt.reverseSide() )
        {
            side = static_cast< rcsc::SideID >( -1 * side );
        }

        if ( side == rcsc::LEFT )
        {
            painter.setPen( dconf.leftTeamPen() );
        }
        else
        {
            painter.setPen( dconf.rightTeamPen() );
        }

        edge_radius = param.kick_radius_;
    }

    painter.setBrush( dconf.transparentBrush() );

    painter.drawEllipse( param.x_ - edge_radius,
                         param.y_ - edge_radius,
                         edge_radius * 2,
                         edge_radius * 2 );

    // body direction line
    if ( ! opt.showBodyShadow() )
    {
        int end_x = opt.absScreenX( param.player_.x() * opt.reverseValue()
                                    + param.player_type_.kickableArea()
                                    * std::cos( param.body_ * rcsc::AngleDeg::DEG2RAD ) );
        int end_y = opt.absScreenY( param.player_.y() * opt.reverseValue()
                                    + param.player_type_.kickableArea()
                                    * std::sin( param.body_ * rcsc::AngleDeg::DEG2RAD ) );
        // same pen & brush
        painter.drawLine( param.x_, param.y_, end_x, end_y );
    }

    // draw stamina status if effort or recovery is decayed.
    if ( param.player_.hasStamina() )
    {
        if ( ! param.have_full_effort_ )
        {
            painter.setPen( dconf.effortDecayedPen() );
            painter.setBrush( dconf.transparentBrush() );
            int radius = param.draw_radius_ + 2;
            painter.drawEllipse( param.x_ - radius,
                                 param.y_ - radius,
                                 radius * 2,
                                 radius * 2 );
        }
        else if ( ! param.player_.hasFullRecovery() )
        {
            painter.setPen( dconf.recoveryDecayedPen() );
            painter.setBrush( dconf.transparentBrush() );
            int radius = param.draw_radius_ + 2;
            painter.drawEllipse( param.x_ - radius,
                                 param.y_ - radius,
                                 radius * 2,
                                 radius * 2 );
        }
    }
}


/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawFuture( QPainter & painter,
                           const PlayerPainter::Param & param ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    // draw future state
    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D pvel( param.player_.deltaX(), param.player_.deltaY() );
    if ( Options::instance().reverseSide() )
    {
        ppos *= -1.0;
        pvel *= -1.0;
    }

    const int last = opt.playerFutureCycle();

    QPoint first_point( opt.absScreenX( ppos.x ),
                        opt.absScreenY( ppos.y ) ) ;
    QPoint last_point = first_point;

    // draw kickable area edge
    rcsc::SideID side = param.player_.side();
    if ( opt.reverseSide() )
    {
        side = static_cast< rcsc::SideID >( -1 * side );
    }

    if ( side == rcsc::LEFT )
    {
        painter.setPen( dconf.rightTeamPen() );
    }
    else
    {
        painter.setPen( dconf.leftTeamPen() );
    }

    painter.setBrush( dconf.transparentBrush() );

    for ( int i = 0; i < last; ++i )
    {
        ppos += pvel;
        pvel *= param.player_type_.playerDecay();

        QPoint pt( opt.absScreenX( ppos.x ),
                   opt.absScreenY( ppos.y ) );
        if ( std::abs( last_point.x() - pt.x() ) < 1
             && std::abs( last_point.y() - pt.y() ) < 1 )
        {
            break;
        }

        painter.drawEllipse( pt.x() - 1,
                             pt.y() - 1,
                             3,
                             3 );
        painter.drawEllipse( pt.x() - param.kick_radius_,
                             pt.y() - param.kick_radius_,
                             param.kick_radius_ * 2,
                             param.kick_radius_ * 2 );

        last_point = pt;
    }

    // draw move line
    painter.setPen( dconf.debugTargetPen() );
    painter.drawLine( first_point, last_point );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawViewArea( QPainter & painter,
                             const PlayerPainter::Param & param ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const double view_width = param.player_.viewWidth();

    const double visible_dist = rcsc::ServerParam::i().visibleDistance();
    const int visible_radius = opt.scale( visible_dist );
    const double view_start_angle = -param.head_ - view_width * 0.5;

    int view_start_angle_int
        = static_cast< int >( rint( view_start_angle * 16 ) );
    int span_angle = static_cast< int >( rint( view_width * 16 ) );

    // draw enlarged view cone and feel area
    if ( opt.isSelectedPlayer( param.player_.side(),
                               param.player_.unum() ) )
    {
        const int UNUM_FAR = opt.scale( 20.0 );
        const int TEAM_FAR = opt.scale( 40.0 );
        const int TEAM_TOOFAR = opt.scale( 60.0 );

        //painter.setRasterOp( Qt::OrROP );
        painter.setPen( dconf.viewAreaPen() );
        //painter.setBrush( dconf.fieldDarkBrush() );
        //painter.setBrush( dconf.transparentBrush() );

        painter.drawArc( param.x_ - UNUM_FAR, // left x
                         param.y_ - UNUM_FAR, // toop y
                         UNUM_FAR * 2, // width
                         UNUM_FAR * 2, // height
                         view_start_angle_int,
                         span_angle );
        painter.drawArc( param.x_ - TEAM_FAR, // left x
                         param.y_ - TEAM_FAR, // toop y
                         TEAM_FAR * 2, // width
                         TEAM_FAR * 2, // height
                         view_start_angle_int,
                         span_angle );
        // pie, no an arc
        painter.drawArc( param.x_ - TEAM_TOOFAR, // left x
                         //painter.drawPie( param.x_ - TEAM_TOOFAR, // left x
                         param.y_ - TEAM_TOOFAR, // toop y
                         TEAM_TOOFAR * 2, // width
                         TEAM_TOOFAR * 2, // height
                         view_start_angle_int,
                         span_angle );

        // draw feeling area circle
        painter.drawArc( param.x_ - visible_radius,
                         param.y_ - visible_radius,
                         visible_radius * 2,
                         visible_radius * 2,
                         0,
                         360 * 16 );

        const double view_start_angle_real = ( param.head_ - view_width * 0.5 ) * rcsc::AngleDeg::DEG2RAD;
        const double view_end_angle_real = ( param.head_ + view_width * 0.5 ) * rcsc::AngleDeg::DEG2RAD;
        // left side view cone end point x
        int lx = param.x_
            + opt.scale( 60.0 * std::cos( view_start_angle_real ) );
        // left side view cone end point y
        int ly = param.y_
            + opt.scale( 60.0  * std::sin( view_start_angle_real ) );
        // right side view cone end point x
        int rx = param.x_
            + opt.scale( 60.0 * std::cos( view_end_angle_real ) );
        // right side view cone end point y
        int ry = param.y_
            + opt.scale( 60.0 * std::sin( view_end_angle_real ) );
        painter.drawLine( lx, ly, param.x_, param.y_ );
        painter.drawLine( rx, ry, param.x_, param.y_ );

        //painter.setRasterOp( Qt::CopyROP );
    }
    else
    {
        // draw normal view cone
        painter.setPen( dconf.viewConePen() );
        painter.setBrush( dconf.transparentBrush() );

        painter.drawPie( param.x_ - visible_radius,
                         param.y_ - visible_radius,
                         visible_radius * 2,
                         visible_radius * 2,
                         view_start_angle_int,
                         span_angle );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawViewDir( QPainter & painter,
                            const PlayerPainter::Param & param ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    double r = ( opt.enlargeMode()
                 ? param.player_type_.kickableArea()
                 : param.player_type_.playerSize() );

    int end_x = opt.absScreenX( param.player_.x() * opt.reverseValue()
                                + r * std::cos( param.head_ * rcsc::AngleDeg::DEG2RAD ) );
    int end_y = opt.absScreenY( param.player_.y() * opt.reverseValue()
                                + r * std::sin( param.head_ * rcsc::AngleDeg::DEG2RAD ) );

    painter.setPen( Qt::black );
    painter.setBrush( dconf.transparentBrush() );
    painter.drawLine( param.x_, param.y_, end_x, end_y );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawCatchableArea( QPainter & painter,
                                  const PlayerPainter::Param & param ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    if ( param.player_.isGoalie() )
    {
        rcsc::SideID side = param.player_.side();
        if ( opt.reverseSide() )
        {
            side = static_cast< rcsc::SideID >( -1 * side );
        }

        int catchable = opt.scale( rcsc::ServerParam::i().catchableArea() );
        painter.setPen( ( side == rcsc::LEFT )
                        ? dconf.leftGoaliePen()
                        : dconf.rightGoaliePen() );
        painter.setBrush( dconf.transparentBrush() );

        painter.drawEllipse( param.x_ - catchable,
                             param.y_ - catchable,
                             catchable * 2,
                             catchable * 2 );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawTackleArea( QPainter & painter,
                               const PlayerPainter::Param & param ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    rcsc::Vector2D ppos( param.player_.x(),
                         param.player_.y() );
    rcsc::Vector2D bpos( param.ball_.x(),
                         param.ball_.y() );
    if ( opt.reverseSide() )
    {
        ppos *= -1.0;
        bpos *= -1.0;
    }

    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - param.body_ );

    // draw tackle area & probability
    double tackle_dist = ( player_to_ball.x > 0.0
                           ? SP.tackleDist()
                           : SP.tackleBackDist() ); //: 0.0 );
    double tackle_prob = ( std::pow( player_to_ball.absX() / tackle_dist,
                                     SP.tackleExponent() )
                           + std::pow( player_to_ball.absY() / SP.tackleWidth(), // 1.25,
                                       SP.tackleExponent() ) );
    if ( tackle_prob < 1.0 )
    {
        rcsc::AngleDeg body_angle = param.body_;
        rcsc::AngleDeg body_angle_side = body_angle + 90.0;
        double body_x = body_angle.cos();
        double body_y = body_angle.sin();
        double forward_x = SP.tackleDist() * body_x;
        double forward_y = SP.tackleDist() * body_y;
        double back_x = SP.tackleBackDist() * -body_x; // 0.0 * -body_x;
        double back_y = SP.tackleBackDist() * -body_y; // 0.0 * -body_y;
        double right_x = SP.tackleWidth() * body_angle_side.cos(); // 1.25 * body_angle_side.cos();
        double right_y = SP.tackleWidth() * body_angle_side.sin(); // 1.25 * body_angle_side.sin();

        QPointArray pts( 5 );
        pts[0].setX( opt.absScreenX( ppos.x + forward_x + right_x ) );
        pts[0].setY( opt.absScreenY( ppos.y + forward_y + right_y ) );
        pts[1].setX( opt.absScreenX( ppos.x + forward_x - right_x ) );
        pts[1].setY( opt.absScreenY( ppos.y + forward_y - right_y ) );
        pts[2].setX( opt.absScreenX( ppos.x + back_x - right_x ) );
        pts[2].setY( opt.absScreenY( ppos.y + back_y - right_y ) );
        pts[3].setX( opt.absScreenX( ppos.x + back_x + right_x ) );
        pts[3].setY( opt.absScreenY( ppos.y + back_y + right_y ) );
        pts[4] = pts[0];

        painter.setPen( dconf.tackleAreaPen() );
        painter.setBrush( dconf.transparentBrush() );

        //painter.drawConvexPolygon( pts, 0, 4 );
        painter.drawPolyline( pts, 0, 5 );

        int text_radius = param.draw_radius_;
        if ( text_radius > 40 )
        {
            text_radius = 40;
        }

        painter.setFont( dconf.playerFont() );
        painter.setPen( dconf.tacklePen() );

        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4 + painter.fontMetrics().ascent(),
                          QString( "TackleProb=%1" ).arg( 1.0 - tackle_prob, 0, 'g', 3 ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawKickAccelArea( QPainter & painter,
                                  const PlayerPainter::Param & param ) const
{
    if ( ! param.ball_.hasDelta() )
    {
        return;
    }

    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D bpos( param.ball_.x(), param.ball_.y() );
    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - param.player_.body() );

    rcsc::Vector2D bnext( bpos.x + param.ball_.deltaX(),
                          bpos.y + param.ball_.deltaY() );
    double ball_dist = player_to_ball.r();

    if ( ball_dist < param.player_type_.kickableArea() )
    {
        double max_kick_accel
            = SP.maxPower()
            * SP.kickPowerRate()
            * ( 1.0 - 0.25 * player_to_ball.th().abs() / 180.0
                - 0.25
                * ( ball_dist - param.player_type_.playerSize() - SP.ballSize() )
                / param.player_type_.kickableMargin() );

        if ( max_kick_accel > SP.ballAccelMax() )
        {
            max_kick_accel = SP.ballAccelMax();
        }

        QPoint bpos_screen( opt.screenX( bpos.x ),
                            opt.screenY( bpos.y ) );
        QPoint bnext_screen( opt.screenX( bnext.x ),
                             opt.screenY( bnext.y ) );
        int max_speed_screen = opt.scale( SP.ballSpeedMax() );
        int max_kick_accel_screen = opt.scale( max_kick_accel );

        painter.setPen( dconf.kickAccelPen() );
        painter.setBrush( dconf.transparentBrush() );

        // draw no noise next ball move only by inertia
        painter.drawLine( bpos_screen, bnext_screen );

        rcsc::Circle2D max_speed_circle( bpos, SP.ballSpeedMax() );
        rcsc::Circle2D max_accel_circle( bnext, max_kick_accel );
        rcsc::Vector2D intersection_1, intersection_2;

        if ( max_speed_circle.intersection( max_accel_circle,
                                            &intersection_1,
                                            &intersection_2 ) != 2 )

        {
            // no intersection points

            // just draw a next ball reachable area by max accel
            painter.drawEllipse( bnext_screen.x() - max_kick_accel_screen,
                                 bnext_screen.y() - max_kick_accel_screen,
                                 max_kick_accel_screen * 2,
                                 max_kick_accel_screen * 2 );
        }
        else
        {
            // exists 2 intersection points

            rcsc::AngleDeg bpos_to_intersection_1 = ( intersection_1 - bpos ).th();
            rcsc::AngleDeg bpos_to_intersection_2 = ( intersection_2 - bpos ).th();

            rcsc::AngleDeg bpos_to_bnext_angle = ( bnext - bpos ).th();

            rcsc::AngleDeg * bpos_start_angle = 0;
            double bpos_angle_span = 0.0;
            if ( bpos_to_intersection_1.isLeftOf( bpos_to_bnext_angle ) )
            {
                bpos_start_angle = &bpos_to_intersection_1;
                bpos_angle_span = ( bpos_to_intersection_2 - bpos_to_intersection_1 ).degree();
                if ( bpos_angle_span < 0.0 )
                {
                    bpos_angle_span += 360.0;
                }
                bpos_angle_span *= -1.0;
            }
            else
            {
                bpos_start_angle = &bpos_to_intersection_2;
                bpos_angle_span = ( bpos_to_intersection_1 - bpos_to_intersection_2 ).degree();
                if ( bpos_angle_span < 0.0 )
                {
                    bpos_angle_span += 360.0;
                }
                bpos_angle_span *= -1.0;
            }

            if ( opt.reverseSide() )
            {
                *bpos_start_angle += 180.0;
            }

            int bpos_start_angle_int
                = static_cast< int >( rint( - bpos_start_angle->degree() * 16 ) );
            int bpos_angle_span_int
                = static_cast< int >( rint( bpos_angle_span * 16 ) );
            painter.drawArc( bpos_screen.x() - max_speed_screen,
                             bpos_screen.y() - max_speed_screen,
                             max_speed_screen * 2,
                             max_speed_screen * 2,
                             bpos_start_angle_int,
                             bpos_angle_span_int  );

            rcsc::AngleDeg bnext_to_intersection_1 = ( intersection_1 - bnext ).th();
            rcsc::AngleDeg bnext_to_intersection_2 = ( intersection_2 - bnext ).th();

            rcsc::AngleDeg bnext_to_bpos_angle = bpos_to_bnext_angle + 180.0;

            rcsc::AngleDeg * bnext_start_angle = 0;
            double bnext_angle_span = 0.0;
            if ( bnext_to_intersection_1.isLeftOf( bnext_to_bpos_angle ) )
            {
                bnext_start_angle = &bnext_to_intersection_1;
                bnext_angle_span = ( bnext_to_intersection_2 - bnext_to_intersection_1 ).degree();
                if ( bnext_angle_span < 0.0 )
                {
                    bnext_angle_span += 360.0;
                }
                bnext_angle_span *= -1.0;
            }
            else
            {
                bnext_start_angle = &bnext_to_intersection_2;
                bnext_angle_span = ( bnext_to_intersection_1 - bnext_to_intersection_2 ).degree();
                if ( bnext_angle_span < 0.0 )
                {
                    bnext_angle_span += 360.0;
                }
                bnext_angle_span *= -1.0;
            }

            if ( opt.reverseSide() )
            {
                *bnext_start_angle += 180.0;
            }

            int bnext_start_angle_int
                = static_cast< int >( rint( - bnext_start_angle->degree() * 16 ) );
            int bnext_angle_span_int
                = static_cast< int >( rint( bnext_angle_span * 16 ) );
            painter.drawArc( bnext_screen.x() - max_kick_accel_screen,
                             bnext_screen.y() - max_kick_accel_screen,
                             max_kick_accel_screen * 2,
                             max_kick_accel_screen * 2,
                             bnext_start_angle_int,
                             bnext_angle_span_int );
        }

        // draw kick info text
        painter.setFont( dconf.playerFont() );
        painter.setPen( dconf.kickAccelPen() );

        //std::snprintf( buf, 32, "MaxAccel=%.3f", max_kick_accel );
        painter.drawText( bnext_screen.x() + 10,
                          bnext_screen.y() + painter.fontMetrics().ascent(),
                          QString( "MaxAccel=%1" ).arg( max_kick_accel, 0, 'g', 4 ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawPointto( QPainter & painter,
                            const PlayerPainter::Param & param ) const
{
    if ( ! param.player_.isPointing() )
    {
        return;
    }

    int px = Options::instance().screenX( param.player_.data().point_x_ );
    int py = Options::instance().screenY( param.player_.data().point_y_ );

    painter.setPen( DrawConfig::instance().pointtoPen() );
    painter.setBrush( DrawConfig::instance().transparentBrush() );

    painter.drawLine( param.x_, param.y_, px, py );
    painter.drawLine( px - 4, py, px + 4, py );
    painter.drawLine( px, py - 4, px, py + 4 );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawAttentionto( QPainter & painter,
                                const PlayerPainter::Param & param ) const
{
    if ( ! param.player_.isFocusing() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getCurrentViewData();

    if ( ! view )
    {
        return;
    }

    const Options & opt = Options::instance();

    const rcsc::SideID self_side = ( Options::instance().reverseSide()
                                     ? static_cast< rcsc::SideID >( - param.player_.side() )
                                     : param.player_.side() );

    const rcsc::SideID side = ( param.player_.data().focus_side_ == 'l'
                                ? rcsc::LEFT
                                : rcsc::RIGHT );
    const int unum = param.player_.data().focus_unum_;

    const std::vector< Player >::const_iterator end = view->players().end();
    for ( std::vector< Player >::const_iterator p = view->players().begin();
          p != end;
          ++p )
    {
        if ( p->side() == side
             && p->unum() == unum )
        {
            if ( self_side == rcsc::LEFT )
            {
                painter.setPen( DrawConfig::instance().leftGoaliePen() );
            }
            else
            {
                painter.setPen( DrawConfig::instance().rightGoaliePen() );
            }
            //painter.setPen( DrawConfig::instance().attentiontoPen() );
            painter.setBrush( DrawConfig::instance().transparentBrush() );


            int px = opt.screenX( p->x() );
            int py = opt.screenY( p->y() );
            //int r = param.draw_radius_ + 3;
            int r = opt.scale( 2.0 );
            int lx = px;
            int ly = py;

            rcsc::Vector2D rel( param.player_.x(),
                                param.player_.y() );
            rel.x -= p->x();
            rel.y -= p->y();

            double len = rel.r();
            if ( len > std::pow( 2.0, 2 ) )
            {
                rel *= 2.0 / len;
                lx = opt.screenX( p->x() + rel.x );
                ly = opt.screenY( p->y() + rel.y );
            }

            painter.drawLine( param.x_, param.y_, lx, ly );
            painter.drawEllipse( px - r, py - r, r * 2, r * 2 );
            break;
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerPainter::drawText( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const int text_radius = std::min( 40, param.draw_radius_ );

    //     painter.save();
    //     painter.translate( param.x_, param.y_ );
    //     painter.rotate( -90.0 );
    //     painter.translate( -param.x_, -param.y_ );

    painter.setFont( dconf.playerFont() );

    if ( opt.showPlayerNumber()
         && opt.showPlayerType() )
    {
        painter.setPen( dconf.playerNumberFontPen() );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4,
                          QString( "%1,t%2" ).arg( param.player_.unum() ).arg( param.player_.type() ) );
    }
    else if ( opt.showPlayerNumber() )
    {
        painter.setPen( dconf.playerNumberFontPen() );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4,
                          QString::number( param.player_.unum() ) );
    }
    else if ( opt.showPlayerType() )
    {
        painter.setPen( dconf.playerNumberFontPen() );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4,
                          QString( "t%1" ).arg( param.player_.type() ) );
    }

    if ( param.player_.hasStamina() )
    {
        QString str;

        if ( opt.showStamina() )
        {
            str += QString::number( static_cast< int >( rint( param.player_.stamina() ) ) );
        }

        if ( opt.showStaminaCapacity()
             && param.player_.hasStaminaCapacity() )
        {
            if ( ! str.isEmpty() ) str += '/';
            str += QString::number( static_cast< int >( rint( param.player_.staminaCapacity() ) ) );
        }

        if ( ! str.isEmpty() )
        {
            // this player is selected
            if ( opt.isSelectedPlayer( param.player_.side(),
                                       param.player_.unum() ) )
            {
                str += QString( ",e%1,r%2" )
                    .arg( param.player_.effort(), 0, 'g', 3 )
                    .arg( param.player_.recovery(), 0, 'g', 3 );
                painter.setPen( dconf.playerNumberFontPen() );
            }
            else
            {
                painter.setPen( dconf.playerStaminaFontPen() );
            }

            painter.drawText( param.x_ - text_radius,
                              param.y_ - text_radius - 3,
                              str );
        }
    }

    //    painter.restore();
}
