// -*-c++-*-

/*!
  \file team_graphic_painter.cpp
  \brief team logo image painter class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "team_graphic_painter.h"

// model
#include "options.h"
#include "main_data.h"

#include <iostream>
#include <cstring>

/*-------------------------------------------------------------------*/
/*!

*/
TeamGraphicPainter::TeamGraphicPainter( const MainData & main_data )
    : M_main_data( main_data )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
TeamGraphicPainter::draw( QPainter & painter )
{
    const Options & opt = Options::instance();

    if ( opt.anonymousMode()
         || ! opt.showTeamLogo() )
    {
        return;
    }

    // update left team pixmap
    if ( M_team_graphic_left_set.size()
         != M_main_data.viewHolder().teamGraphicLeft().tiles().size() )
    {
        if ( M_main_data.viewHolder().teamGraphicLeft().tiles().empty() )
        {
            M_team_graphic_left_set.clear();
            M_team_graphic_pixmap_left.resize( 0, 0 );
        }
        else
        {
            copyTeamGraphic( M_team_graphic_pixmap_left,
                             M_team_graphic_left_set,
                             M_main_data.viewHolder().teamGraphicLeft() );
        }
    }

    // update right team pixmap
    if ( M_team_graphic_right_set.size()
         != M_main_data.viewHolder().teamGraphicRight().tiles().size() )
    {
        if ( M_main_data.viewHolder().teamGraphicRight().tiles().empty() )
        {
            M_team_graphic_right_set.clear();
            M_team_graphic_pixmap_right.resize( 0, 0 );
        }
        else
        {
            copyTeamGraphic( M_team_graphic_pixmap_right,
                             M_team_graphic_right_set,
                             M_main_data.viewHolder().teamGraphicRight() );
        }
    }

    if ( ! M_team_graphic_pixmap_left.isNull() )
    {
        int left_x = 0;
        if ( opt.reverseSide() )
        {
            left_x = opt.canvasWidth() - M_team_graphic_pixmap_left.width();
        }
        painter.drawPixmap( left_x,
                            opt.scoreBoardHeight(),
                            M_team_graphic_pixmap_left );
    }

    if ( ! M_team_graphic_pixmap_right.isNull() )
    {
        int left_x = opt.canvasWidth() - M_team_graphic_pixmap_right.width();
        if ( opt.reverseSide() )
        {
            left_x = 0;
        }
        painter.drawPixmap( left_x,
                            opt.scoreBoardHeight(),
                            M_team_graphic_pixmap_right );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
TeamGraphicPainter::copyTeamGraphic( QPixmap & dst_pixmap,
                                     std::set< rcsc::TeamGraphic::Index > & index_set,
                                     const rcsc::TeamGraphic & team_graphic )
{
    if ( ! team_graphic.isValid() )
    {
        return;
    }

    //copyXpmAll( dst_pixmap, index_set, team_graphic );

    if ( dst_pixmap.width() != team_graphic.width()
         || dst_pixmap.height() != team_graphic.height() )
    {
        dst_pixmap.resize( team_graphic.width(),
                           team_graphic.height() );
        dst_pixmap.fill();
    }

    const rcsc::TeamGraphic::Map::const_reverse_iterator end = team_graphic.tiles().rend();
    for ( rcsc::TeamGraphic::Map::const_reverse_iterator tile = team_graphic.tiles().rbegin();
          tile != end;
          ++tile )
    {
        index_set.insert( tile->first );
        copyTeamGraphicXpmTile( dst_pixmap,
                                tile->first,
                                *(tile->second) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
TeamGraphicPainter::copyTeamGraphicXpmTile( QPixmap & dst_pixmap,
                                            const rcsc::TeamGraphic::Index & index,
                                            const rcsc::TeamGraphic::XpmTile & tile )
{
    const int array_size = 1 + tile.colors().size() + tile.height();

    char ** xpm;
    xpm = new char*[ array_size ];

    // header
    xpm[0] = new char[64];
    snprintf( xpm[0], 64, "%d %d %d %d",
              tile.width(), tile.height(),
              static_cast< int >( tile.colors().size() ),
              tile.cpp() );

    // colors
    std::size_t idx = 1;
    for ( std::vector< boost::shared_ptr< std::string > >::const_iterator col = tile.colors().begin();
          col != tile.colors().end();
          ++col )
    {
        xpm[idx] = new char[ (*col)->length() + 1 ];
        std::strcpy( xpm[idx], (*col)->c_str() );
        ++idx;
    }

    // pixels
    for ( std::vector< std::string >::const_iterator line = tile.pixelLines().begin();
          line != tile.pixelLines().end();
          ++line )
    {
        xpm[idx] = new char[ tile.width() + 1 ];
        std::strcpy( xpm[idx], line->c_str() );
        ++idx;
    }

    copyTeamGraphicXpmTile( dst_pixmap, index.first, index.second, xpm );

    // release memories
    for ( int i = 0; i < array_size; ++i )
    {
        delete [] xpm[i];
    }
    delete [] xpm;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
TeamGraphicPainter::copyTeamGraphicXpmTile( QPixmap & dst_pixmap,
                                            const int x,
                                            const int y,
                                            const char * const * xpm )
{
    if ( dst_pixmap.width() < (x+1) * rcsc::TeamGraphic::TILE_SIZE
         || dst_pixmap.height() < (y+1) * rcsc::TeamGraphic::TILE_SIZE )
    {
        return;
    }

    QPixmap pixmap( xpm );

    if ( pixmap.isNull()
         || pixmap.width() != rcsc::TeamGraphic::TILE_SIZE
         || pixmap.height() != rcsc::TeamGraphic::TILE_SIZE )
    {
        std::cerr << "Failed to create the appropriate pixmap from the xpm tile ("
                  << x << ',' << y << ')'
                  << std::endl;
        return;
    }

    copyBlt( &dst_pixmap,
             x * rcsc::TeamGraphic::TILE_SIZE,
             y * rcsc::TeamGraphic::TILE_SIZE,
             &pixmap );
    //bitBlt( &dst_pixmap, x * 8, y * 8, &pixmap );
    //QPainter painter( &dst_pixmap );
    //painter.drawPixmap( x * 8, y * 8, pixmap );
    //painter.end();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
TeamGraphicPainter::copyXpmAll( QPixmap & dst_pixmap,
                                std::set< rcsc::TeamGraphic::Index > & index_set,
                                const rcsc::TeamGraphic & team_graphic )
{
    if ( team_graphic.tiles().empty()
         || team_graphic.colors().empty()
         || team_graphic.width() <= 0
         || team_graphic.height() <= 0 )
    {
        std::cerr << __FILE__ << ' ' << __LINE__ << ':'
                  << " no data." << std::endl;
        return;
    }


    const int header_size = 1 + team_graphic.colors().size();
    const int array_size = header_size + team_graphic.height();
    const int array_width = team_graphic.width() + 1;

    //
    // allocate memory
    //
    std::cerr << __FILE__ << ' ' << __LINE__ << ':'
              << " create xpm array. size=" << array_size
              << std::endl;
    char ** xpm;
    xpm = new char*[ array_size ];

    //
    // header
    //
    std::cerr << __FILE__ << ' ' << __LINE__ << ':'
              << " print header." << std::endl;
    xpm[0] = new char[64];
    snprintf( xpm[0],
              64, "%d %d %d 1",
              team_graphic.width(), team_graphic.height(),
              static_cast< int >( team_graphic.colors().size() ) );

    int idx = 1;

    //
    // colors
    //
    std::cerr << __FILE__ << ' ' << __LINE__ << ':'
              << " print colors. color_size=" << team_graphic.colors().size()
              << std::endl;
    const std::vector< boost::shared_ptr< std::string > >::const_iterator c_end = team_graphic.colors().end();
    for ( std::vector< boost::shared_ptr< std::string > >::const_iterator color = team_graphic.colors().begin();
          color != c_end;
          ++color, ++idx )
    {
        xpm[idx] = new char[ (*color)->length() + 1 ];
        std::strcpy( xpm[idx], (*color)->c_str() );
    }

    //
    // pixels
    //
    std::cerr << __FILE__ << ' ' << __LINE__ << ':'
              << " print pixels."
              << " image_width="  << team_graphic.width()
              << " image_height=" << team_graphic.height()
              << std::endl;
    for ( ; idx < array_size; ++idx )
    {
        xpm[idx] = new char[ array_width ];
        std::memset( xpm[idx], ' ', array_width - 1 );
        xpm[ array_width - 1 ] = '\0';
    }

    const rcsc::TeamGraphic::Map::const_iterator t_end = team_graphic.tiles().end();
    for ( rcsc::TeamGraphic::Map::const_iterator tile = team_graphic.tiles().begin();
          tile != t_end;
          ++tile )
    {
        index_set.insert( tile->first );

        int x = ( tile->first.first * rcsc::TeamGraphic::TILE_SIZE );
        int y = header_size + ( tile->first.second * rcsc::TeamGraphic::TILE_SIZE );

        std::cerr << __FILE__ << ' ' << __LINE__ << ':'
                  << " print tile (" << tile->first.first << ", " << tile->first.second << ")"
                  << " array=[" << x << ',' << y << ']'
                  << std::endl;

        const std::vector< std::string >::const_iterator l_end = tile->second->pixelLines().end();
        for ( std::vector< std::string >::const_iterator line = tile->second->pixelLines().begin();
              line != l_end;
              ++line, ++y )
        {
            std::memcpy( xpm[y] + x, line->c_str(), line->length() );
        }
    }

    std::cerr << __FILE__ << ' ' << __LINE__ << ':'
              << " create QPixmap." << std::endl;
    dst_pixmap = QPixmap( xpm );

    //
    // release memories
    //
    std::cerr << __FILE__ << ' ' << __LINE__ << ':'
              << " release memory." << std::endl;
    for ( int i = 0; i < array_size; ++i )
    {
        std::cerr << xpm[i] << '\n';
        delete [] xpm[i];
    }
    std::cerr.flush();
    delete [] xpm;
}
