// -*-c++-*-

/*!
  \file font_setting_dialog.cpp
  \brief font setting dialog for the field canvas Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "font_setting_dialog.h"
#include "draw_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
FontButton::FontButton( const QString & name,
                        QFont * font,
                        FontSettingDialog * parent )
    : QPushButton( parent )
    , M_name( name )
    , M_default_font( *font )
    , M_font( font )
{
    updateText();

    connect( this, SIGNAL( clicked() ),
             this, SLOT( fontDialog() ) );
    connect( this, SIGNAL( fontChanged() ),
             parent, SIGNAL( fontChanged() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontButton::setFont( const QFont & font )
{
    if ( *M_font != font )
    {
        *M_font = font;
        this->setFont( font );

        updateText();

        emit fontChanged();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontButton::fontDialog()
{
    bool ok = true;
    QFont font = QFontDialog::getFont( &ok, *M_font );
    if ( ok )
    {
        setFont( font );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontButton::updateText()
{
    QString str = M_name;
    str += tr( " (" );
    str += M_font->family();
    str += tr( "," );
    str += QString::number( M_font->pointSize() );
    str += tr( ")" );
    this->setText( str );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontButton::revert()
{
    setFont( M_default_font );
}

/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/

/*-------------------------------------------------------------------*/
/*!

*/
FontSettingDialog::FontSettingDialog( QWidget * parent )
    : QDialog( parent )
{

    this->setWindowTitle( tr( "Font Settings" ) );

    createControls();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontSettingDialog::createControls()
{
    QVBoxLayout * main_layout = new QVBoxLayout( this );
    main_layout->setSizeConstraint( QLayout::SetFixedSize );

    main_layout->addLayout( createFontButtons() );

    main_layout->addLayout( createAnswerButtons() );
}

/*-------------------------------------------------------------------*/
/*!

*/
FontSettingDialog::~FontSettingDialog()
{
    //std::cerr << "delete FontSettingDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
FontSettingDialog::createFontButtons()
{
    QVBoxLayout * layout = new QVBoxLayout();
    layout->setMargin( 4 );
    layout->setSpacing( 2 );

    {
        FontButton * btn = new FontButton( tr( "Score Board" ),
                                           DrawConfig::instance().scoreBoardFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }
    {
        FontButton * btn = new FontButton( tr( "Ball" ),
                                           DrawConfig::instance().ballFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }
    {
        FontButton * btn = new FontButton( tr( "Player" ),
                                           DrawConfig::instance().playerFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }
    {
        FontButton * btn = new FontButton( tr( "Measure" ),
                                           DrawConfig::instance().measureFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }
    {
        FontButton * btn = new FontButton( tr( "Debug Comment" ),
                                           DrawConfig::instance().debugCommentFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }
    {
        FontButton * btn = new FontButton( tr( "Debug Message" ),
                                           DrawConfig::instance().debugMessageFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }
    {
        FontButton * btn = new FontButton( tr( "Debug Log" ),
                                           DrawConfig::instance().debugLogMessageFontPtr(),
                                           this );
        M_font_buttons.push_back( btn );
        layout->addWidget( btn, 1, Qt::AlignLeft );
    }

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
FontSettingDialog::createAnswerButtons()
{
    QHBoxLayout * layout = new QHBoxLayout();

    QFontMetrics fm = this->fontMetrics();
    int text_width = fm.width( tr( "Cancel" ) ) + 8;

    QPushButton * def = new QPushButton( tr( "Default" ), this );
    //cancel->resize( text_width, fm.height() );
    connect( def, SIGNAL( clicked() ),
             this, SLOT( setDefaultFonts() ) );

    QPushButton * ok = new QPushButton( tr( "OK" ), this );
    ok->resize( text_width, fm.height() );
    connect( ok, SIGNAL( clicked() ),
             this, SLOT( accept() ) );

    QPushButton * cancel = new QPushButton( tr( "Cancel" ), this );
    cancel->resize( text_width, fm.height() );
    connect( cancel, SIGNAL( clicked() ),
             this, SLOT( reject() ) );


    layout->addWidget( def, 0, Qt::AlignLeft );
    layout->addStretch( 1 );
    layout->addWidget( ok, 0, Qt::AlignRight );
    layout->addWidget( cancel, 0, Qt::AlignRight );

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontSettingDialog::revert()
{
    for ( std::vector< FontButton * >::iterator it = M_font_buttons.begin();
          it != M_font_buttons.end();
          ++it )
    {
        (*it)->revert();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontSettingDialog::setDefaultFonts()
{
    DrawConfig::instance().setDefaultFonts();

    for ( std::vector< FontButton * >::iterator it = M_font_buttons.begin();
          it != M_font_buttons.end();
          ++it )
    {
        (*it)->updateText();
    }

    emit fontChanged();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FontSettingDialog::reject()
{
    // reset all changes

    revert();

    QDialog::reject();
}
