// -*-c++-*-

/*!
  \file player_type_dialog.cpp
  \brief player type list ialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "player_type_dialog.h"

#include "main_data.h"

#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>

#include <iostream>
#include <cstdio>

namespace {
const int COL_SIZE = 15; // the number of param string
//const int FONT_SIZE = 10;
//const int CELL_HEIGHT = 18;
}

class PlayerTypeSortProxy
    : public QSortFilterProxyModel {
public:
    PlayerTypeSortProxy( QObject * parent )
        : QSortFilterProxyModel( parent )
      { }

protected:

    virtual
    bool lessThan( const QModelIndex & left,
                   const QModelIndex & right ) const
      {
          const QAbstractItemModel * src = sourceModel();

          double lhs = 0.0;
          double rhs = 0.0;
          std::sscanf( src->data( left ).toString().toAscii(), " %lf ", &lhs );
          std::sscanf( src->data( right ).toString().toAscii(), " %lf ", &rhs );

          return lhs < rhs;
      }
};

/*-------------------------------------------------------------------*/
/*!

*/
PlayerTypeDialog::PlayerTypeDialog( QWidget * parent,
                            const MainData & main_data )
    : QDialog( parent, Qt::MSWindowsFixedSizeDialogHint )
    , M_main_data( main_data )
{
    this->setWindowTitle( tr( "Player Type List" ) );
    this->setMinimumWidth( 640 );

    //QFont font = this->font();
    //font.setPointSize( 8 );
    //this->setFont( font );

    createTable();
}

/*-------------------------------------------------------------------*/
/*!

*/
PlayerTypeDialog::~PlayerTypeDialog()
{
    //std::cerr << "delete PlayerTypeDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerTypeDialog::createTable()
{
    QVBoxLayout * layout = new QVBoxLayout();
    layout->setMargin( 4 );

    M_item_view = new QTableView();
    //M_item_view->setRootIsDecorated( false );
    //M_item_view->setGridStyle( Qt::NoPen );
    M_item_view->verticalHeader()->hide();
    M_item_view->setShowGrid( false );
    M_item_view->setAlternatingRowColors( true );
    M_item_view->setSortingEnabled( true );

    M_item_view->setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
    M_item_view->setVerticalScrollBarPolicy( Qt::ScrollBarAlwaysOff );

    M_item_view->horizontalHeader()->setResizeMode( QHeaderView::ResizeToContents );

    QFont font = M_item_view->font();
    font.setPointSize( 10 );
    M_item_view->setFont( font );


    createModel();

    QSortFilterProxyModel * proxy = new PlayerTypeSortProxy( this );
    proxy->setSourceModel( M_model );
    M_item_view->setModel( proxy );

    //std::cerr << "table row count = " << M_model->rowCount() << std::endl;

    layout->addWidget( M_item_view );
    this->setLayout( layout );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerTypeDialog::createModel()
{
    const int ROW_SIZE = M_main_data.viewHolder().playerTypeCont().size();

    M_model = new QStandardItemModel( ROW_SIZE, 15, this );

    int i = 0;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "id" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Speed Real/Max" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "AccelStep" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "AccelMax" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "StamInc" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Consume" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Kickable" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Decay" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Inertia" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "DashRate" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Size" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "KickMargin" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "KickRand" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "ExtStam" ) ); ++i;
    M_model->setHeaderData( i, Qt::Horizontal, tr( "Effort Max-Min" ) ); ++i;

    updateData();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerTypeDialog::updateData()
{
    if ( ! this->isVisible() )
    {
        return;
    }

    const int ROW_SIZE = M_main_data.viewHolder().playerTypeCont().size();

    if ( M_model->rowCount() != ROW_SIZE )
    {
        std::cerr << "PlayerTypeDialog::updateData() invalid row count"
                  << M_model->rowCount()
                  << std::endl;
        M_model->setRowCount( ROW_SIZE );
    }

    for ( int row = 0; row < ROW_SIZE; ++row )
    {
        const std::map< int , rcsc::PlayerType >::const_iterator it
            = M_main_data.viewHolder().playerTypeCont().find( row );
        if ( it == M_main_data.viewHolder().playerTypeCont().end() )
        {
            continue;
        }

        const rcsc::PlayerType & param = it->second;

        int col = 0;
        char buf[32];

        // id
        M_model->setData( M_model->index( row, col++ ),
                          param.id() );

        // speed real/max
        snprintf( buf, 32, "%5.3f / %5.3f",
                  param.realSpeedMax(), param.playerSpeedMax() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ) );

        // accel step
        M_model->setData( M_model->index( row, col++ ),
                          param.cyclesToReachMaxSpeed() );

        // accel max
        snprintf( buf, 32, "%.4f",
                  rcsc::ServerParam::i().maxPower()
                  * param.dashPowerRate()
                  * param.effortMax() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // stamina inc max
        snprintf( buf, 32, "%.2f",
                  param.staminaIncMax() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // consume
        snprintf( buf, 32, "%.2f",
                  param.getOneStepStaminaComsumption( rcsc::ServerParam::i() ) );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // kickable area
        snprintf( buf, 32, "%.3f",
                  param.playerSize() + param.kickableMargin()
                  + rcsc::ServerParam::i().ballSize() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // decay
        snprintf( buf, 32, "%.3f", param.playerDecay() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // inertia
        snprintf( buf, 32, "%.2f", param.inertiaMoment() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // dash power rate
        snprintf( buf, 32, "%.5f", param.dashPowerRate() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // size
        snprintf( buf, 32, "%.2f", param.playerSize() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // kickable margin
        snprintf( buf, 32, "%.3f", param.kickableMargin() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // kick rand
        snprintf( buf, 32, "%.2f", param.kickRand() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // extra stamina
        snprintf( buf, 32, "%.2f", param.extraStamina() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );

        // effort max - min
        snprintf( buf, 32, "%.3f - %.3f",
                      param.effortMax(), param.effortMin() );
        M_model->setData( M_model->index( row, col++ ),
                          QString::fromAscii( buf ),
                          Qt::DisplayRole );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerTypeDialog::showEvent( QShowEvent * event )
{
    updateData();

    QFontMetrics metrics = M_item_view->fontMetrics();

    //M_item_view->resizeRowsToContents();
    const int row_height = metrics.height() + 2;
    const int ROW_SIZE = M_main_data.viewHolder().playerTypeCont().size();
    for ( int i = 0; i < ROW_SIZE; ++i )
    {
        M_item_view->setRowHeight( i, row_height );
    }

    int i = 0;
    // id
    M_item_view->setColumnWidth( i, metrics.width( "   1" ) + 4 ); ++i;
    // speed max
    M_item_view->setColumnWidth( i, metrics.width( "00.000 / 00.000" ) + 4 ); ++i;
    // accel step
    M_item_view->setColumnWidth( i, metrics.width( "   0" ) + 4 ); ++i;
    // accel max
    M_item_view->setColumnWidth( i, metrics.width( "  0.000000" ) + 4 ); ++i;
    // stamina inc max
    M_item_view->setColumnWidth( i, metrics.width( "  00.00" ) + 4 ); ++i;
    // consume
    M_item_view->setColumnWidth( i, metrics.width( "  00.00" ) + 4 ); ++i;
    // kickable area
    M_item_view->setColumnWidth( i, metrics.width( "  0.0000" ) + 4 ); ++i;
    // decay
    M_item_view->setColumnWidth( i, metrics.width( "  0.0000" ) + 4 ); ++i;
    // inertia
    M_item_view->setColumnWidth( i, metrics.width( "  0.0000" ) + 4 ); ++i;
    // dash power rate
    M_item_view->setColumnWidth( i, metrics.width( "  0.000000" ) + 4 ); ++i;
    // size
    M_item_view->setColumnWidth( i, metrics.width( "  0.00" ) + 4 ); ++i;
    // kickable margin
    M_item_view->setColumnWidth( i, metrics.width( "  0.0000" ) + 4 ); ++i;
    // kick rand
    M_item_view->setColumnWidth( i, metrics.width( "  0.0000" ) + 4 ); ++i;
    // extra stamina
    M_item_view->setColumnWidth( i, metrics.width( "  00.00" ) + 4 ); ++i;
    // effort max - min
    M_item_view->setColumnWidth( i, metrics.width( "  0.000 - 0.000" ) + 4 ); ++i;

    QRect rect = this->geometry();
    QRect child_rect = this->childrenRect();
    int width_diff = rect.width() - child_rect.width();
    int height_diff = rect.height() - child_rect.height();

    rect.setWidth( M_item_view->horizontalHeader()->length()
                   + width_diff );
    rect.setHeight( M_item_view->horizontalHeader()->height()
                    + M_item_view->rowHeight( 0 ) * ROW_SIZE
                    + height_diff );

    this->setGeometry( rect );

    event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerTypeDialog::wheelEvent( QWheelEvent * event )
{
    if ( event->delta() < 0 )
    {
        this->setWindowOpacity( std::max( 0.1, this->windowOpacity() - 0.05 ) );
    }
    else
    {
        this->setWindowOpacity( std::min( 1.0, this->windowOpacity() + 0.05 ) );
    }

    event->accept();
}
