// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_goalie_free_kick.h"

#include "bhv_goalie_basic_move.h"

#include <rcsc/action/body_clear_ball.h>
#include <rcsc/action/body_pass.h>

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/geom/sector_2d.h>

#include <rcsc/player/say_message_builder.h>
#include <rcsc/geom/polygon_2d.h>
/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_GoalieFreeKick::execute( rcsc::PlayerAgent * agent )
{
    static bool s_first_move = false;
    static bool s_second_move = false;
    static int s_second_wait_count = 0;

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        __FILE__": Bhf_GoalieFreeKick" );
    if ( agent->world().gameMode().type() != rcsc::GameMode::GoalieCatch_
         || agent->world().gameMode().side() != agent->world().ourSide()
         || ! agent->world().self().isKickable() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": Bhv_GoalieFreeKick. Not a goalie catch mode" );

        Bhv_GoalieBasicMove().execute( agent );
        return true;
    }


    const long time_diff
        = agent->world().time().cycle()
        - agent->effector().getCatchTime().cycle();
    //- M_catch_time.cycle();

    // reset flags & wait
    if ( time_diff <= 2 )
    {
        s_first_move = false;
        s_second_move = false;
        s_second_wait_count = 0;

        doWait( agent );
        return true;
    }

    // first move
    if ( ! s_first_move )
    {
        //rcsc::Vector2D move_target( rcsc::ServerParam::i().ourPenaltyAreaLine() - 0.8, 0.0 );
        rcsc::Vector2D move_target( rcsc::ServerParam::i().ourPenaltyAreaLineX() - 1.5,
                                    agent->world().ball().pos().y > 0.0 ? -13.0 : 13.0 );
        //rcsc::Vector2D move_target( -45.0, 0.0 );
	const rcsc::PlayerPtrCont & mates = agent->world().teammatesFromSelf();
	const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
	for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
	     it != mates_end;
	     ++it )
	  {
	    if( move_target.dist( (*it)->pos() ) < 3.0 )
	      {
		move_target.x -= 1.5;
	      }
	  }
        s_first_move = true;
        s_second_move = false;
        s_second_wait_count = 0;
        agent->doMove( move_target.x, move_target.y );
        agent->setNeckAction( new rcsc::Neck_ScanField );
        return true;
    }

    // after first move
    // check stamina recovery or wait teammate
    rcsc::Rect2D our_pen( rcsc::Vector2D( -52.5, -40.0 ),
                          rcsc::Vector2D( -36.0, 40.0 ) );
    if ( time_diff < 50
         || agent->world().setplayCount() < 3
         || ( time_diff < rcsc::ServerParam::i().dropBallTime() - 10
              && ( agent->world().self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.9
                   || agent->world().existTeammateIn( our_pen, 20, true )
                   )
              )
         )
    {
        doWait( agent );
        return true;
    }

    // second move
    if ( ! s_second_move )
    {
        rcsc::Vector2D kick_point = getKickPoint( agent );
        agent->doMove( kick_point.x, kick_point.y );
        agent->setNeckAction( new rcsc::Neck_ScanField );
        s_second_move = true;
        s_second_wait_count = 0;
        return true;
    }

    s_second_wait_count++;

    // after second move
    // wait see info
    if ( s_second_wait_count < 5
         || agent->world().seeTime() != agent->world().time() )
    {
        doWait( agent );
        return true;
    }

    s_first_move = false;
    s_second_move = false;
    s_second_wait_count = 0;

    // register kick intention
    doKick( agent );

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Bhv_GoalieFreeKick::getKickPoint( const rcsc::PlayerAgent * agent )
{
    static const double base_x = -43.0;
    static const double base_y = 10.0;

    std::vector< std::pair< rcsc::Vector2D, double > > candidates;
    candidates.reserve( 4 );
    candidates.push_back( std::make_pair( rcsc::Vector2D( base_x, base_y ), 0.0 ) );
    candidates.push_back( std::make_pair( rcsc::Vector2D( base_x, -base_y ), 0.0 ) );
    candidates.push_back( std::make_pair( rcsc::Vector2D( base_x, 0.0 ), 0.0 ) );

    const rcsc::PlayerPtrCont::const_iterator opps_end
        = agent->world().opponentsFromSelf().end();
    for ( rcsc::PlayerPtrCont::const_iterator o
              = agent->world().opponentsFromSelf().begin();
          o != opps_end;
          ++o )
    {
        for ( std::vector< std::pair< rcsc::Vector2D, double > >::iterator it = candidates.begin();
              it != candidates.end();
              ++it )
        {
            it->second += 1.0 / (*o)->pos().dist2( it->first );
        }
    }

    rcsc::Vector2D best_pos = candidates.front().first;
    double min_cong = 10000.0;
    for ( std::vector< std::pair< rcsc::Vector2D, double > >::iterator it = candidates.begin();
          it != candidates.end();
          ++it )
    {
        if ( it->second < min_cong )
        {
            best_pos = it->first;
            min_cong = it->second;
        }
    }

    return best_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Bhv_GoalieFreeKick::doKick( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::Vector2D target_point;
    double pass_speed = 0.0;
    int receiver = 0;

    //direct pass
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates_end;
         ++it )
    {
        bool target_is_free = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     0.0, (*it)->pos().dist( wm.ball().pos() ) + 5.0,
                                     ( (*it)->pos() - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                     ( (*it)->pos() - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps.end();
             ++itr )
        {
            if( sector.contains( (*itr)->pos() ) )
            {
                target_is_free = false;
            }
        }

        if( target_is_free
            && (*it)->posCount() < 3
            && (*it)->pos().x > -30.0
            && (*it)->pos().x > wm.self().pos().x
            && (*it)->pos().dist( wm.self().pos() ) > 10.0 )
        {
            const double ball_decay = rcsc::ServerParam::i().ballDecay();
            const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
            int reach_cycle = (int)(log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( (*it)->pos() ) / ball_speed_max ) / log( ball_decay ));
            double ball_speed = (*it)->playerTypePtr()->kickableArea() * 1.5 * pow( 1.0 / ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            if( rcsc::Body_KickOneStep( (*it)->pos(),
                                        first_speed ).execute( agent ) )
            {
                agent->debugClient().addMessage( "DirectPass" );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             (*it)->pos(),
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                return;
            }
        }
    }

    //advance kick
    double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
    double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
    for( double angle = -80.0; angle <= 80.0; angle += 10.0 )
    {
        bool angle_safety = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     0.0, 20.0,
                                     rcsc::AngleDeg( angle ) - rcsc::AngleDeg( 15.0 ),
                                     rcsc::AngleDeg( angle ) + rcsc::AngleDeg( 15.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps_end;
             ++it )
        {
            if( sector.contains( (*it)->pos() ) )
            {
                angle_safety = false;
            }
        }
        if( angle_safety
            && fabs( angle ) < fabs( best_angle ) )
        {
            best_angle = angle;
        }
    }
    target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
    target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
    if( rcsc::Body_KickOneStep( target_point,
                                ball_speed_max ).execute( agent ) )
    {
        agent->debugClient().addMessage( "DefKickAdvance:%lf",best_angle );
        agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
        return;
    }

    if  ( rcsc::Body_Pass::get_best_pass( agent->world(), &target_point, &pass_speed, &receiver )
          && target_point.dist( rcsc::Vector2D( -50.0, 0.0 ) ) > 20.0 )
    {
        double opp_dist = 100.0;
        const rcsc::PlayerObject * opp
            = agent->world().getOpponentNearestTo( target_point, 20, &opp_dist );
        agent->debugClient().addMessage( "GKickOppDist%.1f", opp ? opp_dist : 1000.0 );
        if ( ! opp || opp_dist > 3.0 )
        {
            rcsc::Body_KickOneStep( target_point,
                                    pass_speed ).execute( agent );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": register goalie kick intention. to (%.1f, %.1f)",
                                target_point.x,
                                target_point.y );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
        }
    }
    else
    {
#if 1
        target_point = getKickPos( agent, receiver );
        if( receiver == 0 )
        {
                rcsc::Body_ClearBall().execute( agent );
                agent->setNeckAction( new rcsc::Neck_ScanField() );
                return;
        }
        pass_speed = getBestSpeed( target_point.dist( agent->world().ball().pos() ) );
        rcsc::Body_KickOneStep( target_point,
                                pass_speed ).execute( agent );
        agent->debugClient().addMessage( "newGoalieKick" );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
#endif
    }
    if ( agent->config().useCommunication()
         && receiver != rcsc::Unum_Unknown )
    {
        rcsc::dlog.addText( rcsc::Logger::ACTION,
                            "%s:%d: execute() set pass communication.freekick"
                            ,__FILE__, __LINE__ );
        rcsc::Vector2D target_buf = target_point - agent->world().self().pos();
        target_buf.setLength( 0.3 );
        agent->addSayMessage( new rcsc::PassMessage( receiver,
                                                     target_point,
                                                     agent->effector().queuedNextBallPos(),
                                                     agent->effector().queuedNextBallVel() ) );
        return;
    }
    rcsc::Body_ClearBall().execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Bhv_GoalieFreeKick::doWait( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D face_target( 0.0, 0.0 );

    if ( wm.self().pos().x > rcsc::ServerParam::i().ourPenaltyAreaLineX()
         - rcsc::ServerParam::i().ballSize()
         - wm.self().playerType().playerSize()
         - 0.5 )
    {
        face_target.assign( rcsc::ServerParam::i().ourPenaltyAreaLineX() - 1.0,
                            0.0 );
    }

    rcsc::Body_TurnToPoint( face_target ).execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
}


/*-------------------------------------------------------------------*/
/*!

*/
double
Bhv_GoalieFreeKick::getBestSpeed( double dist )
{
   double speed = rcsc::ServerParam::i().ballSpeedMax();
   int cycle = 0;
   while( cycle < 10 )
   {
      cycle++;
      double sum = rcsc::calc_sum_geom_series( speed, rcsc::ServerParam::i().ballDecay(), cycle );
      if( sum >= dist )
      {
	 speed = rcsc::calc_first_term_geom_series( dist, rcsc::ServerParam::i().ballDecay(), cycle + 1 );
	 break;
      }
   }
   
   return speed;
}


/*-------------------------------------------------------------------*/
/*!

*/
void
Bhv_GoalieFreeKick::setCandidates( rcsc::Vector2D base_pos, std::vector< rcsc::Vector2D > & candidates )
{
   candidates.clear();
   candidates.push_back( base_pos );
   
   rcsc::Vector2D cand1 = base_pos;
   rcsc::Vector2D cand2 = base_pos;
   rcsc::Vector2D cand3 = base_pos;
   rcsc::Vector2D cand4 = base_pos;
   rcsc::Vector2D cand5 = base_pos;
   rcsc::Vector2D cand6 = base_pos;
   rcsc::Vector2D cand7 = base_pos;
   rcsc::Vector2D cand8 = base_pos;

   double radius = 1.5;
   
   cand1.x += radius;

   cand2.x += radius * cos( M_PI_4 );
   cand2.y += radius * sin( M_PI_4 );

   cand3.y += radius;

   cand4.x += radius * cos( M_PI_4 * 3 );
   cand4.y += radius * sin( M_PI_4 * 3 );

   cand5.x -= radius;

   cand6.x += radius * cos( M_PI_4 * 5 );
   cand6.y += radius * sin( M_PI_4 * 5 );

   cand7.y -= radius;

   cand8.x += radius * cos( M_PI_4 * 7 );
   cand8.y += radius * sin( M_PI_4 * 7 );

   candidates.push_back( cand1 );
   candidates.push_back( cand2 );
   candidates.push_back( cand3 );
   candidates.push_back( cand4 );
   candidates.push_back( cand5 );
   candidates.push_back( cand6 );
   candidates.push_back( cand7 );
   candidates.push_back( cand8 );
   

   for( unsigned int i = 0; i < candidates.size(); i++ )
   {

      if( candidates.at( i ).x > 50.2 )
      {
	 candidates.at( i ).x = 50.5;
      }
      if( candidates.at( i ).y > 32 )
      {
	 candidates.at( i ).y = 32;
      }
      if( candidates.at( i ).y < -32 )
      {
	 candidates.at( i ).y = -32;
      }
   }
   return;
}

rcsc::Vector2D
Bhv_GoalieFreeKick::getKickPos( const rcsc::PlayerAgent * agent, int & receiver )
{
    rcsc::Vector2D target_pos( 0, 0 );
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::AbstractPlayerObject * mate4 = wm.teammate( 4 );
    const rcsc::AbstractPlayerObject * mate5 = wm.teammate( 5 );
    const rcsc::AbstractPlayerObject * target_mate = NULL;

    if( mate4 )
    {
        if( !mate5 )
        {
            target_mate = mate4;
        }
        else
        {
            double dist = wm.self().pos().dist( mate4->pos() );
            if( dist < wm.self().pos().dist( mate5->pos() ) )
            {
                target_mate = mate4;
            }
            else
            {
                target_mate = mate5;
            }
        }
    }
    else if( mate5 )
    {
        target_mate = mate4;
    }
    

    if( target_mate )
    {
        receiver = target_mate->unum();
        rcsc::Vector2D tmp_target = target_mate->pos();
        std::vector< rcsc::Vector2D > candidates;
        std::vector< bool > eval;
        setCandidates( tmp_target, candidates );
        for( unsigned int i = 0; i < candidates.size(); i++ )
        {
            std::vector< rcsc::Vector2D > vertices;
            double tmpY = 0;
            if( wm.self().pos().y > 0 )
            {
                tmpY = candidates.at( i ).y;
            }
            else
            {
                tmpY = candidates.at( i ).y;
            }
            vertices.push_back( rcsc::Vector2D( candidates.at( i ).x + 1, tmpY ) );
            vertices.push_back( rcsc::Vector2D( candidates.at( i ).x - 1, tmpY ) );
            vertices.push_back( wm.self().pos() );
            rcsc::Polygon2D polygon( vertices );
            if( !wm.existOpponentIn( polygon, 3, true )
                && candidates.at( i ).x < 51 )
            {
                eval.push_back( true );
            }
            else
            {
                eval.push_back( false );
            }
            int best = 0;
            double maxY  = 0;
            for( unsigned int i = 0; i < eval.size(); i++ )
            {
                if( eval.at( i ) )
                {
                    if( candidates.at( i ).absY() > maxY )
                    {
                        best = i;
                        maxY = candidates.at( i ).absY();
                    }
                }
            }
            target_pos = candidates.at( best );
        }
    }//end if target_mate
    return target_pos;
}
