// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_opuci_defensive_kick.h"

#include "body_kick_to_corner.h"
#include "neck_opuci_turn_to_next_receiver.h"

#include <rcsc/action/body_advance_ball.h>
#include <rcsc/action/body_dribble.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_clear_ball.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/sector_2d.h>

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_opuciDefensiveKick::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_opuciDefensiveKick"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();

    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    const rcsc::PlayerObject * nearest_opp
        = ( opps.empty()
            ? static_cast< rcsc::PlayerObject * >( 0 )
            : opps.front() );
    const double nearest_opp_dist = ( nearest_opp
                                      ? nearest_opp->distFromSelf()
                                      : 1000.0 );
    const rcsc::Vector2D nearest_opp_pos = ( nearest_opp
                                             ? nearest_opp->pos()
                                             : rcsc::Vector2D( -1000.0, 0.0 ) );

    const rcsc::Rect2D penalty_area( rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength(), -rcsc::ServerParam::i().penaltyAreaHalfWidth() ),
                                     rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength() + rcsc::ServerParam::i().penaltyAreaLength(),
                                                     rcsc::ServerParam::i().penaltyAreaHalfWidth() ) );

    rcsc::Vector2D pass_point;

    if( wm.ball().pos().x < -47.0 && fabs( wm.ball().pos().y ) < 9.5 )
      {
	if( rcsc::Body_ClearBall().execute( agent ) )
	  {
	    agent->debugClient().addMessage( "Clear" );
	    agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
	    return true;
	  }
      }

    //direct pass
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates_end;
         ++it )
    {
        bool target_is_free = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     1.0, (*it)->pos().dist( wm.ball().pos() ) + 3.0,
                                     ( (*it)->pos() - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                     ( (*it)->pos() - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps.end();
             ++itr )
        {
            if( sector.contains( (*itr)->pos() ) )
            {
                target_is_free = false;
            }
        }

        if( target_is_free
            && (*it)->posCount() <= 3
            && (*it)->pos().x > -25.0
            && (*it)->pos().x > wm.self().pos().x - 5.0
            && (*it)->pos().dist( wm.self().pos() ) > 10.0 )
        {
	  pass_point = (*it)->pos() * 0.8 + wm.ball().pos() * 0.2;
            const double ball_decay = rcsc::ServerParam::i().ballDecay();
            const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
            int reach_cycle = (int)(log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( (*it)->pos() ) / ball_speed_max ) / log( ball_decay ));
	    double ball_reach_speed = rcsc::ServerParam::i().kickPowerRate() * rcsc::ServerParam::i().maxPower() * 0.5;
	    double ball_speed = ball_reach_speed / pow( ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            if( penalty_area.contains( wm.ball().pos() ) )
            {
                if( rcsc::Body_KickOneStep( pass_point,
                                            first_speed ).execute( agent ) )
                {
                    agent->debugClient().addMessage( "DirectOneStepPass" );
		    agent->debugClient().setTarget( pass_point );
                    agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                                 pass_point,
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                    agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                    return true;
                }
            }
            else
            {
                rcsc::Body_SmartKick( pass_point,
                                      first_speed,
                                      first_speed * 0.8,
                                      3 ).execute( agent );
                agent->debugClient().addMessage( "DirectPass" );
		agent->debugClient().setTarget( pass_point );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             pass_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
                return true;
            }
        }
    }

    if ( rcsc::Body_Pass::get_best_pass( wm, &pass_point, NULL, NULL ) )
    {
        if ( pass_point.x > wm.self().pos().x - 1.0 )
        {
            bool safety = true;
            const rcsc::Sector2D sector( wm.ball().pos(),
                                         1.0, wm.ball().pos().dist( pass_point ) + 5.0,
                                         ( pass_point - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                         ( pass_point - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
            const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
            for ( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                  it != opps_end;
                  ++it )
            {
                if ( sector.contains( (*it)->pos() ) )
                {
                    safety = false;
                }
            }

            if ( safety )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: do best pass"
                                    ,__FILE__, __LINE__ );
                agent->debugClient().addMessage( "DefKickPass(1)" );
                rcsc::Body_Pass().execute( agent );
                agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
                return true;
            }
        }
    }

    if ( nearest_opp_dist < 7.0 )
    {
        if ( rcsc::Body_Pass().execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: do best pass"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickPass(2)" );
            agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
            return true;
        }
    }

    //don't dribble in penalty area
    if( ! penalty_area.contains( wm.ball().pos() )
	|| nearest_opp_dist > 7.0 )
    {
        // dribble to my body dir
        if ( nearest_opp_dist < 5.0
             && nearest_opp_dist > ( rcsc::ServerParam::i().tackleDist()
                                     + rcsc::ServerParam::i().defaultPlayerSpeedMax() * 1.5 )
             && wm.self().body().abs() < 70.0 )
        {
            const rcsc::Vector2D body_dir_drib_target
                = wm.self().pos()
                + rcsc::Vector2D::polar2vector(5.0, wm.self().body());
            
            int max_dir_count = 0;
            wm.dirRangeCount( wm.self().body(), 20.0, &max_dir_count, NULL, NULL );
            
            if ( body_dir_drib_target.x < rcsc::ServerParam::i().pitchHalfLength() - 1.0
                 && body_dir_drib_target.absY() < rcsc::ServerParam::i().pitchHalfWidth() - 1.0
                 && max_dir_count < 3
            )
            {
                // check opponents
                // 10m, +-30 degree
                const rcsc::Sector2D sector( wm.self().pos(),
                                             0.5, 10.0,
                                             wm.self().body() - 30.0,
                                             wm.self().body() + 30.0 );
                // opponent check with goalie
                if ( ! wm.existOpponentIn( sector, 10, true ) )
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: dribble to my body dir"
                                        ,__FILE__, __LINE__ );
                    agent->debugClient().addMessage( "DefKickDrib(1)" );
                    rcsc::Body_Dribble2008( body_dir_drib_target,
                                            1.0,
                                            rcsc::ServerParam::i().maxDashPower(),
                                            2
                    ).execute( agent );
                    agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                    return true;
                }
            }
        }
        
        rcsc::Vector2D drib_target( 50.0, wm.self().pos().absY() );
        if ( drib_target.y < 20.0 ) drib_target.y = 20.0;
        if ( drib_target.y > 29.0 ) drib_target.y = 27.0;
        if ( wm.self().pos().y < 0.0 ) drib_target.y *= -1.0;
        const rcsc::AngleDeg drib_angle = ( drib_target - wm.self().pos() ).th();
        
        // opponent is behind of me
        if ( nearest_opp_pos.x < wm.self().pos().x + 1.0 )
        {
            // check opponents
            // 15m, +-30 degree
            const rcsc::Sector2D sector( wm.self().pos(),
                                         0.5, 15.0,
                                         drib_angle - 30.0,
                                         drib_angle + 30.0 );
            // opponent check with goalie
            if ( ! wm.existOpponentIn( sector, 10, true ) )
            {
                const int max_dash_step
                    = wm.self().playerType()
                    .cyclesToReachDistance( wm.self().pos().dist( drib_target ) );
                if ( wm.self().pos().x > 35.0 )
                {
                    drib_target.y *= ( 10.0 / drib_target.absY() );
                }
                
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: fast dribble to (%.1f, %.1f) max_step=%d"
                                    ,__FILE__, __LINE__,
                                    drib_target.x, drib_target.y,
                                    max_dash_step );
                agent->debugClient().addMessage( "DefKickDrib(2)" );
                rcsc::Body_Dribble2008( drib_target,
                                        1.0,
                                        rcsc::ServerParam::i().maxDashPower(),
                                        std::min( 5, max_dash_step )
                ).execute( agent );
            }
            else
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: slow dribble to (%f, %f)"
                                    ,__FILE__, __LINE__,
                                    drib_target.x, drib_target.y );
                agent->debugClient().addMessage( "DefKickDrib(3)" );
                rcsc::Body_Dribble2008( drib_target,
                                        1.0,
                                        rcsc::ServerParam::i().maxDashPower(),
                                        2
                ).execute( agent );
                
            }
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return true;
        }
        
        // opp is far from me
        if ( nearest_opp_dist > 5.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp far. dribble(%f, %f)"
                                ,__FILE__, __LINE__,
                                drib_target.x, drib_target.y );
            agent->debugClient().addMessage( "DefKickDrib(4)" );
            rcsc::Body_Dribble2008( drib_target,
                                    1.0,
                                    rcsc::ServerParam::i().maxDashPower() * 0.4,
                                    1
            ).execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return true;
        }

        // opp is near

        // can pass
        if ( rcsc::Body_Pass().execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: pass"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickPass(3)" );
            agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
            return true;
        }
        
        // opp is far from me
        if ( nearest_opp_dist > 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp far. dribble(%f, %f)"
                                ,__FILE__, __LINE__,
                                drib_target.x, drib_target.y );
            agent->debugClient().addMessage( "DefKickDrib(5)" );
            rcsc::Body_Dribble2008( drib_target,
                                    1.0,
                                    rcsc::ServerParam::i().maxDashPower() * 0.2,
                                    1
            ).execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return true;
        }

        if ( nearest_opp_dist > 2.5 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: hold"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickHold" );
            rcsc::Body_HoldBall().execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return true;
        }
    }

    if ( wm.self().pos().x > wm.offsideLineX() - 10.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: kick near side"
                            ,__FILE__, __LINE__ );
        agent->debugClient().addMessage( "DefKickToCorner" );
        Body_KickToCorner( (wm.self().pos().y < 0.0) ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
    }
    else
    {
        rcsc::Vector2D target_point;
        double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
	double first_angle = ( rcsc::Vector2D( -52.5, -33.0 ) - wm.ball().pos() ).th().degree();
	double last_angle = ( rcsc::Vector2D( -52.5, 33.0 ) - wm.ball().pos() ).th().degree();
        double best_angle = 0.0;
	if( wm.ball().pos().y < 0.0 )
	  {
	    best_angle = first_angle;
	  }
	else
	  {
	    best_angle = last_angle;
	  }
        for( double angle = first_angle; angle <= last_angle; angle += 10.0 )
        {
            bool angle_safety = true;
            rcsc::Sector2D sector( wm.ball().pos(),
                                   1.0, 30.0,
                                   angle - 30.0,
                                   angle + 30.0 );
            for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                 it != opps_end;
                 ++it )
            {
                if( sector.contains( (*it)->pos() ) )
                {
                    angle_safety = false;
                }
            }
            if( angle_safety
                && fabs( angle ) < fabs( best_angle ) )
            {
                best_angle = angle;
            }
        }
        target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
        target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
        for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
             it != mates_end;
             ++it )
        {
            rcsc::Sector2D sector( wm.ball().pos(),
                                   1.0, 30.0,
                                   best_angle - 30.0,
                                   best_angle + 30.0 );
            if( sector.contains( (*it)->pos() ) )
            {
                target_point = rcsc::Line2D( wm.ball().pos(), rcsc::AngleDeg( best_angle ) ).projection( (*it)->pos() );
                rcsc::Body_KickOneStep( target_point, ball_speed_max ).execute( agent );
                agent->debugClient().addMessage( "DefAdvancePass(1):%lf",best_angle );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                return true;
            }
            else if( (*it)->pos().dist( target_point ) < 30.0 * rcsc::AngleDeg( 30.0 ).sin() * 0.8 )
            {
                rcsc::Body_KickOneStep( target_point, ball_speed_max ).execute( agent );
                agent->debugClient().addMessage( "DefAdvancePass(2):%lf",best_angle );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                return true;
            }
        }
        if( rcsc::Body_KickOneStep( target_point,
                                    ball_speed_max ).execute( agent ) )
        {
            agent->debugClient().addMessage( "DefKickAdvance(1):%lf",best_angle );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return true;
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: advance"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickAdvance(2)" );
            rcsc::Body_AdvanceBall().execute( agent );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
        }
    }
    return true;
}
