// -*-c++-*-

/*!
  \file bhv_shoot2008.cpp
  \brief advanced shoot planning and behavior.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_opuci_shoot.h"

#include <rcsc/common/logger.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/action/neck_turn_to_goalie_or_scan.h>

#include <rcsc/action/body_smart_kick.h>

OpuciShootTable Bhv_opuciShoot::S_shoot_table;

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_opuciShoot::execute( rcsc::PlayerAgent * agent )
{
    if ( ! agent->world().self().isKickable() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " not ball kickable!"
                  << std::endl;
        rcsc::dlog.addText( rcsc::Logger::ACTION,
                      __FILE__":  not kickable" );
        return false;
    }

    const OpuciShootTable::ShotCont & shots = S_shoot_table.getShots( agent );

    // update
    if ( shots.empty() )
    {
        rcsc::dlog.addText( rcsc::Logger::SHOOT,
                      __FILE__": no shoot route" );
        return false;
    }

    OpuciShootTable::ShotCont::const_iterator shot
        = std::min_element( shots.begin(),
                            shots.end(),
                            OpuciShootTable::ScoreCmp() );

    if ( shot == shots.end() )
    {
        rcsc::dlog.addText( rcsc::Logger::SHOOT,
                      __FILE__": no best shot" );
        //return false;
        agent->debugClient().addMessage( "NoBestShot" );
    }

    // it is necessary to evaluate shoot courses

    rcsc::Vector2D target_point = shot->point_;
    double first_speed = shot->speed_;

    agent->debugClient().addMessage( "PreShoot" );
    agent->debugClient().setTarget( target_point );

    rcsc::Vector2D one_step_vel
        = rcsc::KickTable::calc_max_velocity( ( target_point - agent->world().ball().pos() ).th(),
                                        agent->world().self().kickRate(),
                                        agent->world().ball().vel() );
    double one_step_speed = one_step_vel.r();

    rcsc::dlog.addText( rcsc::Logger::SHOOT,
                  __FILE__": shoot to (%.2f, %.2f) speed=%.2f one_kick_max_speed=%.2f",
                  target_point.x, target_point.y,
                  first_speed,
                  one_step_speed );

    if ( one_step_speed > first_speed * 0.96 )
    {
        rcsc::Body_SmartKick( target_point,
                        one_step_speed,
                        one_step_speed - 0.0001,
                        1 ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_TurnToGoalieOrScan( -1 ) );
        agent->debugClient().addMessage( "Enforce1Step" );
        return true;
    }

    if( rcsc::Body_SmartKick( target_point, first_speed, first_speed * 0.96, 3 ).execute( agent ) )
    {
        if( agent->world().self().tackleProbability() > 0.1 && !agent->world().self().isKickable())
        {
            rcsc::Vector2D r_targetpnt = target_point - agent->world().self().pos();
            double tackle_ang = rcsc::AngleDeg::normalize_angle( r_targetpnt.th().degree() - agent->world().self().body().degree() );
            agent->doTackle( tackle_ang );
            agent->debugClient().addMessage( "BlutalTackleShot" );
        }
    }

    agent->setNeckAction( new rcsc::Neck_TurnToGoalieOrScan( -1 ) );
    return true;
}

