// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_set_play_goal_kick.h"

#include "bhv_set_play.h"
#include "bhv_prepare_set_play_kick.h"
#include "bhv_go_to_static_ball.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_clear_ball.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>
#include <rcsc/action/body_turn_to_angle.h>
#include <rcsc/geom/sector_2d.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_SetPlayGoalKick::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_SetPlayGoalKick"
                        ,__FILE__, __LINE__ );

    if ( isKicker( agent ) )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_SetPlayGoalKick::isKicker( const rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    if ( agent->world().setplayCount() < 30 )
    {
        return false;
    }

    if ( wm.teammatesFromBall().empty() )
    {
        return true;
    }

    const rcsc::PlayerPtrCont::const_iterator end
        = agent->world().teammatesFromBall().end();
    for ( rcsc::PlayerPtrCont::const_iterator it
              = agent->world().teammatesFromBall().begin();
          it != end;
          ++it )
    {
        if ( ! (*it)->goalie()
             && (*it)->distFromBall() < agent->world().ball().distFromSelf() )
        {
            // exist other kicker
            return false;
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Bhv_SetPlayGoalKick::doKick( rcsc::PlayerAgent * agent )
{
    static int S_scan_count = -5;

    // go to point to kick the ball
    if ( Bhv_GoToStaticBall( 0.0 ).execute( agent ) )
    {
        return;
    }

    // already kick point
    if ( ( agent->world().ball().angleFromSelf()
           - agent->world().self().body() ).abs() > 3.0 )
    {
        rcsc::Body_TurnToBall().execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return;
    }

    // wait1
    if ( S_scan_count < 0 )
    {
        S_scan_count++;
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return;
    }

    // wait2
    if ( S_scan_count < 40
         && agent->world().teammatesFromSelf().empty() )
    {
        S_scan_count++;
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return;
    }

    S_scan_count = -5;

    // kick to teammate
    rcsc::Vector2D target_point;

    //check if there is a good segment between opps
    
    // preperation
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();

    // pick up six nearest opponents from you
    std::vector< rcsc::Vector2D > six_opps;
    for( int i = 0 ; i < 6 ; i++ )
        six_opps.push_back( opps[i]->pos() );
    // sort six opponents along y-axis in descending order
    rcsc::Vector2D tmp_2D;
    for( int i = 0 ; i < 6 ; i++ )
    {
        for( int j = i ; j < 6 ; j++ )
        {
            if( six_opps[i].y < six_opps[j].y )
            {
                tmp_2D = six_opps[i];
                six_opps[i] = six_opps[j];
                six_opps[j] = tmp_2D;
            }
        }
    }
    for( int i = 0 ; i < 6 ; i++ )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: six_opps[%d](%f, %f)"
                            ,__FILE__, __LINE__,
                            i, six_opps[i].x, six_opps[i].y );
    }
    // find the segment that the projection of you directly hits
    for( int i = 0 ; i < 5 ; i++ )
    {
        rcsc::Segment2D seg( six_opps[i], six_opps[i+1] );
        rcsc::Line2D line = seg.line();
        if( line.dist( six_opps[i] ) == seg.dist( six_opps[i] ) )
        {
            // found the segment. kick the ball to it.
            rcsc::Vector2D target_pos = (seg.a() + seg.b())/2.0;
            // search for the nearest mate to target_pos
            double min_dist = rcsc::ServerParam::i().pitchLength() * 2.0; // very large
            int unum_receiver;
            for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
                 it != mates_end;
                 ++it )
            {
                if( min_dist > target_pos.dist( (*it)->pos() ) )
                {
                    min_dist = target_pos.dist( (*it)->pos() );
                    unum_receiver = (*it)->unum();
                }
            }
            
            // kick action
            double first_speed = std::min( wm.self().kickRate() * rcsc::ServerParam::i().maxPower(),
                                           rcsc::ServerParam::i().ballSpeedMax() );
            double simed_speed = first_speed;
            double travel_dist = 0.0;
            int elapsed_time = 0;
            while( travel_dist < target_pos.dist( wm.ball().pos() ) )
            {
                elapsed_time++;
                travel_dist += simed_speed;
                simed_speed *= rcsc::ServerParam::i().ballDecay();
            }
            travel_dist -= simed_speed / rcsc::ServerParam::i().ballDecay();
            if( travel_dist > 10 )
                continue;

            rcsc::Vector2D self2tgt = target_pos - wm.ball().pos();
            self2tgt.setLength( travel_dist );
            target_pos = wm.ball().pos() + self2tgt;
            rcsc::Body_KickOneStep( target_pos, first_speed ).execute( agent );

            if ( agent->config().useCommunication() )
            {
                rcsc::dlog.addText( rcsc::Logger::ACTION,
                              __FILE__": goal kick pass communication to %d", unum_receiver );
                agent->addSayMessage( new rcsc::PassMessage( unum_receiver,
                                                             target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
        }
    }
    

    //direct pass
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates_end;
         ++it )
    {
        bool target_is_free = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     0.0, (*it)->pos().dist( wm.ball().pos() ) + 5.0,
                                     ( (*it)->pos() - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                     ( (*it)->pos() - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps.end();
             ++itr )
        {
            if( sector.contains( (*itr)->pos() ) )
            {
                target_is_free = false;
            }
        }

        if( target_is_free
            && (*it)->posCount() < 3
            && (*it)->pos().x > wm.self().pos().x
            && (*it)->pos().dist( wm.self().pos() ) > 10.0 )
        {
            const double ball_decay = rcsc::ServerParam::i().ballDecay();
            const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
            int reach_cycle = (int)(log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( (*it)->pos() ) / ball_speed_max ) / log( ball_decay ));
            double ball_speed = (*it)->playerTypePtr()->kickableArea() * 1.5 * pow( 1.0 / ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            target_point = wm.ball().pos() * 0.2 + (*it)->pos() * 0.8;
            if( target_point.x > -35.0
                || target_point.absY() > 20.0 )
            {
                if( rcsc::Body_KickOneStep( target_point,
                                            first_speed ).execute( agent ) )
                {
                    agent->debugClient().addMessage( "DirectPass" );
                    agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                                 target_point,
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                    agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                    return;
                }
            }
        }
    }

    if  ( rcsc::Body_Pass::get_best_pass( agent->world(),
                                          & target_point,
                                          NULL,
                                          NULL )
          && target_point.dist( rcsc::Vector2D(-50.0, 0.0) ) > 20.0
          )
    {
        double opp_dist = 100.0;
        const rcsc::PlayerObject * opp
            = agent->world().getOpponentNearestTo( target_point,
                                                   10,
                                                   & opp_dist );
        if ( ! opp
             || opp_dist > 5.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: pass to (%f, %f)"
                                ,__FILE__, __LINE__,
                                target_point.x, target_point.y );
            rcsc::Body_Pass().execute( agent );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            return;
        }
    }

    //advance kick
    double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
    double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
    for( double angle = -80.0; angle <= 80.0; angle += 10.0 )
    {
        bool angle_safety = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     0.0, 20.0,
                                     angle - 15.0,
                                     angle + 15.0 );
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps_end;
             ++it )
        {
            if( sector.contains( (*it)->pos() ) )
            {
                angle_safety = false;
            }
        }
        if( angle_safety
            && fabs( angle ) < fabs( best_angle ) )
        {
            best_angle = angle;
        }
    }
    target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
    target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
    target_point = wm.ball().pos() * 0.2 + target_point * 0.8;
    if( target_point.x > -35.0 )
    {
        if( rcsc::Body_KickOneStep( target_point,
                                    ball_speed_max ).execute( agent ) )
        {
            agent->debugClient().addMessage( "DefKickAdvance:%lf",best_angle );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }
    }

    // clear
    rcsc::Body_ClearBall().execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Bhv_SetPlayGoalKick::doMove( rcsc::PlayerAgent * agent )
{
    double dash_power
        = Bhv_SetPlay::get_set_play_dash_power( agent );
    double dist_thr = agent->world().ball().distFromSelf() * 0.07;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    rcsc::Vector2D move_point = M_home_pos;
    move_point.y += agent->world().ball().pos().y * 0.5;

    if ( ! rcsc::Body_GoToPoint( move_point,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        // already there
	if( agent->world().ball().posCount() < 3 ){
	    rcsc::AngleDeg t_angle( 0.0 );
	    rcsc::Body_TurnToAngle( t_angle ).execute( agent );
	}
	else
	    rcsc::Body_TurnToBall().execute( agent );
    }
    agent->setNeckAction( new rcsc::Neck_ScanField() );
}
