// -*-c++-*-

/*!
  \file body_pass.cpp
  \brief advanced pass planning & behavior.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "body_opuci_pass.h"
#include "strategy.h"
//#include "intention_kick.h"
//#include "body_kick_one_step.h"
//#include "body_kick_two_step.h"
//#include "body_kick_multi_step.h"
//#include "body_smart_kick.h"
//#include "body_stop_ball.h"
//#include "body_hold_ball.h"

#include <rcsc/action/obsolete/body_kick_multi_step.h>
#include <rcsc/action/obsolete/intention_kick2007.h>

#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/body_stop_ball.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_dribble.h>

#include <rcsc/player/interception.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/audio_sensor.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/geom/sector_2d.h>
#include <rcsc/soccer_math.h>
#include <rcsc/math_util.h>
#include <rcsc/action/neck_turn_to_point.h>
#include <rcsc/geom/polygon_2d.h>
#include "neck_opuci_turn_to_next_receiver.h"
#include "bhv_opuci_offensive_move.h"

#define DEBUG





/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Body_opuciPass::execute( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::dlog.addText( rcsc::Logger::ACTION,
                        "%s:%d: Body_opuciPass. execute()"
                        ,__FILE__, __LINE__ );

    if ( ! agent->world().self().isKickable() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " not ball kickable!"
                  << std::endl;
        rcsc::dlog.addText( rcsc::Logger::ACTION,
                            "%s:%d:  not kickable"
                            ,__FILE__, __LINE__ );
        return false;
    }

#if 1
    std::vector< rcsc::PlayerObject* > teammates;
    rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->posCount() < 3
            && (*it)->unum() > 1
            && (*it)->pos().x < wm.offsideLineX() )
        {
            teammates.push_back( (*it) );
        }
    }
    //sort
    if( teammates.empty() )
    {
        return false;
    }
    rcsc::PlayerObject* tmp_po;

    for( unsigned int i = 0; i < teammates.size() - 1; i++ )
    {
      for( unsigned int j = i + 1 ; j < teammates.size() ; j++ )
      {
	if( teammates[i]->pos().x < teammates[j]->pos().x )
        {
	  tmp_po = teammates[i];
	  teammates[i] = teammates[j];
	  teammates[j] = tmp_po;
	}
      }
    }
    if( wm.ball().pos().x > 35 )
    {
        for( unsigned int i = 0; i < teammates.size(); i++ )
        {
            if( doShootPass( agent, teammates.at( i ) ) )
            {
                return true;
            }
        }
    }
    if( M_mode == 1 )
    {
      for( unsigned int i = 0; i < teammates.size(); i++ )
	{
	  if( teammates.at( i )->pos().x > wm.self().pos().x )
	    {
	      if( wm.ball().pos().x < 35 )
		{
		  if( doThroughPass( agent, teammates.at( i ) ) )
		    {
		      return true;
		    }
		  if( doAidaPass( agent, teammates.at( i ) ) )
		    {
		      return true;
		    }
		}
	      if( doDirectPass( agent, teammates.at( i ) ) )
		{
		  return true;
		}        
	    }
	}
    }
    else if( M_mode == -1 )
    {
      for( unsigned int i = 0; i < teammates.size(); i++ )
	{
	  if( teammates.at( i )->pos().x <= wm.self().pos().x )
	    {
	      if( doDirectPass( agent, teammates.at( i ) ) )
		{
		  return true;
		}  
	    }
	}
    }
    else
    {
      for( unsigned int i = 0; i < teammates.size(); i++ )
	{
	  if( teammates.at( i )->pos().x > wm.self().pos().x )
	    {
	      if( wm.ball().pos().x < 35 )
		{
		  if( doThroughPass( agent, teammates.at( i ) ) )
		    {
		      return true;
		    }
		  if( doAidaPass( agent, teammates.at( i ) ) )
		    {
		      return true;
		    }
		}
	      if( doDirectPass( agent, teammates.at( i ) ) )
		{
		  return true;
		}        
	    }
	}
      for( unsigned int i = 0; i < teammates.size(); i++ )
	{
	  if( teammates.at( i )->pos().x <= wm.self().pos().x )
	    {
	      if( doDirectPass( agent, teammates.at( i ) ) )
		{
		  return true;
		}  
	    }
	}
    }

#endif

/////////////old//////////////////////////
#if 0
    if( doAidaPass( agent ) )
    {
        return true;
    }
/*
    if( wm.ball().pos().x > 35 )
    {
        if( doShootChancePass( agent ) )
        {
            return true;
        }
    }
*/
    if( Strategy::get_ball_area( wm.ball().pos() ) == Strategy::BA_DribbleAttack
        || Strategy::get_ball_area( wm.ball().pos() ) == Strategy::BA_OffMidField )
    {
        if( doThroughPass( agent ) )
        {
            return true;
        }
        else
        {
            agent->debugClient().addMessage( "NoThrough" );
        }
    }

    if( doDirectPass( agent ) )
    {
        return true;
    }
    else
    {
        agent->debugClient().addMessage( "NoDirect" );
    }
#endif


    return false;
}

double
Body_opuciPass::getBestSpeed( double dist, double move_dist  )
{
    double sm = 0;
    double ps = 0;
    int min_cycle = 0;
    if( move_dist > 3.0 )
    {
        move_dist = 3.0;
    }
    while( sm < move_dist )
    {
        min_cycle++;
        ps +=  rcsc::ServerParam::i().defaultDashPowerRate() * rcsc::ServerParam::i().maxPower();
        sm += ps;
        ps *= rcsc::ServerParam::i().defaultPlayerDecay();
    }
   

/////////////////////////////////////////////////////////
    double speed = rcsc::ServerParam::i().ballSpeedMax();
    int cycle = 0;
    while( cycle < 20 )
    {
        cycle++;
        double sum = rcsc::calc_sum_geom_series( speed, rcsc::ServerParam::i().ballDecay(), cycle );
        if( sum >= dist )
        {
            if( cycle < min_cycle + 1 )
            {
                cycle = min_cycle + 1;
            }
            speed = rcsc::calc_first_term_geom_series( dist, rcsc::ServerParam::i().ballDecay(), cycle );
            break;
        }

    }
   
    return speed;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Body_opuciPass::setCandidates( rcsc::Vector2D base_pos, std::vector< rcsc::Vector2D > & candidates, double radius )
{
    candidates.clear();
    candidates.push_back( base_pos );
    if( radius > 2.0 )
    {
        radius = 2.0;
    }
    rcsc::Vector2D cand1 = base_pos;
    rcsc::Vector2D cand2 = base_pos;
    rcsc::Vector2D cand3 = base_pos;
    rcsc::Vector2D cand4 = base_pos;
    rcsc::Vector2D cand5 = base_pos;
    rcsc::Vector2D cand6 = base_pos;
    rcsc::Vector2D cand7 = base_pos;
    rcsc::Vector2D cand8 = base_pos;

   
    cand1.x += radius;

    cand2.x += radius * cos( M_PI_4 );
    cand2.y += radius * sin( M_PI_4 );

    cand3.y += radius;

    cand4.x += radius * cos( M_PI_4 * 3 );
    cand4.y += radius * sin( M_PI_4 * 3 );

    cand5.x -= radius;

    cand6.x += radius * cos( M_PI_4 * 5 );
    cand6.y += radius * sin( M_PI_4 * 5 );

    cand7.y -= radius;

    cand8.x += radius * cos( M_PI_4 * 7 );
    cand8.y += radius * sin( M_PI_4 * 7 );

    candidates.push_back( cand1 );
    candidates.push_back( cand2 );
    candidates.push_back( cand3 );
    candidates.push_back( cand4 );
    candidates.push_back( cand5 );
    candidates.push_back( cand6 );
    candidates.push_back( cand7 );
    candidates.push_back( cand8 );
   

    for( unsigned int i = 0; i < candidates.size(); i++ )
    {

        if( candidates.at( i ).x > 50.2 )
        {
            candidates.at( i ).x = 50.5;
        }
        if( candidates.at( i ).y > 32 )
        {
            candidates.at( i ).y = 32;
        }
        if( candidates.at( i ).y < -32 )
        {
            candidates.at( i ).y = -32;
        }
    }
    return;
}

rcsc::Vector2D
Body_opuciPass::getKickPos( const rcsc::PlayerAgent * agent, rcsc::Vector2D tmp_target )
{
    rcsc::Vector2D target_pos = tmp_target;
    const rcsc::WorldModel & wm = agent->world();
    std::vector< rcsc::Vector2D > candidates;
    std::vector< bool > eval;
    double radius = tmp_target.dist( wm.ball().pos() ) / 3.0;
    setCandidates( tmp_target, candidates, radius );
    for( unsigned int i = 0; i < candidates.size(); i++ )
    {
        std::vector< rcsc::Vector2D > vertices;
        double tmpY = 0;
        if( wm.self().pos().y > 0 )
        {
	    tmpY = candidates.at( i ).y;
        }
        else
        {
	    tmpY = candidates.at( i ).y;
        }
        vertices.push_back( rcsc::Vector2D( candidates.at( i ).x + 1.0, tmpY ) );
        vertices.push_back( rcsc::Vector2D( candidates.at( i ).x - 1.0, tmpY ) );
        vertices.push_back( wm.self().pos() );
        rcsc::Polygon2D polygon( vertices );
        if( !wm.existOpponentIn( polygon, 5, true ) )
        {
            eval.push_back( true );
            target_pos = candidates.at( i );
        }
        else
        {
	    eval.push_back( false );
        }
        double dist = 0;
        double max = 0;
        int best = 0;
	
        for( unsigned int i = 0; i < eval.size(); i++ )
        {
	    if( eval.at( i ) )
	    {
                if( eval.at( 0 ) )
                {
                    best = 0;
                    break;
                }
                if( true )
                {
                    rcsc::Vector2D mid = ( candidates.at( 0 ) + wm.ball().pos() ) / 2.0;
                    const rcsc::PlayerObject * target_opp = 
                        wm.getOpponentNearestTo( mid, 5, NULL );
                    const rcsc::PlayerObject * nearest_opp =
                        wm.getOpponentNearestTo( candidates.at( 0 ), 5, NULL );
                    
                    if( target_opp )
                    {
                        dist = fabs( target_opp->pos().x - candidates.at( i ).x );
                        if( nearest_opp )
                        {
                            dist += fabs( nearest_opp->pos().x - candidates.at( i ).x );
                        }
                        else
                        {
                            dist += 5.0;
                        }

                        if( max < dist )
                        {
                            max = dist;
                            best = i;
                        }
                    }
                    else
                    {
                        dist = 5.0;
                        if( nearest_opp )
                        {
                            dist += fabs( nearest_opp->pos().x - candidates.at( i ).x );
                        }
                        else
                        {
                            best = i;
                            dist = 100;
                            break;
                        }
                        if( max < dist )
                        {
                            max = dist;
                            best = i;
                        }
                    }
                }
                else
                {
                    dist = candidates.at( i ).x;
                    if( dist > max )
                    {
                        max = dist;
                        best = i;
                    }
                }
	    }
        }//end for loop
        target_pos = candidates.at( best );
    }
    return target_pos;
}



bool
Body_opuciPass::doShootChancePass( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D target_point(50.0, 0.0);
    double first_speed = rcsc::ServerParam::i().ballSpeedMax();
    int receiver = 0;

    const rcsc::PlayerObject * cand_mate1 = NULL;
    const rcsc::PlayerObject * cand_mate2 = NULL;
    const rcsc::PlayerObject * cand_mate3 = NULL;
    const rcsc::PlayerObject * target_mate =NULL;

    const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( !cand_mate1 )
        {
            bool otherSide = false;
            if( wm.self().pos().y >= 0 && (*it)->pos().y < 0 )
            {
                otherSide = true;
            }
            else if( wm.self().pos().y < 0 && (*it)->pos().y >= 0 ) 
            {
                otherSide = true;
            }
            if( (*it)->pos().x > 35
                && (*it)->distFromSelf() < 15
                && ( (*it)->pos().absY() < wm.self().pos().absY() 
                     || ( (*it)->pos().absY() < 10 && otherSide ) ) )
            {
                cand_mate1 = (*it);
            }
        }
        else if( !cand_mate2 )
        {
            bool otherSide = false;
            if( wm.self().pos().y >= 0 && (*it)->pos().y < 0 )
            {
                otherSide = true;
            }
            else if( wm.self().pos().y < 0 && (*it)->pos().y >= 0 ) 
            {
                otherSide = true;
            }
            if( (*it)->pos().x > 35
                && (*it)->distFromSelf() < 15
                && ( (*it)->pos().absY() < wm.self().pos().absY() 
                     || ( (*it)->pos().absY() < 10 && otherSide ) ) )
            {
                cand_mate2 = (*it);
            }
        }
        else if( !cand_mate3 )
        {
            bool otherSide = false;
            if( wm.self().pos().y >= 0 && (*it)->pos().y < 0 )
            {
                otherSide = true;
            }
            else if( wm.self().pos().y < 0 && (*it)->pos().y >= 0 ) 
            {
                otherSide = true;
            }
            if( (*it)->pos().x > 35
                && (*it)->distFromSelf() < 15
                && ( (*it)->pos().absY() < wm.self().pos().absY() 
                     || ( (*it)->pos().absY() < 10 && otherSide ) ) )
            {
                cand_mate3 = (*it);
                break;
            }
        }
    }//end for loop
    
    bool found = false;
    if( cand_mate1 )
    {
        rcsc::Vector2D tmp_target = cand_mate1->pos();
        std::vector< rcsc::Vector2D > candidates1;
        std::vector< bool > eval1;
        double radius = tmp_target.dist( wm.ball().pos() ) / 3.0;
        setCandidates( tmp_target, candidates1, radius );
        for( unsigned int i = 0; i < candidates1.size(); i++ )
        {
            std::vector< rcsc::Vector2D > vertices;
            double tmpY = 0;
            if( wm.self().pos().y > 0 )
            {
                tmpY = candidates1.at( i ).y;
            }
            else
            {
                tmpY = candidates1.at( i ).y;
            }
            vertices.push_back( rcsc::Vector2D( candidates1.at( i ).x + 2, tmpY ) );
            vertices.push_back( rcsc::Vector2D( candidates1.at( i ).x - 2, tmpY ) );
            vertices.push_back( wm.self().pos() );
            rcsc::Polygon2D polygon( vertices );
            if( !wm.existOpponentIn( polygon, 5, true )
                && candidates1.at( i ).x < 51 )
            {
                found = true;
                eval1.push_back( true );
            }
            else
            {
                eval1.push_back( false );
            }
        }
        double minY = 50;
        for( unsigned int i = 0; i < eval1.size(); i++ )
        {
            if( eval1.at( i ) )
            {
                if( candidates1.at( i ).absY() < minY )
                {
                    minY = candidates1.at( i ).absY();
                }
            }
        }
        target_mate = cand_mate1;
        ///////////////////////////////////////////////
        if( cand_mate2 )
        {
            tmp_target = cand_mate2->pos();
            std::vector< rcsc::Vector2D > candidates2;
            std::vector< bool > eval2;
            double radius = tmp_target.dist( wm.ball().pos() ) / 3.0;
            setCandidates( tmp_target, candidates2, radius );
            for( unsigned int i = 0; i < candidates2.size(); i++ )
            {
                std::vector< rcsc::Vector2D > vertices;
                double tmpY = 0;
                if( wm.self().pos().y > 0 )
                {
                    tmpY = candidates2.at( i ).y;
                }
                else
                {
                    tmpY = candidates2.at( i ).y;
                }
                vertices.push_back( rcsc::Vector2D( candidates2.at( i ).x + 2, tmpY ) );
                vertices.push_back( rcsc::Vector2D( candidates2.at( i ).x - 2, tmpY ) );
                vertices.push_back( wm.self().pos() );
                rcsc::Polygon2D polygon( vertices );
                if( !wm.existOpponentIn( polygon, 5, true )
                    && candidates2.at( i ).x < 51 )
                {
                    found = true;
                    eval2.push_back( true );
                }
                else
                {
                    eval2.push_back( false );
                }
            }
            bool better = false;
            for( unsigned int i = 0; i < eval2.size(); i++ )
            {
                if( eval2.at( i ) )
                {
                    if( candidates2.at( i ).absY() < minY )
                    {
                        minY = candidates2.at( i ).absY();
                        better = true;
                    }
                }
            }
            if( better )
            {
                target_mate = cand_mate2;
            }
            ///////////////////////////////////////////////////
            if( cand_mate3 )
            {
                tmp_target = cand_mate3->pos();
                std::vector< rcsc::Vector2D > candidates3;
                std::vector< bool > eval3;
                double radius = tmp_target.dist( wm.ball().pos() ) / 3.0;
                setCandidates( tmp_target, candidates3, radius );
                for( unsigned int i = 0; i < candidates3.size(); i++ )
                {
                    std::vector< rcsc::Vector2D > vertices;
                    double tmpY = 0;
                    if( wm.self().pos().y > 0 )
                    {
                        tmpY = candidates3.at( i ).y;
                    }
                    else
                    {
                        tmpY = candidates3.at( i ).y;
                    }
                    vertices.push_back( rcsc::Vector2D( candidates3.at( i ).x + 2, tmpY ) );
                    vertices.push_back( rcsc::Vector2D( candidates3.at( i ).x - 2, tmpY ) );
                    vertices.push_back( wm.self().pos() );
                    rcsc::Polygon2D polygon( vertices );
                    if( !wm.existOpponentIn( polygon, 5, true )
                        && candidates3.at( i ).x < 51 )
                    {
                        found = true;
                        eval3.push_back( true );
                    }
                    else
                    {
                        eval3.push_back( false );
                    }
                }
                better = false;
                for( unsigned int i = 0; i < eval3.size(); i++ )
                {
                    if( eval3.at( i ) )
                    {
                        if( candidates3.at( i ).absY() < minY )
                        {
                            minY = candidates3.at( i ).absY();
                            better = true;
                        }
                    }
                }
                if( better )
                {
                    target_mate = cand_mate3;
                }
            }//end if cand_mate3
        }//end if cand_mate2
    }//end if cand_mate1
    else
    {
        agent->debugClient().addMessage( "NoGoodTeamMate" );
        return false;
    }
    if( !found )
    {
        agent->debugClient().addMessage( "NoGoodRoute" );
        return false;
    }
    if( target_mate )
    {
        receiver = target_mate->unum();
        target_point = getKickPos( agent, target_mate->pos() );
        first_speed = getBestSpeed( target_point.dist( wm.ball().pos() ), target_mate->pos().dist( wm.ball().pos() ) / 3.0 );
    }
    else
    {
        agent->debugClient().addMessage( "Impossible" );
        return false;
    }
    int kick_step = ( agent->world().gameMode().type() != rcsc::GameMode::PlayOn
                      && agent->world().gameMode().type() != rcsc::GameMode::GoalKick_
                      ? 1
                      : 3 );

    if (  rcsc::Body_SmartKick( target_point,
                                first_speed,
                                first_speed * 0.96,
                                kick_step ).execute( agent ) )
    {
        agent->debugClient().addMessage( "DoOpuciPass" );
        agent->debugClient().setTarget( target_point );
        
        rcsc::KickTable::Sequence seq;
        rcsc::KickTable::instance().simulate( wm,
                                              target_point,
                                              first_speed,
                                              first_speed * 0.94,
                                              3,
                                              seq );
        if( seq.pos_list_.size() == 1 )
        {
            if ( agent->config().useCommunication()
                 && receiver != rcsc::Unum_Unknown )
            {
                rcsc::dlog.addText( rcsc::Logger::ACTION,
                                    "%s:%d: execute() set pass communication.freekick"
                                ,__FILE__, __LINE__ );
                agent->addSayMessage( new rcsc::PassMessage( receiver,
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
        }
        return true;
    }
    else
    {
        agent->debugClient().addMessage( "CantKick" );
        rcsc::Body_HoldBall().execute( agent );
    }
    return false;
}

bool
Body_opuciPass::doThroughPass( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    int receiver = getReceiver( agent );
    if( receiver == 0 )
    {
        return false;
    }
    if( !wm.teammate( receiver ) )
    {
        return false;
    }

    std::vector< rcsc::PlayerObject* > opponents;
    double def_line = 0;
    rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x > 35
            && wm.ball().pos().x < 15 )
        {
            continue;
        }
        if( (*it)->pos().x > def_line )
        {
            def_line = (*it)->pos().x;
        }
    }
    if( wm.offsideLineX() < def_line )
    {
        def_line = wm.offsideLineX();
    }
    if( wm.teammate( receiver )->pos().x < def_line - 5 ) 
    {
        return false;
    }       

    rcsc::Vector2D receiver_pos = wm.teammate( receiver )->pos();
    rcsc::Vector2D top_left;
    rcsc::Vector2D bottom_right;
    if( receiver_pos.y < wm.self().pos().y ) // receiver upper
    {
        top_left = rcsc::Vector2D( receiver_pos.x - 3.0, receiver_pos.y );
        bottom_right = rcsc::Vector2D( def_line + 3.0, wm.self().pos().y - 2.0 );
    }
    else
    {
        top_left = rcsc::Vector2D( receiver_pos.x - 3.0, wm.self().pos().y + 2.0 );
        bottom_right = rcsc::Vector2D( def_line + 3.0, receiver_pos.y );
    }

    agent->debugClient().addLine( rcsc::Vector2D( def_line, -34 ), rcsc::Vector2D( def_line, 34 ) );


    rcsc::Rect2D rect( top_left, bottom_right );
    agent->debugClient().addRectangle( rect );

    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
        if( rect.contains( (*it)->pos() ) )
        {
            agent->debugClient().addMessage( "Check%d", (*it)->unum() );
            opponents.push_back( (*it) );
        }
    }
    if( opponents.empty() )
    {
        return false;
    }
    rcsc::Vector2D target_pos = wm.teammate( receiver )->pos();
    bool success = false;
    double first_speed = 0;
    for( int i = 4; i <= 15; i++ )
    {
        double max = 0;
        int num = 0;
        for( unsigned int j = 0; j < opponents.size(); j++ )
        {
            double sum = rcsc::inertia_n_step_point( opponents.at( j )->pos(),
                                                     opponents.at( j )->vel(),
                                                     i,
                                                     opponents.at( j )->playerTypePtr()->playerDecay() ).x;
            double tmp_speed = 0;
            for( int k = 0; k < i - 1; k++ )
            {
                tmp_speed += opponents.at( j )->playerTypePtr()->dashPowerRate() * 100;
                sum += tmp_speed;
                tmp_speed *= opponents.at( j )->playerTypePtr()->playerDecay();
            }
            sum += opponents.at( j )->playerTypePtr()->kickableMargin() * 2;
            if( max < sum )
            {
                max = sum;
                num = j;
            }
        }
        rcsc::Vector2D tmp_pos( max, opponents.at( num )->pos().y );
        rcsc::Line2D ball_route( wm.ball().pos(), tmp_pos );
        rcsc::Line2D mate_route( wm.teammate( receiver )->pos(), rcsc::AngleDeg( 0 ) );
        target_pos = ball_route.intersection( mate_route );
        rcsc::Vector2D ball_move = tmp_pos - wm.ball().pos();
        if( !target_pos.valid() )
        {
            continue;
        }
        if( target_pos.x > 45 )
        {
            break;
        }
        double ball_speed = rcsc::calc_first_term_geom_series( ball_move.r(),
                                                               rcsc::ServerParam::i().ballDecay(),
                                                               i );
        if( ball_speed > rcsc::ServerParam::i().ballSpeedMax() )
        {
            continue;
        }
        ball_move = target_pos - wm.ball().pos();
        int ball_count = rcsc::calc_length_geom_series( ball_speed,
                                                        ball_move.r(),
                                                        rcsc::ServerParam::i().ballDecay() );
        if( ball_count <= 1 )
        {
            continue;
        }
        if( ball_count > 20 )
        {
            break;
        }
                                                        
        rcsc::Vector2D mate_move = target_pos - rcsc::inertia_n_step_point( wm.teammate( receiver )->pos(),
                                                                            wm.teammate( receiver )->vel(),
                                                                            i,
                                                                            wm.teammate( receiver )->playerTypePtr()->playerDecay() );
        double sum = 0;
        double mate_speed = 0;
        for( int j = 0; j < ball_count - 1; j++ )
        {
            mate_speed += wm.teammate( receiver )->playerTypePtr()->dashPowerRate() * 100;
            sum += mate_speed;
            mate_speed *= wm.teammate( receiver )->playerTypePtr()->playerDecay();
        }
        if( sum >= mate_move.r() )
        {
            success = true;
            first_speed = ball_speed;
            break;
        }
    }

    if( success )
    {
        agent->debugClient().setTarget( target_pos );
        if( rcsc::Body_SmartKick( target_pos, first_speed, first_speed * 0.94, 3 ).execute( agent ) )
        {
            rcsc::KickTable::Sequence seq;
            rcsc::KickTable::instance().simulate( wm,
                                                  target_pos,
                                                  first_speed,
                                                  first_speed * 0.94,
						  3,
                                                  seq );
            if( seq.pos_list_.size() == 1 )
            {
                agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            agent->debugClient().addMessage( "opuciThroughPass%d", receiver );
            return true;
        }
    }

    return false;
}

bool
Body_opuciPass::doDirectPass( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    std::vector< rcsc::Vector2D > teammates;
    std::vector< int > unums;
    std::vector< double > dif;
    rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().dist( wm.self().pos() ) < 25 )
        {
            int ball_cycle = getBallCycle( (*it)->pos().dist( wm.ball().pos() ) );
            if( ball_cycle < 3 )
            {
                continue;
            }
            double x = getDashDist( (*it), ball_cycle - 2 );
            rcsc::Vector2D target_pos = (*it)->pos();
            target_pos.x += x;

            
            rcsc::Vector2D vector = target_pos - wm.self().pos();
            rcsc::AngleDeg angle = vector.th();
            double length = target_pos.dist( wm.self().pos() );
            if( length < 1 )
            {
                length = 1;
            }
            rcsc::Sector2D sector( wm.self().pos(),
                                   0.1, 
                                   length,
                                   angle - 10,
                                   angle + 10 );
            if( !wm.existOpponentIn( sector, 10, true ) 
                && (*it)->unum() > 1 
                && (*it)->posCount() <= 3 
                && (*it)->pos().x > -35
                && (*it)->pos().dist( wm.self().pos() ) > 5 )
            {
                teammates.push_back( target_pos );
                unums.push_back( (*it)->unum() );
                dif.push_back( x );
            }
                             
        }
    }
    if( teammates.size() < 1 )
    {
        return false;
    }

    double max = -10;
    int num = 0;
    std::vector< double > points;
    for( unsigned int i = 0; i < teammates.size(); i++ )
    {
        rcsc::Vector2D top_left;
        rcsc::Vector2D bottom_right;
        if( teammates.at( i ).x > wm.self().pos().x )
        {
            top_left.x = wm.self().pos().x;
            bottom_right.x = teammates.at( i ).x + 5;
        }
        else
        {
            top_left.x = teammates.at( i ).x - 5;
            bottom_right.x = wm.self().pos().x;
        }
        if( teammates.at( i ).y < wm.self().pos().y ) // teammate is upper
        {
            top_left.y = teammates.at( i ).y - 5;
            bottom_right.y = wm.self().pos().y;
        }
        else
        {
            top_left.y = wm.self().pos().y;
            bottom_right.y = teammates.at( i ).y + 5;
        }
        rcsc::Rect2D rect( top_left, bottom_right );
        std::vector< rcsc::Vector2D > opponents;
        rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromSelf().end();
        for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
             it != opp_end;
             ++it )
        {
            if( rect.contains( (*it)->pos() ) )
            {
                opponents.push_back( (*it)->pos() );
            }
        }
        double point = 0;
        if( opponents.size() < 1 )
        {
            point = teammates.at( i ).x - wm.self().pos().x;
            points.push_back( point );
        }
        else
        {
            if( evalDirect( agent, teammates.at( i ), opponents ) > 0 )
            {
                point = teammates.at( i ).x - wm.self().pos().x;
            }
            else
            {
                point = -100;
            }
            points.push_back( point );
        }
    }
    
    for( unsigned int i = 0; i < points.size(); i++ )
    {
        if( points.at( i ) > max )
        {
            max = points.at( i );
            num = i;
        }
        agent->debugClient().addMessage( "Num%d:%d", unums.at( i ), points.at( i ) );
    }

    if( max < 0 )
    {
        return false;
    }
    rcsc::Vector2D target_pos = teammates.at( num );
    int receiver = unums.at( num );
    rcsc::Circle2D circle( target_pos, 3.0 );
    double first_speed = getBestSpeed( target_pos.dist( wm.ball().pos() ), dif.at( num ) );
    if( rcsc::Body_SmartKick( target_pos, first_speed, first_speed * 0.94, 3 ).execute( agent ) )
    {
        rcsc::KickTable::Sequence seq;
        rcsc::KickTable::instance().simulate( wm,
                                              target_pos,
                                              first_speed,
                                              first_speed * 0.94,
                                              3,
                                              seq );
        if( seq.pos_list_.size() == 1 )
        {
            agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                         agent->effector().queuedNextBallPos(),
                                                         agent->effector().queuedNextBallVel() ) );
        }
        agent->debugClient().addMessage( "opuciDirectPass%d", receiver );
        agent->debugClient().setTarget( target_pos );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: opuciDPass Receiver %d: Type ID %d"
                            ,__FILE__, __LINE__, receiver, agent->world().teammateHeteroID( receiver ) );
        
        return true;
    }
    return false;
}

int
Body_opuciPass::getReceiver( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    int receiver = 0;
    int left_opp = 0;
    int right_opp = 0;
    rcsc::PlayerPtrCont::const_iterator opps_end = wm.opponentsFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != opps_end;
         ++it )
    {
        if( (*it)->pos().x > wm.self().pos().x )
        {
            if( (*it)->pos().y > wm.self().pos().y )
            {
                left_opp++;
            }
            else
            {
                right_opp++;
            }
        }
    }
    switch( wm.self().unum() )
    {
    case 11:
        if( wm.teammate( 9 ) )
        {
            if( wm.teammate( 10 ) )
            {
                if( wm.self().pos().dist2( wm.teammate( 9 )->pos() )
                    < wm.self().pos().dist2( wm.teammate( 10 )->pos() ) )
                {
                    receiver = 9;
                }
                else
                {
                    receiver = 10;
                }
            }
            else
            {
                receiver = 9;
            }
        }
        else if( wm.teammate( 10 ) )
        {
            receiver = 10;
        }
        break;
    case 10:
    case 9:
        if( wm.teammate( 11 ) )
        {
            receiver = 11;
        }
        break;
    case 8:
        if( wm.teammate( 10 ) )
        {
            if( right_opp < 3 )
            {
                receiver = 10;
            }
            else if( wm.teammate( 11 ) )
            {
                receiver = 11;
            }
        }
        else if( wm.teammate( 11 ) )
        {
            receiver = 11;
        }
        break;
    case 7:
        if( wm.teammate( 9 ) )
        {
            if( left_opp < 3 )
            {
                receiver = 9;
            }
            else if( wm.teammate( 11 ) )
            {
                receiver = 11;
            }
        }
        else if( wm.teammate( 11 ) )
        {
            receiver = 11;
        }
        break;
    default:
        break;
    }
    return receiver;
}


int 
Body_opuciPass::evalDirect( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos, std::vector< rcsc::Vector2D > & opponents )
{
    const rcsc::WorldModel & wm = agent->world();
    int point = 0;
    int min = 0;
    double ball_speed = getBestSpeed( target_pos.dist( wm.ball().pos() ), 0 );
    for( unsigned int i = 0; i < opponents.size(); i++ )
    {
        rcsc::Segment2D ball_to_target( wm.ball().pos(), target_pos );
        int opp_cycle = 1; // turn first
        rcsc::Vector2D opp_pos = opponents.at( i );
        double opp_dist = ball_to_target.dist( opp_pos );
        double opp_speed = 0;
        double sum = 0;
        while( sum < opp_dist )
        {
            opp_speed += rcsc::ServerParam::i().defaultDashPowerRate() * 100;
            sum += opp_speed;
            opp_speed *= rcsc::ServerParam::i().defaultPlayerDecay();
            opp_cycle++;
        }

        double tmp_speed = ball_speed;
        int ball_cycle1 = 0;
        sum = 0;
        rcsc::Vector2D via_pos = ball_to_target.nearestPoint( opp_pos );
        rcsc::Segment2D opp_line( opp_pos, rcsc::Vector2D( 52.0, opp_pos.y ) );
/*
        if( ball_to_target.existIntersection( opp_line ) )
        {
            via_pos = ball_to_target.intersection( opp_line );
        }
*/
        double ball_dist1 = via_pos.dist( wm.ball().pos() );
        while( sum < ball_dist1 )
        {
            sum += tmp_speed;
            tmp_speed *= rcsc::ServerParam::i().ballDecay();
            ball_cycle1++;
            if( tmp_speed < 0.5 )
            {
                return -10;
            }
        }
        if( ball_cycle1 > opp_cycle )
        {
            return -10;
        }

        double ball_dist2 = target_pos.dist( wm.ball().pos() );
        int ball_cycle2 = 0;
        sum = 0;
        tmp_speed = ball_speed;
        while( sum < ball_dist2 )
        {
            sum += tmp_speed;
            tmp_speed *= rcsc::ServerParam::i().ballDecay();
            ball_cycle2++;
            if( tmp_speed < 0.5 )
            {
                return -10;
            }
        }
/*        
        double opp_dist2 = target_pos.dist( opp_pos );
        int opp_cycle2 = 1;
        sum = 0;
        opp_speed = 0;
        while( sum < opp_dist2 )
        {
            opp_speed += rcsc::ServerParam::i().defaultDashPowerRate() * 100;
            sum += opp_speed;
            opp_speed *= rcsc::ServerParam::i().defaultPlayerDecay();
            opp_cycle2++;
        }
        if( opp_cycle2 < ball_cycle2 )
        {
            return -10;
        }
*/
        point = opp_cycle - ball_cycle1;
        if( target_pos.x < wm.self().pos().x )
        {
            point /= 2;
        }
        if( point < min )
        {
            min = point;
        }
    }   
    return min;
}

bool
Body_opuciPass::doAidaPass( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    std::vector< rcsc::Vector2D > candidates(5);
    Bhv_opuciOffensiveMove::findDefenseLine( agent, &candidates );

    std::vector< int > mate_unum;
    double min = 100;
    int num = -1;
    for( unsigned int i = 0; i < candidates.size(); i++ )
    {
        if( !candidates.at( i ).valid() )
        {
            break;
        }
        agent->debugClient().addCircle( candidates.at( i ), 1 );
        const rcsc::PlayerObject * tmp = wm.getTeammateNearestTo( candidates.at( i ), 5, NULL );
        if( tmp )
        {
            double from_self = candidates.at( i ).dist( wm.self().pos() );
            double from_ball = candidates.at( i ).dist( wm.ball().pos() );
            double length = from_ball;
            if( length < 1 )
            {
                length = 1;
            }
            double from_mate = candidates.at( i ).dist( tmp->pos() );
            double from_opp = 100;
            rcsc::Vector2D ball_to = candidates.at( i ) - wm.ball().pos();
            double width = 15;
            rcsc::Sector2D sector( wm.ball().pos(),
                                   0.1, length,
                                   ball_to.th() - width,
                                   ball_to.th() + width );
//            rcsc::Vector2D a = ball_to.rotatedVector( width ) + wm.ball().pos();
//            rcsc::Vector2D b = ball_to.rotatedVector( -width ) + wm.ball().pos();
//            agent->debugClient().addTriangle( wm.ball().pos(), a, b );
            const rcsc::PlayerObject * opp = wm.getOpponentNearestTo( candidates.at( i ), 10, NULL );
            if( opp )
            {
                from_opp = candidates.at( i ).dist( opp->pos() );
            }
            int ball_cycle = getBallCycle( from_ball );
            if( ball_cycle < 3 ||
                getDashDist( tmp, ball_cycle - 2 ) < from_mate )
            {
                mate_unum.push_back( -1 );
            }
            else if( from_self < 20
                     && from_mate < from_self / 2
                     && !wm.existOpponentIn( sector, 10, true )
                     && tmp->posCount() <= 3 )
            {
                mate_unum.push_back( tmp->unum() );
                if( from_mate < min )
                {
                    min = from_mate;
                    num = i;
                }
            }
            else
            {
                mate_unum.push_back( -1 );
            }
        }
        else
        {
            mate_unum.push_back( -1 );
        }
    }
    if( num >= 0 )
    {
        rcsc::Vector2D target_pos = candidates.at( num );
        double ball_speed = getBestSpeed( wm.ball().pos().dist( target_pos ), 0 );
        int receiver = mate_unum.at( num );
        if( rcsc::Body_SmartKick( target_pos, ball_speed, ball_speed * 0.94, 3 ).execute( agent ) )
        {
            rcsc::KickTable::Sequence seq;
            rcsc::KickTable::instance().simulate( wm,
                                                  target_pos,
                                                  ball_speed,
                                                  ball_speed * 0.94,
                                                  3,
                                                  seq );
            if( seq.pos_list_.size() == 1 )
            {
                agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            
            agent->debugClient().setTarget( target_pos );
            agent->debugClient().addMessage( "AidaPass%d", receiver );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: APass receiver %d, Type ID %d"
                                ,__FILE__, __LINE__, receiver, agent->world().teammateHeteroID( receiver ) );
            return true;
        }
    }

    return false;
}


double
Body_opuciPass::getDashDist( const rcsc::PlayerObject * player, int dash_count )
{
    double dist = 0;
    double speed = 0;
    for( int i = 0; i < dash_count; i++ )
    {
        speed += player->playerTypePtr()->dashPowerRate() * 100;
        dist += speed;
        speed *= player->playerTypePtr()->playerDecay();
    }
    return dist;
}

int
Body_opuciPass::getBallCycle( double dist )
{
    double sum = 0;
    double speed = rcsc::ServerParam::i().ballSpeedMax();
    int cycle = 0;
    while( cycle < 20 )
    {
        sum += speed;
        speed *= rcsc::ServerParam::i().ballDecay();
        cycle++;
        if( sum >= dist )
        {
            return cycle;
        }
    }
    return 20;
}


bool
Body_opuciPass::doThroughPass( rcsc::PlayerAgent * agent, rcsc::PlayerObject * teammate )
{
    const rcsc::WorldModel & wm = agent->world();
    std::vector< rcsc::PlayerObject* > opponents;
    double def_line = 0;
    rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x > 35
            && wm.ball().pos().x < 15 )
        {
            continue;
        }
        if( (*it)->pos().x > def_line )
        {
            def_line = (*it)->pos().x;
        }
    }
    if( wm.offsideLineX() < def_line )
    {
        def_line = wm.offsideLineX();
    }
    if( teammate->pos().x < def_line - 5 )
    {
        return false;
    }
    
    rcsc::Vector2D receiver_pos = teammate->pos();
    int receiver = teammate->unum();
    rcsc::Vector2D top_left;
    rcsc::Vector2D bottom_right;
    
    if( receiver_pos.y < wm.self().pos().y ) // receiver upper
    {
        top_left = rcsc::Vector2D( receiver_pos.x - 3.0, receiver_pos.y );
        bottom_right = rcsc::Vector2D( def_line + 3.0, wm.self().pos().y - 2.0 );
    }
    else
    {
        top_left = rcsc::Vector2D( receiver_pos.x - 3.0, wm.self().pos().y + 2.0 );
        bottom_right = rcsc::Vector2D( def_line + 3.0, receiver_pos.y );
    }
    agent->debugClient().addLine( rcsc::Vector2D( def_line, -34 ), rcsc::Vector2D( def_line, 34 ) );
    rcsc::Rect2D rect( top_left, bottom_right );
    agent->debugClient().addRectangle( rect );
    

    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
        if( rect.contains( (*it)->pos() ) )
        {
            agent->debugClient().addMessage( "Check%d", (*it)->unum() );
            opponents.push_back( (*it) );
        }
    }
    if( opponents.empty() )
    {
        return false;
    }

    rcsc::Vector2D target_pos = teammate->pos();
    bool success = false;
    double first_speed = 0;
    for( int i = 4; i <= 15; i++ )
    {
        double max = 0;
        int num = 0;
        for( unsigned int j = 0; j < opponents.size(); j++ )
        {
            double sum = rcsc::inertia_n_step_point( opponents.at( j )->pos(),
                                                     opponents.at( j )->vel(),
                                                     i,
                                                     opponents.at( j )->playerTypePtr()->playerDecay() ).x;
            double tmp_speed = 0;
            for( int k = 0; k < i - 1; k++ )
            {
                tmp_speed += opponents.at( j )->playerTypePtr()->dashPowerRate() * 100;
                sum += tmp_speed;
                tmp_speed *= opponents.at( j )->playerTypePtr()->playerDecay();
            }
            sum += opponents.at( j )->playerTypePtr()->kickableMargin() * 2;
            if( max < sum )
            {
                max = sum;
                num = j;
            }
        }
        rcsc::Vector2D tmp_pos( max, opponents.at( num )->pos().y );
        rcsc::Line2D ball_route( wm.ball().pos(), tmp_pos );
        rcsc::Line2D mate_route( receiver_pos, rcsc::AngleDeg( 0 ) );
        target_pos = ball_route.intersection( mate_route );
        rcsc::Vector2D ball_move = tmp_pos - wm.ball().pos();
        if( !target_pos.valid() )
        {
            continue;
        }
        if( target_pos.x > 45 )
        {
            break;
        }
        double ball_speed = rcsc::calc_first_term_geom_series( ball_move.r(),
                                                               rcsc::ServerParam::i().ballDecay(),
                                                               i );
        if( ball_speed > rcsc::ServerParam::i().ballSpeedMax() )
        {
            continue;
        }
        ball_move = target_pos - wm.ball().pos();
        int ball_count = rcsc::calc_length_geom_series( ball_speed,
                                                        ball_move.r(),
                                                        rcsc::ServerParam::i().ballDecay() );
        if( ball_count <= 1 )
        {
            continue;
        }
        if( ball_count > 20 )
        {
            break;
        }
                                                        
        rcsc::Vector2D mate_move = target_pos - rcsc::inertia_n_step_point( wm.teammate( receiver )->pos(),
                                                                            wm.teammate( receiver )->vel(),
                                                                            i,
                                                                            wm.teammate( receiver )->playerTypePtr()->playerDecay() );
        double sum = 0;
        double mate_speed = 0;
        for( int j = 0; j < ball_count - 1; j++ )
        {
            mate_speed += wm.teammate( receiver )->playerTypePtr()->dashPowerRate() * 100;
            sum += mate_speed;
            mate_speed *= wm.teammate( receiver )->playerTypePtr()->playerDecay();
        }
        if( sum >= mate_move.r() )
        {
            success = true;
            first_speed = ball_speed;
            break;
        }
    }

    if( success )
    {
        agent->debugClient().setTarget( target_pos );
        if( rcsc::Body_SmartKick( target_pos, first_speed, first_speed * 0.94, 3 ).execute( agent ) )
        {
            rcsc::KickTable::Sequence seq;
            rcsc::KickTable::instance().simulate( wm,
                                                  target_pos,
                                                  first_speed,
                                                  first_speed * 0.94,
						  3,
                                                  seq );
            if( seq.pos_list_.size() == 1 )
            {
                agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            agent->debugClient().addMessage( "opuciThroughPass%d", receiver );
            return true;
        }
    }
    
    return false;
}
bool
Body_opuciPass::doDirectPass( rcsc::PlayerAgent * agent, rcsc::PlayerObject * teammate )
{
    const rcsc::WorldModel & wm = agent->world();
    if(  teammate->pos().dist( wm.self().pos() ) > 30
         || teammate->pos().x < -35 )
    {
        return false;
    }
    int ball_cycle = getBallCycle( teammate->pos().dist( wm.ball().pos() ) );
    if( ball_cycle < 3 )
    {
        return false;
    }
    double x = getDashDist( teammate, ball_cycle - 3 );
    rcsc::Vector2D target_pos = teammate->pos();
    target_pos.x += x;
    int receiver = teammate->unum();
    
    rcsc::Vector2D vector = target_pos - wm.ball().pos();
    rcsc::AngleDeg angle = vector.th();
    double length = target_pos.dist( wm.ball().pos() );
    if( length < 1 )
    {
        length = 1;
    }
    rcsc::Sector2D sector( wm.ball().pos(),
                           0.1,
                           length,
                           angle - 10,
                           angle + 10 );
    if( wm.existOpponentIn( sector, 10, true ) )
    {
        return false;
    }
    
    
    rcsc::Vector2D top_left;
    rcsc::Vector2D bottom_right;
    if( teammate->pos().x > wm.self().pos().x )
    {
        top_left.x = wm.self().pos().x;
        bottom_right.x = teammate->pos().x + 5;
    }
    else
    {
        top_left.x = teammate->pos().x - 5;
        bottom_right.x = wm.self().pos().x;
    }
    if( teammate->pos().y < wm.self().pos().y )
    {
        top_left.y = teammate->pos().y - 5;
        bottom_right.y = wm.self().pos().y;
    }
    else
    {
        top_left.y = wm.self().pos().y;
        bottom_right.y = teammate->pos().y + 5;
    }
    
    rcsc::Rect2D rect( top_left, bottom_right );
    std::vector< rcsc::PlayerObject* > opponents;
    rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
      
      if( rect.contains( (*it)->pos() ) && (*it)->pos().dist( teammate->pos() ) < 10.0) // magic number
        {
	  opponents.push_back( (*it) );
        }
    }    
    int success = false;
    double ball_speed = rcsc::calc_first_term_geom_series( teammate->pos().dist( wm.ball().pos() ),
                                                           rcsc::ServerParam::i().ballDecay(),
                                                           ball_cycle );
                                                                                 

    if( opponents.empty() )
    {
        success = true;
    }
    else
    {
        for( unsigned int i = 0; i < opponents.size(); i++ )
        {
            rcsc::Segment2D ball_to_target( wm.ball().pos(), target_pos );
            int opp_cycle = 1;
            rcsc::Vector2D opp_pos = rcsc::inertia_n_step_point( opponents.at( i )->pos(),
                                                                 opponents.at( i )->vel(),
                                                                 3,
                                                                 opponents.at( i )->playerTypePtr()->playerDecay() );
            double opp_dist = ball_to_target.dist( opp_pos );
            double opp_speed = 0;
            double sum = 0;
            while( sum < opp_dist )
            {
                opp_speed += opponents.at( i )->playerTypePtr()->dashPowerRate() * 100;
		if( opp_speed > opponents.at( i )->playerTypePtr()->playerSpeedMax() )
		  opp_speed = opponents.at( i )->playerTypePtr()->playerSpeedMax();
                sum += opp_speed;
                opp_speed *= opponents.at( i )->playerTypePtr()->playerDecay();
                opp_cycle++;
            }
	    /*
            double ball_dist = opp_pos.dist( wm.ball().pos() );
	    double ball_endspeed = rcsc::ServerParam::i().kickPowerRate() * rcsc::ServerParam::i().maxPower() * 0.6;
	    ball_speed = ball_dist - rcsc::ServerParam::i().ballDecay() * ( ball_dist - ball_endspeed );
	    if( ball_speed >= rcsc::ServerParam::i().ballSpeedMax() ){
	      ball_speed = rcsc::ServerParam::i().ballSpeedMax();
	      ball_endspeed = (ball_dist - ball_speed) / rcsc::ServerParam::i().ballDecay() - ball_dist;
	    }
	    rcsc::dlog.addText( rcsc::Logger::TEAM,
				"%s:%d: ini. b. speed %.00f, fin. b. speed %.00f"
				,__FILE__, __LINE__, ball_speed, ball_endspeed );
	    
	    int ball_cycle2 = (int)( (log( ball_endspeed / ball_speed )) / log( rcsc::ServerParam::i().ballDecay() )) + 1 ;

	    rcsc::dlog.addText( rcsc::Logger::TEAM,
				"%s:%d: necessary time for ball %d"
				,__FILE__, __LINE__, ball_cycle2 );

	    */
            double tmp_speed = ball_speed;
            int ball_cycle2 = 0;
            sum = 0;
            rcsc::Vector2D via_pos = ball_to_target.nearestPoint( opp_pos );
            double ball_dist = via_pos.dist( wm.ball().pos() );
            while( sum < ball_dist )
            {
                sum += tmp_speed;
                tmp_speed *= rcsc::ServerParam::i().ballDecay();
                ball_cycle2++;
                if( ball_cycle2 > 25 )
                {
                    return false;
                }
            }
	    
            if( ball_cycle2 > opp_cycle )
            {
                return false;
            }
            else
            {
                success = true;
            }
        }
    }
                                  
    if( success )
    {
        if( rcsc::Body_SmartKick( target_pos, ball_speed, ball_speed * 0.94, 3 ).execute( agent ) )
        {
            rcsc::KickTable::Sequence seq;
            rcsc::KickTable::instance().simulate( wm,
                                                  target_pos,
                                                  ball_speed,
                                                  ball_speed * 0.94,
                                                  3,
                                                  seq );
            if( seq.pos_list_.size() == 1 )
            {
                agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            agent->debugClient().addMessage( "opuciDirectPass%d", receiver );
            agent->debugClient().setTarget( target_pos );
            return true;
        }
    }
    return false;
}
bool
Body_opuciPass::doAidaPass( rcsc::PlayerAgent * agent, rcsc::PlayerObject * teammate )
{
    const rcsc::WorldModel & wm = agent->world();
    if( teammate->pos().dist( wm.self().pos() ) > 25 
        || teammate->pos().x < wm.self().pos().x )
    {
        return false;
    }
    std::vector< rcsc::Vector2D > candidates(5);
    Bhv_opuciOffensiveMove::findDefenseLine( agent, &candidates );
    double min = 10;
    int num = -1;
    for( unsigned int i = 0; i < candidates.size(); i++ )
    {
        if( !candidates.at( i ).valid() )
        {
            break;
        }
        agent->debugClient().addCircle( candidates.at( i ), 1 );
        rcsc::Vector2D mate_pos = teammate->pos();
                                          
        double dist = candidates.at( i ).dist( mate_pos );
        if( dist < min )
        {
            min = dist;
            num = i;
        }
    }
    if( num < 0 )
    {
        return false;
    }
    int ball_cycle = getBallCycle( candidates.at( num ).dist( wm.ball().pos() ) );
    if( ball_cycle < 3 )
    {
        ball_cycle = 3;
    }
    if( getDashDist( teammate, ball_cycle - 2 ) < min )
    {
        return false;
    }
    double length = candidates.at( num ).dist( wm.ball().pos() );
    if( length < 1 )
    {
        length = 1;
    }
    double width = 15;
    rcsc::Vector2D ball_to = candidates.at( num ) - wm.ball().pos();
    rcsc::Sector2D sector( wm.ball().pos(),
                           0.1, length,
                           ball_to.th() - width,
                           ball_to.th() + width );
    if( wm.existOpponentIn( sector, 10, true ) )
    {
        return false;
    }
    double ball_speed = rcsc::calc_first_term_geom_series( ball_to.r(),
                                                           rcsc::ServerParam::i().ballDecay(),
                                                           ball_cycle );
    
    rcsc::Vector2D target_pos = candidates.at( num );
    int receiver = teammate->unum();
    if( rcsc::Body_SmartKick( target_pos, ball_speed, ball_speed * 0.94, 3 ).execute( agent ) )
    {
        rcsc::KickTable::Sequence seq;
        rcsc::KickTable::instance().simulate( wm,
                                              target_pos,
                                              ball_speed,
                                              ball_speed * 0.94,
                                              3,
                                              seq );
        if( seq.pos_list_.size() == 1 )
        {
            agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                         agent->effector().queuedNextBallPos(),
                                                         agent->effector().queuedNextBallVel() ) );
        }
        
        agent->debugClient().setTarget( target_pos );
        agent->debugClient().addMessage( "AidaPass%d", receiver );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: APass receiver %d, Type ID %d"
                            ,__FILE__, __LINE__, receiver, agent->world().teammateHeteroID( receiver ) );
        return true;        
    }

 
                           
    
    return false;
}

bool
Body_opuciPass::doShootPass( rcsc::PlayerAgent * agent, rcsc::PlayerObject * teammate )
{
    const rcsc::WorldModel & wm = agent->world();
    if( teammate->pos().dist( wm.self().pos() ) > 30 
        || teammate->pos().x < 30 )
    {
        return false;
    }
    int ball_cycle = getBallCycle( teammate->pos().dist( wm.ball().pos() ) );
    if( ball_cycle < 3 )
    {
        ball_cycle = 3;
    }
    rcsc::Vector2D target_pos = rcsc::inertia_n_step_point( teammate->pos(),
                                                   teammate->vel(),
                                                   ball_cycle,
                                                   teammate->playerTypePtr()->playerDecay() );
    
    std::vector< rcsc::PlayerObject* > opponents;
    rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x > 30 )
        {
            opponents.push_back( (*it) );
        }
    }
    bool success = false;
    double ball_speed = rcsc::calc_first_term_geom_series( target_pos.dist( wm.ball().pos() ),
                                                           rcsc::ServerParam::i().ballDecay(),
                                                           ball_cycle );
    if( opponents.empty() )
    {
        success = true;
    }
    else
    {
        for( unsigned int i = 0; i < opponents.size(); i++ )
        {
            rcsc::Segment2D ball_to_target( wm.ball().pos(), target_pos );
            int opp_cycle = 1;
            rcsc::Vector2D opp_pos = rcsc::inertia_n_step_point( opponents.at( i )->pos(),
                                                                 opponents.at( i )->vel(),
                                                                 3,
                                                                 opponents.at( i )->playerTypePtr()->playerDecay() );
            double opp_dist = ball_to_target.dist( opp_pos );
            double opp_speed = 0;
            double sum = 0;
            while( sum < opp_dist )
            {
                opp_speed += opponents.at( i )->playerTypePtr()->dashPowerRate() * 100;
		if( opp_speed > opponents.at( i )->playerTypePtr()->playerSpeedMax() )
		  opp_speed = opponents.at( i )->playerTypePtr()->playerSpeedMax();
                sum += opp_speed;
                opp_speed *= opponents.at( i )->playerTypePtr()->playerDecay();
                opp_cycle++;
            }
	    /*
            double ball_dist = opp_pos.dist( wm.ball().pos() );
	    double ball_endspeed = rcsc::ServerParam::i().kickPowerRate() * rcsc::ServerParam::i().maxPower() * 0.6;
	    ball_speed = ball_dist - rcsc::ServerParam::i().ballDecay() * ( ball_dist - ball_endspeed );
	    if( ball_speed >= rcsc::ServerParam::i().ballSpeedMax() ){
	      ball_speed = rcsc::ServerParam::i().ballSpeedMax();
	      ball_endspeed = (ball_dist - ball_speed) / rcsc::ServerParam::i().ballDecay() - ball_dist;
	    }
	    rcsc::dlog.addText( rcsc::Logger::TEAM,
				"%s:%d: ini. b. speed %.00f, fin. b. speed %.00f"
				,__FILE__, __LINE__, ball_speed, ball_endspeed );
	    
	    int ball_cycle2 = (int)( (log( ball_endspeed / ball_speed )) / log( rcsc::ServerParam::i().ballDecay() )) + 1 ;

	    rcsc::dlog.addText( rcsc::Logger::TEAM,
				"%s:%d: necessary time for ball %d"
				,__FILE__, __LINE__, ball_cycle2 );
            */
	    
            double tmp_speed = ball_speed;
            int ball_cycle2 = 0;
            sum = 0;
            rcsc::Vector2D via_pos = ball_to_target.nearestPoint( opp_pos );
            double ball_dist = via_pos.dist( wm.ball().pos() );
            while( sum < ball_dist )
            {
                sum += tmp_speed;
                tmp_speed *= rcsc::ServerParam::i().ballDecay();
                ball_cycle2++;
                if( ball_cycle2 > 25 )
                {
                    return false;
                }
            }
	    
            if( ball_cycle2 > opp_cycle )
            {
                return false;
            }
            else
            {
                success = true;
            }
        }
    }
    int receiver = teammate->unum();
    if( success )
    {
        if( rcsc::Body_SmartKick( target_pos, ball_speed, ball_speed * 0.94, 3 ).execute( agent ) )
        {
            rcsc::KickTable::Sequence seq;
            rcsc::KickTable::instance().simulate( wm,
                                                  target_pos,
                                                  ball_speed,
                                                  ball_speed * 0.94,
                                                  3,
                                                  seq );
            if( seq.pos_list_.size() == 1 )
            {
                agent->addSayMessage( new rcsc::PassMessage( receiver, target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            agent->debugClient().addMessage( "ShootPass%d", receiver );
            agent->debugClient().setTarget( target_pos );
            return true;
        }
    }
    return false;
}
