// -*-c++-*-

/*!
  \file body_smart_kick.cpp
  \brief smart kick action class source file.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "body_opuci_smart_kick.h"

#include <rcsc/action/kick_table.h>

#include <rcsc/action/body_stop_ball.h>
#include <rcsc/action/body_hold_ball2008.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

#include <algorithm>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_opuciSmartKick::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::KICK,
                  __FILE__": Body_opuciSmartKick" );

    const WorldModel & wm = agent->world();

    if ( ! wm.self().isKickable() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " not ball kickable!"
                  << std::endl;
        dlog.addText( Logger::ACTION,
                      __FILE__":  not kickable" );
        return false;
    }

    if ( ! wm.ball().velValid() )
    {
        dlog.addText( Logger::KICK,
                      __FILE__". unknown ball vel" );
        return Body_StopBall().execute( agent );
    }

    double first_speed = std::min( M_first_speed, ServerParam::i().ballSpeedMax() );
    double first_speed_thr = std::max( 0.0, M_first_speed_thr );
    int max_step = std::max( 1, M_max_step );

    if ( rcsc::KickTable::instance().simulate( wm,
                                               M_target_point,
                                               first_speed,
                                               first_speed_thr,
                                               max_step,
                                               M_sequence )
         || M_sequence.speed_ >= first_speed_thr )
    {
        agent->debugClient().addMessage( "opuciSmartKick%d", (int)M_sequence.pos_list_.size() );
#ifdef DEBUG
        for ( std::vector< Vector2D >::const_iterator p = M_sequence.pos_list_.begin();
              p != M_sequence.pos_list_.end();
              ++p )
        {
            agent->debugClient().addCircle( *p, 0.05 );
        }
#endif
        dlog.addText( rcsc::Logger::KICK,
                      __FILE__": Success! target=(%.2f %.2f)"
                      " speed=%.3f first_speed_thr=%.3f"
                      " max_step=%d -> achieved_speed=%.3f power=%.2f step=%d",
                      M_target_point.x, M_target_point.y,
                      first_speed,
                      first_speed_thr,
                      max_step,
                      M_sequence.speed_,
                      M_sequence.power_,
                      (int)M_sequence.pos_list_.size() );

        Vector2D vel = M_sequence.pos_list_.front() - wm.ball().pos();
        Vector2D kick_accel = vel - wm.ball().vel();
        agent->doKick( kick_accel.r() / wm.self().kickRate(),
                       kick_accel.th() - wm.self().body() );
        return true;
    }


    //
    // TODO: force mode
    //
    if( M_force )
    {
        static int holdcycle = 0;
        agent->debugClient().addMessage( "opuciForceKick" );
        double power = ServerParam::i().maxPower();
        Vector2D rel = M_target_point - wm.ball().pos();
        Vector2D vel = rel - wm.ball().vel();
        AngleDeg angle = vel.th() - wm.self().body();
       
        const PlayerObject * blocker = NULL; 
        const PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
        for( PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
             it != end;
             ++it )
        {
            if( (*it)->pos().x > wm.self().pos().x )
            {
                blocker = (*it);
            }
        }
        if( blocker )
        {
            if( penetrable( agent, blocker->pos() ) )
            {
                agent->doKick( power, angle );
                agent->debugClient().addMessage( "PenetrateKick" );
                return true;
            }
            else if( wm.time().cycle() - 1 == holdcycle )
            {
                agent->doKick( power, angle );
                agent->debugClient().addMessage( "HoldedNextKick" );
                return true;
            }
            else
            {
                Vector2D next_self = wm.self().pos() + wm.self().vel();
                Vector2D target = next_self;
                if( blocker->pos().y > wm.self().pos().y )
                {
                    target.y -= 0.5;
                }
                else
                {
                    target.y += 0.5;
                }
                target.x += 0.3;
                rel = target - wm.ball().pos();
                vel = rel - wm.ball().vel();
                angle = vel.th() - wm.self().body();
                power = vel.r() / wm.self().kickRate();
                if( power > 100 )
                {
                    power = 100;
                }
                agent->doKick( power, angle );
                holdcycle = wm.time().cycle();
                agent->debugClient().addMessage( "opuciHold" );
                return false;
            }
        }
        else
        {
            agent->debugClient().addMessage( "NoOpponentKick" );
            agent->doKick( power, angle );
            return true;
        }
    }

    // failed to search the kick sequence

    agent->debugClient().addMessage( "opuciSmartKick.Hold" );
    dlog.addText( rcsc::Logger::KICK,
                  __FILE__": Failed! target=(%.2f %.2f)"
                  " speed=%.3f first_speed_thr=%.3f"
                  " max_step=%d -> speed=%.3f power=%.2f step=%d",
                  M_target_point.x, M_target_point.y,
                  first_speed,
                  first_speed_thr,
                  max_step,
                  M_sequence.speed_,
                  M_sequence.power_,
                  (int)M_sequence.pos_list_.size() );

    Body_HoldBall2008( false, M_target_point, M_target_point ).execute( agent );
    return false;
}

bool
Body_opuciSmartKick::penetrable( PlayerAgent * agent, Vector2D opponent_pos )
{
    const WorldModel & wm = agent->world();
    double kickRate = wm.self().kickRate();
    double power = 100;
    double effect = kickRate * power;
    
    Vector2D ball_pos = wm.ball().pos();
    Vector2D kick_dir = M_target_point - ball_pos;
    Vector2D ball_speed = wm.ball().vel();
    kick_dir.setLength( effect );
    ball_speed += kick_dir;
    Vector2D current_ball = ball_pos;
    Vector2D current_opponent = opponent_pos;
    Vector2D opponent_dir = wm.ball().pos() - opponent_pos;
    opponent_dir.setLength( 0.5 );
    for( int i = 0; i < 5; i++ )
    {
        Vector2D next_ball = current_ball + ball_speed;
        Circle2D current_circle( current_opponent, 1.0 );
        Vector2D next_opponent = current_opponent + opponent_dir;
        Circle2D next_circle( next_opponent, 1.0 );
        if( !current_circle.contains( current_ball )
            && !next_circle.contains( next_ball )
            && next_ball.x > opponent_pos.x )
        {
            return true;
        }
        ball_speed *= ServerParam::i().ballDecay();
        current_ball = next_ball;
        current_opponent = next_opponent;
    }
    return false;
}
}//end name space
