// -*-c++-*-

/*!
  \file intention_dribble2008.cpp
  \brief queued dribble behavior
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "opuci_intention_dribble.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_goalie_or_scan.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/view_synch.h>

#include <rcsc/player/audio_sensor.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/audio_memory.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>
//#define USE_CHANGE_VIEW


/*-------------------------------------------------------------------*/
/*!

*/
bool
OpuciIntentionDribble::finished( const rcsc::PlayerAgent* agent )
{
    if ( M_turn_step + M_dash_step == 0 )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": finished() check finished. empty queue" );
        return true;
    }


    // NOTE: use operator+  operator- has bugs
    if ( M_last_execute_time.cycle() + 1 != agent->world().time().cycle() )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": finished(). last execute time is not match" );
        return true;
    }
/*
    if ( agent->world().ball().pos().dist2( M_target_point ) < 2.0 * 2.0
         && agent->world().self().pos().dist2( M_target_point ) < 2.0 * 2.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": finished(). reached target point" );
        return true;
    }
*/

    if ( agent->world().audioMemory().passRequestTime() == agent->world().time() )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": finished(). heard pass request." );
        return true;
    }

    // playmode is checked in rcsc::PlayerAgent::parse()
    // and intention queue is totally managed at there.

    rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                  __FILE__": finished(). not finished yet." );

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
OpuciIntentionDribble::execute( rcsc::PlayerAgent * agent )
{
    if ( M_turn_step + M_dash_step == 0 )
    {
        agent->debugClient().addMessage( "Empty" );
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": execute. empty queue." );
        return false;
    }
    const rcsc::WorldModel & wm = agent->world();
    if(  !agent->world().self().isKickable()
         && wm.ball().pos().x < wm.self().pos().x )
    {
        agent->debugClient().addMessage( "LostBall" );
        this->clear();
        return false;
    }
    if( wm.existKickableOpponent() )
    {
        agent->debugClient().addMessage( "OppKickable" );
        this->clear();
        return false;
    }
    const rcsc::Vector2D ball_next = wm.ball().pos() + wm.ball().vel();
    if ( ball_next.absX() > rcsc::ServerParam::i().pitchHalfLength() - 0.5
         || ball_next.absY() > rcsc::ServerParam::i().pitchHalfWidth() - 0.5 )
    {
        agent->debugClient().addMessage( "BallOut" );
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": execute(). ball will be out of pitch. stop intention." );
        return false;
    }

        
/*
    if ( checkOpponent( wm ) )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": execute(). but exist interfere opponent. cancel intention." );
        return false;
    }
*/
    /*--------------------------------------------------------*/
    // execute action
    if ( M_turn_step > 0 )
    {
        if ( ! doTurn( agent ) )
        {
            agent->debugClient().addMessage( "MissTurn" );
            rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                          __FILE__": exuecute() failed to. turn. clear intention" );
            this->clear();
            return false;
        }
    }
    else if ( M_dash_step > 0 )
    {
        if ( ! doDash( agent ) )
        {
            agent->debugClient().addMessage( "MissDash" );
            rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                          __FILE__": execute() failed to. dash.  clear intention" );
            this->clear();
            return false;
        }
    }
    else
    {
        agent->debugClient().addMessage( "NoCommand" );
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": execute(). No command queue" );
        this->clear();
        return false;
    }

    if( wm.ball().velCount() >= 1 )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    }
    else
    {
        if ( wm.gameMode().type() == rcsc::GameMode::PenaltyTaken_ )
        {
            const rcsc::PlayerObject * opp_goalie = wm.getOpponentGoalie();
            if ( opp_goalie )
            {
                agent->setNeckAction( new rcsc::Neck_TurnToPoint( opp_goalie->pos() ) );
            }
            else
            {
                agent->setNeckAction( new rcsc::Neck_ScanField() );
            }
        }
        else
        {
            if ( wm.self().pos().x > 36.0
                 && wm.self().pos().absY() < 20.0 )
            {
                agent->setNeckAction( new rcsc::Neck_TurnToGoalieOrScan() );
            }
            else
            {
                agent->setNeckAction( new rcsc::Neck_ScanField() );
            }
        }
        
#ifdef USE_CHANGE_VIEW
        if ( wm.gameMode().type() != rcsc::GameMode::PlayOn
             || M_turn_step + M_dash_step <= 1 )
        {
            agent->setViewAction( new rcsc::View_Synch() );
        }
        else
        {
            agent->setViewAction( new rcsc::View_Normal() );
        }
#endif
    }
    M_last_execute_time = wm.time();

    rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                  __FILE__": execute(). done" );
    agent->debugClient().addMessage( "DribbleIntention" );
    agent->debugClient().setTarget( M_target_point );
    if( M_turn_step == 0 
        && M_dashed > 0 
        && wm.ball().velCount() == 0 
        && !wm.self().isKickable() )
    {
        //check correct angle 
        rcsc::Vector2D ball_end = wm.ball().inertiaPoint( M_dash_step  );
        rcsc::Vector2D self_end = getFinalPoint( agent );
        rcsc::Circle2D keep_margin( self_end,
                                    wm.self().playerType().kickableMargin() * 2 );
        rcsc::Line2D ball_route( wm.ball().pos(), ball_end );
        if( ball_end.dist( wm.ball().pos() ) < 0.3 )
        {
            //nothing
        } 
        else if( keep_margin.intersection( ball_route, NULL, NULL ) < 1 
                 && !keep_margin.contains( ball_end ) )
        {
            agent->debugClient().addCircle( self_end, wm.self().playerType().kickableMargin() );
            agent->debugClient().addCircle( ball_end, 0.3 );
            agent->debugClient().addMessage( "NeedAdditionalTurn" );
            rcsc::Vector2D inertia_self
                = wm.self().playerType().inertiaPoint( wm.self().pos(),
                                                       wm.self().vel(),
                                                       M_dash_step + 2 );
            rcsc::Vector2D inertia_ball 
                = wm.ball().inertiaPoint( M_dash_step + 2 );
            rcsc::Vector2D rel = inertia_ball - inertia_self;
            rcsc::AngleDeg dif = rel.th() - wm.self().body();
            ++M_dash_step;
            agent->doTurn( dif );
            return true;
        }
    }    

    if ( M_turn_step + M_dash_step > 0
         && agent->config().useCommunication() )
    {
        rcsc::dlog.addText( rcsc::Logger::ACTION,
                      __FILE__":  set dribble communication." );
        agent->debugClient().addMessage( "SayD" );
        agent->addSayMessage( new rcsc::DribbleMessage( M_target_point,
                                                  M_turn_step + M_dash_step ) );
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
OpuciIntentionDribble::checkOpponent( const rcsc::WorldModel & world )
{
    const rcsc::Vector2D ball_next = world.ball().pos() + world.ball().vel();

    /*--------------------------------------------------------*/
    // exist near opponent goalie in NEXT cycle
    if ( ball_next.x > rcsc::ServerParam::i().theirPenaltyAreaLineX()
         && ball_next.absY() < rcsc::ServerParam::i().penaltyAreaHalfWidth() )
    {
        const rcsc::PlayerObject * opp_goalie = world.getOpponentGoalie();
        if ( opp_goalie
             && opp_goalie->distFromBall() < ( rcsc::ServerParam::i().catchableArea()
                                               + rcsc::ServerParam::i().defaultPlayerSpeedMax() )
             )
        {
            rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                          __FILE__": existOpponent(). but exist near opponent goalie" );
            this->clear();
            return true;
        }
    }

    const rcsc::PlayerObject * nearest_opp = world.getOpponentNearestToSelf( 5 );

    if ( ! nearest_opp )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": existOppnent(). No opponent" );
        return false;
    }

    /*--------------------------------------------------------*/
    // exist very close opponent in CURRENT cycle
    if (  nearest_opp->distFromBall() < rcsc::ServerParam::i().defaultKickableArea() + 0.2 )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": existOpponent(). but exist kickable opp" );
        this->clear();
        return true;
    }

    /*--------------------------------------------------------*/
    // exist near opponent in NEXT cycle
    const double opp2ball_dist_next = nearest_opp->pos().dist( ball_next );
    if ( opp2ball_dist_next < ( rcsc::ServerParam::i().defaultPlayerSpeedMax()
                                + rcsc::ServerParam::i().defaultKickableArea() + 0.3 ) )
    {
        const rcsc::Vector2D opp_next = nearest_opp->pos() + nearest_opp->vel();
        // oppopnent angle is known
        if ( nearest_opp->bodyCount() == 0
             || nearest_opp->vel().r() > 0.2 )
        {
            rcsc::Line2D opp_line( opp_next,
                             ( nearest_opp->bodyCount() == 0
                               ? nearest_opp->body()
                               : nearest_opp->vel().th() ) );
            if ( opp_line.dist( ball_next ) > 1.2 )
            {
                // never reach
                rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                              __FILE__": existOpponent(). opp never reach." );
            }
            else if ( opp_next.dist( ball_next ) < 0.6 + 1.2 )
            {
                rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                              __FILE__": existOpponent(). but opp may reachable(1)." );
                this->clear();
                return true;
            }

            rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                          __FILE__": existOpponent(). opp angle is known. opp may not be reached." );
        }
        // opponent angle is not known
        else
        {
            if ( opp_next.dist( ball_next ) < 1.2 + 1.2 ) //0.6 + 1.2 )
            {
                rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                              __FILE__": existOpponent(). but opp may reachable(2)." );
                this->clear();
                return true;
            }
        }

        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": existOpponent(). exist near opp. but avoidable?" );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
OpuciIntentionDribble::doTurn( rcsc::PlayerAgent * agent )
{
    if ( M_turn_step <=0 )
    {
        return false;
    }

    const rcsc::WorldModel & wm = agent->world();

    --M_turn_step;

    rcsc::Vector2D my_final = wm.self().inertiaFinalPoint();
    rcsc::AngleDeg target_angle = ( M_target_point - my_final ).th();
    rcsc::AngleDeg angle_diff = target_angle - wm.self().body();
    if ( M_back_dash_mode )
    {
        angle_diff -= 180.0;
    }

    double target_dist = ( M_target_point - my_final ).r();
    double angle_margin
        = std::max( 15.0,
                    std::fabs( rcsc::AngleDeg::atan2_deg( M_dist_thr,
                                                    target_dist ) ) );
/*
    if ( angle_diff.abs() < angle_margin )
    {
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": doTurn.  but already facing. diff = %.1f  margin=%.1f",
                      angle_diff.degree(), angle_margin );
        this->clear();
        return false;
    }
*/
    rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                  __FILE__": doTurn. turn to (%.2f, %.2f)",
                  M_target_point.x, M_target_point.y );

    agent->doTurn( angle_diff );

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
OpuciIntentionDribble::doDash( rcsc::PlayerAgent * agent )
{
    if ( M_dash_step <= 0 )
    {
        return false;
    }

    const rcsc::WorldModel & wm = agent->world();

    --M_dash_step;
    ++M_dashed;

    rcsc::AngleDeg accel_angle = ( M_back_dash_mode
                             ? wm.self().body() - 180.0
                             : wm.self().body() );
    double used_power
        = wm.self().getSafetyDashPower( M_dash_power_abs
                                        * ( M_back_dash_mode ? -1.0 : 1.0 ) );
    const double max_accel_mag = std::fabs( used_power )
        * wm.self().playerType().dashPowerRate()
        * wm.self().effort();
    double accel_mag = max_accel_mag;
    if ( wm.self().playerType().normalizeAccel( wm.self().vel(),
                                                accel_angle,
                                                &accel_mag ) )
    {
        used_power *=  accel_mag / max_accel_mag;
    }

    rcsc::Vector2D dash_accel = rcsc::Vector2D::polar2vector( accel_mag, accel_angle );

    // uenishi    rcsc::Vector2D my_next = wm.self().pos() + wm.self().vel() + dash_accel;
    rcsc::Vector2D my_next = wm.self().pos() + dash_accel; // nazo
    rcsc::Vector2D ball_next = wm.ball().pos() + wm.ball().vel();
    rcsc::Vector2D ball_next_rel = ( ball_next - my_next ).rotatedVector( - accel_angle );
    double ball_next_dist = ball_next_rel.r();

    if ( ball_next_dist < ( wm.self().playerType().playerSize()
                            + rcsc::ServerParam::i().ballSize() )//+ 0.1 )
         )
    {
        agent->debugClient().addMessage( "Collision" );
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": doDash() collision may occur. ball_dist = %f",
                      ball_next_dist );
        this->clear();
        return false;
    }

    /*
    if ( ball_next_rel.absY() > wm.self().kickableArea() - 0.1 )
    {
        agent->debugClient().addMessage( "yOber" );
        rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                      __FILE__": doDash() next Y difference is over. y_diff = %f",
                      ball_next_rel.absY() );
        this->clear();
        return false;
    }
    */

    // this dash is the last of queue
    // but at next cycle, ball is NOT kickable
    if ( M_dash_step <= 0 )
    {
        agent->debugClient().addMessage( "LastIntention" );
        if ( ball_next_dist > wm.self().kickableArea() - 0.15 )
        {
            agent->debugClient().addMessage( "CantKeep");
            rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                          __FILE__": doDash() last dash. but not kickable at next. ball_dist=%f",
                          ball_next_dist );
            this->clear();
            return false;
        }
    }
/*
    if ( M_dash_step > 0 )
    {
        // remain dash step. but next dash cause over run.
        rcsc::AngleDeg ball_next_angle = ( ball_next - my_next ).th();
        if ( ( accel_angle - ball_next_angle ).abs() > 90.0
             && ball_next_dist > wm.self().kickableArea() - 0.2 )
        {
            agent->debugClient().addMessage( "BallOver" );
            rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                          __FILE__": doDash() dash. but run over. ball_dist=%f",
                          ball_next_dist );
            this->clear();
            return false;
        }
    }
*/
    rcsc::dlog.addText( rcsc::Logger::DRIBBLE,
                  __FILE__": doDash() power=%.1f  accel_mag=%.2f",
                  used_power, accel_mag );
    agent->doDash( 100 );

    return true;
}

rcsc::Vector2D
OpuciIntentionDribble::getFinalPoint( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D final_point;
    
    int dash_count = M_dash_step;
    double dist = 0;
    double self_speed = 0;
    double stamina = wm.self().stamina();
    double effort = wm.self().effort();
    double recovery = wm.self().recovery();
    for( int i = 0; i < dash_count; i++ )
    {
        self_speed += effort * wm.self().playerType().dashPowerRate() * 100;
        if( self_speed > wm.self().playerType().playerSpeedMax() )
        {
            self_speed = wm.self().playerType().playerSpeedMax();
        }
        dist += self_speed;
        self_speed *= wm.self().playerType().playerDecay();

        stamina -= wm.self().playerType().getOneStepStaminaComsumption( rcsc::ServerParam::i() );
        
        if( stamina <= 
            rcsc::ServerParam::i().staminaMax() 
            * rcsc::ServerParam::i().recoverDecThr() )
        {
            if( recovery > rcsc::ServerParam::i().recoverMin() )
            {
                recovery -= rcsc::ServerParam::i().recoverDec();
            }
            if( recovery > rcsc::ServerParam::i().recoverMin() )
            {
                recovery = rcsc::ServerParam::i().recoverMin();
            }
        }
        if( stamina < 
            rcsc::ServerParam::i().staminaMax() 
            * rcsc::ServerParam::i().effortDecThr() )
        {
            if( effort > wm.self().playerType().effortMin() )
            {
                effort -= rcsc::ServerParam::i().effortDec();
            }
            if( effort < wm.self().playerType().effortMin() )
            {
                effort = wm.self().playerType().effortMin();
            }
        }
        if( stamina >=
            rcsc::ServerParam::i().staminaMax()
            * rcsc::ServerParam::i().effortIncThr() )
        {
            if( effort < wm.self().playerType().effortMax() )
            {
                effort += rcsc::ServerParam::i().effortInc();
                if( effort > wm.self().playerType().effortMax() )
                {
                    effort = wm.self().playerType().effortMax();
                }
            }
        }
        stamina += recovery * wm.self().playerType().staminaIncMax();
    }
    
    rcsc::Vector2D inertia_self 
        = wm.self().playerType().inertiaPoint( wm.self().pos(),
                                               wm.self().vel(),
                                               dash_count );
    rcsc::Vector2D dash_vec( dist, 0 );
    dash_vec.rotate( wm.self().body() );
    final_point = inertia_self + dash_vec;
    return final_point;
}
