// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_goalie.h"

#include "bhv_opuci_goalie_basic_move.h"
#include "bhv_opuci_goalie_chase_ball.h"
#include "bhv_goalie_free_kick.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>
#include <rcsc/action/body_clear_ball.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/world_model.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/geom/sector_2d.h>

#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/player/say_message_builder.h>
/*-------------------------------------------------------------------*/
/*!

*/
void
RoleGoalie::execute( rcsc::PlayerAgent* agent )
{
    static const
        rcsc::Rect2D our_penalty( rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength(),
                                                  -rcsc::ServerParam::i().penaltyAreaHalfWidth() + 1.0 ),
                                  rcsc::Size2D( rcsc::ServerParam::i().penaltyAreaLength() - 1.0,
                                                rcsc::ServerParam::i().penaltyAreaWidth() - 2.0 ) );

    //////////////////////////////////////////////////////////////
    // play_on play

    // catchable
    if ( agent->world().time().cycle() > M_last_goalie_kick_time.cycle() + 5
         && agent->world().ball().distFromSelf() < agent->world().self().catchableArea() - 0.05
         && our_penalty.contains( agent->world().ball().pos() ) )
    {
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": catchable. ball dist=%.1f, my_catchable=%.1f",
                            agent->world().ball().distFromSelf(),
                            agent->world().self().catchableArea() );
        agent->doCatch();
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }
    else if ( agent->world().self().isKickable() )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleGoalie::doKick( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::Vector2D target_point;
    double pass_speed = 0.0;
    int receiver = 0;

    //direct pass
    const rcsc::Rect2D penalty_area( rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength(), -rcsc::ServerParam::i().penaltyAreaHalfWidth() ),
                                     rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength() + rcsc::ServerParam::i().penaltyAreaLength(),
                                                     rcsc::ServerParam::i().penaltyAreaHalfWidth() ) );
    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates_end;
         ++it )
    {
        bool target_is_free = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     1.0, (*it)->pos().dist( wm.ball().pos() ) + 3.0,
                                     ( (*it)->pos() - wm.ball().pos() ).th() - rcsc::AngleDeg( 30.0 ),
                                     ( (*it)->pos() - wm.ball().pos() ).th() + rcsc::AngleDeg( 30.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps.end();
             ++itr )
        {
            if( sector.contains( (*itr)->pos() ) )
            {
                target_is_free = false;
            }
        }

        if( target_is_free
            && (*it)->posCount() < 3
            && (*it)->pos().x > -25.0
            && (*it)->pos().x > wm.self().pos().x - 5.0
            && (*it)->pos().dist( wm.self().pos() ) > 10.0 )
        {
            int reach_cycle = log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( (*it)->pos() ) / ball_speed_max ) / log( ball_decay );
            double ball_speed = (*it)->playerTypePtr()->kickableArea() * 1.5 * pow( 1.0 / ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            if( penalty_area.contains( wm.ball().pos() ) )
            {
                if( rcsc::Body_SmartKick( (*it)->pos(),
                                          first_speed,
                                          first_speed * 0.8,
                                          1 ).execute( agent ) )
                {
                    agent->debugClient().addMessage( "DirectOneStepPass" );
                    agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                                 (*it)->pos(),
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                    agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                    return;
                }
            }
            else
            {
                rcsc::Body_SmartKick( (*it)->pos(),
                                      first_speed,
                                      first_speed * 0.8,
                                      3 ).execute( agent );
                agent->debugClient().addMessage( "DirectPass" );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             (*it)->pos(),
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                return;
            }
        }
    }

    if  ( rcsc::Body_Pass::get_best_pass( agent->world(), &target_point, &pass_speed, &receiver )
          && target_point.dist( rcsc::Vector2D( -50.0, 0.0 ) ) > 20.0 )
    {
        double opp_dist = 100.0;
        const rcsc::PlayerObject * opp
            = agent->world().getOpponentNearestTo( target_point, 20, &opp_dist );
        agent->debugClient().addMessage( "GKickOppDist%.1f", opp ? opp_dist : 1000.0 );
        if ( ! opp || opp_dist > 3.0 )
        {
            rcsc::Body_SmartKick( target_point,
                                  pass_speed,
                                  pass_speed * 0.96,
                                  3 ).execute( agent );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": register goalie kick intention. to (%.1f, %.1f)",
                                target_point.x,
                                target_point.y );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            if ( agent->config().useCommunication()
                 && receiver != rcsc::Unum_Unknown )
            {
                rcsc::dlog.addText( rcsc::Logger::ACTION,
                                    "%s:%d: execute() set pass communication.freekick"
                                    ,__FILE__, __LINE__ );
                rcsc::Vector2D target_buf = target_point - agent->world().self().pos();
                target_buf.setLength( 0.3 );
                agent->addSayMessage( new rcsc::PassMessage( receiver,
                                                            target_point,
                                                            agent->effector().queuedNextBallPos(),
                                                            agent->effector().queuedNextBallVel() ) );
                return;
            }
        }
    }

    double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
    for( double angle = -90.0; angle <= 90.0; angle += 10.0 )
    {
        bool angle_safety = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     1.0, 20.0,
                                     rcsc::AngleDeg( angle ) - rcsc::AngleDeg( 20.0 ),
                                     rcsc::AngleDeg( angle ) + rcsc::AngleDeg( 20.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps_end;
             ++it )
        {
            if( sector.contains( (*it)->pos() ) )
            {
                angle_safety = false;
            }
        }
        if( angle_safety
            && abs( angle ) < abs( best_angle ) )
        {
            best_angle = angle;
        }
    }
    target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
    target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
    if( rcsc::Body_SmartKick( target_point,
                              ball_speed_max,
                              ball_speed_max * 0.8,
                              1 ).execute( agent ) )
    {
        agent->debugClient().addMessage( "GoalieKickAdvance" );
        agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
        return;
    }
    rcsc::Body_ClearBall().execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleGoalie::doMove( rcsc::PlayerAgent * agent )
{
    if ( Bhv_OpuciGoalieChaseBall::is_ball_chase_situation( agent ) )
    {
        Bhv_OpuciGoalieChaseBall().execute( agent );
    }
    else
    {
        Bhv_OpuciGoalieBasicMove().execute( agent );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
double
RoleGoalie::getBestSpeed( double dist )
{

   double speed = rcsc::ServerParam::i().ballSpeedMax();
   int cycle = 0;
   while( cycle < 10 )
   {
      cycle++;
      double sum = rcsc::calc_sum_geom_series( speed, rcsc::ServerParam::i().ballDecay(), cycle );
      if( sum >= dist )
      {
	 speed = rcsc::calc_first_term_geom_series( dist, rcsc::ServerParam::i().ballDecay(), cycle + 1 );
	 break;
      }
   }
   
   return speed;
}


/*-------------------------------------------------------------------*/
/*!

*/
void
RoleGoalie::setCandidates( rcsc::Vector2D base_pos, std::vector< rcsc::Vector2D > & candidates )
{
   candidates.clear();
   candidates.push_back( base_pos );
   
   rcsc::Vector2D cand1 = base_pos;
   rcsc::Vector2D cand2 = base_pos;
   rcsc::Vector2D cand3 = base_pos;
   rcsc::Vector2D cand4 = base_pos;
   rcsc::Vector2D cand5 = base_pos;
   rcsc::Vector2D cand6 = base_pos;
   rcsc::Vector2D cand7 = base_pos;
   rcsc::Vector2D cand8 = base_pos;

   double radius = 1.5;
   
   cand1.x += radius;

   cand2.x += radius * cos( M_PI_4 );
   cand2.y += radius * sin( M_PI_4 );

   cand3.y += radius;

   cand4.x += radius * cos( M_PI_4 * 3 );
   cand4.y += radius * sin( M_PI_4 * 3 );

   cand5.x -= radius;

   cand6.x += radius * cos( M_PI_4 * 5 );
   cand6.y += radius * sin( M_PI_4 * 5 );

   cand7.y -= radius;

   cand8.x += radius * cos( M_PI_4 * 7 );
   cand8.y += radius * sin( M_PI_4 * 7 );

   candidates.push_back( cand1 );
   candidates.push_back( cand2 );
   candidates.push_back( cand3 );
   candidates.push_back( cand4 );
   candidates.push_back( cand5 );
   candidates.push_back( cand6 );
   candidates.push_back( cand7 );
   candidates.push_back( cand8 );
   

   for( unsigned int i = 0; i < candidates.size(); i++ )
   {

      if( candidates.at( i ).x > 50.2 )
      {
	 candidates.at( i ).x = 50.5;
      }
      if( candidates.at( i ).y > 32 )
      {
	 candidates.at( i ).y = 32;
      }
      if( candidates.at( i ).y < -32 )
      {
	 candidates.at( i ).y = -32;
      }
   }
   return;
}

rcsc::Vector2D
RoleGoalie::getKickPos( const rcsc::PlayerAgent * agent, int & receiver )
{
    rcsc::Vector2D target_pos( 0, 0 );
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::AbstractPlayerObject * mate4 = wm.teammate( 4 );
    const rcsc::AbstractPlayerObject * mate5 = wm.teammate( 5 );
    const rcsc::AbstractPlayerObject * target_mate = NULL;

    if( mate4 )
    {
        if( !mate5 )
        {
            target_mate = mate4;
        }
        else
        {
            double dist = wm.self().pos().dist( mate4->pos() );
            if( dist < wm.self().pos().dist( mate5->pos() ) )
            {
                target_mate = mate4;
            }
            else
            {
                target_mate = mate5;
            }
        }
    }
    else if( mate5 )
    {
        target_mate = mate4;
    }
    

    if( target_mate )
    {
        receiver = target_mate->unum();
        rcsc::Vector2D tmp_target = target_mate->pos();
        std::vector< rcsc::Vector2D > candidates;
        std::vector< bool > eval;
        setCandidates( tmp_target, candidates );
        for( unsigned int i = 0; i < candidates.size(); i++ )
        {
            std::vector< rcsc::Vector2D > vertices;
            double tmpY = 0;
            if( wm.self().pos().y > 0 )
            {
                tmpY = candidates.at( i ).y;
            }
            else
            {
                tmpY = candidates.at( i ).y;
            }
            vertices.push_back( rcsc::Vector2D( candidates.at( i ).x + 1, tmpY ) );
            vertices.push_back( rcsc::Vector2D( candidates.at( i ).x - 1, tmpY ) );
            vertices.push_back( wm.self().pos() );
            rcsc::Polygon2D polygon( vertices );
            if( !wm.existOpponentIn( polygon, 3, true )
                && candidates.at( i ).x < 51 )
            {
                eval.push_back( true );
            }
            else
            {
                eval.push_back( false );
            }
            int best = 0;
            double maxY  = 0;
            for( unsigned int i = 0; i < eval.size(); i++ )
            {
                if( eval.at( i ) )
                {
                    if( candidates.at( i ).absY() > maxY )
                    {
                        best = i;
                        maxY = candidates.at( i ).absY();
                    }
                }
            }
            target_pos = candidates.at( best );
        }
    }//end if target_mate
    return target_pos;
}
