// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_offensive_half.h"

#include "strategy.h"

#include "bhv_basic_offensive_kick.h"
#include "bhv_basic_move.h"
#include "bhv_opuci_offensive_move.h"
#include "bhv_opuci_defensive_move.h"
#include "bhv_opuci_shootchance_move.h"
#include "bhv_opuci_offensive_kick.h"
#include "bhv_opuci_defensive_kick.h"
#include "neck_opuci_turn_to_next_receiver.h"

#include <rcsc/formation/formation.h>

#include "bhv_basic_tackle.h"
#include "body_kick_to_corner.h"
#include "body_opuci_pass.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/neck_turn_to_ball.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>
#include <rcsc/action/neck_turn_to_point.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/obsolete/body_dribble2007.h>
#include <rcsc/action/body_dribble2008.h>
#include <rcsc/action/body_advance_ball.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_pass.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleOffensiveHalf::execute( rcsc::PlayerAgent * agent )
{
    bool kickable = agent->world().self().isKickable();
    if ( agent->world().existKickableTeammate()
         && agent->world().teammatesFromBall().front()->distFromBall()
         < agent->world().ball().distFromSelf() )
    {
        kickable = false;
    }

    if ( kickable )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleOffensiveHalf::doKick( rcsc::PlayerAgent * agent )
{
    switch ( Strategy::get_ball_area( agent->world().ball().pos() ) ) {
    case Strategy::BA_CrossBlock:
    case Strategy::BA_Stopper:
    case Strategy::BA_Danger:
    case Strategy::BA_DribbleBlock:
    case Strategy::BA_DefMidField:
      //Bhv_opuciDefensiveKick().execute( agent );
      doOffensiveKick( agent );
      break;
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
      doOffensiveKick( agent );
      break;
    case Strategy::BA_Cross:
    case Strategy::BA_ShootChance:
      Bhv_opuciOffensiveKick().execute( agent );
      break;
    default:
      //      Bhv_opuciOffensiveKick().execute( agent );
      doOffensiveKick( agent );
      break;
    }
}
/*-------------------------------------------------------------------*/
/*!

*/
void
RoleOffensiveHalf::doMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    int ball_reach_step = 0;
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent() )
    {
        ball_reach_step
            = std::min( wm.interceptTable()->teammateReachCycle(),
                        wm.interceptTable()->opponentReachCycle() );
    }
    const rcsc::Vector2D base_pos
        = wm.ball().inertiaPoint( ball_reach_step );

    rcsc::Vector2D home_pos
        = formation().getPosition( agent->config().playerNumber(),
                                   base_pos );
    if ( rcsc::ServerParam::i().useOffside() )
    {
        home_pos.x = std::min( home_pos.x, wm.offsideLineX() - 1.0 );
    }

    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        "%s: HOME POSITION=(%.2f, %.2f) base_point(%.1f %.1f)"
                        ,__FILE__,
                        home_pos.x, home_pos.y,
                        base_pos.x, base_pos.y );

    switch ( Strategy::get_ball_area( base_pos ) ) {
    case Strategy::BA_CrossBlock:
    case Strategy::BA_Stopper:
    case Strategy::BA_Danger:
    case Strategy::BA_DribbleBlock:
    case Strategy::BA_DefMidField:
      doDefensiveMove( agent );
      break;
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
      if( wm.interceptTable()->teammateReachCycle() < wm.interceptTable()->opponentReachCycle() 
	  || wm.existKickableTeammate() )
          doOffensiveMove( agent );
      else
          doDefensiveMove( agent );
      break;
    case Strategy::BA_Cross:
      Bhv_opuciShootChanceMove( home_pos ).execute( agent );
      break;
    case Strategy::BA_ShootChance:
      Bhv_opuciShootChanceMove( home_pos ).execute( agent );
      //doOffensiveMove( agent );
      break;
    default:
      doOffensiveMove( agent );
      break;
    }
}
/*-------------------------------------------------------------------*/
/*!

*/
void
RoleOffensiveHalf::doOffensiveKick( rcsc::PlayerAgent * agent )
{

    const rcsc::WorldModel & wm = agent->world();

    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    const rcsc::PlayerObject * nearest_opp
        = ( opps.empty()
            ? static_cast< rcsc::PlayerObject * >( 0 )
            : opps.front() );
    const double nearest_opp_dist = ( nearest_opp
                                      ? nearest_opp->distFromSelf()
                                      : 1000.0 );
    const rcsc::Vector2D nearest_opp_pos = ( nearest_opp
                                             ? nearest_opp->pos()
                                             : rcsc::Vector2D( -1000.0, 0.0 ) );

    const rcsc::Rect2D penalty_area( rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength(),
                                                     -rcsc::ServerParam::i().penaltyAreaHalfWidth() ),
                                     rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength() + rcsc::ServerParam::i().penaltyAreaLength(),
                                                     rcsc::ServerParam::i().penaltyAreaHalfWidth() ) );
    
    rcsc::Vector2D pass_point;
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();

    for( int target = 11; target >= 2; --target )
      {
	if( wm.teammate( target ) && wm.teammate( target )->pos().x > wm.offsideLineX() )
	//	if( wm.teammate( target )->pos().x > wm.offsideLineX() )
	  continue;

	if( doDirectPass( agent, target ) )
	  {
	    agent->debugClient().addMessage("target:%d", target );
	    return;
	  }
	if( doThroughPass( agent, target ) )
	  {
	    agent->debugClient().addMessage("target:%d", target );
	    return;
	  }
      }

    int receiver;
    if ( rcsc::Body_Pass::get_best_pass( wm, &pass_point, NULL, &receiver ) )
    {
        if ( pass_point.x > wm.self().pos().x - 1.0 )
        {
            bool safety = true;
            const rcsc::Sector2D sector( wm.ball().pos(),
                                         1.0, wm.ball().pos().dist( pass_point ) + 3.0,
                                         ( pass_point - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                         ( pass_point - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
            const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
            for ( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                  it != opps_end;
                  ++it )
            {
                if ( sector.contains( (*it)->pos() ) )
                {
                    safety = false;
                }
            }

            if ( safety )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: do best pass"
                                    ,__FILE__, __LINE__ );
                agent->debugClient().addMessage( "DefKickPass(1)" );
		agent->debugClient().setTarget( pass_point );
		agent->addSayMessage( new rcsc::PassMessage( receiver,
							     pass_point,
							     agent->effector().queuedNextBallPos(),
							     agent->effector().queuedNextBallVel() ) );
                            rcsc::Body_Pass().execute( agent );
                agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
                return;
            }
        }
    }

    if ( nearest_opp_dist < 7.0 )
    {
      //        if ( rcsc::Body_Pass().execute( agent ) )
        if ( Body_opuciPass().execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: do best pass"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickPass(2)" );
            agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
            return;
        }
    }

    //don't dribble in penalty area
    if( ! penalty_area.contains( wm.ball().pos() ) )
    {
        // dribble to my body dir
        if ( nearest_opp_dist < 5.0
             && nearest_opp_dist > ( rcsc::ServerParam::i().tackleDist()
                                     + rcsc::ServerParam::i().defaultPlayerSpeedMax() * 1.5 )
             && wm.self().body().abs() < 70.0 )
        {
            const rcsc::Vector2D body_dir_drib_target
                = wm.self().pos()
                + rcsc::Vector2D::polar2vector(5.0, wm.self().body());
            
            int max_dir_count = 0;
            wm.dirRangeCount( wm.self().body(), 20.0, &max_dir_count, NULL, NULL );
            
            if ( body_dir_drib_target.x < rcsc::ServerParam::i().pitchHalfLength() - 1.0
                 && body_dir_drib_target.absY() < rcsc::ServerParam::i().pitchHalfWidth() - 1.0
                 && max_dir_count < 3
            )
            {
                // check opponents
                // 10m, +-30 degree
                const rcsc::Sector2D sector( wm.self().pos(),
                                             0.5, 10.0,
                                             wm.self().body() - 30.0,
                                             wm.self().body() + 30.0 );
                // opponent check with goalie
                if ( ! wm.existOpponentIn( sector, 10, true ) )
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: dribble to my body dir"
                                        ,__FILE__, __LINE__ );
                    agent->debugClient().addMessage( "DefKickDrib(1)" );
                    rcsc::Body_Dribble2008( body_dir_drib_target,
                                            1.0,
                                            rcsc::ServerParam::i().maxDashPower(),
                                            2
                    ).execute( agent );
                    agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                    return;
                }
            }
        }
        
        rcsc::Vector2D drib_target( 50.0, wm.self().pos().absY() );
        if ( drib_target.y < 20.0 ) drib_target.y = 20.0;
        if ( drib_target.y > 29.0 ) drib_target.y = 27.0;
        if ( wm.self().pos().y < 0.0 ) drib_target.y *= -1.0;
        const rcsc::AngleDeg drib_angle = ( drib_target - wm.self().pos() ).th();
        
        // opponent is behind of me
        if ( nearest_opp_pos.x < wm.self().pos().x + 1.0 )
        {
            // check opponents
            // 15m, +-30 degree
            const rcsc::Sector2D sector( wm.self().pos(),
                                         0.5, 15.0,
                                         drib_angle - 30.0,
                                         drib_angle + 30.0 );
            // opponent check with goalie
            if ( ! wm.existOpponentIn( sector, 10, true ) )
            {
                const int max_dash_step
                    = wm.self().playerType()
                    .cyclesToReachDistance( wm.self().pos().dist( drib_target ) );
                if ( wm.self().pos().x > 35.0 )
                {
                    drib_target.y *= ( 10.0 / drib_target.absY() );
                }
                
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: fast dribble to (%.1f, %.1f) max_step=%d"
                                    ,__FILE__, __LINE__,
                                    drib_target.x, drib_target.y,
                                    max_dash_step );
                agent->debugClient().addMessage( "DefKickDrib(2)" );
                rcsc::Body_Dribble2008( drib_target,
                                        1.0,
                                        rcsc::ServerParam::i().maxDashPower(),
                                        std::min( 5, max_dash_step )
                ).execute( agent );
            }
            else
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: slow dribble to (%f, %f)"
                                    ,__FILE__, __LINE__,
                                    drib_target.x, drib_target.y );
                agent->debugClient().addMessage( "DefKickDrib(3)" );
                rcsc::Body_Dribble2008( drib_target,
                                        1.0,
                                        rcsc::ServerParam::i().maxDashPower(),
                                        2
                ).execute( agent );
                
            }
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }
        
        // opp is far from me
        if ( nearest_opp_dist > 5.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp far. dribble(%f, %f)"
                                ,__FILE__, __LINE__,
                                drib_target.x, drib_target.y );
            agent->debugClient().addMessage( "DefKickDrib(4)" );
            rcsc::Body_Dribble2008( drib_target,
                                    1.0,
                                    rcsc::ServerParam::i().maxDashPower() * 0.4,
                                    1
            ).execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }

        // opp is near

        // can pass
        if ( rcsc::Body_Pass().execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: pass"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickPass(3)" );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }
        
        // opp is far from me
        if ( nearest_opp_dist > 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp far. dribble(%f, %f)"
                                ,__FILE__, __LINE__,
                                drib_target.x, drib_target.y );
            agent->debugClient().addMessage( "DefKickDrib(5)" );
            rcsc::Body_Dribble2008( drib_target,
                                    1.0,
                                    rcsc::ServerParam::i().maxDashPower() * 0.2,
                                    1
            ).execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }

        if ( nearest_opp_dist > 2.5 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: hold"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickHold" );
            rcsc::Body_HoldBall().execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }
    }

    if ( wm.self().pos().x > wm.offsideLineX() - 10.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: kick near side"
                            ,__FILE__, __LINE__ );
        agent->debugClient().addMessage( "DefKickToCorner" );
        Body_KickToCorner( (wm.self().pos().y < 0.0) ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
    }
    else
    {
        rcsc::Vector2D target_point;
        double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
        double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
        for( double angle = -90.0; angle <= 90.0; angle += 10.0 )
        {
            bool angle_safety = true;
            rcsc::Sector2D sector( wm.ball().pos(),
                                   1.0, 30.0,
                                   angle - 20.0,
                                   angle + 20.0 );
            for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                 it != opps_end;
                 ++it )
            {
                if( sector.contains( (*it)->pos() ) )
                {
                    angle_safety = false;
                }
            }
            if( angle_safety
                && fabs( angle ) < fabs( best_angle ) )
            {
                best_angle = angle;
            }
        }
        target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
        target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
        for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
             it != mates_end;
             ++it )
        {
            rcsc::Sector2D sector( wm.ball().pos(),
                                   1.0, 30.0,
                                   best_angle - 20.0,
                                   best_angle + 20.0 );
            if( sector.contains( (*it)->pos() ) )
            {
                target_point = rcsc::Line2D( wm.ball().pos(), rcsc::AngleDeg( best_angle ) ).projection( (*it)->pos() );
                rcsc::Body_KickOneStep( target_point, ball_speed_max ).execute( agent );
                agent->debugClient().addMessage( "DefAdvancePass(1):%lf",best_angle );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                return;
            }
            else if( (*it)->pos().dist( target_point ) < 20.0 * rcsc::AngleDeg( 20.0 ).sin() * 0.8 )
            {
                rcsc::Body_KickOneStep( target_point, ball_speed_max ).execute( agent );
                agent->debugClient().addMessage( "DefAdvancePass(2):%lf",best_angle );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
                return;
            }
        }
        if( rcsc::Body_KickOneStep( target_point,
                                    ball_speed_max ).execute( agent ) )
        {
            agent->debugClient().addMessage( "DefKickAdvance(1):%lf",best_angle );
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
            return;
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: advance"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickAdvance(2)" );
            rcsc::Body_AdvanceBall().execute( agent );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
        }
    }
    return;
}
/*-------------------------------------------------------------------*/
/*!

*/
bool
RoleOffensiveHalf::doDirectPass( rcsc::PlayerAgent * agent, const int target_unum )
{
  const rcsc::WorldModel & wm = agent->world();

  rcsc::Vector2D pass_point;
  rcsc::Vector2D modified_tgt;
  const double ball_decay = rcsc::ServerParam::i().ballDecay();
  const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
  const double ball_speed_max08 = ball_speed_max * 0.8;

  if( wm.self().unum() == target_unum )
    {
      return false;
    }

  if( ! wm.teammate( target_unum ) )
    {
      return false;
    }

  int reach_cycle = (int)(log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( wm.teammate( target_unum )->pos() ) / ball_speed_max ) / log( ball_decay ));
  modified_tgt = wm.teammate( target_unum )->pos();
  if( wm.teammate( target_unum )->pos().x > wm.self().pos().x )
    modified_tgt.x += wm.teammate( target_unum )->playerTypePtr()->kickableArea() * 0.5 * reach_cycle; // 0.5 is magic number
  else
    modified_tgt.x += wm.teammate( target_unum )->playerTypePtr()->kickableArea() * 0.8 * 0.5 * reach_cycle; // 0.5 is magic number, 0.8 is internal point

  bool target_is_free = true;
  const rcsc::Sector2D sector( wm.ball().pos(),
			       1.0, modified_tgt.dist( wm.ball().pos() ) + 3.0,
			       ( modified_tgt - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
			       ( modified_tgt - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );

  const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
  const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
  for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
       it != opps_end;
       ++it )
    {
      if( sector.contains( (*it)->pos() )
	  && ! (*it)->isTackling() )
	{
	  target_is_free = false;
	}
    }
  
  if( target_is_free
      && wm.teammate( target_unum )->posCount() < 3
      && modified_tgt.x > -25.0
      && modified_tgt.x > wm.self().pos().x - 5.0
      && modified_tgt.dist( wm.self().pos() ) > 5.0 )
    {
      if( wm.teammate( target_unum )->pos().x > wm.self().pos().x )
	{
	  pass_point = modified_tgt;
	}
      else
	{
	  pass_point = modified_tgt * 0.8 + wm.ball().pos() * 0.2;
	}

      double ball_reach_speed = rcsc::ServerParam::i().kickPowerRate() * rcsc::ServerParam::i().maxPower() * 0.5;
      double ball_speed = ball_reach_speed / pow( ball_decay, reach_cycle - 1 );
      double first_speed = std::min( ball_speed_max, ball_speed );		       
      if( rcsc::Body_SmartKick( pass_point, first_speed, first_speed * 0.9, 3 ).execute( agent ) )
        {
	  rcsc::KickTable::Sequence seq;
	  rcsc::KickTable::instance().simulate( wm,
						pass_point,
						first_speed,
						first_speed * 0.9,
						3,
						seq );
	  if( seq.pos_list_.size() == 1 )
            {
	      agent->addSayMessage( new rcsc::PassMessage( target_unum, pass_point,
							   agent->effector().queuedNextBallPos(),
							   agent->effector().queuedNextBallVel() ) );
            }
	}

      agent->debugClient().addMessage( "DirectPass:%lf",first_speed );
      rcsc::dlog.addText( rcsc::Logger::TEAM,
			  "%s:%d: reach_cycle %d, modified mate%d(%0.00f, %0.00f), pass_point(%0.00f, %0.00f)"
			  ,__FILE__, __LINE__, reach_cycle, target_unum, modified_tgt.x, modified_tgt.y, pass_point.x, pass_point.y );
      agent->debugClient().setTarget( pass_point );
      agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
      return true;
    }

  return false;
}
/*-------------------------------------------------------------------*/
/*!

*/
bool
RoleOffensiveHalf::doThroughPass( rcsc::PlayerAgent * agent, const int target_unum )
{
  const rcsc::WorldModel & wm = agent->world();

  if( wm.self().unum() == target_unum )
    {
      return false;
    }
  if( ! wm.teammate( target_unum ) )
    {
      return false;
    }
  if( wm.teammate( target_unum )->pos().x < wm.self().pos().x )
    {
      return false;
    }
  if( wm.self().pos().x > 30.0 )
    {
      return false;
    }
  if( wm.offsideLineX() - wm.teammate( target_unum )->pos().x > 15.0 )
    {
      return false;
    }

  rcsc::Vector2D target_pos = wm.teammate( target_unum )->pos();

  rcsc::Vector2D first_opp_pos( 500.0, 0.0 );
  rcsc::Vector2D second_opp_pos( 1000.0, 0.0 );
  const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
  const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
  for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
       it != opps_end;
       ++it )
    {
      bool is_defend_line = true;
      rcsc::Vector2D offside_point( wm.offsideLineX(), (*it)->pos().y );
      for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
	   itr != opps_end;
	   ++itr )
	{
	  if( offside_point.dist( (*itr)->pos() ) < offside_point.dist( (*it)->pos() ) )
	    {
	      is_defend_line = false;
	      break;
	    }
	}
      if( is_defend_line )
	{
	  if( target_pos.dist( (*it)->pos() ) < target_pos.dist( first_opp_pos )
	      && target_pos.dist( (*it)->pos() ) < target_pos.dist( second_opp_pos ) )
	    {
	      second_opp_pos = first_opp_pos;
	      first_opp_pos = (*it)->pos();
	    }
	  else if( target_pos.dist( (*it)->pos() ) < target_pos.dist( second_opp_pos ) )
	    {
	      second_opp_pos = (*it)->pos();
	    }
	}
    }

  rcsc::Vector2D pass_point = first_opp_pos * 0.6 + second_opp_pos * 0.4;
  pass_point.assign( 35.0, 
		     wm.ball().pos().y + ( 35.0 - wm.ball().pos().x ) 
		     * ( pass_point.y - wm.ball().pos().y ) / ( pass_point.x - wm.ball().pos().x ) );
  double target_angle = fabs( ( first_opp_pos - wm.ball().pos() ).th().degree() 
			  - ( second_opp_pos - wm.ball().pos() ).th().degree() );

  if( target_angle < 30.0 )
    {
      return false;
    }

  rcsc::Triangle2D target_triangle( wm.ball().pos(), 
				    first_opp_pos * 0.9 + pass_point * 0.1, 
				    second_opp_pos * 0.9 + pass_point * 0.1 );
  for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
       it != opps_end;
       ++it )
    {
      if( target_triangle.contains( (*it)->pos() ) )
	{
	  return false;
	}
      if( wm.ball().pos().dist( (*it)->pos() ) < 5.0 )
	{
	  return false;
	}
    }

  double line_dist = ( pass_point - wm.ball().pos() ).r();
  if( line_dist < 10.0 )
    {
      return false;
    }

  const double ball_decay = rcsc::ServerParam::i().ballDecay();
  const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
  rcsc::Vector2D mate_pos = wm.teammate( target_unum )->pos();
  int mate_poscount = wm.teammate( target_unum )->posCount();
  int mate_reach_cycle = wm.teammate( target_unum )->playerTypePtr()->cyclesToReachDistance( wm.self().pos().dist( pass_point ) ) + 2;
  double first_speed = wm.ball().pos().dist( pass_point ) * ( 1.0 - ball_decay ) / ( 1.0 - pow( ball_decay, mate_reach_cycle ) );
  double ball_speed = std::min( ball_speed_max, first_speed );
  if( pass_point.absY() < 25.0
      && mate_poscount < 3 )
    {
      if( rcsc::Body_SmartKick( pass_point, ball_speed, ball_speed * 0.9, 3 ).execute( agent ) )
        {
	  rcsc::KickTable::Sequence seq;
	  rcsc::KickTable::instance().simulate( wm,
						pass_point,
						ball_speed,
						ball_speed * 0.9,
						3,
						seq );
	  if( seq.pos_list_.size() == 1 )
            {
	      agent->addSayMessage( new rcsc::PassMessage( target_unum, pass_point,
							   agent->effector().queuedNextBallPos(),
							   agent->effector().queuedNextBallVel() ) );
            }
	}

      agent->debugClient().addMessage( "ThroughPass:%lf", target_angle );
      agent->debugClient().setTarget( pass_point );
      agent->debugClient().addLine( wm.ball().pos(), first_opp_pos );
      agent->debugClient().addLine( wm.ball().pos(), second_opp_pos );
      agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
      return true;
    }

  return false;
}
/*-------------------------------------------------------------------*/
/*!

*/
void
RoleOffensiveHalf::doOffensiveMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    int ball_reach_step = 0;
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent() )
    {
        ball_reach_step
            = std::min( wm.interceptTable()->teammateReachCycle(),
                        wm.interceptTable()->opponentReachCycle() );
    }
    const rcsc::Vector2D base_pos
        = wm.ball().inertiaPoint( ball_reach_step );

    rcsc::Vector2D home_pos
        = formation().getPosition( agent->config().playerNumber(),
                                   base_pos );
    if ( rcsc::ServerParam::i().useOffside() )
    {
        home_pos.x = std::min( home_pos.x, wm.offsideLineX() - 1.0 );
    }

    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        "%s: HOME POSITION=(%.2f, %.2f) base_point(%.1f %.1f)"
                        ,__FILE__,
                        home_pos.x, home_pos.y,
                        base_pos.x, base_pos.y );
    // tackle
    if ( Bhv_BasicTackle( 0.9, 80.0 ).execute( agent ) )
    {
        agent->debugClient().addMessage( "offmovetackle" );
        return;
    }

    // check ball owner
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    /*bhv_basic_move*/
    if ( ! wm.existKickableTeammate()
         && ( self_min <= 3
              || ( self_min < mate_min + 3
                   && self_min < opp_min + 4 )
             )
        )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: intercept"
                            ,__FILE__, __LINE__ );
        rcsc::Body_Intercept2008().execute( agent );
        if ( wm.ball().distFromSelf()
             < rcsc::ServerParam::i().visibleDistance() )
        {
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );
        }
        else
        {
            agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        }
        agent->debugClient().addMessage( "OffMoveIntercept" );
        return;
    }

    // go to home position
    rcsc::Vector2D target_pos = home_pos;
    if( mate_min <= opp_min
        && mate_min <= self_min )
    {
        rcsc::Line2D sight_line( wm.ball().pos(), rcsc::AngleDeg( 120.0 ) );
        if( wm.ball().pos().y > wm.self().pos().y )
        {
            sight_line.assign( wm.ball().pos(), rcsc::AngleDeg( -120.0 ) );
        }
        rcsc::Line2D front_line( wm.self().pos(), rcsc::AngleDeg( 0.0 ) );
        target_pos.x = sight_line.intersection( front_line ).x;
        target_pos.y = home_pos.y;

        const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
        bool nearest_opp = false;
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps.end();
             ++it )
        {
            if( home_pos.dist( (*it)->pos() ) < 5.0 )
            {
                nearest_opp = true;
            }
        }
        if( nearest_opp )
        {
            target_pos = target_pos * 0.8 + wm.ball().pos() * 0.2;
        }
    }
    else if( opp_min < self_min && opp_min < mate_min )
        target_pos.x = base_pos.x;

    /*
    if( abs( target_pos.y - base_pos.y ) < 5.0 )
    {
        if( target_pos.y > base_pos.y )
        {
            target_pos.y += 3.0;
        }
        else
        {
            target_pos.y -= 3.0;
        }
    }
    */

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    const double dash_power = getDashPower( agent, target_pos );
    agent->debugClient().addMessage( "OHoffensiveMove%.0f", dash_power );
    agent->debugClient().setTarget( target_pos );

    if ( ! rcsc::Body_GoToPoint( target_pos, dist_thr, dash_power
             ).execute( agent ) )
    {
        rcsc::Body_TurnToBall().execute( agent );
    }

    if ( wm.ball().posCount() > 2 )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }
    else
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    }
    return;
}
/*-------------------------------------------------------------------*/
/*!

*/
void
RoleOffensiveHalf::doDefensiveMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    int ball_reach_step = 0;
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent() )
    {
        ball_reach_step
            = std::min( wm.interceptTable()->teammateReachCycle(),
                        wm.interceptTable()->opponentReachCycle() );
    }
    const rcsc::Vector2D base_pos
        = wm.ball().inertiaPoint( ball_reach_step );

    rcsc::Vector2D home_pos
        = formation().getPosition( agent->config().playerNumber(),
                                   base_pos );
    if ( rcsc::ServerParam::i().useOffside() )
    {
        home_pos.x = std::min( home_pos.x, wm.offsideLineX() - 1.0 );
    }

    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        "%s: HOME POSITION=(%.2f, %.2f) base_point(%.1f %.1f)"
                        ,__FILE__,
                        home_pos.x, home_pos.y,
                        base_pos.x, base_pos.y );

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_opuciOffensiveHalf-DefensiveMove"
                        ,__FILE__, __LINE__ );

    // tackle
    if ( Bhv_BasicTackle( 0.9, 80.0 ).execute( agent ) )
    {
        return;
    }
    else if( wm.ball().pos().x < -38.0 && wm.ball().distFromSelf() < 4.0 )
    {
	if( Bhv_BasicTackle( 0.75, 80.0 ).execute( agent ) )
	{
            return;
	}
    }

    // check ball owner

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( ! wm.existKickableTeammate()
         && ( self_min <= 3
              || ( self_min < mate_min + 3
                   && self_min < opp_min + 4 )
              )
         )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: intercept"
                            ,__FILE__, __LINE__ );
        rcsc::Body_Intercept2008().execute( agent );
        if ( wm.ball().distFromSelf()
             < rcsc::ServerParam::i().visibleDistance() )
        {
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );;
        }
        else
        {
            agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        }
        return;
    }

    // go to home position
    rcsc::Vector2D mark_home_pos;
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    rcsc::Vector2D nearest_opp_pos( 1000.0, 0.0 );
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        if( home_pos.dist( (*it)->pos() ) < home_pos.dist( nearest_opp_pos ) )
        {
            nearest_opp_pos = (*it)->pos();
        }
    }
    mark_home_pos = ( base_pos * 0.3 + nearest_opp_pos * 0.7 ) * 0.5 + home_pos * 0.5;
/*
    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const double ball_speed = rcsc::ServerParam::i().ballSpeedMax();
    int right_back_player = 3;
    int left_back_player = 2;
    if( wm.self().unum() == 7 )
    {
        right_back_player = 2;
        left_back_player = 4;
    }
    else if( wm.self().unum() == 8 )
    {
        right_back_player = 5;
        left_back_player = 3;
    }
    if( wm.teammate(right_back_player) && wm.teammate(left_back_player) )
    {
        rcsc::Line2D first_line( wm.self().pos(), wm.teammate(right_back_player)->pos() );
        rcsc::Line2D second_line( wm.self().pos(), wm.teammate(left_back_player)->pos() );
        rcsc::Line2D third_line( rcsc::Vector2D( wm.defenseLineX(), -1.0 ), rcsc::Vector2D( wm.defenseLineX(), 1.0 ) );
        rcsc::Triangle2D center_triangle( first_line.intersection( second_line ),
                                          first_line.intersection( third_line ),
                                          second_line.intersection( third_line ) );
        rcsc::Vector2D opp_pos = center_triangle.centroid();
        rcsc::Vector2D opp_vel( 0.0, 0.0 );
        bool opp_in_center_triangle = false;
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps_end;
             ++it )
        {
            if( center_triangle.contains( (*it)->pos() ) )
            {
                opp_in_center_triangle = true;
                opp_pos = (*it)->pos();
                opp_vel = (*it)->vel();
            }
        }
        if( opp_in_center_triangle )
        {
            //kickable opponent in triangle
            if( opp_pos.dist( wm.ball().pos() ) < 3.0 )
            {
                agent->debugClient().addMessage("KickableOppInTriangle:%lf",opp_vel.r() );
                mark_home_pos = base_pos + wm.ball().vel();
            }
            else
            {
                agent->debugClient().addMessage("OppInTriangle:%lf",opp_vel.r() );
                mark_home_pos = home_pos * 0.5 + opp_pos * 0.5;
                
                rcsc::Vector2D projection = rcsc::Line2D( base_pos, opp_pos ).projection( mark_home_pos );
                int ball_reach_cycle = log( 1.0 - ( 1.0 - ball_decay ) * base_pos.dist( opp_pos ) / ball_speed ) / log( ball_decay );
                int self_cycle = wm.self().playerType().cyclesToReachDistance( mark_home_pos.dist( projection ) );
                if( ( ball_reach_cycle < self_cycle
                      || ball_reach_cycle < 5 )
                    && ( wm.self().body().degree() < -90.0
                         || wm.self().body().degree() > 90.0 ) )
                {
                    agent->debugClient().addMessage("GoToPassLine:%lf",opp_vel.r() );
                    mark_home_pos = projection;
                }
            }
        }
        else
        {
            rcsc::Vector2D near_to_left( 1000.0, 0.0 );
            rcsc::Vector2D near_to_left_vel( 0.0, 0.0 );
            rcsc::Vector2D near_to_right( 1000.0, 0.0 );
            rcsc::Vector2D near_to_right_vel( 0.0, 0.0 ); 
            for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                 it != opps_end;
                 ++it )
            {
                if( wm.teammate(left_back_player)->pos().dist( near_to_left ) > wm.teammate(left_back_player)->pos().dist( (*it)->pos() ) )
                {
                    near_to_left = (*it)->pos();
                    near_to_left_vel = (*it)->vel(); 
                }
                if( wm.teammate(right_back_player)->pos().dist( near_to_right ) > wm.teammate(right_back_player)->pos().dist( (*it)->pos() ) )
                {
                    near_to_right = (*it)->pos();
                    near_to_right_vel = (*it)->vel();
                }
            }
            if( base_pos.dist( near_to_left ) > base_pos.dist( near_to_right ) )
            {
                agent->debugClient().addMessage("MarkLeftOpponent:%lf",near_to_left_vel.r() );
                mark_home_pos = home_pos * 0.5 + near_to_left * 0.5;

                rcsc::Vector2D projection = rcsc::Line2D( base_pos, near_to_left ).projection( mark_home_pos );
                int ball_reach_cycle = log( 1.0 - ( 1.0 - ball_decay ) * base_pos.dist( near_to_left ) / ball_speed ) / log( ball_decay );
                int self_cycle = wm.self().playerType().cyclesToReachDistance( mark_home_pos.dist( projection ) );
                if( ( ball_reach_cycle < self_cycle
                      || ball_reach_cycle < 5 )
                    && ( wm.self().body().degree() < -90.0
                         || wm.self().body().degree() > 90.0 ) )
                {
                    agent->debugClient().addMessage("GoToPassLine:%lf",near_to_left_vel.r() );
                    mark_home_pos = projection;
                }
            }
            else
            {
                agent->debugClient().addMessage("MarkRightOpponent:%lf",near_to_right_vel.r() );
                mark_home_pos = home_pos * 0.5 + near_to_right * 0.5;

                rcsc::Vector2D projection = rcsc::Line2D( base_pos, near_to_right ).projection( mark_home_pos );
                int ball_reach_cycle = log( 1.0 - ( 1.0 - ball_decay ) * base_pos.dist( near_to_right ) / ball_speed ) / log( ball_decay );
                int self_cycle = wm.self().playerType().cyclesToReachDistance( mark_home_pos.dist( projection ) );
                if( ( ball_reach_cycle < self_cycle
                      || ball_reach_cycle < 5 )
                    && ( wm.self().body().degree() < -90.0
                         || wm.self().body().degree() > 90.0 ) )
                {
                    agent->debugClient().addMessage("GoToPassLine:%lf",near_to_right_vel.r() );
                    mark_home_pos = projection;
                }
            }
        }
    }
    else
    {
        mark_home_pos = home_pos;
    }
*/
    if( mark_home_pos.x < wm.defenseLineX() + 1.0 )
    {
        mark_home_pos.x = wm.defenseLineX() + 1.0;
    }
    /*
    if( opp_min < self_min && opp_min < mate_min && wm.ball().pos().x < wm.self().pos().x )
      mark_home_pos.x = wm.ball().inertiaPoint( opp_min + 1 ).x - 1.0;
    */
    const double dash_power = getDashPower( agent, mark_home_pos );

    double dist_thr = wm.ball().distFromSelf() * 0.2;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    agent->debugClient().addMessage( " BasicMove%.0f", dash_power );
    agent->debugClient().setTarget( mark_home_pos );

    if ( ! rcsc::Body_GoToPoint( mark_home_pos, dist_thr, dash_power, 1
                                 ).execute( agent ) )
    {
        rcsc::Body_TurnToBall().execute( agent );
    }

    if( wm.ball().posCount() > 3 )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
        return;
    }
/*
    if( wm.teammate(right_back_player) )
    {
        if( wm.teammate(right_back_player)->posCount() > 5 )
        {
            agent->setNeckAction( new rcsc::Neck_TurnToPoint( wm.teammate(right_back_player)->pos() ) );
        }
    }
    if( wm.teammate(left_back_player) )
    {
        if( wm.teammate(left_back_player)->posCount() > 5 )
        {
            agent->setNeckAction( new rcsc::Neck_TurnToPoint( wm.teammate(left_back_player)->pos() ) );
        }
    }
*/
    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    return;
}
/*-------------------------------------------------------------------*/
/*!

*/
double
RoleOffensiveHalf::getDashPower( const rcsc::PlayerAgent * agent,
                                 const rcsc::Vector2D & target_point )
{
    static bool s_recover_mode = false;

    const rcsc::WorldModel & wm = agent->world();
    /*--------------------------------------------------------*/
    // check recover
    if (( wm.ball().pos().x < 5.0 && wm.existKickableOpponent() )
	|| wm.ball().pos().x < wm.self().pos().x + 5.0 )
    {
        s_recover_mode = false;
    }
    else if ( wm.self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.5 )
    {
        s_recover_mode = true;
    }
    else if ( wm.self().stamina() > rcsc::ServerParam::i().staminaMax() * 0.7 )
    {
        s_recover_mode = false;
    }

    /*--------------------------------------------------------*/
    double dash_power = rcsc::ServerParam::i().maxDashPower();
    const double my_inc
        = wm.self().playerType().staminaIncMax()
        * wm.self().recovery();


    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();


    double dist_x = wm.self().pos().x - target_point.x;
    if( dist_x > 5.0 )
    {
        dash_power = rcsc::ServerParam::i().maxDashPower();
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": go back immediately" );
    }
    else if ( s_recover_mode )
    {
        dash_power = my_inc - 25.0; // preffered recover value
        if ( dash_power < 0.0 ) dash_power = 0.0;

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": recovering" );
    }
    else if ( wm.existKickableTeammate() )
    {
        // exist kickable teammate but behind the home position
        if( wm.ball().distFromSelf() > 15.0 && dist_x < 0.0
	    || wm.ball().pos().x > 25.0 )
        {
            dash_power = rcsc::ServerParam::i().maxDashPower();
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: exist kickable teammate and should go there. dash_power= %f"
                                ,__FILE__, __LINE__,
                                dash_power );
        }
        // exist kickable teammate and position is almost ok
        else
        {
            dash_power = std::min( my_inc - 20.0,
                                   rcsc::ServerParam::i().maxPower() * 0.5 );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: exist kickable teammate and not in a hurry. dash_power= %f"
                                ,__FILE__, __LINE__,
                                dash_power );
        }
    }
    else if( self_min < mate_min )
    {
      dash_power = rcsc::ServerParam::i().maxDashPower();
      rcsc::dlog.addText( rcsc::Logger::TEAM,
			  "%s:%d: aggressive move by nearest agent. dash_power= %f"
			  ,__FILE__, __LINE__,
			  dash_power );
    }
    else if( wm.ball().pos().x < wm.self().pos().x )
    {
      dash_power = rcsc::ServerParam::i().maxDashPower() * 0.7;
      rcsc::dlog.addText( rcsc::Logger::TEAM,
			  "%s:%d: bad situation. dash_power= %f"
			  ,__FILE__, __LINE__,
			  dash_power );
    }
    // in offside area
    else if ( wm.self().pos().x > wm.offsideLineX() )
    {
        dash_power = rcsc::ServerParam::i().maxDashPower();
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": in offside area. dash_power=%.1f",
                            dash_power );
    }
    //nomal
    else
    {
        dash_power = std::min( my_inc * 1.3,
                               rcsc::ServerParam::i().maxPower() * 0.75 );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: normal mode dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    return dash_power;
}
