// -*-c++-*-

/*!
  \file serializer_v3.h
  \brief v3 format rcg serializer class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 3 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_RCG_SERIALIZER_V3_H
#define RCSC_RCG_SERIALIZER_V3_H

#include <rcsc/rcg/serializer.h>

#include <map>

namespace rcsc {
namespace rcg {

/*!
  \class SerializerV3
  \brief rcg data serializer interface class
*/
class SerializerV3
    : public Serializer {
private:

public:

    /*!
      \brief destruct all
    */
    ~SerializerV3()
      { }

    /*!
      \brief write header
      \param os reference to the output stream
      \return serialization result
    */
    virtual
    std::ostream & serializeHeader( std::ostream & os );

    /*!
      \brief write parameter message
      \param os reference to the output stream
      \param msg server parameter message
      \return reference to the output stream
    */
    virtual
    std::ostream & serializeParam( std::ostream & os,
                                   const std::string & msg );

    /*!
      \brief write server param
      \param os reference to the output stream
      \param param network byte order data
      \return serialization result
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const server_params_t & param );

    /*!
      \brief write player param
      \param os reference to the output stream
      \param pparam network byte order data
      \return serialization result
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const player_params_t & pparam );

    /*!
      \brief write player type param
      \param os reference to the output stream
      \param type network byte order data
      \return serialization result
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const player_type_t & type );

    /*!
      \brief write dispinfo_t.
      \param os reference to the output stream
      \param disp network byte order data
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const dispinfo_t & disp );

    /*!
      \brief write showinfo_t.
      \param os reference to the output stream
      \param show network byte order data
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const showinfo_t & show );

    /*!
      \brief write showinfo_t2
      \param os reference to the output stream
      \param show2 data to be output
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const showinfo_t2 & show2 );

    /*!
      \brief write short_showinfo_t2.
      \param os reference to the output stream
      \param show2 network byte order data
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const short_showinfo_t2 & show2 );

    /*!
      \brief write message info
      \param os reference to the output stream
      \param msg network byte order data
      \return serialization result
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const msginfo_t & msg );

    /*!
      \brief write message info
      \param os reference to the output stream
      \param board message board type
      \param msg message string
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const Int16 board,
                              const std::string & msg );

   /*!
      \brief write drawinfo_t
      \param os reference to the output stream
      \param draw drawinfo_t variable
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const drawinfo_t & draw );

    /*!
      \brief write playmode
      \param os reference to the output stream
      \param playmode play mode variable
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const char playmode );

    /*!
      \brief write team info
      \param os reference to the output stream
      \param team_l left team variable
      \param team_r right team variable
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const team_t & team_l,
                              const team_t & team_r );

    /*!
      \brief write team info
      \param os reference to the output stream
      \param team_l left team variable
      \param team_r right team variable
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const TeamT & team_l,
                              const TeamT & team_r );

    /*!
      \brief write ShowInfoT
      \param os reference to the output stream
      \param show network byte order data
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const ShowInfoT & show );

    /*!
      \brief write DispInfoT
      \param os reference to the output stream
      \param disp data to be written
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const DispInfoT & disp );

private:

    /*!
      \brief analyze parameter message and store all elements
      \param msg source message
      \param param_map map variable to store all elements
      \return parsing result
     */
    bool parseParam( const std::string & msg,
                     std::map< std::string, std::string > & param_map );

    /*!
      \brief convert server message to parameters
      \param from source message
      \param to destination variable
      \return parsing result
     */
    bool parseParam( const std::string & from,
                     server_params_t & to );

    /*!
      \brief convert server message to parameters
      \param from source message
      \param to destination variable
      \return parsing result
     */
    bool parseParam( const std::string & from,
                     player_params_t & to );

    /*!
      \brief convert server message to parameters
      \param from source message
      \param to destination variable
      \return parsing result
     */
    bool parseParam( const std::string & from,
                     player_type_t & to );

};

} // end of namespace rcg
} // end of namespace rcsc

#endif
