/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.controller {
	
	import flash.display.DisplayObject;
	import flash.events.Event;
	import flash.events.EventDispatcher;
	import flash.utils.Dictionary;
	import flash.utils.describeType;
	
	import jp.co.fujitsu.reffi.client.flex.action.AbstractAction;
	import jp.co.fujitsu.reffi.client.flex.action.BaseAction;
	import jp.co.fujitsu.reffi.client.flex.error.ICoreError;
	import jp.co.fujitsu.reffi.client.flex.error.Messages;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.manager.ComponentManager;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	
	import mx.core.Application;
	import mx.core.IMXMLObject;
	import mx.managers.SystemManager;
	import mx.rpc.events.FaultEvent;

	/**
	 * <p>[概 要] </p>
	 * アクションとモデルを制御する基底コントローラクラスです.
	 * 
	 * <p>[詳 細] </p>
	 * コントローラは大きく分けて二つの役割を担います。
	 * <p>
	 * １．<b>コンポーネント、イベント、アクションの紐付け</b>
	 * <p>
	 * BaseControllerを継承した具象コントローラクラスのbind(EventBinder)をオーバーライド実装することで、<br>
	 * コンポーネント、イベント、アクションの紐付けを行います。<br>
	 * 画面表示リストに該当のコンポーネントが挿入された時、紐付けられた情報を元にイベントリスナ、
	 * ハンドラ登録を行います。
	 * <p>
	 * 
	 * ２．<b>イベント発生時の処理フロー形成</b>
	 * <p>
	 * １で登録されたイベントが発生した時、handlerFacade()が全てのイベントハンドリングの
	 * 入り口となります。<br>
	 * イベント処理を委譲されたBaseControllerは以下の処理フローを作ります。
	 * 
	 * <p>
	 * <pre>
	 * 	<table width="100%" border="1" style="border-collapse:collapse;">
	 * 		<tr>
	 * 			<td>実行順序</td>
	 * 			<td>BaseAction</td>
	 * 			<td>BaseController</td>
	 * 			<td>BaseModel</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">1</td>
	 * 			<td>　</td>
	 * 			<td>handlerFacade()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">2</td>
	 * 			<td>　</td>
	 * 			<td>　invoke()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">3</td>
	 * 			<td>　</td>
	 * 			<td>　　runAction()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">4</td>
	 * 			<td>run()</td>
	 * 			<td>　</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">5</td>
	 * 			<td>　</td>
	 * 			<td>　　runModels()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">5´</td>
	 * 			<td>　</td>
	 * 			<td>　　runModelsAndNoWait()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">6</td>
	 * 			<td>　</td>
	 * 			<td>　</td>
	 * 			<td>run()</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">7</td>
	 * 			<td>　</td>
	 * 			<td>　　モデル処理監視リスナ（成功）</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">8</td>
	 * 			<td>successForward()</td>
	 * 			<td>　</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">7´</td>
	 * 			<td>　</td>
	 * 			<td>　　モデル処理監視リスナ（失敗）</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">8´</td>
	 * 			<td>failureForward()</td>
	 * 			<td>　</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">9</td>
	 * 			<td>　</td>
	 * 			<td>　　trap()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">10</td>
	 * 			<td>　</td>
	 * 			<td>　　handlerFinalize()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">11</td>
	 * 			<td>　</td>
	 * 			<td>modelsDone()</td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 		<tr>
	 * 			<td align="center">12</td>
	 * 			<td>complete()</td>
	 * 			<td></td>
	 * 			<td>　</td>
	 * 		</tr>
	 * 	</table>
	 * </pre>
	 * この過程で呼ばれるアクション（run）とモデル（run）の中では別途処理フローが作られ、
	 * 機能実装者にフックポイントを提供します。
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @example 典型的な具象コントローラ（BaseController継承クラス）
	 * <listing version="3.0">
        package demo.client.controller {
            import demo.client.chat.action.ChannelStartAction;
            import demo.client.chat.action.ChannelStopAction;
            import demo.client.chat.action.ChatWindowCloseAction;
            import demo.client.chat.action.EnterChannelAction;
            import demo.client.chat.action.LeaveChannelAction;
            import demo.client.chat.action.RemarkSendAction;
            import demo.client.componentSearch.action.ComponentSearchCreationCompleteAction;
            import demo.client.componentSearch.action.SearchAction;
            import demo.client.custom.action.AbortRowDataAction;
            import demo.client.custom.action.ChangeColumnAction;
　　　　　　　　　　　:
　　　　　　　　　　　:
　　　　　　　　　　　:

            import flash.events.Event;
            import flash.events.FocusEvent;
            import flash.events.MouseEvent;
            
            import jp.co.fujitsu.reffi.client.flex.controller.BaseController;
            import jp.co.fujitsu.reffi.client.flex.controller.EventBinder;
            
            import mx.events.CloseEvent;
            import mx.events.FlexEvent;
            import mx.events.IndexChangedEvent;
        
            public class DemoController extends BaseController {
            
                 // コントローラ（アプリケーション）初期化処理を実装します 
                override protected function initialize(clientConfig:ClientConfig):void {
                
                }      
                  
                  // コンポーネントイベントとアクションの紐付けを定義します
                override protected function bind(eventBinder:EventBinder):void {
                    // メニュー画面アクション
                    eventBinder.addEventBinding("Index.openTop", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openChat", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openForm", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openWindowGroupMenu", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.lineupWindows", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openServerPush", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openPDF", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openDataGrid", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openWSRequest", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openUserInteractive", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openComponentSearch", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.openSpring", MouseEvent.CLICK, Class(demo.client.index.action.ViewStackChangeAction));
                    eventBinder.addEventBinding("Index.blackStyleRadioButton", MouseEvent.CLICK, Class(demo.client.index.action.StyleRadioButtonClickAction));
                    eventBinder.addEventBinding("Index.blueStyleRadioButton", MouseEvent.CLICK, Class(demo.client.index.action.StyleRadioButtonClickAction));
                    eventBinder.addEventBinding("Index.greenStyleRadioButton", MouseEvent.CLICK, Class(demo.client.index.action.StyleRadioButtonClickAction));
                    eventBinder.addEventBinding("Index.grayStyleRadioButton", MouseEvent.CLICK, Class(demo.client.index.action.StyleRadioButtonClickAction));
        
                    // トップ画面
                    eventBinder.addEventBinding("Top.menuLogoImage", FlexEvent.CREATION_COMPLETE, Class(demo.client.top.action.ImageCreationCompleteAction));
                    
                    // チャットデモ画面アクション
                    eventBinder.addEventBinding("chat.channelStartButton", MouseEvent.CLICK, Class(demo.client.chat.action.ChannelStartAction));
                    eventBinder.addEventBinding("chat.channelStopButton", MouseEvent.CLICK, Class(demo.client.chat.action.ChannelStopAction));
                    eventBinder.addEventBinding("chat.enterChannel", MouseEvent.CLICK, Class(demo.client.chat.action.EnterChannelAction));
                    eventBinder.addEventBinding("chat.handleName", FlexEvent.ENTER, Class(demo.client.chat.action.EnterChannelAction));
                    eventBinder.addEventBinding("chat.leaveChannel", MouseEvent.CLICK, Class(demo.client.chat.action.LeaveChannelAction));
                    eventBinder.addEventBinding("chat.remarkSend", MouseEvent.CLICK, Class(demo.client.chat.action.RemarkSendAction));
                    eventBinder.addEventBinding("chat.chatRemark", FlexEvent.ENTER, Class(demo.client.chat.action.RemarkSendAction));
                    eventBinder.addEventBinding("chat.chatWindow", Event.CLOSE, Class(demo.client.chat.action.ChatWindowCloseAction));
        
                }
                 
                // 毎イベント最終共通処理を実装します
                override protected function handlerFinalize(parameterMapping:ParameterMapping):void {
                
                }
            }
        }
	 * </listing>
	 *
	 * @example 具象コントローラの登録方法
	 * <listing version="3.0">
        &lt;?xml version="1.0" encoding="utf-8"?&gt;
        &lt;mx:Application xmlns:mx="http://www.adobe.com/2006/mxml" <font color="FF0000">xmlns:controller="demo.client.controller.&#42;"</font>&gt;
            <font color="FF0000">&lt;controller:DemoController name="demoController" /&gt;</font>
                  :
                  :
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class BaseController extends EventDispatcher implements IMXMLObject, IController{

		// このコントローラの名前 
		private var _name:String;

		// このコントローラが制御するmx:Applicationインスタンス
		private var _application:Application;
		
		// 一アプリケーション（一SWF）内のイベント定義情報が格納されたオブジェクト
		private var _eventBinder:EventBinder;
		
		// フレームワークの挙動情報を保持するオブジェクト
		private static var _clientConfig:ClientConfig = new ClientConfig();

		// nアプリケーション(nSWF)間で共有可能な恒久領域
		private static var _permanent:Dictionary = new Dictionary();

		// AVMにロードされているBaseController継承オブジェクト群を保持するオブジェクト
		private static var _controllers:Dictionary = new Dictionary();


		/**
		 * <p>[概 要]</p>
		 * このコントローラの名前です。デフォルトではクラス型名が入ります.
		 * 
		 * <p>[詳 細]</p>
		 * AVM上に複数のSWF(Application)がロードされる場合、Applicationを管理する
		 * コントローラを識別する為のキーとして使用されます。<br>
		 * 
		 * <p>[備 考]</p>
		 * 読み込まれる各SWFを管理するコントローラクラス名が同一である場合、
		 * コントローラを識別することが出来ません。<br>
		 * 作成する具象コントローラは可能な限り別クラス名にするか、
		 * もしくはinitialize時にユニークなnameプロパティを設定してください。
		 * <p>
		 * 
		 * @see #controllers
		 */
		public function get name():String{
			return this._name;
		}
		public function set name(name:String):void{
			this._name = name;
		}
		
		/**
		 * <p>[概 要]</p>
		 * このコントローラが制御するmx:Applicationインスタンスです.
		 * 
		 * <p>[詳 細]</p>
		 * このコントローラを記述したタグの親ノードとなるApplicationを示すプロパティです。
		 * <listing version="3.0">
	        &lt;?xml version="1.0" encoding="utf-8"?&gt;
	        &lt;mx:Application name="testApp" 
	            xmlns:mx="http://www.adobe.com/2006/mxml" 
	            xmlns:controller="demo.client.controller.&#42;"&gt;
	            
	            &lt;controller:TestController name="testController" /&gt;
	                  :
	                  :
		 * </listing>
		 * 上記の場合、applicationプロパティにはtestAppインスタンスが格納されます。
		 * 
		 * <p>[備 考]</p>
		 */
		public function get application():Application{
			return this._application;
		}
		public function set application(application:Application):void{
			this._application = application;
		}

		/**
		 * <p>[概 要]</p>
		 * 一アプリケーション（一SWF）内のイベント定義情報が格納されたオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @see jp.co.fujitsu.reffi.client.flex.controller.EventBinder
		 */
		public function get eventBinder():EventBinder{
			return this._eventBinder;
		}
		public function set eventBinder(eventBinder:EventBinder):void{
			this._eventBinder = eventBinder;
		}
		
		/**
		 * <p>[概 要]</p>
		 * フレームワークの挙動情報を保持するオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * コントローラ初期化時（BaseController#initilize()）に引数として渡るオブジェクトです。<br>
		 * initialize内で動作の制御を設定します。
		 * 
		 * <p>[備 考]</p>
		 */
		public static function get clientConfig():ClientConfig {
			return _clientConfig;
		}
		public static function set clientConfig(clientConfig:ClientConfig):void {
			_clientConfig = clientConfig;
		}
		 
		/**
		 * <p>[概 要]</p>
		 * nアプリケーション(nSWF)間で共有可能な恒久領域です.
		 * 
		 * <p>[詳 細]</p>
		 * A.swfとB.swfが読み込まれている場合、AControllerとBControllerで同じ領域が参照されます。
		 * 
		 * <p>[備 考]</p>
		 */
		public function get permanent():Dictionary{
			return _permanent;
		}
		public function set permanent(permanent:Dictionary):void{
			_permanent = permanent;
		}
		
		/**
		 * <p>[概 要]</p>
		 * AVMにロードされているBaseController継承オブジェクト群を保持するオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * A.swf(AControllerクラスが管理)とB.swf(BControllerクラスが管理)が読み込まれている場合、
		 * 以下のようにコントローラインスタンスが保持されます。
		 * 
		 * <pre>
		 * controllers:Dictionary
		 *     "AControllerのnameプロパティ値" = AController instance
		 *     "BControllerのnameプロパティ値" = BController instance
		 * </pre>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @see #name
		 */
		public static function get controllers():Dictionary{
			return _controllers;
		}
		public static function set controllers(controllers:Dictionary):void{
			_controllers = controllers;
		}

		/**
		 * <p>[概 要]</p>
		 * 引数applicationを管理しているBaseControllerインスタンスを返却します.
		 * 
		 * <p>[詳 細]</p>
		 * controllersプロパティを走査して引数applicationと同じapplicationを持つ
		 * コントローラを返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param application コントローラを識別する為のApplicationオブジェクト
		 */
		public static function getController(application:Application):BaseController {
			var ret:BaseController = null;
			
			for each(var controller:BaseController in controllers) {
				if(controller.application == application) {
					ret = controller;
					break;
				}
			}
			
			return ret;
		}
		
		/**
		 * <p>[概 要]</p>
		 * デフォルトコンストラクタです.
		 * 
		 * <p>[詳 細]</p>
		 * describeType(this)の結果、取得するXMLのtypeノードからname属性を取り出します。
		 * 取り出したname属性はクラス型名のみ切り出します。
		 * 取得されたクラス名はnameプロパティに設定されます。
		 * 
		 * <p>[備 考]</p>
		 * nameプロパティに設定された名前は、BaseController#controllersに格納される、
		 * このコントローラオブジェクトのキーになります。
		 * <p/>
		 * 
		 */
		public function BaseController(){
			// nameプロパティにthisクラス型名を設定
			this.name = String(describeType(this).@name).match(/::(.*)/)[1];
			
			// イベント紐付けオブジェクトを生成
			this.eventBinder = new EventBinder(this);
			
			// 具象コントローラで定義されているイベント紐付け情報を汲み取る
			bind(this.eventBinder);
		}
		
		/**
		 * <p>[概 要]</p>
		 * このコントローラタグがApplicationの子要素としてパースされた時点でAVMによってコールバックされるメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * このコントローラの親要素であるApplication、及びApplicationを管理するSystemManagerに対して、<br>
		 * ADDEDイベント（ハンドラ：bindEvents）、REMOVEDイベント（ハンドラ：removeEvents）リスナを追加します。<br>
		 * 
		 * その他、nameプロパティへの値設定（クラス名を設定）、initializeメソッドのテンプレートコールを行います。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param document このコントローラがタグとして配置されたApplication
		 * @param id タグとして配置されたコントローラのid属性値
		 */
		public final function initialized(document:Object, id:String):void{
			// このコントローラが制御するmx:Applicationインスタンスを取得
			var application:Application = document as Application;		
			// フィールドに設定
			this.application = application;
			
			// Applicationのaddedイベントで、追加コンポーネントのイベントとアクションを紐付ける
			application.addEventListener(Event.ADDED, bindEvents);
			// removedイベントで削除されたコンポーネントのイベントリスナを解放
			application.addEventListener(Event.REMOVED, removeEvents);

			// Applicationを管理しているSystemManagerインスタンスを取得
			var systemManager:SystemManager = SystemManager.getSWFRoot(application) as SystemManager;

			// SystemManagerのaddedイベントで、追加コンポーネントのイベントとアクションを紐付ける
			// ポップアップで表示リストに入った画面はこのイベントでイベントリスナ登録される
			systemManager.addEventListener(Event.ADDED, bindEvents);
			// removedイベントで削除されたコンポーネントのイベントリスナを解放
			systemManager.addEventListener(Event.REMOVED, removeEvents);

			// このコントローラインスタンスをnameプロパティをキーにして恒久領域に保存
			// 他SWF内のBaseController継承コントローラからthisにアクセスを可能にする
			BaseController.controllers[this.name] = this;

			// 具象コントローラ#initializeでクライアント設定を可能にする			
			initialize(clientConfig);
			
			// 設定されたClientConfigを元に設定処理実行
			postInitialize(clientConfig);
		}
		
		/**
		 * <p>[概 要] </p>
		 * 初期化処理が記述可能なメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * 具象コントローラがApplicationの子要素として認識されたタイミングでテンプレートコールされます。<br>
		 * デフォルトの処理は有りません。
		 * 
		 * <p>[備 考] </p>
		 * 業務固有の初期化処理が必要な場合は、具象コントローラ内でこのメソッドをオーバーライドして下さい。<br>
		 * 又、フレームワークの挙動設定をこのタイミングで設定することが出来ます。
		 * </p>
		 * 
		 * @example 
		 * <listing version="3.0">
              override protected function initialize(clientConfig:ClientConfig):void {
                 
                  // アクションでモデル予約が無かった場合に動作させるデフォルトモデルクラスを設定
                  clientConfig.defaultModelClass = Class(DemoSystemDefaultModel);
         
                  // エラー発生時にコントローラによるエラー表示を抑止する
                  clientConfig.showAlertOnError = false;
              }
		 * </listing>
		 * 
		 * @param clientConfig フレームワーク挙動情報を保持するオブジェクトです
		 */
		protected function initialize(clientConfig:ClientConfig):void {
		}
		
		/**
		 * <p>[概 要]</p>
		 * ClientConfigベースの初期化処理を行います.
		 * 
		 * <p>[詳 細]</p>
		 * initializeで設定されたClientConfigを元に、システムローカルな初期化設定を行います。
		 * 
		 * <p>[備 考]</p>
		 * BaseController内のローカル初期化メソッドです。
		 * </p>
		 * 
		 * @param event このコントローラが管理するApplicationにコンポーネントが追加されたaddedイベント 
		 */
		private function postInitialize(clientConfig:ClientConfig):void {
//			var self:BaseController = this;
//			if(clientConfig.enableConsole) {
//				application.addEventListener(FlexEvent.APPLICATION_COMPLETE, function(evt:FlexEvent):void {
//					application.stage.addEventListener(KeyboardEvent.KEY_UP, function(event:KeyboardEvent):void {
//						if(event.keyCode == 123) {
//							var console:Console = new Console(self);
//							WindowManager.getInstance().addSingletonPopUp("console", console, false, true);
//						}
//					});
//				});
//			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * 追加された画面コンポーネントに対してイベントリスナ登録を行います.
		 * 
		 * <p>[詳 細]</p>
		 * 追加されたコンポーネントがeventBinderプロパティに登録されているname属性を持つ場合、<br>
		 * 該当するイベントタイプのハンドラとしてthis.handlerFacade(event:Event)を追加します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param event このコントローラが管理するApplicationにコンポーネントが追加されたaddedイベント 
		 */
		private function bindEvents(event:Event):void{
			// 画面に追加されたのがDisplayObject継承オブジェクトである場合
			if(event.target is DisplayObject){
				var addedComponent:DisplayObject = DisplayObject(event.target);
				
				// イベント定義情報を元に、コンポーネントにイベントリスナ登録
				if(this.eventBinder.hasEventBinding(addedComponent.name)){
					var eventTypes:Array = this.eventBinder.getEventTypes(addedComponent.name);
					for each(var eventType:String in eventTypes){
						// 一回のイベント発生で複数アクション起動する為、
						// handlerFacadeへの重複登録を避ける
						if(addedComponent.hasEventListener(eventType)) {
							addedComponent.removeEventListener(eventType, handlerFacade);
						}
						// 追加されたコンポーネントにイベントリスナ追加、ハンドラはthis.handlerFacade
						addedComponent.addEventListener(eventType, handlerFacade);
					}
				}
			}
		}

		/**
		 * <p>[概 要]</p>
		 * 削除された画面コンポーネントに付与されているイベントリスナを削除します.
		 * 
		 * <p>[詳 細]</p>
		 * 削除されたコンポーネントがeventBinderプロパティに登録されているname属性を持つ場合、<br>
		 * this.handlerFacade(event:Event)を呼ぶイベントリスナを削除します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param event このコントローラが管理するAppicationからコンポーネントが削除されたremovedイベント
		 */
		public function removeEvents(event:Event):void {
			// 削除されたのがDisplayObject、且つイベント紐付け定義がされていたコンポーネントの場合
			if(event.target is DisplayObject && 
			   eventBinder.hasEventBinding(DisplayObject(event.target).name)) 
			{
				// 削除されたDisplayObjectインスタンス
				var removedComponent:DisplayObject = DisplayObject(event.target);
				// 予約されていたイベントタイプ数分、イベントリスナを削除				
				var eventTypes:Array = eventBinder.getEventTypes(removedComponent.name);
				for each(var eventType:String in eventTypes) {
					removedComponent.removeEventListener(eventType, handlerFacade);
				}
			}
		}
		
		/**
		 * <p>[概 要] </p>
		 * 全ユーザ定義イベントをハンドルする入り口になるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * 発生したイベントタイプ、イベントを起こしたコンポーネント名を元に、<br>
		 * イベント紐付けオブジェクト（EventBinder）から対応するアクションクラス型を取得します。<br>
		 * 取得したアクションクラス型を引数にしてinvoke(Class, ParameterMapping)に処理委譲します。
		 * 
		 * <p>[備 考] </p>
		 * bind(EventBinder)で定義したイベントは全てこのメソッドがハンドリングします。
		 * <p>
		 * 
		 * @param event 発生イベント
		 */
		public function handlerFacade(event:Event):void {
    		// 一アクションシーケンス中だけ保持されるパラメータマッピングオブジェクトを生成
			var parameterMapping:ParameterMapping = createParameterMapping(event);

			// イベントタイプに紐付いたAction数分、Actionを実行
			var actionClasses:Array = 
				eventBinder.getActionClasses(event.currentTarget.name, event.type);
			for each(var actionClass:Class in actionClasses){
				// 一アクションを起動
				invoke(actionClass, parameterMapping); 
			}
		}

		/**
		 * <p>[概 要] </p>
		 * MVC各レイヤを巡回するParameteraMappingオブジェクトを生成、初期化します.
		 * 
		 * <p>[詳 細] </p>
		 * イベント発生の際、Controller、Action、Modelを流れるデータマップを作成します。<br>
		 * このメソッドが呼ばれるフロー段階では発生したEventオブジェクト、
		 * イベント発生させたオブジェクト等が初期値として設定されます。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @return MVC各レイヤを巡回するパラメータオブジェクト
		 */
		protected function createParameterMapping(event:Event):ParameterMapping {
			var mapping:ParameterMapping = new ParameterMapping();

			mapping.isRunModelsAndNoWait = false;
			mapping.models = new Array();
			mapping.validators = new Array();
			mapping.waitModelsDone = false;
			mapping.actionInstance = null;
			mapping.event = event;
			mapping.eventSourceObject = DisplayObject(event.currentTarget);

			return mapping;
		}
		
		/**
		 * <p>[概 要]</p>
		 * コントローラの主幹メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * コントローラ処理フローの幹を形成します。
		 * このメソッドのtryスコープで以下が行われます。
		 * 	<ol>
		 * 		<li>runAction(Class, ParameterMapping) アクションの実行</li>
		 * 		<li>runModels(Array, ParameterMapping, int, ModelProcessEvent) or <br>
		 *          runModelsAndNoWait(Array, ParameterMapping) アクションで予約されたモデル群の実行
		 *      </li>
		 * 	</ol>
		 * 
		 * 上記の処理中に例外が発生した場合、trapメソッドがテンプレートコールされます。<br>
		 * 最終的にfinallyスコープに入るとhanderFinalizeメソッドがテンプレートコールされます。
		 * </p>
		 * 
		 * イベントハンドリング時では無く、任意にアクションを起動する場合、
		 * 第3引数completeHandler、第4引数failureHanlerを指定すると、
		 * invokeの成功、失敗をアクションの外部からハンドル出来ます。<br>
		 * 指定されたcompleteHandlerはアクション#completeが実行された後、
		 * failureHanlderはアクション#failureForwardが実行された後のタイミングで実行されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param event 画面コンポーネントから発生したイベントオブジェクト
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @param successFunction invoke成功時に呼ばれる関数(デフォルト：null)
		 * @param failureFunction invoke失敗時に呼ばれる関数(デフォルト：null)
		 */
		public function invoke(actionClass:Class, 
								 parameterMapping:ParameterMapping, 
								 completeHandler:Function = null, 
								 failureHandler:Function = null):void
		{
			try {
				if(completeHandler != null) {
					parameterMapping.invokeCompleteHandler = completeHandler;
				}
				if(failureHandler != null) {
					parameterMapping.invokeFailureHandler = failureHandler;
				}		
			
				// アクションを実行
				parameterMapping = runAction(actionClass, parameterMapping);
				// BaseAction#prepareがfalse返却、又はvalidationFaultが発生した場合
				if(parameterMapping == null) {
					return;
				}
				
				// BaseAction#isWaitModelsDone()がtrueだった場合
				if (parameterMapping.waitModelsDone) {
					// 全モデルが終了するまで、モーダル画面を表示
					var modelWaitingDialog:ModelWaitingDialog = new ModelWaitingDialog();
					ComponentManager.getInstance().addPopUp("modelWaitingDialog", modelWaitingDialog, null, true, true);
				}

				// アクションで予約されたモデル群を取得
				var models:Array = parameterMapping.models;
				// BaseAction#reserveModelsによる登録が無い場合、デフォルトモデルを予約
				if(models.length == 0) {
					models.push(clientConfig.defaultModelClass);
				}
				parameterMapping.modelReservedNum = models.length;
				
				// アクションでreserveModelsがオーバーライドされず、モデルが予約されなかった場合
				if (parameterMapping.isRunModelsAndNoWait) {
					// モデル群を非同期に実行
					runModelsAndNoWait(models, parameterMapping);
				} else {
					// モデル群をシーケンシャルに実行
					runModels(models, parameterMapping);
				}
			} catch (e:Error) {
				// 同期エラー処理
				trap(e);
			} finally {
				// 最終処理
				handlerFinalize(parameterMapping);
			}
		}

		/**
		 * <p>[概 要] </p>
		 * イベントに対応するアクションを実行します.
		 * 
		 * <p>[詳 細] </p>
		 * 引数で指定されたアクションクラス型をインスタンス化、
		 * 実行（BaseAction#run(ParameterMapping)）します。<br>
		 * execute実行後、結果戻り値であるParameterMappingを返却します。
		 * 
		 * <p>[備 考] </p>
		 * この結果がnullだった場合はコントローラの以降の処理は中止されます。
		 * BaseAction#run(ParameterMapping)がnullを返却するのは、
		 * prepareがfalseを返却、もしくはvalidate結果がエラーだった場合があります。
		 * <p/>
		 * 
		 * @param actionClass 実行するアクションクラスの型
		 * @param parameterMapping MVCを巡回するパラメータマッピング
		 * @return アクション実行後のParameterMappingオブジェクト
		 */
		public function runAction(actionClass:Class, parameterMapping:ParameterMapping):ParameterMapping {
			var action:AbstractAction = new actionClass;
			action.controller = this;

			// アクションを実行、実行するモデルクラス群、バリデータ群等の情報を取得
			parameterMapping = action.run(parameterMapping);
			// BaseAction#prepareメソッドでfalseが返却された場合は以降の処理は中止
			if(parameterMapping == null){
				return null;
			}
			// paramterMappingにアクションインスタンスを設定
			// モデル層でもアクション機能を利用出来るようにする
			parameterMapping.actionInstance = action;

			return parameterMapping;
		}

		/**
		 * <p>[概 要] </p>
		 * アクションで予約されたモデル群をインスタンス化して連続実行します.
		 * 
		 * <p>[詳 細] </p>
		 * BaseAction#reserveModels(Array)で予約されたBaseModel実装モデル群を実行します。<br>
		 * 登録モデルが複数有る場合、前回モデルの実行終了を待ってから次回モデルが実行されます。<p>
		 * 
		 * モデル実行直前にBaseAction#nextModel(int, ModelProcessEvent, Model)がコールバックされます。<br>
		 * 実行モデルへのパラメータ設定を上記メソッドで行うことが出来ます。<p>
		 * 
		 * モデル実行後、成功時はBaseAction#successForward(int, Model, Object)が、
		 * 失敗時はBaseAction#failureForward(int, Model, Exception)がコールバックされます。<br>
		 * モデルの実行結果は上記メソッドで取得することが出来ます。
		 * 
		 * <p>[備 考] </p>
		 * BaseAction#isRunModelsAndNoWait()
		 * がfalseの場合、同期モードで実行されます。<br>
		 * 同メソッドはデフォルトでfalseを返却します。
		 * <p/>
		 * 
		 * @param modelClasses BaseAction#reserveModels(Array) で予約されたモデルクラス群
		 * @param parameterMapping MVC各レイヤを伝播するパラメータ
		 * @param executeIndex モデル実行順序インデックス
		 * @param modelProcessEvent 直前に実行したモデルの処理結果イベントオブジェクト
		 * @see #runModelsAndNoWait()
		 */
		public function runModels(models:Array, mapping:ParameterMapping, index:int=0, modelProcessEvent:ModelProcessEvent=null):void {
			var action:BaseAction = BaseAction(mapping.actionInstance);
			// 実行予約モデルインスタンスが1以上ある場合
			if(models.length > 0) {
				// 実行するモデルクラスを取り出してインスタンス化
				var model:BaseModel = new (models.shift());
				// モデルに対してパラメータマッピングを設定
				model.parameterMapping = mapping;
				model.executeIndex = index;
				model.controller = this;

				// モデルに対して成功時イベントハンドラを設定
				model.addEventListener(ModelProcessEvent.SUCCESS, function(event:ModelProcessEvent):void{
					try{
						var m:BaseModel = BaseModel(event.target);
						var a:BaseAction = BaseAction(m.parameterMapping.actionInstance);
						a.successForward(m.executeIndex, m, event.cause);
						// 実行成功の場合、次のモデルを実行
						// 継続的にSUCCESSイベントを発行するモデルの場合、
						// 初回SUCCESSの時だけ次のモデルを実行
						if(m.successCount == 1) {
							runModels(models, mapping, ++index, event);
						}
					}catch(error:Error) {
						trap(error);
					}
				});

				// モデルに対して失敗時イベントハンドラを設定
				model.addEventListener(ModelProcessEvent.FAILURE, function(event:ModelProcessEvent):void{
					try{
						var m:BaseModel = BaseModel(event.target);
						var a:BaseAction = BaseAction(m.parameterMapping.actionInstance);

						// 全モデル終了を待つモーダル画面を消去
						ComponentManager.getInstance().removePopUp("modelWaitingDialog");

						// アクション処理に失敗フォワード処理を委譲
						var faultEvent:Event = a.failureForward(m.executeIndex, m, event.cause);
						if(faultEvent != null) {
							trap(faultEvent);
						}
						// invoke失敗時のハンドラが予約されている場合はコール
						if(m.parameterMapping.invokeFailureHandler != null) {
							m.parameterMapping.invokeFailureHandler.call();
						}
					}catch(error:Error) {
						trap(error);
					}
				});

				// モデルに対して完了時イベントハンドラを設定
				model.addEventListener(ModelProcessEvent.FINISHED, function(event:ModelProcessEvent):void{
					try{
						var m:BaseModel = BaseModel(event.target);
						var a:BaseAction = BaseAction(m.parameterMapping.actionInstance);
						
						m.parameterMapping.modelFinishedNum++;
						if(m.parameterMapping.modelReservedNum == m.parameterMapping.modelFinishedNum) {
							// 全モデル終了を待つモーダル画面を消去
							ComponentManager.getInstance().removePopUp("modelWaitingDialog");
							// 全モデル終了タイミングフックメソッドをテンプレートコール
							modelsDone(m.parameterMapping);
							// アクションの全モデル終了通知メソッドをテンプレートコール
							a.complete();
							// invoke完了時のハンドラが予約されている場合はコール
							if(m.parameterMapping.invokeCompleteHandler != null) {
								m.parameterMapping.invokeCompleteHandler.call();
							}
							// モデルの完了カウントをクリア
							m.parameterMapping.modelFinishedNum = 0;
						}
					}catch(error:Error) {
						trap(error);
					}
				});

				// BaseAction#nextModelをコールバックしてモデル実行前タイミングをフック可能にする
				var isProceed:Boolean = action.nextModel(index, modelProcessEvent, model);
				// BaseAction#nextModelの返却値がtrueの場合
				if(!isProceed){
					return;
				}
				// モデルのスキップフラグがfalseであればモデルを実行
				if (!model.skip) {
					// モデル実行
					model.run();
				}else{
					runModels(models, mapping, ++index, null);
				}
			}else{
				// BaseAction#nextModelをコールバックして最終モデルの実行結果をハンドリング可能にする
				action.nextModel(index, modelProcessEvent, null);
			}
		}

		/**
		 * <p>[概 要] </p>
		 * アクションで予約されたモデル群をインスタンス化して連続実行します.
		 * 
		 * <p>[詳 細] </p>
		 * BaseAction#reserveModels(Array)で予約されたBaseModel実装モデル群を実行します。<br>
		 * 登録モデルが複数有る場合、前回モデルの実行終了を待たずに次回モデルが実行されます。<br>
		 * 
		 * モデル実行直前にBaseAction#nextModel(int, ModelProcessEvent, Model)がコールバックされます。<br>
		 * 実行モデルへのパラメータ設定を上記メソッドで行うことが出来ます。<br>
		 * 非同期モードでモデル実行した場合は前回モデルの結果を待たずに次回モデルを実行する為、
		 * 第二引数prev:ModelProcessEventが常時nullになります。<br>
		 * 前回モデルの結果を判断して、次回モデルのパラメータ設定をすることは出来ません。<p>
		 * 
		 * モデル実行後、成功時はBaseAction#successForward(int, Model, Object)が、
		 * 失敗時はBaseAction#failureForward(int, Model, Exception)がコールバックされます。<br>
		 * モデルの実行結果は上記メソッドで取得することが出来ます。
		 * 
		 * <p>[備 考] </p>
		 * BaseAction#isRunModelsAndNoWait()
		 * がtrueの場合、非同期モードで実行されます。<br>
		 * 非同期モードで実行する場合は、BaseAction#isRunModelsAndNoWait()を
		 * オーバーライドしてtrueを返却して下さい。
		 * <p/>
		 * 
		 * @example
		 * <listing version="3.0">
		     override protected function isRunModelsAndNoWait():Boolean {
		        return true;
		     }
		 * </listing>
		 * 
		 * @param modelClasses BaseAction#reserveModels(Array) で予約されたモデルクラス群
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @see #runModels()
		 */
		public function runModelsAndNoWait(models:Array, mapping:ParameterMapping):void {
			var action:BaseAction = BaseAction(mapping.actionInstance);
			// 実行モデルインデックス
			var index:int = 0;
			
			while (true) {
				if (models.length == 0) {
					break;
				}
				// 実行するモデルクラスを取り出してインスタンス化
				var model:BaseModel = new (models.shift());
				// モデルに対してパラメータマッピングを設定
				model.parameterMapping = mapping;
				model.executeIndex = index;
				model.controller = this;
				
				// モデルに対して成功時イベントハンドラを設定
				model.addEventListener(ModelProcessEvent.SUCCESS, function(event:ModelProcessEvent):void{
					try{
						var m:BaseModel = BaseModel(event.target);
						var a:BaseAction = BaseAction(m.parameterMapping.actionInstance);
						a.successForward(m.executeIndex, m, event.cause);
					}catch(error:Error) {
						trap(error);
					}
				});
				
				// モデルに対して失敗時イベントハンドラを設定
				model.addEventListener(ModelProcessEvent.FAILURE, function(event:ModelProcessEvent):void{
					try{
						var m:BaseModel = BaseModel(event.target);
						var a:BaseAction = BaseAction(m.parameterMapping.actionInstance);
						
						// 全モデル終了を待つモーダル画面を消去
						ComponentManager.getInstance().removePopUp("modelWaitingDialog");
						
						var faultEvent:Event = a.failureForward(m.executeIndex, m, event.cause);
						if(faultEvent != null) {
							trap(faultEvent);
						}
						// invoke失敗時のハンドラが予約されている場合はコール
						if(m.parameterMapping.invokeFailureHandler != null) {
							m.parameterMapping.invokeFailureHandler.call();
						}
					}catch(error:Error) {
						trap(error);
					}
				});
				
				// モデルに対して完了時イベントハンドラを設定
				model.addEventListener(ModelProcessEvent.FINISHED, function(event:ModelProcessEvent):void{
					try{
						var m:BaseModel = BaseModel(event.target);
						var a:BaseAction = BaseAction(m.parameterMapping.actionInstance);
						
						m.parameterMapping.modelFinishedNum++;
						if(m.parameterMapping.modelReservedNum == m.parameterMapping.modelFinishedNum) {
							// 全モデル終了を待つモーダル画面を消去
							ComponentManager.getInstance().removePopUp("modelWaitingDialog");
							// 全モデル終了タイミングフックメソッドをテンプレートコール
							modelsDone(m.parameterMapping);
							// アクションの全モデル終了通知メソッドをテンプレートコール
							a.complete();
							// invoke完了時のハンドラが予約されている場合はコール
							if(m.parameterMapping.invokeCompleteHandler != null) {
								m.parameterMapping.invokeCompleteHandler.call();
							}
							// モデルの完了カウントをクリア
							m.parameterMapping.modelFinishedNum = 0;
						}
					}catch(error:Error) {
						trap(error);
					}
				});

				// BaseAction#nextModelをコールバックしてモデル実行前タイミングをフック可能にする
				var isProceed:Boolean = action.nextModel(index, null, model);
				// BaseAction#nextModelの返却値がtrueの場合
				if (isProceed) {
					// モデルのスキップフラグがfalseであればモデルを実行
					if (!model.skip) {
						// モデル実行
						model.run();
					}
				}
				index++;
			}
			action.nextModel(index, null, null);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 機能モデルを単体実行します.
		 * 
		 * <p>[詳 細]</p>
		 * BaseAction#reserveModels等で登録した以外のモデルを実行する際に使用します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param model 実行する機能モデルインスタンス
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 */
		public function runModel(model:BaseModel, parameterMapping:ParameterMapping):void {
			model.parameterMapping = parameterMapping;
			model.controller = this;
			model.run();
		}
		
		/**
		 * <p>[概 要]</p>
		 * Controller、Action、Model各レイヤで発生したエラーを最終的に一元ハンドルするメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * MVC各レイヤで発生したエラーが最終的にキャッチされます。<br>
		 * ClientConfigオブジェクトのshowAlertOnErrorプロパティがtrueの場合、
		 * 発生したエラーの情報を画面表示します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param error 発生したエラー、またはエラーイベント
		 */
		public function trap(error:*):void {
			var title:String = "";
			var message:String = ""; 

			if(error is Error) {
				if(error is ICoreError){
					title = ICoreError(error).getId();
					message = ICoreError(error).getMessage();
				}else if(error is Error){
					title = Error(error).errorID as String;
					message = Error(error).message;
				}
				trace(Error(error).getStackTrace());
			}else if(error is FaultEvent) {
				title = FaultEvent(error).fault.faultCode + ":" + FaultEvent(error).fault.faultString;
				message = FaultEvent(error).fault.faultDetail; 
			}else{
				title = "Error";
				message = Messages.getMessage("EFC0000");
			}
			
			if (clientConfig.showAlertOnError) {
				var dialog:ErrorDialog = new ErrorDialog();
				dialog.errorTitle = title;
				dialog.errorMessage = message;
				ComponentManager.getInstance().addPopUp("errorDialog" ,dialog ,null, true ,true);
   			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * MXMLで定義されたコンポーネントと、アクションを紐付けるメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 具象コントローラでこのメソッドをオーバーライドして、コンポーネントのイベント発生時に
		 * 実行するアクションクラスを紐付けます。<br>
		 * eventBinder.addEventBinding("name属性値", "イベントタイプ", BaseAction継承クラス);<BR>
		 * のように紐付け処理を列挙して下さい。
		 * 紐付け情報を保持するEventBinderオブジェクトは、BaseControllerのeventBinderプロパティ
		 * として保持されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @example
		 * name属性値loginを持つコンポーネントがクリックされるとLoginActionが起動する紐付け処理
		 * 
		 * <listing version="3.0">
			eventBinder.addEventBinding("login", MouseEvent.CLICK, Class(LoginAction));
		 * <listing/>
		 * 
		 * @param eventBinder イベント紐付け情報保持オブジェクト
		 * @see jp.co.fujitsu.reffi.client.flex.controller.EventBinder
		 */
		protected function bind(eventBinder:EventBinder):void {
		}

		/**
		 * <p>[概 要] </p>
		 * 各ユーザ定義イベントハンドリングの最後にテンプレートコールされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * デフォルト処理は有りません。
		 * 
		 * <p>[備 考] </p>
		 * このメソッドを具象コントローラでオーバーライドすると、全イベントアクションの
		 * 共通実行後処理を実装出来ます。
		 * <p/>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 */
		protected function handlerFinalize(parameterMapping:ParameterMapping):void {
		}
		
		/**
		 * <p>[概 要] </p>
		 * １アクションに登録されている機能モデル群が全て終了した契機でコールされます.
		 * 
		 * <p>[詳 細] </p>
		 * アクションでの登録モデル群が全て完了した時の共通処理を記述します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 */
		protected function modelsDone(parameterMapping:ParameterMapping):void {
		}
	}
}
