/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.common.nexaweb.dialog;

/**
 * <p>[概 要]</p>
 * メッセージダイアログクラス
 * <p>[詳 細]</p>
 * <p>[備 考]</p>
 * 
 * @author Project Reffi
 */
public class CoreDialog implements CoreDialogIF {

	private String id;
	/** name */
	private String name;
	/** タイトル */
	private String title;
	/** メッセージ */
	private String message;
	/** ダイアログ種別 */
	private String dialogType;
	/** 背景色 */
	private String bgColor;
	/** 背景画像 */
	private String bgImage;
	/** 幅 */
	private Integer width;
	/** 高さ */
	private Integer height;
	/** X位置 */
	private Integer x;
	/** Y位置 */
	private Integer y;
	/** センター配置可否 */
	private Boolean centered;
	/** サイズ変更可否 */
	private Boolean resizable;
	/** リッチテキスト可否 */
	private Boolean richText;
	/** モーダル可否 */
	private Boolean modal;
	/** 最大化可否 */
	private Boolean maximized;
	/** タイトルバー表示可否 */
	private Boolean showTitleBar;
	/** メッセージダイアログフラグ */
	private Boolean messageDialogFlg;

	@Override
	public String getTagXml() {
		StringBuilder sb = new StringBuilder();

		//メッセージダイアログエレメントを作成
		if (messageDialogFlg != null && messageDialogFlg) {
			sb.append("<messageDialog ");
		} else {
			sb.append("<dialog ");
		}

		sb.append("xmlns=\"http://openxal.org/ui/java\"");

		//nameの設定
		if (!isBlank(id)) {
			sb.append("id=\"" + id + "\" ");
		}

		//nameの設定
		if (!isBlank(name)) {
			sb.append("name=\"" + name + "\" ");
		}

		//タイトルの設定
		if (!isBlank(title) && (showTitleBar != null && showTitleBar == true)) {
			sb.append("title=\"" + title + "\" ");
		}
		//メッセージの設定
		if (!isBlank(message)) {
			sb.append("message=\"" + message + "\" ");
		}

		//ダイアログタイプが指定されていればappend
		if (!isBlank(dialogType)) {
			sb.append("type=\"" + dialogType + "\" ");
		}

		//背景色の設定
		if (!isBlank(bgColor)) {
			sb.append("backgroundColor=\"" + bgColor + "\" ");
		}

		//背景画像の設定
		if (!isBlank(bgImage)) {
			sb.append("backgroundImage=\"" + bgImage + "\" ");
		}

		//横の設定
		if (width != null) {
			sb.append("width=\"" + width + "\" ");
		}

		//縦の設定
		if (height != null) {
			sb.append("height=\"" + height + "\" ");
		}

		//xの設定
		if (x != null) {
			sb.append("x=\"" + x + "\" ");
		}

		//yの設定
		if (y != null) {
			sb.append("y=\"" + y + "\" ");
		}

		//センター表示の設定
		if (centered != null) {
			sb.append("centered=\"" + centered + "\" ");
		}

		if (resizable != null) {
			sb.append("resizable=\"" + resizable + "\" ");
		}

		if (richText != null) {
			sb.append("richText=\"" + richText + "\" ");
		}

		if (modal != null) {
			sb.append("modal=\"" + modal + "\" ");
		}

		if (maximized != null) {
			sb.append("maximized=\"" + maximized + "\" ");
		}

		if (showTitleBar != null) {
			sb.append("showTitleBar=\"" + showTitleBar + "\" ");
		}

		sb.append("/>");

		return sb.toString();

	}

	private boolean isBlank(String str) {

		boolean result = true;

		if (str != null && str.length() > 0) {
			result = false;
		}

		return result;
	}

	/**
	 * <p>[概 要]</p>
	 * メッセージダイアログコンストラクタ
	 * <p>[詳 細]</p>
	 * 指定したビルダでメッセージダイアログを生成します。
	 * <p>[備 考]</p>
	 * 
	 * @param builder メッセージビルダ
	 */
	public CoreDialog(Builder builder) {
		this.id = builder.id;
		this.name = builder.name;
		this.title = builder.title;
		this.message = builder.message;
		this.dialogType = builder.dialogType;
		this.bgColor = builder.bgColor;
		this.bgImage = builder.bgImage;
		this.width = builder.width;
		this.height = builder.height;
		this.x = builder.x;
		this.y = builder.y;
		this.centered = builder.centered;
		this.resizable = builder.resizable;
		this.richText = builder.richText;
		this.modal = builder.modal;
		this.maximized = builder.maximized;
		this.showTitleBar = builder.showTitleBar;
		this.messageDialogFlg = builder.messageDialogFlg;
	}

	/**
	 * <p>[概 要]</p>
	 * メッセージダイアログタイプクラス
	 * <p>[詳 細]</p>
	 * <p>[備 考]</p>
	 * 
	 * @author Project Reffi
	 */
	public enum MessageDialogType {
		/** メッセージ種別定数 */
		ERROR("error"), QUESTION("question"), INFO("info"), WARNING("warning"), PLAIN(
				"plain");
		/** メッセージ種別 */
		private String type;

		private MessageDialogType(String str) {
			this.type = str;
		}

		/**
		 * <p>[概 要]</p>
		 * メッセージ種別取得
		 * <p>[詳 細]</p>
		 * 設定されているメッセージ種別を取得します。
		 * <p>[備 考]</p>
		 * 
		 * @return メッセージ種別
		 */
		public String getType() {
			return this.type;
		}
	}

	public static class Builder {
		//必須パラメータ
		private String title;
		private String message;

		//カスタムパラメータ
		private String id = null;
		private String name = null;
		private String dialogType = MessageDialogType.PLAIN.getType();
		private String bgColor = null;
		private String bgImage = null;
		private Integer width = null;
		private Integer height = null;
		private Integer x = null;
		private Integer y = null;
		private Boolean centered = true;
		private Boolean resizable = null;
		private Boolean richText = null;
		private Boolean modal = true;
		private Boolean maximized = null;
		private Boolean showTitleBar = true;
		private Boolean messageDialogFlg = true;

		public Builder id(String id) {
			this.id = id;
			return this;
		}

		public Builder name(String name) {
			this.name = name;
			return this;
		}

		public Builder dialogType(String dialogType) {
			this.dialogType = dialogType;
			return this;
		}

		public Builder bgColor(String bgColor) {
			this.bgColor = bgColor;
			return this;
		}

		public Builder bgImage(String bgImage) {
			this.bgImage = bgImage;
			return this;
		}

		public Builder width(Integer width) {
			this.width = width;
			return this;
		}

		public Builder height(Integer height) {
			this.height = height;
			return this;
		}

		public Builder x(Integer x) {
			this.x = x;
			return this;
		}

		public Builder y(Integer y) {
			this.y = y;
			return this;
		}

		public Builder centered(Boolean centered) {
			this.centered = centered;
			return this;
		}

		public Builder resizable(Boolean resizable) {
			this.resizable = resizable;
			return this;
		}

		public Builder richText(Boolean richText) {
			this.richText = richText;
			return this;
		}

		public Builder modal(Boolean modal) {
			this.modal = modal;
			return this;
		}

		public Builder maximized(Boolean maximized) {
			this.maximized = maximized;
			return this;
		}

		public Builder showTitleBar(Boolean showTitleBar) {
			this.showTitleBar = showTitleBar;
			return this;
		}

		public Builder messageDialogFlg(Boolean messageDialogFlg) {
			this.messageDialogFlg = messageDialogFlg;
			return this;
		}

		public Builder(String title, String message) {
			this.title = title;
			this.message = escape(message);
		}

		public CoreDialog build() {
			return new CoreDialog(this);
		}

		private String escape(String s) {
			s = s.replace("<", "&lt;");
			s = s.replace(">", "&gt;");
			s = s.replace("\"", "&quot;");
			return s;
		}
	}
}
