/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.messaging;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

/**
 * <p>[概 要]</p>
 * メッセージ送信マネージャ。
 * 
 * <p>[詳 細]</p>
 * 複数のメッセージ送信クラスを管理します。
 * <br />
 * 登録するメッセージ送信クラスは{@link AbstractMessageSender}を継承したクラスである必要があります。
 * <br />
 * また、登録する際には名前を定義する必要があり、既に同一の名前で登録されている場合は、あとからの送信
 * クラスは登録されません。
 * 
 * <p>[備 考]</p>
 *
 * @author Project Reffi
 */
public enum SenderManager {
	/** インスタンス */
	instance;
	/** 遅延開始時間 */
	public static final int DELAY_START = 0;
	/** 標準実行時間 */
	public static final int DEFAULT_TIME = 1000;
	/** 登録プロデューサリスト */
	private Map<String, Timer> _timerMap = new HashMap<String, Timer>();
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ送信登録
	 * 
	 * <p>[詳 細]</p>
	 * 実装したメッセージ送信を登録します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param name メッセージ送信名
	 * @param messageSender 実装されたメッセージ送信
	 */
	synchronized public void addMessageSender(final String name, final AbstractMessageSender messageSender) {
		addMessageSender(name, messageSender, SenderManager.DEFAULT_TIME);
	}

	/**
	 * <p>[概 要]</p>
	 * メッセージ送信登録
	 * 
	 * <p>[詳 細]</p>
	 * 実装したメッセージ送信を登録します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param name メッセージ送信名
	 * @param messageSender 実装されたメッセージ送信
	 * @param time メッセージ送信実行時間(ミリ秒にて指定)
	 */
	synchronized public void addMessageSender(final String name, final AbstractMessageSender messageSender, int time) {
		if (this._timerMap.containsKey(name)) {
			return;
		}
		Timer _timer = new Timer(true);
		_timer.schedule(new TimerTask() {
			public void run() {
				messageSender.runSender();
			}
		}, SenderManager.DELAY_START, time);
		this._timerMap.put(name, _timer);
	}
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ送信実行解除
	 * <p>[詳 細]</p>
	 * 指定した名前の登録されているメッセージ送信実行を解除します。
	 * <p>[備 考]</p>
	 * 
	 * @param name メッセージ送信名
	 */
	synchronized public void cancelMessageSender(final String name) {
		if (this._timerMap.containsKey(name)) {
			this._timerMap.get(name).cancel();
			this._timerMap.remove(name);
		}
	}

	/**
	 * <p>[概 要]</p>
	 * メッセージ送信登録確認
	 * <p>[詳 細]</p>
	 * 指定した名前のメッセージ送信が登録されているか確認をします。
	 * <p>[備 考]</p>
	 * 
	 * @param name メッセージ送信名
	 */
	synchronized public boolean hasMessageSender(final String name) {
		return this._timerMap.containsKey(name);
	}

	/**
	 * <p>[概 要]</p>
	 * 全メッセージ送信実行解除
	 * <p>[詳 細]</p>
	 * 登録されている全てのメッセージ送信実行を解除します。
	 * <p>[備 考]</p>
	 */
	synchronized public void cancelAllMessageSender() {
		for (Iterator<Timer> itr = this._timerMap.values().iterator(); itr.hasNext(); ) {
			itr.next().cancel();
		}
		this._timerMap.clear();
	}
}
