/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.manager {
	import flash.events.TimerEvent;
	import flash.utils.Dictionary;
	import flash.utils.Timer;
	
	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.error.CoreLogicError;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.model.timer.TimerProcessCore;
	
	/**
	 * <p>[概 要]</p>
	 * TimerとTimerProcessCoreをIDで管理するマネージャクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * インターバル管理Timerと、タイマーイベントハンドリングを行うTimerProcessCoreを
	 * 任意のIDで対応管理します。<br>
	 * 
	 * <p>[備 考]</p>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */ 
	public class TimerProcessCoreManager {
		
		// TimerとTimerProcessCoreの対応マッププロパティ
		private var _timers:Dictionary = new Dictionary();
		
		/**
		 * <p>[概 要]</p>
		 * TimerとTimerProcessCoreの対応マッププロパティです.
		 *  
		 * <p>[詳 細]</p>
		 * 以下のような構造を持つプロパティです。
		 * 
		 * <listing version="3.0">
		 * timers:Dictionary
		 *     "timerId1" = timerMap:Dictionary
		 *                     "timer" = Timer
		 *                     "core" = TimerCore
		 *     "timerId2" = timerMap:Dictionary
		 *                     "timer" = Timer
		 *                     "core" = TimerCore
		 * </listing>
		 * 
		 * <p>[備 考]</p>
		 * 既に実行中のタイマーと同じtimerIdを付けることは出来ません。
		 * <p/>
		 * 
		 */
		public function get timers():Dictionary {
			return this._timers;
		}
		public function set timers(timers:Dictionary):void {
			this._timers = timers;
		}
		
		/**
		 * <p>[概 要]</p>
		 * コンストラクタです.
		 * 
		 * <p>[詳 細]</p>
		 * 本クラスはシングルトンクラスのため外部からの呼び出しは行えません。
		 * 
		 * <p>[備 考]</p>
		 * TimerProcessCoreManagerのインスタンスを取得する場合は、getInstance()を使用してください。
		 * 
		 * <p>
		 * @param blocker シングルトン生成用ブロックインスタンス
		 * @see #getInstance()
		 */
		public function TimerProcessCoreManager(blocker:TimerBlocker) {
			if (blocker == null) {
				throw new Error("This class is singleton.");
			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * タイマーマネージャインスタンスを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * シングルトンタイマーマネージャインスタンスを取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return タイマーマネージャインスタンス
		 */
		public static function getInstance():TimerProcessCoreManager {
			return TimerBlocker.instance;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数timerIdが現在登録されているか確認します.
		 * 
		 * <p>[詳 細]</p>
		 * timersタイマー管理プロパティから、引数timerIdが登録されているかを調べます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param timerId 登録されているか調べる識別子
		 * @return 登録されている場合はtrue、それ以外はfalse
		 */
		public function isRegist(timerId:String):Boolean {
			return this.timers.hasOwnProperty(timerId);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数timerIdが現在実行中であるか確認します.
		 * 
		 * <p>[詳 細]</p>
		 * timersタイマー管理プロパティから、引数timerIdに対応するタイマーを取得し、
		 * Timer#runningプロパティを返却します。<br>
		 * timerIdに対応するTimerが登録されていない場合はfalseを返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param timerId 登録されているか調べる識別子
		 * @return 登録されている場合はtrue、それ以外はfalse
		 */
		public function isRunning(timerId:String):Boolean {
			var ret:Boolean = false;
			
			if(isRegist(timerId)) {
				var timerMap:Dictionary = this.timers[timerId];
				var timer:Timer = timerMap["timer"];
				ret = timer.running;
			}
			
			return ret;
		}
		
		/**
		 * <p>[概 要]</p>
		 * タイマーを開始します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 開始するタイマーの識別子
		 * @param timerCore 開始するTimerProcessCore
		 */
		public function start(timerId:String, timerCore:TimerProcessCore):void {
			
			// タイマー識別子が指定されていない場合
			if(timerId == "" || timerId == null){
				throw new CoreLogicError("EFC0014");
			}

			// 既に同じタイマー識別子が登録されている場合			
			if(this.timers.hasOwnProperty(timerId)){
				throw new CoreLogicError("EFC0016");
			}
			
			// TimerProcessCoreの設定情報を元に実行タイマーインスタンス生成
			try{
				var timer:Timer = new Timer(timerCore.interval, timerCore.repeat);
			}catch(e:Error) {
				throw new CoreLogicError("EFC0015");
			}
			// repeat数分インターバル実行した場合はstop
			timer.addEventListener(TimerEvent.TIMER_COMPLETE, function():void {
				stop(timerCore.timerId);
			});
			// TimerProcessCoreにアクションが登録されている場合、interval処理としてActionを実行
			if(timerCore.intervalAction) {
			    // BaseController#invoke()を呼ぶとParameterMapping#actionInstanceが上書きされてしまう為、cloneする
			    var clonedParameterMapping:ParameterMapping = timerCore.parameterMapping.clone();

				timer.addEventListener(TimerEvent.TIMER, function(event:TimerEvent):void {
					timerCore.controller.invoke(timerCore.intervalAction, clonedParameterMapping,
					function():void {
						var modelSuccessEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
						timerCore.dispatchModelSuccess(modelSuccessEvent);
					}, 
					function():void {
						var modelFailureEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
						timerCore.dispatchModelFailure(modelFailureEvent);
					});
				});
			} else {
				timer.addEventListener(TimerEvent.TIMER, timerCore.resultHandler);
			}

			// TimerとTimerProcessCoreの対応を保存
			var timerMap:Dictionary = new Dictionary();
			timerMap["timer"] = timer;
			timerMap["core"] = timerCore;
			this.timers[timerId] = timerMap;

			// tick開始			
			timer.start();
		}
		
		/**
		 * <p>[概 要]</p>
		 * タイマーを終了します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 終了するタイマーの識別子
		 */
		public function stop(timerId:String):void {
			
			if (!this.timers.hasOwnProperty(timerId)) {
				return;
			}
			
			var timerMap:Dictionary = this.timers[timerId];
			var timer:Timer = timerMap["timer"] as Timer;
			var timerCore:TimerProcessCore = timerMap["core"] as TimerProcessCore;
			
			timer.removeEventListener(TimerEvent.TIMER, timerCore.resultHandler);
			var modelProcessEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
			timerCore.dispatchModelFinished(modelProcessEvent);

			timer.stop();
			delete this.timers[timerId];
		}
	}
}

import jp.co.fujitsu.reffi.client.flex.manager.TimerProcessCoreManager;

class TimerBlocker {
	public static var instance:TimerProcessCoreManager = new TimerProcessCoreManager(new TimerBlocker());
}
