/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.action {
	
	import flash.display.DisplayObject;
	import flash.display.DisplayObjectContainer;
	import flash.utils.Dictionary;
	
	import jp.co.fujitsu.reffi.client.flex.controller.BaseController;
	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.manager.ComponentManager;
	import jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil;
	import jp.co.fujitsu.reffi.client.flex.validator.ValidateError;
	import jp.co.fujitsu.reffi.client.flex.validator.ValidateErrors;
	
	import mx.core.Application;
	import mx.core.IFlexDisplayObject;
	import mx.events.ValidationResultEvent;
	import mx.validators.ValidationResult;
	import mx.validators.Validator;

	/**
	 * <p>[概 要] </p>
	 * 全アクションクラスの抽象基底アクションクラスです.
	 * 
	 * <p>[詳 細] </p>
	 * アクションクラスとして動作する為の必要最低限機能と、
	 * ウィンドウ操作やエレメント取得の為のAPIを提供します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * 
	 * @author Project Reffi 
	 */
	public class AbstractAction implements IAction {

		// このアクションを起動したコントローラオブジェクト
		private var _controller:BaseController;
		
		// このアクション（イベント）を起こしたUIComponentが所属するdocumentオブジェクト
		private var _document:Object;

		// MVC各レイヤを伝播するパラメータオブジェクト
		private var _parameterMapping:ParameterMapping;
		
		
		/**
		 * <p>[概 要]</p>
		 * このアクションを起動したコントローラオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * このアクションのインスタンスを生成して、アクションと機能モデルの
		 * やりとりを制御するコントローラです。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get controller():BaseController{
			return _controller;	
		}
		public function set controller(controller:BaseController):void{
			this._controller = controller;
		} 

		/**
		 * <p>[概 要]</p>
		 * このアクション（イベント）を起こしたUIComponentが所属するdocumentオブジェクトを設定します.
		 * 
		 * <p>[詳 細]</p>
		 * 以下のようなApplication構造の場合、
		 * <listing version="3.0">
		 * &lt;Application&gt;
		 *     &lt;MXMLコンポーネント name="customComponent"&gt;
		 *         &lt;Button name="btnTest"&gt;
		 * </listing>
		 * btnTestボタンから発生したアクション内に保持されるdocumentは
		 * customComponentになります。
		 * 
		 * <p>[備 考]</p>
		 * documentオブジェクトは、Flex アプリケーション、MXML コンポーネント、
		 * または ActionScript コンポーネント階層の最上位オブジェクトを表します。
		 * </p>
		 * 
		 */
		public function get document():Object{
			return _document;
		}
		public function set document(document:Object):void{
			this._document = document;
		} 

		/**
		 * <p>[概 要]</p>
		 *  MVC各レイヤを伝播するパラメータオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * 1イベントハンドリングに付き、1ParameterMappingオブジェクト
		 * が生成されます。<br>
		 * アクションが並列的に動作する場合、それぞれのアクションは
		 * ユニークなParameterMappingオブジェクトを保持します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get parameterMapping():ParameterMapping {
			return this._parameterMapping;
		}
		public function set parameterMapping(parameterMapping:ParameterMapping):void {
			this._parameterMapping = parameterMapping;
		}
		
		/**
		 * <p>[概 要]</p>
		 * コントローラにコールされるアクションの主幹メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 以下を行います。
		 * <ul>
		 *   <li>引数parameterMappingをプロパティに設定</li>
		 *   <li>イベント発生元オブジェクトからdocumentを取得してプロパティに設定</li>
		 * </ul>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 */
		public function run(parameterMapping:ParameterMapping):ParameterMapping {
        	// パラメータマッピングをフィールドに保存
        	this.parameterMapping = parameterMapping;
        	
			// このアクションを起動したコンポーネントが所属するdocumentオブジェクトをフィールドに設定
			this.document = CompositionSearchUtil.searchDocument(getEventSourceObject());

			return parameterMapping;
		}
		
		/**
		 * <p>[概 要] </p>
		 * バリデーションを行うメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * 引数validatorsの数分、バリデーションを行います。<br>
		 * バリデーションエラーが発生した場合、エラー情報がValidateErrorオブジェクトに設定され、
		 * 戻り値であるValidateErrorsオブジェクトに追加されます。
		 * <p/>
		 * 
		 * FlexのValidatorは一度source設定してvalidateメソッドを実行すると、
		 * 設定されたValidatorはvalueCommitイベント毎に動作するようになります。<br>
		 * valueCommitイベント毎では無く、アクション実行毎にバリデーションを行いたい場合は
		 * このメソッドが有効です。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @example Validator#validateメソッドと、AbstractAction#validateメソッドの挙動
		 * <listing version="3.0">
            public class AddNumberAction extends FlexibleAction {
                
                override public function execute(parameterMapping:ParameterMapping):void {
                    
                    var tiNum1:TextInput = TextInput(getComponentByName("panel.tiNum1"));
                    var tiNum2:TextInput = TextInput(getComponentByName("panel.tiNum2"));
        
                    // 実行後、アクション実行時以外（フォーカスアウトイベント等）でもバリデーションが実行されます
                    var tiNum1Validator:NumberValidator = new NumberValidator();
                    tiNum1Validator.source = tiNum1;
                    tiNum1Validator.property = "text";
                    tiNum1Validator.domain = "int";
                    tiNum1Validator.required = true;
                    tiNum1Validator.validate();
         
                    // アクション実行時のみバリデーションが実行されます
                    var tiNum2Validator:NumberValidator = new NumberValidator();
                    tiNum2Validator.source = tiNum2;
                    tiNum2Validator.property = "text";
                    tiNum2Validator.domain = "int";
                    tiNum2Validator.required = true;
                    var validators:Array = new Array();            
                    validators.push(tiNum2Validator);
                    validate(validators);
                }
            }
		 * </listing>
		 * <p/>
		 *  
		 * @param validators validatorsメソッドで設定されたバリデータオブジェクト群
		 * @return バリデーションエラー保持リストオブジェクト
		 */ 
		protected function validate(validators:Array):ValidateErrors{

			// 返却用エラー集約オブジェクト生成
			var errors:ValidateErrors = new ValidateErrors();

			// validatorsメソッド実装でバリデータが登録されている場合
			if(validators.length != 0){
				
				// 登録されているバリデータ数分バリデート実行
				for each(var validator:Validator in validators) {

					// 既に対象コンポーネントがエラー状態にある場合は実行しない
					if(errors.contains(validator.source)) {
						// バリデータの対象コンポーネントへの関連付けを外す
						validator.source = null;
						continue;						
					}

					// バリデート実行
					var validationResultEvent:ValidationResultEvent = validator.validate();
					
					// バリデーションエラー発生
					if (validationResultEvent.type == ValidationResultEvent.INVALID) {
						// エラー発生元オブジェクト取得
						var errorSourceObject:Object = validator.source;
						var errorSourceName:String = errorSourceObject.name;
						var results:Array = validationResultEvent.results;
						var message:String = ValidationResult(results[0]).errorMessage;

						// エラー情報オブジェクトを構築して返却オブジェクトに集約						 
						var error:ValidateError = 
							new ValidateError(errorSourceName, message, errorSourceObject, results);
						errors.addError(error);
					}
					// バリデータの対象コンポーネントへの関連付けを外す
					validator.source = null;
				}
			}

			return errors;		
		}

		
		/**
		 * <p>[概 要]</p>
		 * 第一引数namePatternをname属性として持つ画面コンポーネントを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 画面が複数のDocumentで構成されている場合、以下の順番で検索を行います。
		 * <ol>
		 *   <li>自Document（このアクションを起こしたコンポーネントが所属するDocument）</li>
		 *   <li>SystemManagerに所属する自Document以外のDocument</li>
		 * </ol>
		 * 
		 * <p>[備 考]</p>
		 * @example 「taLog」という名前のコンポーネントを取得する
		 * <listing version="3.0">
		 * var taLog:TextArea = getComponentByName("taLog") as TextArea;
		 * </listing>
		 * 
		 * @param namePattern 取得するコンポーネントのname属性値
		 * @param searchRange コンポーネントを探す範囲（デフォルト：CompositionSearchUtil.ALL_RANGE）
		 * @return namePatternと合致したname属性を持つコンポーネント
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentByName(namePattern:String, searchRange:int=0x03):DisplayObject {
			return CompositionSearchUtil.searchComponentByName(this.document, namePattern, searchRange);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 第一引数namePatternをname属性として持つ画面コンポーネントを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 自Document（このアクションを起こしたコンポーネントが所属するDocument）を含め、
		 * 順に親のDocumentを辿ってコンポーネントの検索を行います。
		 * 
		 * <p>[備 考]</p>
		 * @example 「taLog」という名前のコンポーネントを取得する
		 * <listing version="3.0">
		 * var taLog:TextArea = getComponentByNameParentTrace("taLog") as TextArea;
		 * </listing>
		 * 
		 * @param namePattern 取得するコンポーネントのname属性値
		 * @return namePatternと合致したname属性を持つコンポーネント
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentByNameParentTrace(namePattern:String):DisplayObject {
			return CompositionSearchUtil.searchComponentByNameParentTrace(this.document, namePattern);
		}

		/**
		 * <p>[概 要]</p>
		 * 第一引数namePatternをname属性として持つ画面コンポーネント配列を取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 画面が複数のDocumentで構成されている場合、以下の順番で検索を行います。
		 * <ol>
		 *   <li>自Document（このアクションを起こしたコンポーネントが所属するDocument）</li>
		 *   <li>SystemManagerに所属する自Document以外のDocument</li>
		 * </ol>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @example AVM上のname属性の付いた全DisplayObjectを取得する
		 * <listing version="3.0">
		 * var allNamedComponents:Array = getComponentsByName(".&#42;");
		 * </listing>
		 * 
		 * @example 「btn」という文字列を含むname属性値が付与されている全DisplayObjectを取得する
		 * <listing version="3.0">
		 * var allNamedButtons:Array = getComponentsByName(".&#42;btn.&#42;");
		 * </listing>
		 * 
		 * @param namePattern 取得するコンポーネントのname属性値
		 * @param searchRange コンポーネントを探す範囲（デフォルト：CompositionSearchUtil.ALL_RANGE）
		 * @return namePatternと合致したname属性を持つコンポーネント配列 
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentsByName(namePattern:String, searchRange:int=0x03):Array {
			return CompositionSearchUtil.searchComponentsByName(this.document, namePattern, searchRange);
		}

		/**
		 * <p>[概 要]</p>
		 * 第一引数containerから、第二引数namePatternをname属性として持つ画面コンポーネントを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * containerの子要素として存在する、namePatternをname属性として持つコンポーネントを取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * getComponentByName、getComponentsByNameは自分のDocumentにコンポーネントが見つからなかった場合
		 * 別Documentを検索しますが、getComponentByNameFromは指定されたコンテナのみ検索対象にします。
		 * <p/>
		 * 
		 * @param container コンポーネント検索対象コンテナオブジェクト
		 * @param namePattern 取得するコンポーネントのname属性値
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentByNameFrom(container:DisplayObjectContainer, namePattern:String):DisplayObject {
			return CompositionSearchUtil.searchComponentByName(container, namePattern, CompositionSearchUtil.DOCUMENT_RANGE); 
		}
		
		/**
		 * <p>[概 要]</p>
		 * 第一引数containerから、第二引数namePatternをname属性として持つ画面コンポーネント配列を取得します.
		 * 
		 * <p>[詳 細]</p>
		 * containerの子要素として存在する、namePatternをname属性として持つコンポーネント群を取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * getComponentByName、getComponentsByNameは自分のDocumentにコンポーネントが見つからなかった場合
		 * 別Documentを検索しますが、getComponentByNameFromは指定されたコンテナのみ検索対象にします。
		 *
		 * @param container コンポーネント検索対象コンテナオブジェクト
		 * @param namePattern 取得するコンポーネントのname属性値
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentsByNameFrom(container:DisplayObjectContainer, namePattern:String):Array {
			return CompositionSearchUtil.searchComponentsByName(container, namePattern, CompositionSearchUtil.DOCUMENT_RANGE); 
		}

		
		/**
		 * <p>[概 要]</p>
		 * 第一引数idをid属性として持つ画面コンポーネントを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 画面が複数のDocumentで構成されている場合、以下の順番で検索を行います。
		 * <ol>
		 *   <li>自Document（このアクションを起こしたコンポーネントが所属するDocument）</li>
		 *   <li>SystemManagerに所属する自Document以外のDocument</li>
		 * </ol>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 取得するコンポーネントのid属性値
		 * @param searchRange コンポーネントを探す範囲（デフォルト：CompositionSearchUtil.ALL_RANGE）
		 * @return idと合致したid属性を持つコンポーネント 
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentById(id:String, searchRange:int=0x03):DisplayObject{
			return CompositionSearchUtil.searchComponentById(this.document, id, searchRange);
		}

		/**
		 * <p>[概 要]</p>
		 * 第一引数idをid属性として持つ画面コンポーネント配列を取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 画面が複数のDocumentで構成されている場合、以下の順番で検索を行います。
		 * <ol>
		 *   <li>自Document（このアクションを起こしたコンポーネントが所属するDocument）</li>
		 *   <li>SystemManagerに所属する自Document以外のDocument</li>
		 * </ol>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 取得するコンポーネントのid属性値
		 * @param searchRange コンポーネントを探す範囲（デフォルト：CompositionSearchUtil.ALL_RANGE）
		 * @return idと合致したid属性を持つコンポーネント 
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentsById(id:String, searchRange:int=0x03):Array{
			return CompositionSearchUtil.searchComponentsById(this.document, id, searchRange);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 第一引数containerから、第二引数idをid属性として持つ画面コンポーネントを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * containerのプロパティとして存在する、引数idをid属性値として持つコンポーネントを取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 取得するコンポーネントid属性値
		 * @return idを基に取得した画面コンポーネント
		 * @see jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil
		 */
		protected function getComponentByIdFrom(container:DisplayObjectContainer, id:String):DisplayObject {
			return CompositionSearchUtil.searchComponentById(container, id, CompositionSearchUtil.DOCUMENT_RANGE);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数idをid属性として持つオブジェクトを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 自Documentから引数idをid属性値として持つオブジェクトを取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 取得するコンポーネントid属性値
		 * @return idを基に取得したオブジェクト
		 */
		protected function getObjectById(id:String):Object{
			var ret:Object = null;

			if(this.document.hasOwnProperty(id)){
				ret = this.document[id];
			}
			
			return ret;
		}
		
		/**
		 * <p>[概 要]</p>
		 * nSWF間で共有可能な恒久領域から、引数keyに対応する値を取得します.
		 * 
		 * <p>[詳 細]</p>
		 * クライアントサイドの恒久メモリ領域を取得します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param key nSWF間で共有可能な恒久領域のキー 
		 * @return 引数keyに対応する値
		 */
		protected function getPermanent(key:Object):Object{
			var permanent:Dictionary = controller.permanent;
			
			return permanent[key];
		}

		/**
		 * <p>[概 要]</p>
		 * nSWF間で共有可能な恒久領域に、引数keyで値valueを設定します.
		 * 
		 * <p>[詳 細]</p>
		 * クライアントサイドの恒久メモリ領域に値を保存します。<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param key nSWF間で共有可能な恒久領域のキー
		 * @param value 引数keyに対応する値 
		 */
		protected function addPermanent(key:Object, value:Object):void{
			var permanent:Dictionary = controller.permanent;
			permanent[key] = value;
		}
		
		/**
		 * <p>[概 要]</p>
		 * nSWF間で共有可能な恒久領域から、引数keyを削除します.
		 * 
		 * <p>[詳 細]</p>
		 * クライアントサイドの恒久メモリ領域から値を削除します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param key nSWF間で共有可能な恒久領域のキー
		 */
		protected function removePermanent(key:Object):void{
			var permanent:Dictionary = controller.permanent;
			delete permanent[key];
		}
		
		/**
		 * <p>[概 要]</p>
		 * ポップアップ画面を表示します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数nameという名前で、第二引数displayObjectをポップアップさせます。<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param windowName ポップアップ表示する画面の名前
		 * @param displayObject ポップアップする画面オブジェクト
		 * @param modal true : モーダル、false : モードレス
		 * @param centered true : センタリング、false : センタリングなし
		 */
		protected function addPopUp(name:String, 
									displayObject:IFlexDisplayObject, 
									modal:Boolean = false,
									centered:Boolean = false):void
		{
			ComponentManager.getInstance().addPopUp(name, displayObject, null, modal, centered);
		}
		
		/**
		 * <p>[概 要]</p>
		 * シングルトンポップアップ画面を表示します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数nameという名前で、第二引数displayObjectをポップアップさせます。<br> 
		 * 既に同じ名前でポップアップが表示されている場合、表示されていたポップアップは消去されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param name ポップアップ表示する画面の名前
		 * @param displayObject ポップアップする画面オブジェクト
		 * @param modal true : モーダル、false : モードレス
		 * @param centered true : センタリング、false : センタリングなし
		 */
		protected function addSingletonPopUp(name:String, 
										displayObject:IFlexDisplayObject, 
										modal:Boolean = false,
										centered:Boolean = false):void 
		{
			ComponentManager.getInstance().addSingletonPopUp(name, displayObject, null, modal, centered);
		}

		/**
		 * <p>[概 要]</p>
		 * 引数windowNameに対応するポップアップ画面を画面から消去します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数nameとして表示されているポップアップ画面を消去します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param name 画面から消去するポップアップ画面の名前
		 */
		protected function removePopUp(name:String):Boolean{
			return ComponentManager.getInstance().removePopUp(name);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 全てのポップアップ画面を画面から消去します.
		 * 
		 * <p>[詳 細]</p>
		 * 表示中の全ての名前のポップアップ画面を消去します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		protected function removeAllPopUp():Boolean{
			return ComponentManager.getInstance().removeAllPopUp();
		}

		/**
		 * <p>[概 要]</p>
		 * 既に表示されているポップアップ画面インスタンス配列を取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数nameで表示されている全てのポップアップ画面を取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param name 取得するポップアップ画面の名前
		 */
		protected function getPopUp(name:String):Array{
			return ComponentManager.getInstance().getPopUp(name);
		}

		/**
		 * <p>[概 要]</p>
		 * 既に表示されている重複を許可しないポップアップ画面インスタンスを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数nameで表示されているポップアップ画面を取得します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param name 取得するポップアップ画面の名前
		 */
		protected function getSingletonPopUp(name:String):IFlexDisplayObject{
			return ComponentManager.getInstance().getSingletonPopUp(name);
		}
			
		/**
		 * <p>[概 要]</p>
		 * 引数で指定した名前を持つポップアップ画面が既に画面表示されているかどうか調べます.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数nameとして表示されているポップアップ画面が有るか調べます。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param name 表示されているかどうか調べる画面の名前
		 */
		protected function isExistPopUp(name:String):Boolean{				
			return ComponentManager.getInstance().isExistPopUp(name);
		}
	
		/**
		 * <p>[概 要]</p>
		 * このアクションを起動したイベントのtargetを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * イベント発行したオブジェクトを取得します。<br>
		 * ボタン押下時に発生したアクションであれば、Buttonインスタンスが取得出来ます。
		 * 
		 * <p>[備 考]</p>
		 * paramterMapping.eventからtargetプロパティと同値です。
		 * <p/>
		 * 
		 * @return イベントターゲット
		 */
		protected function getEventTarget():Object {
			return parameterMapping.event.target;
		}
		
		/**
		 * <p>[概 要]</p>
		 * このアクションを起動したイベントのcurrentTargetを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * イベントフェーズのカレント対象オブジェクトを取得します。<br>
		 * イベントフェーズの変遷によって内容物が変化します。
		 * 
		 * <p>[備 考]</p>
		 * paramterMapping.eventからcurrentTargetプロパティと同値です。
		 * <p/>
		 * 
		 * @return イベントカレントターゲット
		 */
		protected function getEventCurrentTarget():Object {
			return parameterMapping.event.currentTarget;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 恒久的なイベントカレントターゲットです.
		 * 
		 * <p>[詳 細]</p>
		 * アクション起動時にイベントカレントターゲットが入ります。<br>
		 * このメソッドで取得するオブジェクトは、イベントフェーズによって変化しません。
		 * 
		 * <p>[備 考]</p>
		 *
		 */
		protected function getEventSourceObject():DisplayObject {
			return parameterMapping.eventSourceObject;
		}
		
		/**
		 * <p>[概 要]</p>
		 * このアクションを起動したコンポーネントが所属するApplicationを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * イベントを発生させたオブジェクトのルートApplication要素を取得します。
		 * 
		 * <p>[備 考]</p>
		 *
		 */
		protected function getApplication():Application {
			return CompositionSearchUtil.searchApplication(this.document);
		}
		
		/**
		 * <p>[概 要]</p>
		 * AVM上にロードされているBaseControllerのマッピングを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * コントローラが保持する、コントローラ管理マップを取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return コントローラ管理マップ（BaseController.controllers）
		 */
		protected function getLoadedControllers():Dictionary {
			return BaseController.controllers;
		}
		
		/**
		 * <p>[概 要]</p>
		 * AVM上にロードされているBaseControllerから、引数nameをnameとして持つものを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * コントローラが保持するコントローラ管理マップから、引数nameをnameを持つものを取得、
		 * 返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param name コントローラの名前
		 * @return 引数nameをnameプロパティとして持つコントローラ
		 */
		protected function getLoadedController(name:String):BaseController {
			return BaseController.controllers[name];
		}
	}
}
