/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.web.controller;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.co.fujitsu.reffi.server.invoker.Invoker;
import jp.co.fujitsu.reffi.server.invoker.InvokerFactory;

/**
 * <p>[概 要]</p>
 * サーバコントローラクラス。

 * <p>[詳 細]</p>
 * Servletにて起動するモデル実行コントローラクラスです。
 * <p>
 * 本クラスでは、クライアントのからのリクエストをマップ化し、開発者が実装したモデルを実行する
 * ための{@link Invoker}を呼び出します。
 * </p>
 * <p>
 * レスポンスでは、クライアントからのリクエストにキー「forward.page」でJSPが指定されている
 * 場合、リクエストパラメータ情報とモデル実行結果をJSPへ委譲します。
 * 指定がない場合はモデル実行結果をオブジェクトとして直接クライアントへ送信します。
 * </p>
 * <p>
 * 本クラスは通常のHttpServletなので、web.xmlへ定義するだけで利用可能です。
 * <br />
 * 以下はweb.xmlでの設定例となります。
 * <pre class="samplecode">
 *&lt;servlet&gt;
 *	&lt;servlet-name&gt;WebController&lt;/servlet-name&gt;
 *	&lt;servlet-class&gt;jp.co.fujitsu.reffi.server.web.controller.WebController&lt;/servlet-class&gt;
 *&lt;/servlet&gt;
 *&lt;servlet-mapping&gt;
 *	&lt;servlet-name&gt;WebController&lt;/servlet-name&gt;
 *	&lt;url-pattern&gt;/WebController&lt;/url-pattern&gt;
 *&lt;/servlet-mapping&gt;
 * </pre>
 * </p>
 * 
 * <p>[備 考]</p>
 *
 * @author Project Reffi
 */
public class WebController extends HttpServlet {
	/** シリアルバージョン番号 */
	private static final long serialVersionUID = -3020426299773945196L;
	/** 遷移先ページ */
	public static final String FORWARD_PAGE = "forward.page";
	/** 戻り値定数 */
	public static final String MODEL_RESULT = "model.result";
	/** */
	public static final String MODEL_PARAMS = "model.params";

	@Override
	protected void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		try {
			Map<Object, Object> parameters = createRequestParameterMap(request);
			Invoker invoker = InvokerFactory.getInvoker();
			Object returnObject = invoker.invoke(parameters);
			String forwardPage = (String)parameters.get(FORWARD_PAGE);
			if (forwardPage != null) {
				parameters.put(MODEL_RESULT, returnObject);
				request.setAttribute(MODEL_PARAMS, parameters);
				request.getRequestDispatcher(forwardPage).forward(request, response);
			} else if (returnObject != null) {
				response.setContentType("application/octet-stream");
				ObjectOutputStream objectOutputStream = new ObjectOutputStream(response.getOutputStream());
				objectOutputStream.writeObject(returnObject);
				objectOutputStream.flush();
			}
		} catch (IOException e) {
			throw e;
		} catch (Exception e) {
			throw trap(e);
		}
	}

	@SuppressWarnings("unchecked")
	private Map<Object, Object> createRequestParameterMap(HttpServletRequest request) {
		Map<Object, Object> map = new HashMap<Object, Object>();
		Enumeration<String> names = request.getParameterNames();
		while (names.hasMoreElements()) {
			String name = (String)names.nextElement();
			String[] values = request.getParameterValues(name);
			if (values.length == 0) {
				map.put(name, null);
			} else if (values.length == 1) {
				map.put(name, values[0]);
			} else {
				map.put(name, values);
			}
		}
		Enumeration<String> anames = request.getAttributeNames();
		while (anames.hasMoreElements()) {
			String name = (String)names.nextElement();
			Object attr = request.getAttribute(name);
			map.put(name, attr);
		}
		return map;
	}
	
	protected ServletException trap(Throwable e) {
		if (e instanceof ServletException) {
			return (ServletException)e;
		}
		return new ServletException(e);
	}
}
