<?
/**
 * Filter_Validate
 * 
 * Validate処理を行うFilter。
 * 
 * @package    Samurai
 * @subpackage Filter
 * @copyright  Befool,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 */
class Filter_Validate extends Samurai_Filter
{
    private
        /** @var        array   必須項目保持 */
        $_required = array(),
        /** @var        array   チェックルール */
        $_rules = array(),
        /** @var        array   stopper */
        $_stoppers = array();
    public
        /** @var        object  ValidateManagerコンポーネント */
        $Manager,
        /** @var        object  Requestコンポーネント */
        $Request,
        /** @var        object  ActionChainコンポーネント */
        $ActionChain;
    
    
    /**
     * prefilter.
     * @override
     */
    public function _prefilter()
    {
        parent::_prefilter();
        $this->_createManager();
        $this->_build();
        $this->_validate();
        //エラー格納
        if($this->ErrorList->isExists()){
            $this->ErrorList->setType(Samurai_Config::get("error.validate"));
        }
    }
    
    
    
    
    /**
     * validateルール構築。
     * @access     private
     */
    private function _build()
    {
        foreach($this->getAttributes() as $_key => $_val){
            $_key = trim($_key);
            $_val = trim($_val);
            if($_key=="" || $_val=="") Samurai_Logger::error("Validatorの指定が不正です。-> %s", array("{$_key}/{$_val}"));
            //キーの分割
            list($attribute, $validator, $group, $negative) = $this->_resolveKeys($_key);
            //バリューの分割
            list($stopper, $message, $params) = $this->_resolveValues($_val);
            //必須項目は無条件にストッパーになる
            if($validator=="required"){
                $this->_required[] = $group ? $group : $attribute;
                $stopper = 1;
            }
            //Ruleの組み立て
            $Rule = new Filter_Validate_Rule();
            $Rule->attribute = $attribute;
            $Rule->validator = $validator;
            $Rule->group = $group;
            $Rule->stopper = $stopper;
            $Rule->message = $message;
            $Rule->params = $params;
            $Rule->negative = $negative;
            $this->_rules[] = $Rule;
        }
    }
    
    
    /**
     * キーを分割する。
     * <code>
     *     !attribute.validator:group
     * </code>
     * のパターン
     * @access     private
     * @param      string  $key   キー
     * @return     array   分割された配列
     */
    private function _resolveKeys($key)
    {
        $keys = explode(".", $key);
        if(count($keys)!=2) Samurai_Logger::error("Validatorの指定が不正です。-> %s", array("{$_key}/{$_val}"));
        $attribute = $keys[0];
        if(preg_match("/:/", $keys[1])){
            $keys2 = explode(":", $keys[1]);
            $validator = array_shift($keys2);
            $group = array_shift($keys2);
        } else {
            $validator  = $keys[1];
            $group = "";
        }
        //ネガチブ判断
        $negative = false;
        if(preg_match("/^!/", $validator)){
            $negative = true;
            $validator = substr($validator, 1);
        }
        return array($attribute, $validator, $group, $negative);
    }
    
    
    /**
     * バリューを分割する。
     * <code>
     *     stopper,error_message,...(validate_params)
     * </code>
     * のパターン
     * @access     private
     * @param      string  $value   バリュー
     * @return     array   分割された配列
     */
    private function _resolveValues($value)
    {
        $values = explode(",", $value);
        if(count($values) < 2) Samurai_Logger::error("Validatorの指定が不正です。-> %s", array("{$_key}/{$_val}"));
        $stopper = (int)array_shift($values);
        $message = (string)array_shift($values);
        return array($stopper, $message, $values);
    }
    
    
    
    
    
    /**
     * validate実処理。
     * @access     private
     */
    private function _validate()
    {
        foreach($this->_rules as $Rule){
            //STOP状態ならチェックしない
            if($this->_isStopping($Rule)) continue;
            //REQUEST値の取得
            $empty = true;
            if(preg_match("/,/", $Rule->attribute)){
                $value = array();
                foreach(explode(",", $Rule->attribute) as $attribute){
                    $parameter = $this->Request->getParameter($attribute);
                    if($parameter!==NULL && $parameter != "") $empty = false;
                    $value[] = $parameter;
                }
            } else {
                $parameter = $this->Request->getParameter($Rule->attribute);
                if($parameter!==NULL && $parameter != "") $empty = false;
                $value = $parameter;
            }
            //必須項目ではなく、かつ値が空の場合はチェックしない
            if(!$this->_isRequired($Rule) && $empty) continue;
            //Validate!!
            $result = $this->Manager->validate($Rule->validator, $value, $Rule->params);
            if(!$this->_isSuccess($result, $Rule)){
                $key = $Rule->group ? $Rule->group : $Rule->attribute;
                $this->ErrorList->add($key, $Rule->message);
                if($Rule->stopper) $this->_stoppers[] = $key;
            }
        }
    }
    
    
    /**
     * STOP状態かどうか。
     * @access     private
     * @param      object  $Rule   Filter_Validate_Ruleインスタンス
     * @return     boolean STOP状態かどうか
     */
    private function _isStopping(Filter_Validate_Rule $Rule)
    {
        if($Rule->group){
            return in_array($Rule->group, $this->_stoppers);
        } else {
            return in_array($Rule->attribute, $this->_stoppers);
        }
    }
    
    
    /**
     * 必須項目かどうか。
     * @access     private
     * @param      object  $Rule   Filter_Validate_Ruleインスタンス
     * @return     boolean 必須項目かどうか
     */
    private function _isRequired(Filter_Validate_Rule $Rule)
    {
        if($Rule->group){
            return in_array($Rule->group, $this->_required);
        } else {
            return in_array($Rule->attribute, $this->_required);
        }
    }
    
    
    /**
     * 検証が成功したかどうか。
     * @access     private
     * @param      boolean $result   検証結果
     * @param      object  $Rule     Filter_Validate_Ruleインスタンス
     * @return     boolean 検証に成功したかどうか
     */
    private function _isSuccess($result, Filter_Validate_Rule $Rule)
    {
        return (!$Rule->negative && $result) || ($Rule->negative && !$result);
    }
    
    
    
    
    
    /**
     * ValidatorManagerの生成。
     * @access     private
     */
    private function _createManager()
    {
        Samurai_Loader::loadByClass("Filter_Validate_Manager");
        Samurai_Loader::loadByClass("Filter_Validate_Rule");
        $this->Manager = new Filter_Validate_Manager();
        $this->ErrorList = $this->ActionChain->getCurrentErrorList();
    }
}
