<?
/**
 * ActiveGatewayManager
 * 
 * ActiveGatewayの取得、設定情報の管理などを行う。
 * このクラスはsingletonで動作する。
 * 
 * @package    ActiveGateway
 * @subpackage ActiveGateway
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 */
include_once(dirname(__FILE__)."/ActiveGateway.class.php");
include_once(dirname(__FILE__)."/ActiveGatewayRecord.class.php");
include_once(dirname(__FILE__)."/ActiveGatewayRecords.class.php");
include_once(dirname(__FILE__)."/ActiveGatewayCondition.class.php");
include_once(dirname(__FILE__)."/ActiveGatewayUtils.class.php");
include_once(dirname(__FILE__)."/Driver/Driver.abstract.php");
class ActiveGatewayManager
{
    private
        /** @var        array   DSN(スレーブ)情報を保持 */
        $_dsn_slave = array(),
        /** @var        array   DSN(マスター)情報を保持 */
        $_dsn_master = array(),
        /** @var        array   設定ファイル保持 */
        $_config_files = array(),
        /** @var        array   ActiveGatewayインスタンス保持 */
        $_active_gateways = array();
    private static
        /** @var        object  自身のインスタンス */
        $_instance,
        /** @var        array   発行されたクエリーを保持 */
        $_querys = array();
    
    
    /**
     * コンストラクタ。
     * @access     private
     */
    private function __construct()
    {
        
    }
    
    
    /**
     * 唯一のActiveGatewayManagerインスタンスの返却。
     * @access     public
     * @return     object  ActiveGatewayManager唯一のインスタンス
     */
    public function singleton()
    {
        if(self::$_instance===NULL){
            self::$_instance = new ActiveGatewayManager();
        }
        return self::$_instance;
    }
    
    
    
    
    
    /**
     * 設定ファイルを読み込む。
     * @access     public
     * @param      string  $config_file   設定ファイル
     */
    public function import($config_file)
    {
        $config = ActiveGatewayUtils::loadYaml($config_file);
        //情報のセット
        foreach($config as $alias => $_val){
            //スレーブのセット
            if(isset($_val['dsn']) && $_val['dsn']){
                $this->_dsn_slave[$alias] = $_val['dsn'];
            }
            //マスターのセット
            if(isset($_val['dsn_master']) && $_val['dsn_master']){
                $this->_dsn_master[$alias] = $_val['dsn_master'];
            } elseif(!isset($this->_dsn_master[$alias])) {
                $this->_dsn_master[$alias] = $this->_dsn_slave[$alias];
            }
            //confファイルの決定
            if(isset($_val['conf']) && $_val['conf']){
                if(!isset($this->_config_files[$alias])){
                    $this->_config_files[$alias] = $_val['conf'];
                } else {
                    $this->_config_files[$alias] = array_merge($this->_config_files[$alias], $_val['conf']);
                }
            }
        }
    }
    
    
    /**
     * ActiveGatewayの取得。
     * DSN文字列から、該当するドライバーを選択し、インスタンス化したのち返却する。
     * @access     public
     * @param      string  $alias   対象DSNのエイリアス名
     * @return     object  ActiveGatewayインスタンス
     */
    public function getActiveGateway($alias)
    {
        //静的参照許可
        if(!is_object($this) || !$this instanceof ActiveGatewayManager){
            $Manager = ActiveGatewayManager::singleton();
            return $Manager->getActiveGateway($alias);
        }
        //既に作成済みの場合
        if($this->hasActiveGateway($alias)){
            return $this->_active_gateways[$alias];
        }
        //新規作成
        elseif($this->hasDsn($alias)){
            $ActiveGateway = $this->makeActiveGateway($this->_pick($this->_dsn_slave[$alias]),
                                            $this->_pick($this->_dsn_master[$alias]),
                                            isset($this->_config_files[$alias]) ? $this->_config_files[$alias] : "");
            $this->_active_gateways[$alias] = $ActiveGateway;
            return $ActiveGateway;
        //不正
        } else {
            trigger_error("[ActiveGatewayManager]:DSN is Not Found -> {$alias}", E_USER_ERROR);
        }
    }
    
    
    /**
     * ActiveGatewayを作成する。
     * @access     public
     * @param      string  $dsn_slave    DSN(スレーブ)
     * @param      string  $dsn_master   DSN(マスター)
     * @param      string  $conf_file    設定ファイルパス
     * @return     object  ActiveGatewayインスタンス
     */
    public function makeActiveGateway($dsn_slave, $dsn_master=NULL, $conf_file="")
    {
        //ActiveGatewayの生成
        $ActiveGateway = new ActiveGateway();
        $ActiveGateway->setDsn($dsn_slave);
        $ActiveGateway->setDsnMaster($dsn_master !== NULL ? $dsn_master : $dsn_slave);
        $ActiveGateway->import($conf_file);
        //ドライバーの生成
        $driver_name = $this->_getDriverName($dsn_slave);
        $driver_file = $this->_getDriverFile($dsn_slave);
        if(file_exists($driver_file)){
            include_once($driver_file);
            $Driver = new $driver_name();
            if(is_object($Driver)){
                $ActiveGateway->setDriver($Driver);
            } else {
                trigger_error("[ActiveGatewayManager]:Driver generate failed... -> {$driver_file}", E_USER_ERROR);
            }
        } else {
            trigger_error("[ActiveGatewayManager]:Driver is Not Found -> {$driver_file}", E_USER_ERROR);
        }
        return $ActiveGateway;
    }
    
    
    /**
     * Driverのクラス名取得。
     * @access     private
     * @param      string  $dsn   DSN情報
     * @return     string  ドライバークラス名
     */
    private function _getDriverName($dsn)
    {
        $dsn_info = parse_url($dsn);
        if(isset($dsn_info['scheme']) && $dsn_info['scheme']){
            return "ActiveGateway_Driver_".ucfirst($dsn_info['scheme']);
        } else {
            trigger_error("[ActiveGatewayManager]:DSN is invalid format. -> {$dsn}", E_USER_ERROR);
        }
    }
    
    
    /**
     * Driverのファイル名取得。
     * @access     private
     * @param      string  $dsn   DSN情報
     * @return     string  ドライバーファイル名
     */
    private function _getDriverFile($dsn)
    {
        $driver_name = $this->_getDriverName($dsn);
        return sprintf("%s/Driver/%s.class.php", dirname(__FILE__), preg_replace("/^ActiveGateway_Driver_/", "", $driver_name));
    }
    
    
    
    
    
    /**
     * ActiveGatewayを既に保持しているかどうか。
     * @access     public
     * @param      string  $alias   Alias名
     * @return     boolean 既に保持しているかどうか
     */
    public function hasActiveGateway($alias)
    {
        return isset($this->_active_gateways[$alias]) && is_object($this->_active_gateways[$alias]);
    }
    
    
    /**
     * DSN情報を保持しているかどうか。
     * @access     public
     * @param      string  $alias   Alias名
     * @return     boolean 既に保持しているかどうか
     */
    public function hasDsn($alias)
    {
        return isset($this->_dsn_slave[$alias]) && $this->_dsn_slave[$alias];
    }
    
    
    
    
    
    /**
     * 実行クエリーのプール。
     * @access     public
     * @param      string  $query   クエリー文字列
     * @param      int     $time    実行時間
     */
    public function poolQuery($query, $time=0)
    {
        self::$_querys[] = array(
            "query" => $query,
            "time"  => $time,
        );
    }
    /**
     * プールされた実行クエリーの取得。
     * @access     public
     * @return     array   プールされたクエリー全部
     */
    public function getPoolQuery()
    {
        return self::$_querys;
    }
    
    
    /**
     * 配列からランダムにピック。
     * 配列でない場合はそのまま返却。
     * @access     private
     * @param      mixed   $array
     * @return     mixed   ピック結果を返却
     */
    private function _pick($array)
    {
        if(is_array($array)){
            return $array[array_rand($array)];
        } else {
            return $array;
        }
    }
}
