<?
/**
 * ActiveGatewayRecord
 * 
 * ActiveGatewayで使用するレコードを表すクラス。
 * 
 * @package    ActiveGateway
 * @subpackage ActiveGateway
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 */
class ActiveGatewayRecord {
    private
        /** @var        array   オリジナルの値を保持 */
        $_original = array(),
        /** @var        boolean 新規レコードかどうか */
        $_new_record = true,
        /** @var        string  基本エイリアス */
        $_base_alias = "",
        /** @var        string  テーブル名 */
        $_table_name = "",
        /** @var        array   テーブル情報 */
        $_table_info = array(),
        /** @var        string  プライマリキー */
        $_primary_key = "id";
    
    
    /**
     * コンストラクタ。
     * @access     public
     * @param      mixed   $row          PDOStatement->fetchで取得した値
     * @param      boolean $new_record   新規レコードかどうかの判断値
     * @param      string  $alias        仮想テーブル名
     */
    public function __construct($row=NULL, $new_record=true, $alias=NULL)
    {
        //$rowの展開
        if($row!==NULL){
            if(is_object($row)) $row = (array)$row;
            if(!is_array($row)){
                trigger_error("[ActiveGatewayRecord]:\$row is Not Array");
            }
            $this->_original = $row;
            foreach($row as $_key => $_val){
                if(!preg_match("/^_/", $_key)){
                    $this->$_key = $_val;
                }
            }
        }
        //その他情報のセット
        $this->_new_record = $new_record;
        $this->_base_alias = $alias;
        $this->_table_name = $alias;
    }
    
    
    
    
    
    /**
     * テーブル名の設定。
     * @access     public
     * @param      string  $table_name   テーブル名
     */
    public function setTableName($table_name)
    {
        $this->_table_name = (string)$table_name;
    }
    /**
     * テーブル名の取得。
     * @access     public
     * @return     string  テーブル名
     */
    public function getTableName()
    {
        return $this->_table_name;
    }
    /**
     * テーブル情報の設定。
     * @access     public
     * @param      array   $table_info   テーブル情報
     */
    public function setTableInfo($table_info)
    {
        $this->_table_info = $table_info;
    }
    
    
    /**
     * プライマリキーの設定。
     * @access     public
     * @param      string  $primary_key   プライマリキー名
     */
    public function setPrimaryKey($primary_key)
    {
        $this->_primary_key = (string)$primary_key;
    }
    /**
     * プライマリキーの取得。
     * @access     public
     * @return     string  プライマリキー名
     */
    public function getPrimaryKey()
    {
        return $this->_primary_key;
    }
    /**
     * プライマリキーの値の取得。
     * @access     public
     * @return     mixed   プライマリキーの値
     */
    public function getPrimaryValue()
    {
        $primary_key = $this->getPrimaryKey();
        return $this->$primary_key;
    }
    
    
    /**
     * Alias名の取得。
     * @access     public
     * @return     string  alias名
     */
    public function getAlias()
    {
        return $this->_base_alias;
    }
    
    
    /**
     * Attributesの取得。
     * @access     public
     * @param      boolean $updated   更新されたものに限定するかどうか
     * @return     array   attributes
     */
    public function getAttributes($updated=false)
    {
        $attributes = get_object_vars($this);
        foreach($attributes as $_key => $_val){
            if(preg_match("/^_/", $_key)) unset($attributes[$_key]);
            if($updated){
                if(isset($this->_original[$_key]) && $attributes[$_key]==$this->_original[$_key]){
                    unset($attributes[$_key]);
                }
            }
        }
        return $attributes;
    }
    
    
    /**
     * オリジナルの値取得。
     * @access     public
     * @param      string  $key   キー
     * @return     mixed   オリジナルの値
     */
    public function getOriginalValue($key)
    {
        switch($key){
            case 'primary_key':
                $value = $this->_original[$this->getPrimaryKey()];
                break;
            default:
                if(isset($this->_original[$key])){
                    $value = $this->_original[$key];
                } else {
                    $value = NULL;
                }
                break;
        }
        return $value;
    }
    
    
    
    
    
    /**
     * 新規レコードかどうかの判断。
     * @access     public
     * @return     boolean 新規レコードかどうか
     */
    public function isNewRecord()
    {
        return $this->_new_record;
    }
    
    
    /**
     * 指定のフィールを保持しているかどうか。
     * @access     public
     * @return     boolean 指定のフィールドがあるかどうか
     */
    public function hasField($field_name)
    {
        return isset($this->_table_info[$field_name]);
    }
    
    
    /**
     * ActiveGatewayによる、saveが可能であるかを返却。
     * 判断基準は、テーブル名が取得できるかどうかである。
     * 現在、findSqlもしくはfindAllSqlを直接使用すると、このsaveは使用できないようになっている。
     * テーブル名(Alias名)が取得しにくいからだ。
     * @access     public
     * @return     boolean saveメソッドが使用可能かどうか
     */
    public function enableSave()
    {
        if($this->_table_name===NULL) return false;
        return true;
    }
    public function isSavable()
    {
        return $this->enableSave();
    }
    
    
    /**
     * 論理削除が可能かどうか。
     * 判断基準は、規約通りに「active」フィールドがあるかどうかだ。
     * @access     public
     * @return     boolean 論理削除がかのうかどうか
     */
    public function enableDeleteByLogical()
    {
        return isset($this->_table_info['active']);
    }
    
    
    
    
    
    /**
     * 配列に変換する。
     * @access     public
     * @return     array   自身を配列に変換したもの
     */
    public function toArray()
    {
        return ActiveGatewayUtils::object2Array($this->getAttributes());
    }
}
