#include "config.h"
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <oniguruma.h>
#include <sys/stat.h>
#include <signal.h>

#include <limits.h>

#if defined(HAVE_CURSES_H)
#include <curses.h>
#elif defined(HAVE_NCURSES_H)
#include <ncurses.h>
#elif defined(HAVE_NCURSES_NCURSES_H)
#include <ncurses/ncurses.h>
#endif

#include <limits.h>

#if defined(__DARWIN__)
#include <sys/syslimits.h>
#endif

#include "saphire.h"
#include "saphire_inner.h"

///////////////////////////////////////////////////
/// statment_tree_internal_commandsの子関数
///////////////////////////////////////////////////
static void statment_tree_internal_commands_selector(string_obj* output, string_obj* result, int* scrolltop, int* cursor, enum eKanjiCode code, BOOL multiple, enum eLineField lf)
{
    if(tcsetpgrp(0, getpid()) < 0) {
        switch(errno) {
            case EBADF:
                puts("EBADF");
                break;

            case EINVAL:
                puts("EINVAL");
                break;

            case ENOTTY:
                puts("ENOTTY");
                break;

            case EPERM:
                puts("EPERM");
                break;

            default:
                puts("default");
                break;
        }
        perror("tcsetpgrp inner command");
        exit(1);
    }

#if !defined(__FREEBSD__)
    msave_screen();
#endif
    msave_ttysettings();
    minitscr();

    const int maxx = mgetmaxx();
    const int maxy = mgetmaxy();

    vector_obj* lines = VECTOR_NEW(100);

    /// 入力ごとにlinesに格納 ///
    string_obj* tmp = STRING_NEW("");
    char* p = string_c_str(output);
    while(*p) {
        if(is_line_field2(lf, p)) {
            if(lf == kCRLF) {
                p+=2;
            }
            else {
                p++;
            }
            vector_add(lines, tmp);
            tmp = STRING_NEW("");
        }
        else {
            string_push_back2(tmp, *p++);
        }
    }
    vector_add(lines, tmp);

    static int line_num = -1;
    if(line_num != -1 && vector_size(lines) != line_num) {
        *scrolltop = 0;
        *cursor = 0;
    }
    line_num = vector_size(lines);

    int* markfiles = MALLOC(sizeof(int)*vector_size(lines));
    memset(markfiles, 0, sizeof(int)*vector_size(lines));

    if(vector_size(lines) > 0) {
        while(1) {
            /// 描写 ///
            mclear();
            int n = *scrolltop;
            int y = 0;
            while(y < maxy && n < vector_size(lines)) {
                int attrs = 0;
                if(n == *cursor) {
                    attrs |= kCAReverse;
                }
                else if(markfiles[n]) {
                    attrs |= kCACyan;
                    attrs |= kCABold;
                }
                if(attrs) mattron(attrs);

                char* line = string_c_str(vector_item(lines, n));

                if(str_termlen(code, line) <= maxx) {
                    mmvprintw(y, 0, "%s", line);
                    y++;
                }
                else {
                    char* p = line;
                    while(p < line + strlen(line)) {
                        char one_line[BUFSIZ];
                        str_cut(code, p, maxx, one_line, BUFSIZ);
                        mmvprintw(y, 0, "%s", one_line);
                        y++;

                        p+=strlen(one_line);
                    }
                }

                if(attrs) {
                    mattroff();
                }

                n++;
            }
            mrefresh();

            /// インプット ///
            int meta;
            int key = mgetch(&meta);

            if(key == 14 || key == KEY_DOWN) {
                (*cursor)++;
            }
            else if(key == 16 || key == KEY_UP) {
                (*cursor)--;
            }
            else if(key == 4 || key == KEY_NPAGE) {
                (*cursor) += (maxy / 2);
            }
            else if(key == 21 || key == KEY_PPAGE) {
                (*cursor) -= (maxy /2);
            }
            else if(key == 12) {
                mclear();
                mrefresh();
            }
            else if(key == 'q' || key == 3 | key == 27 || key == 7) {
                break;
            }
            else if(key == ' ' && multiple) {
                markfiles[*cursor] = !markfiles[*cursor];
                (*cursor)++;
            }
            else if(key == 10 || key == 13) {
                if(multiple) {
                    BOOL flg_mark = FALSE;
                    int k;
                    for(k=0; k<vector_size(lines); k++) {
                        if(markfiles[k]) {
                            string_push_back(result
                                , string_c_str(vector_item(lines, k)));

                            string_push_back(result, "\n");

                            flg_mark = TRUE;
                        }
                    }

                    if(flg_mark) {
                        string_chomp(result);
                    }
                    else {
                        string_put(result 
                            , string_c_str(vector_item(lines, *cursor)));
                    }
                }
                else {
                    string_put(result 
                        , string_c_str(vector_item(lines, *cursor)));
                }
                break;
            }

            /// 修正 ///
            if(*cursor < 0) {
                *cursor = 0;
            }
            if(*cursor >= vector_size(lines)) {
                *cursor = vector_size(lines)-1;
            }
            if(*cursor >= n) {
                *scrolltop = n;
            }
            if(*cursor < *scrolltop) {
                int i = *cursor;
                int width_sum = 0;
                while(width_sum < maxy) {
                    char* line = string_c_str(vector_item(lines, i));
                    int width = str_termlen(code, line) / maxx + 1;
                    width_sum += width;
                    i--;
                    if(i < 0) {
                        i = -2;
                        break;
                    }
                }
                
                *scrolltop = i +2;
            }
        }
    }

//mclear();
//mrefresh();
    mendwin();
#if !defined(__FREEBSD__)
    mrestore_screen();
#endif
    mrestore_ttysettings();

    FREE(markfiles);

    int i;
    for(i=0; i<vector_size(lines); i++) {
        string_delete(vector_item(lines, i));
    }
    vector_delete(lines);
}

static double calculate_number(char** p) 
{
    double value = 0;

    BOOL minus;
    if(**p == '-') {
        (*p)++;
        minus = TRUE;

        while(**p == ' ' | **p == '\t') (*p)++;
    }
    else {
        minus = FALSE;
    }

    double keta = 1;
    char* tmp = *p;
    while(*tmp >= '0' && *tmp <= '9') {
        keta = keta * 10;
        tmp++;
    }

    keta = keta / 10;

    while(**p >= '0' && **p <= '9') {
        value = value + keta * (**p-'0');
        keta = keta / 10;
        (*p)++;
    }

    if(**p == '.') {
        (*p)++;

        while(**p >= '0' && **p <= '9') {
            value = value + keta * (**p-'0');
            keta = keta / 10;
            (*p)++;
        }
    }

    if(minus)
        return -value;
    else
        return value;
}

static double calculate_expr_add_sub(char** p, BOOL* result, char* sname, int line);

static double calculate_node(char** p, BOOL* result, char* sname, int sline)
{
    while(**p == ' ' || **p == '\t') (*p)++;

    if(**p == '(') {
        (*p)++;
        
        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        while(**p == ' ' || **p == '\t') (*p)++;
        
        if(**p == ')') {
            (*p)++;
            *result = TRUE;
        }
        else {
            err_msg("calc: need )", sname, sline);
            *result = FALSE;
        }

        return value;
    }
    else if(**p == '-' || **p >= '0' && **p <= '9') {
        *result = TRUE;
        return calculate_number(p);
    }
    else if(**p == 'p' && *(*p+1) == 'a' && *(*p+2) == 'i') {
        (*p)+=3;

        *result = TRUE;
        return 16*atan((double)1/5) - 4 * atan((double)1/239);
    }
    else if(**p == 'l' && *(*p+1) == 'o' && *(*p+2) == 'g') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return log(value);
    }
    else if(**p == 'i' && *(*p+1) == 'n' && *(*p+2) == 't') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return (int)(value);
    }
    else if(**p == 'c' && *(*p+1) == 'e' && *(*p+2) == 'i' && *(*p+3) == 'l') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return ceil(value);
    }
    else if(**p == 'f' && *(*p+1) == 'l' && *(*p+2) == 'o' && *(*p+3) == 'o'
            && *(*p+4) == 'r') 
    {
        (*p)+=5;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return floor(value);
    }
    else if(**p == 'l' && *(*p+1) == 'o' && *(*p+2) == 'g' && *(*p+3) == '1'
            && *(*p+4) == '0') 
    {
        (*p)+=5;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return log10(value);
    }
    else if(**p == 'a' && *(*p+1) == 'b' && *(*p+2) == 's') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return fabs(value);
    }
    else if(**p == 'e' && *(*p+1) == 'x' && *(*p+2) == 'p') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return exp(value);
    }
    else if(**p == 's' && *(*p+1) == 'i' && *(*p+2) == 'n') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return sin(value);
    }
    else if(**p == 'c' && *(*p+1) == 'o' && *(*p+2) == 's') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return cos(value);
    }
    else if(**p == 't' && *(*p+1) == 'a' && *(*p+2) == 'n') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return tan(value);
    }
    else if(**p == 'a' && *(*p+1) == 's' && *(*p+2) == 'i' && *(*p+3) == 'n') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return asin(value);
    }
    else if(**p == 'a' && *(*p+1) == 'c' && *(*p+2) == 'o' && *(*p+3) == 's') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return acos(value);
    }
    else if(**p == 'a' && *(*p+1) == 't' && *(*p+2) == 'a' && *(*p+3) == 'n') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return atan(value);
    }
    else if(**p == 's' && *(*p+1) == 'i' && *(*p+2) == 'n' && *(*p+3) == 'h') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return sinh(value);
    }
    else if(**p == 'c' && *(*p+1) == 'o' && *(*p+2) == 's' && *(*p+3) == 'h')
    {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return cosh(value);
    }
    else if(**p == 't' && *(*p+1) == 'a' && *(*p+2) == 'n' && *(*p+3) == 'h')
    {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return tanh(value);
    }
    else if(**p == 's' && *(*p+1) == 'q' && *(*p+2) == 'r' && *(*p+3) == 't') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result, sname, sline);
        if(!*result) return 0;

        return sqrt(value);
    }
    else {
        char buf[64];
        snprintf(buf, 64, "calc: invalid character (%c)", **p);
        err_msg(buf, sname, sline);
        *result = FALSE;
        return 0.0;
    }
}

static double calculate_expr_pow(char** p, BOOL* result,char* sname, int sline)
{
    double value = calculate_node(p, result, sname, sline);
    if(!*result) return 0;

    while(1) {
        if(**p == '*' && *(*p+1) == '*') {
            (*p)+=2;
            value = pow(value, calculate_node(p, result, sname, sline));
            if(!*result) {
                return 0;
            }
        }
        else if(**p == ' ' || **p == '\t') {
            (*p)++;
        }
        else {
            break;
        }
    }
    
    return value;
}

static double calculate_expr_mul_div(char** p, BOOL* result, char* sname, int sline)
{
    double value = calculate_expr_pow(p, result, sname, sline);
    if(!*result) return 0;

    while(1) {
        if(**p == '*') {
            (*p)++;
            value *= calculate_expr_pow(p, result, sname, sline);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == '/') {
            (*p)++;
            double waruhou = calculate_expr_pow(p, result, sname, sline);
            if(waruhou == 0.0) {
                err_msg("calc: divide 0", sname, sline);
                *result = FALSE;
                return 0;
            }
            value /= waruhou;
            if(!*result) {
                return 0;
            }
        }
        else if(**p == '%') {
            (*p)++;
            double waruhou = calculate_expr_pow(p, result, sname, sline);
            if(waruhou == 0.0) {
                err_msg("calc: divide 0", sname, sline);
                *result = FALSE;
                return 0;
            }
            value = fmod(value, waruhou);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == ' ' || **p == '\t') {
            (*p)++;
        }
        else {
            break;
        }
    }
    
    return value;
}

static double calculate_expr_add_sub(char** p, BOOL* result, char* sname, int sline)
{
    double value = calculate_expr_mul_div(p, result, sname, sline);
    if(!*result) return 0;

    while(1) {
        if(**p == '+') {
            (*p)++;
            value += calculate_expr_mul_div(p, result, sname, sline);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == '-') {
            (*p)++;
            value -= calculate_expr_mul_div(p, result, sname, sline);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == ' ' || **p == '\t') {
            (*p)++;
        }
        else {
            break;
        }
    }
    
    return value;
}

static BOOL calculate(double* result, char** p, char* sname, int sline) 
{
    BOOL err;
    *result = calculate_expr_add_sub(p, &err, sname, sline);

    if(**p != 0) {
        char buf[64];
        snprintf(buf, 64, "calc: invalid character (%c)\n", **p);
        err_msg(buf, sname, sline);
        return FALSE;
    }

    return err;
}

// 出力をCTRL-Cで止めたいので、この関数を経由して書き込む
// 戻り値 FALSE --> CTRL-Cで止められた
// 戻り値 TRUE --> 正常終了
BOOL statment_tree_internal_commands_write_nextout(sWFd* nextout, char* str)
{
    return sWFd_push_back(nextout, str);
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin(sRFd* nextin, string_obj* str)
{
    return sRFd_read_all(nextin, str);
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin_preserve(sRFd* nextin, string_obj* str)
{
    return sRFd_read_all_preserve(nextin, str);
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin_oneline(sRFd* nextin, string_obj* str)
{
    return sRFd_read_oneline(nextin, str);
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin_oneline_preserve(sRFd* nextin, string_obj* str)
{
    return sRFd_read_oneline_preserve(nextin, str);
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin_oneline_preserve_num(sRFd* nextin, string_obj* str, int line_num)
{
    return sRFd_read_oneline_preserve_num(nextin, str, line_num);
}

static BOOL statment_tree_internal_commands_test(int* rcode, vector_obj* argv, string_obj* input, char* sname, int sline)
{
    char* arg_last = string_c_str(
            vector_item(argv, vector_size(argv)-1));

    if(strcmp(arg_last, "]") != 0) {
        err_msg("test: missing ]", sname, sline);
        return FALSE;
    }
    else {
        BOOL reverse = FALSE;

        char* arg1;
        char* arg2;
        char* arg3;
        char* arg4;
        char* arg5;

        if(vector_size(argv) > 1) {
            arg1 = string_c_str(vector_item(argv, 1));
            if(strcmp(arg1, "!") == 0) {
                reverse = TRUE;
            }
        }

        if(reverse) {
            if(input) {
                if(vector_size(argv) == 4) {
                    arg1 = string_c_str(vector_item(argv, 2));
                    arg2 = string_c_str(input);
                }
                else if(vector_size(argv) == 5) {
                    arg1 = string_c_str(input);
                    arg2 = string_c_str(vector_item(argv, 2));
                    arg3 = string_c_str(vector_item(argv, 3));
                }
                else {
                    return TRUE;
                    /*
                    err_msg("test: invalid [ arguments 1", sname, sline);
                    return FALSE;
                    */
                }
            }
            else {
                if(vector_size(argv) == 5) {
                    arg1 = string_c_str(vector_item(argv, 2));
                    arg2 = string_c_str(vector_item(argv, 3));
                }
                else if(vector_size(argv) == 6) {
                    arg1 = string_c_str(vector_item(argv, 2));
                    arg2 = string_c_str(vector_item(argv, 3));
                    arg3 = string_c_str(vector_item(argv, 4));
                }
                else {
                    return TRUE;
                    /*
                    err_msg("test: invalid [ arguments 2", sname, sline);
                    return FALSE;
                    */
                }
            }
        }
        else {
            if(input) {
                if(vector_size(argv) == 3) {
                    arg2 = string_c_str(input);
                }
                else if(vector_size(argv) == 4) {
                    arg1 = string_c_str(input);
                    arg2 = string_c_str(vector_item(argv, 1));
                    arg3 = string_c_str(vector_item(argv, 2));
                }
                else {
                    return TRUE;
                }
            }
            else {
                if(vector_size(argv) == 4) {
                    arg2 = string_c_str(vector_item(argv, 2));
                }
                else if(vector_size(argv) == 5) {
                    arg2 = string_c_str(vector_item(argv, 2));
                    arg3 = string_c_str(vector_item(argv, 3));
                }
                else {
                    return TRUE;
                }
            }
        }
        if(!reverse && (vector_size(argv) + (input ? 1: 0)) == 4 
            || reverse && (vector_size(argv) + (input ? 1:0)) == 5) 
        {
            /// string ///
            if(strcmp(arg1, "-n") == 0) {
                if(!reverse && strcmp(arg2, "") != 0) {
                    *rcode = 0;
                }
                else if(reverse && ! strcmp(arg2, "") != 0) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg1, "-z") == 0) {
                if(!reverse && strcmp(arg2, "") == 0) {
                    *rcode = 0;
                }
                else if(reverse && ! strcmp(arg2, "") == 0) {
                    *rcode = 0;
                }
            }
            
            else {
                /// file type ///
                struct stat stat_;
                if(stat(arg2, &stat_) == 0) {
                    if(strcmp(arg1, "-b") == 0) {
                        if(!reverse && S_ISBLK(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse && ! S_ISBLK(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-c") == 0) {
                        if(!reverse && S_ISCHR(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISCHR(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-d") == 0) {
                        if(!reverse && S_ISDIR(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse && ! S_ISDIR(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-f") == 0) {
                        if(!reverse && S_ISREG(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISREG(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-h") == 0 
                             || strcmp(arg1, "-L") == 0) 
                    {
                        if(!reverse && S_ISLNK(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISLNK(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-p") == 0) {
                        if(!reverse && S_ISFIFO(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISFIFO(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-S") == 0) {
                        if(!reverse && S_ISSOCK(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && S_ISSOCK(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-t") == 0) {
                    }
                    /// permission ///
                    else if(strcmp(arg1, "-g") == 0) {
                        if(!reverse && (stat_.st_mode & S_ISGID)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !(stat_.st_mode & S_ISGID)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-k") == 0) {
#if defined(S_ISTXT)
                        if(!reverse && (stat_.st_mode & S_ISTXT)) {
                            *rcode = 0;
                        }
#endif
#if defined(S_ISVTX)
                        if(!reverse && (stat_.st_mode & S_ISVTX)) {
                            *rcode = 0;
                        }
#endif
#if defined(S_ISTXT)
                        else if(reverse && !(stat_.st_mode & S_ISTXT)) {
                            *rcode = 0;
                        }
#endif
#if defined(S_ISVTX)
                        else if(reverse && !(stat_.st_mode & S_ISVTX)) {
                            *rcode = 0;
                        }
#endif
                    }
                    else if(strcmp(arg1, "-u") == 0) {
                        if(!reverse 
                            && (stat_.st_mode & S_ISUID)) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && ! (stat_.st_mode & S_ISUID)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-r") == 0) {
                        if(!reverse 
                            && access(arg2, R_OK) == 0) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !(access(arg2, R_OK) == 0)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-w") == 0) {
                        if(!reverse 
                            && access(arg2, W_OK) == 0) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && ! ( access(arg2, W_OK) == 0)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-x") == 0) {
                        if(!reverse && access(arg2, X_OK) == 0) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !(access(arg2, X_OK) == 0)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-O") == 0) {
                        if(!reverse && stat_.st_uid == getuid()) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !stat_.st_uid == getuid()) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-G") == 0) {
                        if(!reverse && stat_.st_gid == getgid()) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! stat_.st_gid == getgid()) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-e") == 0) {
                        if(!reverse) *rcode = 0;
                    }
                    else if(strcmp(arg1, "-s") == 0) {
                        if(!reverse && stat_.st_size > 0) {
                            *rcode = 0;
                        }
                        else if(reverse && !(stat_.st_size > 0)) {
                            *rcode = 0;
                        }
                    }
                    else {
                        return TRUE;
                        /*
                        err_msg("test: invalid [ arguments 5", sname, sline);
                        return FALSE;
                        */
                    }
                }
                else {
                    if(reverse) *rcode = 0;
                }
            }
        }
        else if(!reverse && (vector_size(argv) + (input ? 1:0)) == 5 
            || reverse && (vector_size(argv) + (input ? 1:0)) == 6) 
        {
            /// string ///
            if(strcmp(arg2, "=") == 0) {
                if(!reverse && strcmp(arg1, arg3) == 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) == 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "!=") == 0) {
                if(!reverse && strcmp(arg1, arg3) != 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) != 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-slt") == 0) {
                if(!reverse && strcmp(arg1, arg3) < 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) >= 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-sgt") == 0) {
                if(!reverse && strcmp(arg1, arg3) > 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) <= 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-sle") == 0) {
                if(!reverse && strcmp(arg1, arg3) <= 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) > 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-sge") == 0) {
                if(!reverse && strcmp(arg1, arg3) >= 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) < 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-silt") == 0) {
                if(!reverse && strcasecmp(arg1, arg3) < 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcasecmp(arg1, arg3) >= 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-sigt") == 0) {
                if(!reverse && strcasecmp(arg1, arg3) > 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcasecmp(arg1, arg3) <= 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-sile") == 0) {
                if(!reverse && strcasecmp(arg1, arg3) <= 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcasecmp(arg1, arg3) > 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-sige") == 0) {
                if(!reverse && strcasecmp(arg1, arg3) >= 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcasecmp(arg1, arg3) < 0)) {
                    *rcode = 0;
                }
            }
            else if(arg2[0] == '-' && arg2[1] == 'r' && arg2[2] == 'e')
            {
                char* target = arg1;
                char* regex = arg3;

                int k;
                for(k=1; k<10; k++) {
                    char name[3];
                    snprintf(name, 3, "%d", k);
                    saphire_delete_local_var(name);
                }
                saphire_delete_local_var("PREMATCH");
                saphire_delete_local_var("MATCH");
                saphire_delete_local_var("POSTMATCH");

                OnigEncoding enc;
                if(gKanjiCode == kUtf8) {
                    enc = ONIG_ENCODING_UTF8;
                }
                else if(gKanjiCode == kEucjp) {
                    enc = ONIG_ENCODING_EUC_JP;
                }
                else {
                    enc = ONIG_ENCODING_SJIS;
                }

                char* regex2 = regex;

                int r;
                regex_t* reg;

                if(strcmp(arg2, "-reim") == 0 
                            || strcmp(arg2, "-remi") == 0) 
                {
                    reg = hash_item(gRegexsIM, regex);

                    if(reg == NULL) {
                        OnigErrorInfo err_info;
                        r = onig_new(&reg, regex2
                            , regex2 + strlen((char*)regex2)
                            , ONIG_OPTION_DEFAULT | ONIG_OPTION_IGNORECASE | ONIG_OPTION_MULTILINE
                            , enc
                            , ONIG_SYNTAX_DEFAULT
                            , &err_info);

                        if(r == ONIG_NORMAL) {
                            hash_put(gRegexs, regex, reg);
                        }
                        else {
                            onig_free(reg);
                        }
                    }
                    else {
                        r = ONIG_NORMAL;
                    }
                }
                else if(strcmp(arg2, "-rei") == 0) {
                    reg = hash_item(gRegexsI, regex);

                    if(reg == NULL) {
                        OnigErrorInfo err_info;
                        r = onig_new(&reg, regex2
                            , regex2 + strlen((char*)regex2)
                            , ONIG_OPTION_DEFAULT | ONIG_OPTION_IGNORECASE
                            , enc
                            , ONIG_SYNTAX_DEFAULT
                            , &err_info);

                        if(r == ONIG_NORMAL) {
                            hash_put(gRegexs, regex, reg);
                        }
                        else {
                            onig_free(reg);
                        }
                    }
                    else {
                        r = ONIG_NORMAL;
                    }
                }
                else if(strcmp(arg2, "-rem") == 0) {
                    reg = hash_item(gRegexsI, regex);

                    if(reg == NULL) {
                        OnigErrorInfo err_info;
                        r = onig_new(&reg, regex2
                            , regex2 + strlen((char*)regex2)
                            , ONIG_OPTION_DEFAULT|ONIG_OPTION_MULTILINE
                            , enc
                            , ONIG_SYNTAX_DEFAULT
                            , &err_info);

                        if(r == ONIG_NORMAL) {
                            hash_put(gRegexs, regex, reg);
                        }
                        else {
                            onig_free(reg);
                        }
                    }
                    else {
                        r = ONIG_NORMAL;
                    }
                }
                else {
                    reg = hash_item(gRegexsI, regex);

                    if(reg == NULL) {
                        OnigErrorInfo err_info;
                        r = onig_new(&reg, regex2
                            , regex2 + strlen((char*)regex2)
                            , ONIG_OPTION_DEFAULT
                            , enc
                            , ONIG_SYNTAX_DEFAULT
                            , &err_info);

                        if(r == ONIG_NORMAL) {
                            hash_put(gRegexs, regex, reg);
                        }
                        else {
                            onig_free(reg);
                        }
                    }
                    else {
                        r = ONIG_NORMAL;
                    }
                }
                

                if(r == ONIG_NORMAL) {
                    OnigRegion* region = onig_region_new();
                    int r2 = onig_search(reg, target
                       , target + strlen(target)
                       , target, target + strlen(target)
                       , region, ONIG_OPTION_NONE);

                    if(region->num_regs > 0) {
                        if(region->beg[0] > 0) {
                            /// マッチした文字列の前 ///
                            char* tmp = MALLOC(region->beg[0] + 1);

                            memcpy(tmp, target, region->beg[0]);
                            tmp[region->beg[0]] = 0;

                            saphire_set_local_var("PREMATCH", tmp);

                            FREE(tmp);
                        }
                        else {
                            saphire_set_local_var("PREMATCH", "");
                        }

                        /// マッチした文字列 ///
                        char* tmp = MALLOC(
                                region->end[0]-region->beg[0] + 1);

                        memcpy(tmp, target + region->beg[0]
                         , region->end[0]-region->beg[0]);

                        tmp[region->end[0]
                            - region->beg[0]] = 0;

                        saphire_set_local_var("MATCH", tmp);

                        FREE(tmp);

                        /// マッチした文字列の後 ///
                        const int n = strlen(target)-region->end[0];
                        if(n > 0) {
                            char* tmp = MALLOC(n + 1);

                            memcpy(tmp, target + region->end[0], n);

                            tmp[n] = 0;

                            saphire_set_local_var("POSTMATCH", tmp);

                            FREE(tmp);
                        }
                        else {
                            saphire_set_local_var("POSTMATCH", "");
                        }
                    }
                    if(r2 >= 0) {
                        int i;
                        for (i=1; i<region->num_regs; i++) {
                            char* tmp =
                                MALLOC(region->end[i]-region->beg[i]+1);

                            memcpy(tmp, target + region->beg[i]
                             , region->end[i]-region->beg[i]);

                            tmp[region->end[i]
                                - region->beg[i]] = 0;

                            char name[16];
                            snprintf(name, 16, "%d", i);

                            saphire_set_local_var(name, tmp);

                            FREE(tmp);
                        }
                    }

                    if(r2 >= 0 && !reverse || r2 < 0 && reverse) {
                        *rcode = 0;
                    }

                    onig_region_free(region, 1);
                }
                else {
                    err_msg("test: invalid regex", sname, sline);

                    return FALSE;
                }
            }

            /// number ///
            else if(strcmp(arg2, "-eq") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l == r) {
                    *rcode = 0;
                }
                else if(reverse && !(l == r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-ne") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l != r) {
                    *rcode = 0;
                }
                else if(reverse && !(l != r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-lt") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l < r) {
                    *rcode = 0;
                }
                else if(reverse && !(l < r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-le") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l <= r) {
                    *rcode = 0;
                }
                else if(reverse && !(l <= r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-gt") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l > r) {
                    *rcode = 0;
                }
                else if(reverse && !(l > r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-ge") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l >= r) {
                    *rcode = 0;
                }
                else if(reverse && !(l >= r)) {
                    *rcode = 0;
                }
            }

            /// file ///
            else if(strcmp(arg2, "-nt") == 0) {
                struct stat lstat_;
                struct stat rstat_;

                if(lstat(arg1, &lstat_) == 0) {
                    if(lstat(arg3, &rstat_) == 0) {
                        if(!reverse 
                            && lstat_.st_mtime > rstat_.st_mtime) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && !(lstat_.st_mtime > rstat_.st_mtime)) 
                        {
                            *rcode = 0;
                        }
                    }
                }
            }
            else if(strcmp(arg2, "-ot") == 0) {
                struct stat lstat_;
                struct stat rstat_;

                if(lstat(arg1, &lstat_) == 0) {
                    if(lstat(arg3, &rstat_) == 0) {
                        if(!reverse 
                            && lstat_.st_mtime < rstat_.st_mtime) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && !(lstat_.st_mtime < rstat_.st_mtime)) 
                        {
                            *rcode = 0;
                        }
                    }
                }
            }
            else if(strcmp(arg2, "-ef") == 0) {
            }
            else {
                return TRUE;
                //err_msg("test: invalid [ arguments 6", sname, sline);
                //return FALSE;
            }
        }
    }

    return TRUE;
}

// p: 対象文字列文字列内のポインタ
// start:対象文字列の最初の位置 
// word: 検索文字列
static char* strstr_back(char* p, char* start, char* word, char* sname, int sline)
{
//printf("p %s start %s word %s\n", p, start, word);
    int n = strlen(word);

    while(p >= start) {
        BOOL flg = TRUE;
        int i;
        for(i=-1; i>=-n; i--) {
            if(p[i] != word[n+i]) {
                flg = FALSE;
                break;
            }

            if(gKitutukiSigInt) {
                err_msg("interrupt", sname, sline);
                return NULL;
            }
        }

        if(flg) {
            return p -n;
        }
        else {
            p--;
        }
    }

    return NULL;
}

static BOOL statment_tree_internal_commands_match(char* str2
  , UChar* regex2, int* rcode, sWFd* nextout, BOOL print, BOOL line_field
  , BOOL line_num, int n, string_obj* field, BOOL line_oriented
  , int* match_count, BOOL ignore_case, BOOL multiline, BOOL restart, char* sname, int sline)
{
    int k;
    for(k=1; k<10; k++) {
        char name[3];
        snprintf(name, 3, "%d", k);
        saphire_delete_local_var(name);
    }
    saphire_delete_local_var("PREMATCH");
    saphire_delete_local_var("MATCH");
    saphire_delete_local_var("POSTMATCH");

    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, (char*)regex2);
    else if(ignore_case)
        reg = hash_item(gRegexsI, (char*)regex2);
    else if(multiline) 
        reg = hash_item(gRegexsM, (char*)regex2);
    else
        reg = hash_item(gRegexs, (char*)regex2);
        
    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT | ONIG_OPTION_IGNORECASE | ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, (char*)regex2, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT | ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, (char*)regex2, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT|ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, (char*)regex2, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, (char*)regex2, reg);
            }
            else {
                onig_free(reg);
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        int position;
        if(restart) {
            position = (int)hash_item(gMatchRestart, str2);
        }
        else {
            position = 0;
        }

        OnigRegion* region = onig_region_new();
        int r2 = onig_search(reg, str2
           , str2 + strlen(str2)
           , str2 + position, str2 + strlen(str2)
           //, region, ONIG_OPTION_CAPTURE_GROUP);
           , region, ONIG_OPTION_NONE);

        if(r2 >= 0) {
            *rcode = 0;
            (*match_count)++;

            if(region->num_regs > 0) {
                /// マッチした文字列の前 ///
                if(region->beg[0] > 0) {
                    char* tmp = MALLOC(region->beg[0] + 1);

                    memcpy(tmp, str2, region->beg[0]);
                    tmp[region->beg[0]] = 0;

                    saphire_set_local_var("PREMATCH", tmp);

                    FREE(tmp);
                }
                else {
                    saphire_set_local_var("PREMATCH", "");
                }

                /// マッチした文字列 ///
                char* tmp = MALLOC(
                        region->end[0]-region->beg[0] + 1);

                memcpy(tmp, str2 + region->beg[0]
                 , region->end[0]-region->beg[0]);

                tmp[region->end[0]
                    - region->beg[0]] = 0;

                saphire_set_local_var("MATCH", tmp);

                if(restart) {
                    hash_put(gMatchRestart, str2, (void*)region->end[0]);
                }


                FREE(tmp);

                /// マッチした文字列の後 ///
                const int n = strlen(str2)-region->end[0];
                if(n > 0) {
                    char* tmp = MALLOC(n + 1);

                    memcpy(tmp, str2 + region->end[0], n);

                    tmp[n] = 0;

                    saphire_set_local_var("POSTMATCH", tmp);

                    FREE(tmp);
                }
                else {
                    saphire_set_local_var("POSTMATCH", tmp);
                }
            }

            /// マッチしたグループ化文字列 ///
            int i;
            for (i=1; i<region->num_regs; i++) {
                char* tmp = MALLOC(
                    region->end[i]-region->beg[i] + 1);

                memcpy(tmp, str2 + region->beg[i]
                 , region->end[i]-region->beg[i]);

                tmp[region->end[i]
                    - region->beg[i]] = 0;

                char name[16];
                snprintf(name, 16, "%d", i);

                saphire_set_local_var(name, tmp);

                FREE(tmp);
            }

            if(print) {
                if(line_oriented) {
                    if(line_num) {
                        char num[128];
                        snprintf(num, 128, "%d:", n);
                        if(!statment_tree_internal_commands_write_nextout(nextout, num))
                        {
                            return FALSE;
                        }
                    }
                    
                    if(!statment_tree_internal_commands_write_nextout(nextout, str2) )
                    {
                        return FALSE;
                    }
                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n") )
                        {
                            return FALSE;
                        }
                    }
                }
                else if(region->num_regs == 1) {
                    if(line_num) {
                        char num[128];
                        snprintf(num, 128, "%d:", n);
                        if(!statment_tree_internal_commands_write_nextout(nextout, num) )
                        {
                            return FALSE;
                        }
                    }
                    
                    char* match = saphire_get_local_var("MATCH");

                    if(match) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, match) )
                        {
                            return FALSE;
                        }
                    }
                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n") )
                        {
                            return FALSE;
                        }
                    }
                }
                else if(region->num_regs > 1) {
                    int i;
                    for(i=1; i<region->num_regs; i++) {
                        char name[16];
                        snprintf(name, 16, "%d", i);

                        char* match = saphire_get_local_var(name);

                        if(match) {
                            if(line_num) {
                                char num[128];
                                snprintf(num, 128, "%d:", n);
                                if(!statment_tree_internal_commands_write_nextout(nextout, num)) 
                                {
                                    return FALSE;
                                }
                            }

                            if(!statment_tree_internal_commands_write_nextout(nextout, match)) 
                            {
                                return FALSE;
                            }
                        }
                        if(i==region->num_regs-1) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                                {
                                    return FALSE;
                                }
                            }
                        }
                        else {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(field))) 
                            {
                                return FALSE;
                            }
                        }
                    }
                }
            }
        }
        else {
            hash_put(gMatchRestart, str2, 0);
        }

        onig_region_free(region, 1);
    }
    else {
        err_msg("match: invalid regex", sname, sline);

        return FALSE;
    }

    return TRUE;
}

// sort_funにデータを渡すためのグローバル変数
static char* gSortBlock;
static sWFd* gSortNextOut;
static sRFd* gSortNextIn;
static int gSortNextErr;

static int sort_fun(void* left, void* right)
{
    char* left2 = string_c_str(left);
    char* right2 = string_c_str(right);

    saphire_set_local_var("a", left2);
    saphire_set_local_var("b", right2);
    
    int rcode = saphire_shell(gSortBlock, "sort_block"
                    , gSortNextOut, gSortNextIn
                    , gSortNextErr);

    if(rcode < 0) {
        return -1;
    }

    return rcode == 0;
}

BOOL statment_tree_internal_commands_printf(char* format, vector_obj* strings, sWFd* nextout, BOOL check, BOOL line_field, char* sname, int sline, enum eKanjiCode code)
{
    string_obj* output = STRING_NEW("");
    char* p = format;

    int strings_num = 0;

    while(*p) {
        if(*p == '%') {
            p++;

            /// オプション ///
            BOOL minus = FALSE;
            if(*p == '-') {
                p++;
                minus = TRUE;
            }

            int width = -1;
            BOOL range_enable = FALSE;
            int range1 = 0;
            int range2 = 0;

            if(*p >= '0' && *p <= '9') {
                string_obj* str = STRING_NEW("");

                while(*p >= '0' && *p <= '9') {
                    string_push_back2(str, *p++);
                }

                width = atoi(string_c_str(str));

                string_delete(str);
            }

            if(*p == '(') {
                p++;

                /// 範囲指定一つ目 ///
                string_obj* str1 = STRING_NEW("");

                if(*p != '-' && (*p < '0' || *p > '9')) {
                    err_msg("invalid format", sname, sline);
                    string_delete(str1);
                    string_delete(output);
                    return FALSE;
                }

                if(*p == '-') {
                    string_push_back2(str1, *p++);
                }

                while(*p >= '0' && *p <= '9') {
                    string_push_back2(str1, *p++);
                }

                if(*p != '.') {
                    err_msg("invalid format", sname, sline);
                    string_delete(str1);
                    string_delete(output);
                    return FALSE;
                }

                p++;

                if(*p != '.') {
                    err_msg("invalid format", sname, sline);
                    string_delete(str1);
                    string_delete(output);
                    return FALSE;
                }

                p++;

                /// 範囲指定２つ目 ///

                string_obj* str2 = STRING_NEW("");

                if(*p != '-' && (*p < '0' || *p > '9')) {
                    err_msg("invalid format", sname, sline);
                    string_delete(str1);
                    string_delete(str2);
                    string_delete(output);
                    return FALSE;
                }

                if(*p == '-') {
                    string_push_back2(str2, *p++);
                }

                while(*p >= '0' && *p <= '9') {
                    string_push_back2(str2, *p++);
                }

                if(*p != ')') {
                    err_msg("invalid format", sname, sline);
                    string_delete(str1);
                    string_delete(str2);
                    string_delete(output);
                    return FALSE;
                }

                p++;

                range1 = atoi(string_c_str(str1));
                range2 = atoi(string_c_str(str2));

                range_enable = TRUE;

                string_delete(str1);
                string_delete(str2);
            }

            if(*p == 's') {
                p++;

                if(strings_num >= vector_size(strings)) {
                    if(check) {
                        err_msg("invalid % number", sname, sline);
                        string_delete(output);
                        return FALSE;
                    }
                }
                else {
                    string_obj* string = vector_item(strings, strings_num++);

                    if(range_enable) {
                        if(range1 < 0) {
                            range1 = str_kanjilen(code, string_c_str(string)) + range1;
                        }
                        if(range2 < 0) {
                            range2 = str_kanjilen(code, string_c_str(string)) + range2;
                        }

                        if(range1 >= str_kanjilen(code, string_c_str(string))) {
                            range1 = str_kanjilen(code, string_c_str(string)) -1;
                        }

                        if(range2 >= str_kanjilen(code, string_c_str(string))) {
                            range2 = str_kanjilen(code, string_c_str(string)) -1;
                        }

                        if(range1 < 0) {
                            range1 = 0;
                        }

                        if(range2 < 0) {
                            range2 = 0;
                        }

                        if(range1 > range2) {
                            err_msg("invalid range", sname, sline);
                            string_delete(output);
                            return FALSE;
                        }

                        char* tmp = MALLOC(string_length(string) + 1);
                        int len = str_kanjipos2pointer(code, string_c_str(string), range2+1) - str_kanjipos2pointer(code, string_c_str(string), range1);
                        memcpy(tmp, str_kanjipos2pointer(code, string_c_str(string), range1), len);
                        tmp[len] = 0;

                        string_put(string, tmp);

                        FREE(tmp);
                    }

                    if(width >= 0) {
                        string_obj* new_string = STRING_NEW("");

                        /// 左詰 ///
                        if(minus) {
                            if(str_kanjilen(code, string_c_str(string)) >= width) {
                                int len = str_kanjipos2pointer(code, string_c_str(string), width) - string_c_str(string);
                                char* str2 = MALLOC(len+1);

                                memcpy(str2, string_c_str(string), len);
                                str2[len] = 0;

                                string_put(new_string, str2);

                                FREE(str2);
                            }
                            else {
                                char* spaces = MALLOC(width-str_kanjilen(code, string_c_str(string))+1);
                                char* p = spaces;
                                int i;
                                int len = width - str_kanjilen(code, string_c_str(string));
                                for(i=0; i<len; i++) {
                                    *p++ = ' ';
                                }
                                *p = 0;

                                string_put(new_string, string_c_str(string));
                                string_push_back(new_string, spaces);

                                FREE(spaces);
                            }
                        }
                        /// 右詰 ///
                        else {
                            if(str_kanjilen(code, string_c_str(string)) >= width) {
                                int over_width = str_kanjilen(code, string_c_str(string)) - width;
                                char* str2 = MALLOC(string_length(string)+1);

                                char* p = str_kanjipos2pointer(code, string_c_str(string), over_width);
                                memcpy(str2, p, string_c_str(string) + strlen(string_c_str(string)) - p);
                                str2[string_c_str(string) + strlen(string_c_str(string))-p] = 0;

                                string_put(new_string, str2);

                                FREE(str2);
                            }
                            else {
                                char* spaces = MALLOC(width- str_kanjilen(code, string_c_str(string))+1);
                                char* p = spaces;
                                int i;
                                int len = width - str_kanjilen(code, string_c_str(string));
                                for(i=0; i<len; i++) {
                                    *p++ = ' ';
                                }
                                *p = 0;

                                string_put(new_string, spaces);
                                string_push_back(new_string, string_c_str(string));

                                FREE(spaces);
                            }
                        }

                        string_put(string, string_c_str(new_string));

                        string_delete(new_string);
                    }

                    string_push_back(output, string_c_str(string));
                }
            }
            else if(*p == '%') {
                p++;

                string_push_back2(output, '%');
            }
            else {
                err_msg("non support format string", sname, sline);
                string_delete(output);
                return FALSE;
            }
        }
        else {
            string_push_back2(output, *p++);
        }
    }

    if(line_field) {
        string_push_back2(output, '\n');
    }

    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(output)))
    {
        string_delete(output);
        return FALSE;
    }

    string_delete(output);

    return TRUE;
}

typedef struct {
    int kind;   // 0: num 1: range 2: baisuu
    int num1;
    int num2;
    sStatments* statments;
    string_obj* condition;
} sLinesArg;

BOOL vcat_expand_env(string_obj* pipeout, char* env, vector_obj* numbers,char* sname, int sline)
{
    if(vector_size(numbers) == 0)
    {
        string_push_back(pipeout, env);
    }
    else {
        int i;
        for(i=0; i<vector_size(numbers); i++) {
            sIndex* index = vector_item(numbers, i);

            if(index->number_num == 2) {
                int n = index->number;
                int n2 = index->number2;

                if(n < 0) {
                    n = str_kanjilen(gKanjiCode, env) + n;
                }

                if(n2 < 0) {
                    n2 = str_kanjilen(gKanjiCode, env) + n2;
                }

                if(n< 0) { n = 0; }
                if(n>=str_kanjilen(gKanjiCode, env)) {
                    n = str_kanjilen(gKanjiCode, env) -1;
                }
                if(n2< 0) { n2 = 0; }
                if(n2>=str_kanjilen(gKanjiCode, env)) {
                    n2 = str_kanjilen(gKanjiCode, env) -1;
                }

                if(n <= n2) {
                    char* p1 = str_kanjipos2pointer(gKanjiCode, env, n);
                    char* p2 = str_kanjipos2pointer(gKanjiCode, env, n2+1);

                    char* buf = MALLOC(p2-p1+1);
                    memcpy(buf, p1, p2-p1);
                    buf[p2-p1] = 0;
            
                    string_push_back(pipeout, buf);

                    FREE(buf);
                }
                else {
                    err_msg("vcat_expand_env: invalid [] range", sname, sline);
                    return FALSE;
                }
            }
            else {
                int n = index->number;

                if(n < 0) {
                    n = str_kanjilen(gKanjiCode, env) + n;
                }

                if(n< 0) { n = 0; }
                if(n>=str_kanjilen(gKanjiCode, env)) {
                    n = str_kanjilen(gKanjiCode, env) -1;
                }

                char str[MB_CUR_MAX+1];
                char* p1 = str_kanjipos2pointer(gKanjiCode, env, n);
                char* p2 = str_kanjipos2pointer(gKanjiCode, env, n + 1);

                memcpy(str, p1, p2-p1);
                str[p2-p1] = 0;

                string_push_back(pipeout, str);
            }
        }
    }

    return TRUE;
}

BOOL statment_tree_internal_commands_global_match(vector_obj* argv, string_obj* field, BOOL input, sWFd* nextout, sRFd* nextin, int *rcode, BOOL ignore_case, BOOL multiline, char* sname, int sline, BOOL quiet)
{
    int match_count = 0;

    char* regex;

    string_obj* str;
    if(input) {
        if(vector_size(argv) == 2) {
            str = STRING_NEW("");
            int ret = statment_tree_internal_commands_read_nextin
                    (nextin, str); 
            if(ret == 1) {
                *rcode = 1;
                return TRUE;
            }
            else if(ret < 0)
            {
                string_delete(str);
                return FALSE;
            }
            regex = string_c_str(vector_item(argv, 1));
        }
        else {
            err_msg("invalid match arguments", sname, sline);
            return FALSE;
        }
    }
    else {
        if(vector_size(argv) == 3) {
            str = STRING_NEW(string_c_str(vector_item(argv, 1)));
            regex = string_c_str(vector_item(argv, 2));
        }
        else {
            err_msg("invalid match arguments", sname, sline);
            return FALSE;
        }
    }

    char* str2 = string_c_str(str);
    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                        |ONIG_OPTION_MULTILINE
                    , enc
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        char* p = str2;
        while(1) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                string_delete(str);
                return FALSE;
            }
//puts("looop hennkan");
            OnigRegion* region = onig_region_new();

            int r2 = onig_match(reg, str2
               , str2 + strlen(str2)
               , p
               , region, ONIG_OPTION_NONE);

            if(r2 >= 0) {
                *rcode = 0;
                match_count++;

                if(region->num_regs == 1) {
                    if(!quiet) {
                        /// マッチした文字列 ///
                        char* tmp = MALLOC(region->end[0]
                                        -region->beg[0] + 1);

                        memcpy(tmp, str2 + region->beg[0]
                         , region->end[0]-region->beg[0]);

                        tmp[region->end[0]
                            - region->beg[0]] = 0;

                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }

                        FREE(tmp);
                    }
                }
                else {
                    if(!quiet) {
                        /// グループ化 ///
                        int i;
    //printf("(region->num_regs: %d)\n", region->num_regs);
                        for (i=1; i<region->num_regs; i++) {
                            char* tmp = MALLOC(
                                region->end[i]-region->beg[i] + 1);

                            memcpy(tmp, str2 + region->beg[i]
                             , region->end[i]-region->beg[i]);

                            tmp[region->end[i]
                                - region->beg[i]] = 0;

                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(tmp);
                                string_delete(str);
                                onig_region_free(region, 1);
                                return FALSE;
                            }

                            if(i==region->num_regs-1) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(tmp);
                                    string_delete(str);
                                    onig_region_free(region, 1);
                                    return FALSE;
                                }
                            }
                            else {
                                if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(field)))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(tmp);
                                    string_delete(str);
                                    onig_region_free(region, 1);
                                    return FALSE;
                                }
                            }

                            FREE(tmp);
                        }
                    }
                }

                p = str2 + region->end[0];
            }
            else {
//printf("(検索失敗)\n");
                p = p + 1;
                /*
                /// 検索失敗したら抜ける ///
                onig_region_free(region, 1);
                break;
                */
            }

            onig_region_free(region, 1);

            /// 最後まで行ったら抜ける ///
            if(p >= str2 + strlen(str2)) {
//printf("(最後まで行った)\n");
                break;
            }
        }

        char buf[256];
        snprintf(buf, 256, "%d", match_count);
        saphire_set_local_var("MATCH_COUNT", buf);
    }
    else {
        err_msg("match: invalid regex", sname, sline);

        string_delete(str);

        return FALSE;
    } 

    string_delete(str);

    return TRUE;
}

BOOL statment_tree_internal_commands_global_match_oneline(vector_obj* argv, string_obj* field, BOOL input, sWFd* nextout, sRFd* nextin, int *rcode, BOOL ignore_case, BOOL multiline, char* sname, int sline, BOOL quiet)
{
    int match_count = 0;

    if(vector_size(argv) != 2) {
        err_msg("invalid match arguments", sname, sline);
        return FALSE;
    }

    char* regex = string_c_str(vector_item(argv, 1));
    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                        |ONIG_OPTION_MULTILINE
                    , enc
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        while(1) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return FALSE;
            }

            string_obj* line = STRING_NEW("");
            int result = statment_tree_internal_commands_read_nextin_oneline(nextin, line);
            if(result < 0) {
                err_msg("interrupt", sname, sline);
                string_delete(line);
                return FALSE;
            }
            else if(result == 1) {
                string_delete(line);
                break;
            }

            char* p = string_c_str(line);
            char* str2 = p;
            while(1) {
                OnigRegion* region = onig_region_new();

                int r2 = onig_match(reg, str2
                   , str2 + strlen(str2)
                   , p
                   , region, ONIG_OPTION_NONE);

                if(r2 >= 0) {
                    *rcode = 0;
                    match_count++;

                    if(region->num_regs == 1) {
                        if(!quiet) {
                            /// マッチした文字列 ///
                            char* tmp = MALLOC(region->end[0]
                                            -region->beg[0] + 1);

                            memcpy(tmp, str2 + region->beg[0]
                             , region->end[0]-region->beg[0]);

                            tmp[region->end[0]
                                - region->beg[0]] = 0;

                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(tmp);
                                string_delete(line);
                                onig_region_free(region, 1);
                                return FALSE;
                            }
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(tmp);
                                string_delete(line);
                                onig_region_free(region, 1);
                                return FALSE;
                            }
                            FREE(tmp);
                        }
                    }
                    else {
                        if(!quiet) {
                            /// グループ化 ///
                            int i;
        //printf("(region->num_regs: %d)\n", region->num_regs);
                            for (i=1; i<region->num_regs; i++) {
                                char* tmp = MALLOC(
                                    region->end[i]-region->beg[i] + 1);

                                memcpy(tmp, str2 + region->beg[i]
                                 , region->end[i]-region->beg[i]);

                                tmp[region->end[i]
                                    - region->beg[i]] = 0;

                                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(tmp);
                                    string_delete(line);
                                    onig_region_free(region, 1);
                                    return FALSE;
                                }

                                if(i==region->num_regs-1) {
                                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                    {
                                        err_msg("signal interrupt", sname, sline);
                                        FREE(tmp);
                                        string_delete(line);
                                        onig_region_free(region, 1);
                                        return FALSE;
                                    }
                                }
                                else {
                                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(field)))
                                    {
                                        err_msg("signal interrupt", sname, sline);
                                        FREE(tmp);
                                        string_delete(line);
                                        onig_region_free(region, 1);
                                        return FALSE;
                                    }
                                }

                                FREE(tmp);
                            }
                        }
                    }

                    p = str2 + region->end[0];
                }
                else {
    //printf("(検索失敗)\n");
                    p = p + 1;
                    /*
                    /// 検索失敗したら抜ける ///
                    onig_region_free(region, 1);
                    break;
                    */
                }

                onig_region_free(region, 1);

                /// 最後まで行ったら抜ける ///
                if(p >= str2 + strlen(str2)) {
    //printf("(最後まで行った)\n");
                    break;
                }
            }

            string_delete(line);
        }

        char buf[256];
        snprintf(buf, 256, "%d", match_count);
        saphire_set_local_var("MATCH_COUNT", buf);
    }
    else {
        err_msg("match: invalid regex", sname, sline);

        return FALSE;
    } 

    char buf[256];
    snprintf(buf, 256, "%d", match_count);
    saphire_set_local_var("MATCH_COUNT", buf);

    return TRUE;
}

BOOL statment_tree_internal_commands_erase(vector_obj* argv, BOOL input, sWFd* nextout, sRFd* nextin, int *rcode, BOOL ignore_case, BOOL multiline,char* sname, int sline)
{
    char* regex;

    string_obj* str;
    if(input) {
        if(vector_size(argv) == 2) {
            str = STRING_NEW("");
            int ret = statment_tree_internal_commands_read_nextin
                    (nextin, str); 
            if(ret == 1) {
                *rcode = 1;
                return TRUE;
            }
            else if(ret < 0)
            {
                string_delete(str);
                return FALSE;
            }
            regex = string_c_str(vector_item(argv, 1));
        }
        else {
            err_msg("invalid erase arguments", sname, sline);
            return FALSE;
        }
    }
    else {
        if(vector_size(argv) == 3) {
            str = STRING_NEW(string_c_str(vector_item(argv, 1)));
            regex = string_c_str(vector_item(argv, 2));
        }
        else {
            err_msg("invalid erase arguments", sname, sline);
            return FALSE;
        }
    }

    char* str2 = string_c_str(str);
    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                        |ONIG_OPTION_MULTILINE
                    , enc
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        char* p = str2;
        OnigRegion* region = onig_region_new();

        int r2 = onig_search(reg, str2
           , str2 + strlen(str2)
           , str2, str2 + strlen(str2)
           , region, ONIG_OPTION_NONE);

        if(r2 >= 0) {
            *rcode = 0;

            if(region->num_regs == 1) {
                /// マッチしなかった文字列 ///
                int len = region->beg[0]+strlen(str2)-region->end[0] + 1;
                char* tmp = MALLOC(len);

                memcpy(tmp, str2, region->beg[0]);
                memcpy(tmp + region->beg[0], str2 + region->end[0], strlen(str2)-region->end[0]);
                tmp[len-1] = 0;

                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                {
                    err_msg("signal interrupt", sname, sline);
                    FREE(tmp);
                    string_delete(str);
                    onig_region_free(region, 1);
                    return FALSE;
                }

                FREE(tmp);
            }
            else {
                /// グループ化 ///
                int i;
                int p = 0;
                for (i=1; i<region->num_regs; i++) {
                    if(region->beg[i] >= 0) {
                        int len = region->beg[i]-p + 1;
                        char* tmp = MALLOC(len);
                        memcpy(tmp, str2 + p, region->beg[i]-p);
                        tmp[len-1] = 0;

                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }

                        FREE(tmp);

                        p = region->end[i];
                    }
                }

                int len = strlen(str2) - p + 1;
                char* tmp = MALLOC(len);
                memcpy(tmp, str2 + p, strlen(str2)-p);
                tmp[len-1] = 0;

                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                {
                    err_msg("signal interrupt", sname, sline);
                    FREE(tmp);
                    string_delete(str);
                    onig_region_free(region, 1);
                    return FALSE;
                }

                FREE(tmp);
            }
        }
        else {
            if(!statment_tree_internal_commands_write_nextout(nextout, str2))
            {
                err_msg("signal interrupt", sname, sline);
                string_delete(str);
                onig_region_free(region, 1);
                return FALSE;
            }
        }

        onig_region_free(region, 1);
    }
    else {
        err_msg("erase: invalid regex", sname, sline);

        string_delete(str);

        return FALSE;
    } 

    string_delete(str);

    return TRUE;
}

BOOL statment_tree_internal_commands_erase_global(vector_obj* argv, BOOL input, sWFd* nextout, sRFd* nextin, int *rcode, BOOL ignore_case, BOOL multiline, char* sname, int sline)
{
    char* regex;

    string_obj* str;
    if(input) {
        if(vector_size(argv) == 2) {
            str = STRING_NEW("");
            int ret = statment_tree_internal_commands_read_nextin
                    (nextin, str); 
            if(ret == 1) {
                *rcode = 1;
                return TRUE;
            }
            else if(ret < 0)
            {
                string_delete(str);
                return FALSE;
            }
            regex = string_c_str(vector_item(argv, 1));
        }
        else {
            err_msg("invalid erase arguments", sname, sline);
            return FALSE;
        }
    }
    else {
        if(vector_size(argv) == 3) {
            str = STRING_NEW(string_c_str(vector_item(argv, 1)));
            regex = string_c_str(vector_item(argv, 2));
        }
        else {
            err_msg("invalid erase arguments", sname, sline);
            return FALSE;
        }
    }

    char* str2 = string_c_str(str);
    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                        |ONIG_OPTION_MULTILINE
                    , enc
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        char* p = str2;
        int match = -1;
        while(1) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                string_delete(str);
                return FALSE;
            }
            OnigRegion* region = onig_region_new();

            int r2 = onig_search(reg, str2
               , str2 + strlen(str2)
               , p, p + strlen(p)
               , region, ONIG_OPTION_NONE);
            if(r2 >= 0) {
                *rcode = 0;

                if(region->num_regs == 1) {
                    /// マッチしなかった文字列 ///
                    int len = region->beg[0]-(p-str2)+ 1;
                    char* tmp = MALLOC(len);

                    memcpy(tmp, p, region->beg[0]-(p-str2));
                    tmp[len-1] = 0;

                    if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                    {
                        err_msg("signal interrupt", sname, sline);
                        FREE(tmp);
                        string_delete(str);
                        onig_region_free(region, 1);
                        return FALSE;
                    }

                    FREE(tmp);
                }
                else {
                    /// マッチしなかった文字列 ///
                    int len = region->beg[0]-(p-str2)+ 1;
                    char* tmp = MALLOC(len);

                    memcpy(tmp, p, region->beg[0]-(p-str2));
                    tmp[len-1] = 0;

                    if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                    {
                        err_msg("signal interrupt", sname, sline);
                        FREE(tmp);
                        string_delete(str);
                        onig_region_free(region, 1);
                        return FALSE;
                    }

                    FREE(tmp);


                    /// グループ化 ///
                    int i;
                    int p2 = region->beg[0];
                    for (i=1; i<region->num_regs; i++) {
                        if(region->beg[i] >= 0) {
                            int len = region->beg[i]-p2 + 1;
                            char* tmp = MALLOC(len);
                            memcpy(tmp, str2 + p2, region->beg[i]-p2);
                            tmp[len-1] = 0;

                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(tmp);
                                string_delete(str);
                                onig_region_free(region, 1);
                                return FALSE;
                            }

                            FREE(tmp);

                            p2 = region->end[i];
                        }
                    }

                    len = region->end[0]-p2 + 1;
                    tmp = MALLOC(len);
                    memcpy(tmp, str2 + p2, region->end[0]-p2);
                    tmp[len-1] = 0;

                    if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                    {
                        err_msg("signal interrupt", sname, sline);
                        FREE(tmp);
                        string_delete(str);
                        onig_region_free(region, 1);
                        return FALSE;
                    }

                    FREE(tmp);
                }

                match = region->end[0];
                p = str2 + region->end[0];
            }
            else {
                if(match >= 0) {
                    char* tmp = MALLOC(strlen(str2) - match + 1);
                    strcpy(tmp, str2 + match);

                    if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                    {
                        err_msg("signal interrupt", sname, sline);
                        FREE(tmp);
                        string_delete(str);
                        onig_region_free(region, 1);
                        return FALSE;
                    }

                    FREE(tmp);
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, str2))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        onig_region_free(region, 1);
                        return FALSE;
                    }
                }

                onig_region_free(region, 1);
                break;
            }

            onig_region_free(region, 1);

            if(p >= str2 + strlen(str2)) {
                break;
            }
        }
    }
    else {
        err_msg("erase: invalid regex", sname, sline);

        string_delete(str);

        return FALSE;
    } 

    string_delete(str);

    return TRUE;
}

BOOL statment_tree_internal_commands_erase_global_oneline(vector_obj* argv, BOOL input, sWFd* nextout, sRFd* nextin, int *rcode, BOOL ignore_case, BOOL multiline, char* sname, int sline)
{
    char* regex;

    if(vector_size(argv) == 2) {
        regex = string_c_str(vector_item(argv, 1));
    }
    else {
        err_msg("invalid erase arguments", sname, sline);
        return FALSE;
    }

    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                        |ONIG_OPTION_MULTILINE
                    , enc
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        |ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL_NG
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        while(1) {
            string_obj* str = STRING_NEW("");
            int ret = statment_tree_internal_commands_read_nextin_oneline
                    (nextin, str); 
            if(ret == 1) {
                string_delete(str);
                break;
            }
            else if(ret < 0)
            {
                string_delete(str);
                return FALSE;
            }

            char* str2 = string_c_str(str);
            char* p = str2;
            int match = -1;
            while(1) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    string_delete(str);
                    return FALSE;
                }
                OnigRegion* region = onig_region_new();

                int r2 = onig_search(reg, str2
                   , str2 + strlen(str2)
                   , p, p + strlen(p)
                   , region, ONIG_OPTION_NONE);
                if(r2 >= 0) {
                    *rcode = 0;

                    if(region->num_regs == 1) {
                        /// マッチしなかった文字列 ///
                        int len = region->beg[0]-(p-str2)+ 1;
                        char* tmp = MALLOC(len);

                        memcpy(tmp, p, region->beg[0]-(p-str2));
                        tmp[len-1] = 0;

                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }

                        FREE(tmp);
                    }
                    else {
                        /// マッチしなかった文字列 ///
                        int len = region->beg[0]-(p-str2)+ 1;
                        char* tmp = MALLOC(len);

                        memcpy(tmp, p, region->beg[0]-(p-str2));
                        tmp[len-1] = 0;

                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }

                        FREE(tmp);


                        /// グループ化 ///
                        int i;
                        int p2 = region->beg[0];
                        for (i=1; i<region->num_regs; i++) {
                            if(region->beg[i] >= 0) {
                                int len = region->beg[i]-p2 + 1;
                                char* tmp = MALLOC(len);
                                memcpy(tmp, str2 + p2, region->beg[i]-p2);
                                tmp[len-1] = 0;

                                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(tmp);
                                    string_delete(str);
                                    onig_region_free(region, 1);
                                    return FALSE;
                                }

                                FREE(tmp);

                                p2 = region->end[i];
                            }
                        }

                        len = region->end[0]-p2 + 1;
                        tmp = MALLOC(len);
                        memcpy(tmp, str2 + p2, region->end[0]-p2);
                        tmp[len-1] = 0;

                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }

                        FREE(tmp);
                    }

                    match = region->end[0];
                    p = str2 + region->end[0];
                }
                else {
                    if(match >= 0) {
                        char* tmp = MALLOC(strlen(str2) - match + 1);
                        strcpy(tmp, str2 + match);

                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            err_msg("signal interrupt", sname, sline);
                            FREE(tmp);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }

                        FREE(tmp);
                    }
                    else {
                        if(!statment_tree_internal_commands_write_nextout(nextout, str2))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            onig_region_free(region, 1);
                            return FALSE;
                        }
                    }

                    onig_region_free(region, 1);
                    break;
                }

                onig_region_free(region, 1);

                if(p >= str2 + strlen(str2)) {
                    break;
                }
            }

            string_delete(str);
        }
    }
    else {
        err_msg("erase: invalid regex", sname, sline);

        return FALSE;
    } 

    return TRUE;
}

BOOL statment_tree_internal_commands_split_oneline(char* regex, sRFd* nextin, sWFd* nextout, string_obj* field, BOOL multiline, BOOL ignore_case, int* rcode, char* sname, int sline)
{
    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    char* regex2 = regex;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        OnigErrorInfo err_info;
        r = onig_new(&reg, regex2
                , regex2 + strlen((char*)regex2)
                , ONIG_OPTION_DEFAULT
                , enc
                //, ONIG_SYNTAX_RUBY
                , ONIG_SYNTAX_PERL
                //, ONIG_SYNTAX_DEFAULT
                , &err_info);

        if(r == ONIG_NORMAL) {
            hash_put(gRegexs, regex, reg);
        }
        else {
            onig_free(reg);
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        while(1) {
            string_obj* line = STRING_NEW("");
            int result = statment_tree_internal_commands_read_nextin_oneline(nextin, line);
            if(result < 0) {
                err_msg("interrupt", sname, sline);
                string_delete(line);
                return FALSE;
            }
            else if(result == 1) {
                *rcode = 1;
                string_delete(line);
                break;
            }
            
            string_chomp(line);

            char* p = string_c_str(line);
            string_obj* new_line = STRING_NEW("");

            while(*p) {
                char* target = p;
                OnigRegion* region = onig_region_new();
                int r2 = onig_search(reg, target
                   , target + strlen(target)
                   , target, target + strlen(target)
                   , region, ONIG_OPTION_NONE);

                if(region->num_regs > 0 && region->beg[0] == 0 && region->end[0] == 0) 
                {
                    /// マッチした文字列の前 ///
                    char* tmp = MALLOC(2);
                    memcpy(tmp, target, 1);
                    tmp[1] = 0;

                    string_put(new_line, tmp);

                    FREE(tmp);

                    /// マッチした文字列の後 ///
                    p = target + 1;

                    /// 出力 ///
                    if(p < p + strlen(p)) string_push_back(new_line, string_c_str(field));
                    if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                    {
                        string_delete(new_line);
                        string_delete(line);
                        err_msg("singal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                else if(region->num_regs > 0 && region->beg[0] >= 0 && region->end[0] > 0) 
                {
                    /// マッチした文字列の前 ///
                    char* tmp 
                        = MALLOC(region->beg[0] + 1);
                    memcpy(tmp, target, region->beg[0]);
                    tmp[region->beg[0]] = 0;

                    string_put(new_line, tmp);

                    FREE(tmp);

                    /// マッチした文字列の後 ///
                    p = target + region->end[0];

                    /// 出力 ///
                    // perl likeならこれが必要。あと上と下を
                    // 微調整
                    //if(string_c_str(new_line)[0] != 0) {
                        string_push_back(new_line, string_c_str(field));
                        if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                        {
                            string_delete(new_line);
                            string_delete(line);
                            err_msg("singal interrupt", sname, sline);
                            return FALSE;
                        }
                    //}
                }
                else {
                    string_put(new_line, p);
                    p = p + strlen(p);

                    /// 出力 ///
                    if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                    {
                        string_delete(new_line);
                        string_delete(line);
                        err_msg("singal interrupt", sname, sline);
                        return FALSE;
                    }
                }

                onig_region_free(region, 1);
            }
            string_delete(new_line);
            string_delete(line);

            if(!statment_tree_internal_commands_write_nextout(nextout,  "\n"))
            {
                err_msg("singal interrupt", sname, sline);
                return FALSE;
            }
        }
    }
    else {
        err_msg("split: invalid regex", sname, sline);

        return FALSE;
    }

    return TRUE;
}

BOOL statment_tree_internal_commands_split(string_obj* line, char* regex, sWFd* nextout, string_obj* field, BOOL multiline, BOOL ignore_case, char* sname, int sline)
{
    string_chomp(line);
    char* p = string_c_str(line);
    
    regex_t* reg;
    if(ignore_case && multiline) 
        reg = hash_item(gRegexsIM, regex);
    else if(ignore_case)
        reg = hash_item(gRegexsI, regex);
    else if(multiline) 
        reg = hash_item(gRegexsM, regex);
    else
        reg = hash_item(gRegexs, regex);

    int r;

    char* regex2 = regex;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        OnigErrorInfo err_info;
        r = onig_new(&reg, regex2
                , regex2 + strlen((char*)regex2)
                , ONIG_OPTION_DEFAULT
                , enc
                //, ONIG_SYNTAX_RUBY
                , ONIG_SYNTAX_PERL
                //, ONIG_SYNTAX_DEFAULT
                , &err_info);

        if(r == ONIG_NORMAL) {
            hash_put(gRegexs, regex, reg);
        }
        else {
            onig_free(reg);
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        string_obj* new_line = STRING_NEW("");
        while(*p) {
            char* target = p;
            OnigRegion* region = onig_region_new();
            int r2 = onig_search(reg, target
               , target + strlen(target)
               , target, target + strlen(target)
               , region, ONIG_OPTION_NONE);

            if(region->num_regs > 0 && region->beg[0] == 0 && region->end[0] == 0) 
            {
                /// マッチした文字列の前 ///
                char* tmp = MALLOC(2);
                memcpy(tmp, target, 1);
                tmp[1] = 0;

                string_put(new_line, tmp);

                FREE(tmp);

                /// マッチした文字列の後 ///
                p = target + 1;

                /// 出力 ///
                if(p < p + strlen(p)) string_push_back(new_line, string_c_str(field));
                if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                {
                    string_delete(new_line);
                    err_msg("singal interrupt", sname, sline);
                    return FALSE;
                }
            }
            else if(region->num_regs > 0 && region->beg[0] >= 0 && region->end[0] > 0) 
            {
                /// マッチした文字列の前 ///
                char* tmp 
                    = MALLOC(region->beg[0] + 1);
                memcpy(tmp, target, region->beg[0]);
                tmp[region->beg[0]] = 0;

                string_put(new_line, tmp);

                FREE(tmp);

                /// マッチした文字列の後 ///
                p = target + region->end[0];

                /// 出力 ///
                // perl likeならこれが必要。あと上と下を
                // 微調整
                //if(string_c_str(new_line)[0] != 0) {
                    string_push_back(new_line, string_c_str(field));
                    if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                    {
                        string_delete(new_line);
                        err_msg("singal interrupt", sname, sline);
                        return FALSE;
                    }
                //}
            }
            else {
                string_put(new_line, p);
                p = p + strlen(p);

                /// 出力 ///
                if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                {
                    string_delete(new_line);
                    err_msg("singal interrupt", sname, sline);
                    return FALSE;
                }
            }

            onig_region_free(region, 1);
        }
        string_delete(new_line);
    }
    else {
        err_msg("split: invalid regex", sname, sline);

        return FALSE;
    }

    return TRUE;
}

BOOL statment_tree_internal_commands_sub(string_obj* str, char* regex, char* destination, BOOL global, BOOL quiet, BOOL multiline, BOOL check, BOOL ignore_case, int* rcode, sWFd* nextout, char* sname, int sline)
{
    BOOL check_all = FALSE;
    BOOL check_all_no = FALSE;
    int hennkann_kaisuu = 0;

    int i;

    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) {
        reg = hash_item(gRegexsIM, regex);
    }
    else if(ignore_case) {
        reg = hash_item(gRegexsI, regex);
    }
    else if(multiline) {
        reg = hash_item(gRegexsM, regex);
    }
    else {
        reg = hash_item(gRegexs, regex);
    }
    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        | ONIG_OPTION_IGNORECASE
                        | ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        | ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        | ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    char* str2 = string_c_str(str);
    if(r == ONIG_NORMAL) {
        if(global) {
            int point = 0;
            string_obj* hennkan_mojiretu 
                            = STRING_NEW("");
            while(1) {
                if(gKitutukiSigInt) {
                    break;
                }

                OnigRegion* region = onig_region_new();
                OnigErrorInfo err_info;
                int r2 = onig_search(reg, str2
                   , str2 + strlen(str2)
                   , str2 + point
                   , str2 + strlen(str2)
                   , region, ONIG_OPTION_NONE);

                if(r2 == ONIG_MISMATCH) {
                    //point++;
                    onig_region_free(region, 1);
                    break;
                }
                else {
//printf("beg %d end %d\n", region->beg[0], region->end[0]);
                    /// チェック ///
                    BOOL quetion;

                    if(check_all_no) {
                        quetion = FALSE;
                    }
                    else if(check_all) {
                        quetion = TRUE;
                    }
                    else if(check) 
                    {
                        msave_ttysettings();
                        msave_screen();
                        minitscr();

                        mclear();

                        /// 変換場所から行前を表示 ///
                        int count = 0;
                        char* p = str2+region->beg[0];
                        while(p > str2) {
                            if(is_line_field2(gLineField, p)) {
                                if(gLineField == kCRLF) {
                                    count++;
                                    if(count == 6) {
                                        p+=2;
                                        break;
                                    }
                                }
                                else {
                                    count++;
                                    if(count == 6) {
                                        p++;
                                        break;
                                    }
                                }

                                p--;
                            }
                            else {
                                p--;
                            }
                        }

                        char* tmp = MALLOC(str2+region->beg[0]-p+1);
                        memcpy(tmp, p, str2+region->beg[0]-p);
                        tmp[str2+region->beg[0]-p] = 0;

                        mprintw(tmp);
                        FREE(tmp);
                        

                        /// 変換文字列を表示 ///
                        tmp = MALLOC(region->end[0]-region->beg[0]+1);
                        memcpy(tmp, str2 + region->beg[0], region->end[0]-region->beg[0]);
                        tmp[region->end[0]-region->beg[0]] = 0;

                        mattron(kCAReverse);
                        mprintw(tmp);
                        mattroff();

                        FREE(tmp);

                        /// 変換場所から行後を表示 ///
                        count = 0;
                        p = str2+region->end[0];
                        while(p < str2+strlen(str2)) {
                            if(is_line_field2(gLineField, p)) {
                                if(gLineField == kCRLF) {
                                    count++;
                                    if(count == 6) {
                                        p-=2;
                                        break;
                                    }
                                }
                                else {
                                    count++;
                                    if(count == 6) {
                                        p--;
                                        break;
                                    }
                                }

                                p++;
                            }
                            else {
                                p++;
                            }
                        }

                        tmp = MALLOC(p-str2-region->end[0]+1);
                        memcpy(tmp, str2+region->end[0], p-str2-region->end[0]);
                        tmp[p-str2-region->end[0]] = 0;

                        mprintw(tmp);
                        FREE(tmp);

                        mprintw("\n");
                        mprintw("sub ok? (y)es/(n)o/(a)ll yes/(A)ll no");
                        mrefresh();

                        int meta;
                        int key = mgetch(&meta);
                        
                        mendwin();
                        mrestore_ttysettings();
                        mrestore_screen();
    
                        if(key == 3) {
                            break;
                        }
                        quetion = key == 'y' || key == 'a';
                        check_all = key == 'a';
                        check_all_no = key == 'A';
                    }
                    else {
                        quetion = TRUE;
                    }

                    /// グループ化したものを
                    /// ローカル変数に代入
                    int i;
                    for (i=1; i<region->num_regs; i++)
                    {
                        char* tmp =
                            MALLOC(region->end[i]
                                -region->beg[i]+1);

                        memcpy(tmp
                         , str2 + region->beg[i]
                         , region->end[i]-region->beg[i]);

                        tmp[region->end[i]
                            - region->beg[i]] = 0;

                        char name[16];
                        snprintf(name, 16, "%d", i);

                        saphire_set_local_var(name, tmp);

                        FREE(tmp);
                    }
                    /// グループ化したものを
                    ///  $1 $2などに変換 ///
                    if(point == 0) {
                        char* p = destination;

                        while(*p) {
                            if(*p == '$' 
                                && *(p+1) == '$')
                            {
                                p++;
                                string_push_back2(
                                    hennkan_mojiretu
                                        , *p++);
                            }
                            else if(*p == '$' 
                                && *(p+1) >= '1'
                                && *(p+1) <= '9')
                            {
                                char tmp[BUFSIZ];
                                int n = *(p+1) - '0';

                                if(n < region->num_regs) 
                                {
                                    p+=2;
                                    memcpy(tmp
                                        , str2 
                                        + region->beg[n]
                                        , region->end[n]
                                        -region->beg[n]);

                                    tmp[region->end[n]
                                      - region->beg[n]] 
                                        = 0;

                                    char* p3 = tmp;
                                    while(*p3) {
                                       string_push_back2(
                                        hennkan_mojiretu
                                         , *p3++);
                                    }
                                }
                                else {
                                    string_push_back2(
                                        hennkan_mojiretu
                                        , *p++);
                                    string_push_back2(
                                        hennkan_mojiretu
                                        , *p++);
                                }
                            }
                            else { 
                                string_push_back2(
                                    hennkan_mojiretu
                                        , *p++);
                            }
                        }
                    }

                    /// 変換開始 ///
                    if(quetion) {
                        /// マッチした文字列の前 ///
                        char* tmp = MALLOC(
                            region->beg[0] - point + 1);

                        memcpy(tmp, str2 + point
                            , region->beg[0] - point);
                        tmp[region->beg[0] - point] = 0;
                        if(!quiet) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                string_delete(hennkan_mojiretu);
                                return FALSE;
                            }
                        }
                        FREE(tmp);

                        /// 変換文字列を入れる ///
                        if(!quiet) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(hennkan_mojiretu)))
                            {
                                string_delete(hennkan_mojiretu);
                                return FALSE;
                            }
                        }

                        if(region->beg[0] == region->end[0]) 
                        {
                            char buf[2];
                            buf[0] = str2[region->beg[0]];
                            buf[1] = 0;
                            if(!quiet) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                {
                                    string_delete(hennkan_mojiretu);
                                    return FALSE;
                                }
                            }
                            point = region->end[0] + 1;
                        }
                        else {
                            point = region->end[0];
                        }

                        hennkann_kaisuu++;
                    }
                    else {
                        /// マッチした文字列の前 ///
                        char* tmp = MALLOC(
                            region->beg[0] - point + 1);

                        memcpy(tmp, str2 + point
                            , region->beg[0] - point);
                        tmp[region->beg[0] - point] = 0;
                        if(!quiet) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                string_delete(hennkan_mojiretu);
                                return FALSE;
                            }
                        }

                        FREE(tmp);

                        if(region->beg[0] == region->end[0]) {
                            char buf[2];
                            buf[0] = str2[region->beg[0]];
                            buf[1] = 0;
                            if(!quiet) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                {
                                    string_delete(hennkan_mojiretu);
                                    return FALSE;
                                }
                            }
                            point = region->end[0] + 1;
                        }
                        else {
                            char* tmp = MALLOC(region->end[0] - region->beg[0] + 1);
                            memcpy(tmp, str2 + region->beg[0], region->end[0]-region->beg[0]);
                            tmp[region->end[0]-region->beg[0]] = 0;
                            if(!quiet) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                                {
                                    string_delete(hennkan_mojiretu);
                                    return FALSE;
                                }
                            }

                            point = region->end[0];

                            FREE(tmp);
                        }
                    }

                    onig_region_free(region, 1);

                    if(str2 + point>=str2 + strlen(str2)) 
                    {
                        break;
                    }
                }
            }

            if(point < strlen(str2)) {
                if(!quiet) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, str2 + point))
                    {
                        string_delete(hennkan_mojiretu);
                        return FALSE;
                    }
                }
            }

            string_delete(hennkan_mojiretu);
        }
        else {
            OnigRegion* region = onig_region_new();

            int r2 = onig_search(reg, str2
               , str2 + strlen((char*)str2)
               , str2, str2 + strlen((char*)str2)
               , region, ONIG_OPTION_NONE);

            if(r2 >= 0) {
                /// チェック ///
                BOOL quetion;

                if(check_all_no) {
                    quetion = FALSE;
                }
                else if(check_all) {
                    quetion = TRUE;
                }
                else if(check) 
                {
                    msave_ttysettings();
                    msave_screen();
                    minitscr();

                    mclear();

                    /// 変換場所から行前を表示 ///
                    int count = 0;
                    char* p = str2+region->beg[0];
                    while(p > str2) {
                        if(is_line_field2(gLineField, p)) {
                            if(gLineField == kCRLF) {
                                count++;
                                if(count == 6) {
                                    p+=2;
                                    break;
                                }
                            }
                            else {
                                count++;
                                if(count == 6) {
                                    p++;
                                    break;
                                }
                            }

                            p--;
                        }
                        else {
                            p--;
                        }
                    }

                    char* tmp = MALLOC(str2+region->beg[0]-p+1);
                    memcpy(tmp, p, str2+region->beg[0]-p);
                    tmp[str2+region->beg[0]-p] = 0;

                    mprintw(tmp);
                    FREE(tmp);
                    

                    /// 変換文字列を表示 ///
                    tmp = MALLOC(region->end[0]-region->beg[0]+1);
                    memcpy(tmp, str2 + region->beg[0], region->end[0]-region->beg[0]);
                    tmp[region->end[0]-region->beg[0]] = 0;

                    mattron(kCAReverse);
                    mprintw(tmp);
                    mattroff();

                    FREE(tmp);

                    /// 変換場所から行後を表示 ///
                    count = 0;
                    p = str2+region->end[0];
                    while(p < str2+strlen(str2)) {
                        if(is_line_field2(gLineField, p)) {
                            if(gLineField == kCRLF) {
                                count++;
                                if(count == 6) {
                                    p-=2;
                                    break;
                                }
                            }
                            else {
                                count++;
                                if(count == 6) {
                                    p--;
                                    break;
                                }
                            }

                            p++;
                        }
                        else {
                            p++;
                        }
                    }

                    tmp = MALLOC(p-str2-region->end[0]+1);
                    memcpy(tmp, str2+region->end[0], p-str2-region->end[0]);
                    tmp[p-str2-region->end[0]] = 0;

                    mprintw(tmp);
                    FREE(tmp);

                    mprintw("\n");
                    mprintw("sub ok? (y)es/(n)o/(a)ll yes/(A)ll no");
                    mrefresh();

                    int meta;
                    int key = mgetch(&meta);
                    
                    mendwin();
                    mrestore_ttysettings();
                    mrestore_screen();

                    if(key == 3) {
                        onig_region_free(region, 1);
                        return TRUE;
                    }

                    quetion = key == 'y' || key == 'a';
                    check_all = key == 'a';
                    check_all_no = key == 'A';
                }
                else {
                    quetion = TRUE;
                }

                /// グループ化したものを
                /// ローカル変数に代入
                int i;
                for (i=1; i<region->num_regs; i++) {
                    char* tmp =
                        MALLOC(region->end[i]-region->beg[i]+1);

                    memcpy(tmp, str2 + region->beg[i]
                     , region->end[i]-region->beg[i]);

                    tmp[region->end[i]
                        - region->beg[i]] = 0;

                    char name[16];
                    snprintf(name, 16, "%d", i);

                    saphire_set_local_var(name, tmp);

                    FREE(tmp);
                }

                /// グループ化したものを
                /// $1 $2などに変換 ///
                
                char* p = destination;

                string_obj* hennkan_mojiretu
                    = STRING_NEW("");
                while(*p) {
                    if(*p == '$' && *(p+1) == '$') {
                        p++;
                        string_push_back2(
                            hennkan_mojiretu, *p++);
                    }
                    else if(*p == '$' 
                       && *(p+1) >= '1' && *(p+1) <= '9')
                    {
                        int n = *(p+1) - '0';
                        if(n < region->num_regs) {
                            p+=2;
                            char* tmp = MALLOC(
                          region->end[n]-region->beg[n]+1);

                            memcpy(tmp
                                , str2 + region->beg[n]
                                , region->end[n]
                                    -region->beg[n]);

                            tmp[region->end[n]
                                - region->beg[n]] = 0;
                            char* p3 = tmp;
                            while(*p3) {
                                string_push_back2(
                                    hennkan_mojiretu
                                    , *p3++);
                            }

                            FREE(tmp);
                        }
                        else {
                            string_push_back2(
                                hennkan_mojiretu, *p++);
                            string_push_back2(
                                hennkan_mojiretu, *p++);
                        }
                    }
                    else { 
                        string_push_back2(
                            hennkan_mojiretu, *p++);
                    }
                }

                if(quetion) {
                    char* tmp = MALLOC(region->beg[0] + 1);

                    memcpy(tmp, str2, region->beg[0]);
                    tmp[region->beg[0]] = 0;
                    if(!quiet) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                        {
                            string_delete(hennkan_mojiretu);
                            onig_region_free(region, 1);
                            return FALSE;
                        }
                    }

                    if(!quiet) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(hennkan_mojiretu)))
                        {
                            string_delete(hennkan_mojiretu);
                            onig_region_free(region, 1);
                            return FALSE;
                        }
                    }

                    if(!quiet) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, str2 + region->end[0]))
                        {
                            string_delete(hennkan_mojiretu);
                            onig_region_free(region, 1);
                            return FALSE;
                        }
                    }

                    hennkann_kaisuu++;

                    FREE(tmp);
                }
                else {
                    if(!quiet) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, str2))
                        {
                            string_delete(hennkan_mojiretu);
                            onig_region_free(region, 1);
                            return FALSE;
                        }
                    }
                }

                string_delete(hennkan_mojiretu);
            }
            else {
                if(!quiet) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, str2))
                    {
                        onig_region_free(region, 1);
                        return FALSE;
                    }
                }
            }

            onig_region_free(region, 1);
        }
    }

    /// 出力 ///

    char buf[128];
    snprintf(buf, 128, "%d", hennkann_kaisuu);
    saphire_set_local_var("SUB_COUNT", buf);

    if(hennkann_kaisuu > 0) {
        *rcode = 0;
    }

    return TRUE;
}

BOOL statment_tree_internal_commands_sub_global_oneline(char* regex, char* destination, BOOL global, BOOL quiet, BOOL multiline, BOOL check, BOOL ignore_case, int* rcode, sWFd* nextout, sRFd* nextin, char* sname, int sline)
{
    BOOL check_all = FALSE;
    BOOL check_all_no = FALSE;
    int hennkann_kaisuu = 0;

    int i;

    char* regex2 = regex;

    regex_t* reg;
    if(ignore_case && multiline) {
        reg = hash_item(gRegexsIM, regex);
    }
    else if(ignore_case) {
        reg = hash_item(gRegexsI, regex);
    }
    else if(multiline) {
        reg = hash_item(gRegexsM, regex);
    }
    else {
        reg = hash_item(gRegexs, regex);
    }
    int r;

    if(reg == NULL) {
        OnigEncoding enc;
        if(gKanjiCode == kUtf8) {
            enc = ONIG_ENCODING_UTF8;
        }
        else if(gKanjiCode == kEucjp) {
            enc = ONIG_ENCODING_EUC_JP;
        }
        else {
            enc = ONIG_ENCODING_SJIS;
        }

        if(ignore_case && multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        | ONIG_OPTION_IGNORECASE
                        | ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsIM, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
        else if(ignore_case) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        | ONIG_OPTION_IGNORECASE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsI, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
        else if(multiline) {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                        | ONIG_OPTION_MULTILINE
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexsM, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
        else {
            OnigErrorInfo err_info;
            r = onig_new(&reg, regex2
                    , regex2 + strlen((char*)regex2)
                    , ONIG_OPTION_DEFAULT
                    , enc
                    //, ONIG_SYNTAX_RUBY
                    //, ONIG_SYNTAX_PERL
                    , ONIG_SYNTAX_DEFAULT
                    , &err_info);

            if(r == ONIG_NORMAL) {
                hash_put(gRegexs, regex, reg);
            }
            else {
                onig_free(reg);

                err_msg("sub: invalid regex", sname, sline);
                return FALSE;
            }
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        while(1) {
            string_obj* str = STRING_NEW("");
            int r = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
            if(r < 0) {
                err_msg("interrupt", sname, sline);
                string_delete(str);
                return FALSE;
            }
            else if(r == 1) {
                string_delete(str);
                break;
            }

            char* str2 = string_c_str(str);
            int point = 0;
            string_obj* hennkan_mojiretu 
                            = STRING_NEW("");
            while(1) {
                if(gKitutukiSigInt) {
                    err_msg("interrupt", sname, sline);
                    gKitutukiSigInt = FALSE;
                    string_delete(str);
                    string_delete(hennkan_mojiretu);
                    return FALSE;
                }

                OnigRegion* region = onig_region_new();
                OnigErrorInfo err_info;
                int r2 = onig_search(reg, str2
                   , str2 + strlen(str2)
                   , str2 + point
                   , str2 + strlen(str2)
                   , region, ONIG_OPTION_NONE);

                if(r2 == ONIG_MISMATCH) {
                    //point++;
                    onig_region_free(region, 1);
                    break;
                }
                else {
    //printf("beg %d end %d\n", region->beg[0], region->end[0]);
                    /// チェック ///
                    BOOL quetion;

                    if(check_all_no) {
                        quetion = FALSE;
                    }
                    else if(check_all) {
                        quetion = TRUE;
                    }
                    else if(check) 
                    {
                        msave_ttysettings();
                        msave_screen();
                        minitscr();

                        mclear();

                        /// 変換場所から行前を表示 ///
                        int count = 0;
                        char* p = str2+region->beg[0];
                        while(p > str2) {
                            if(is_line_field2(gLineField, p)) {
                                if(gLineField == kCRLF) {
                                    count++;
                                    if(count == 6) {
                                        p+=2;
                                        break;
                                    }
                                }
                                else {
                                    count++;
                                    if(count == 6) {
                                        p++;
                                        break;
                                    }
                                }

                                p--;
                            }
                            else {
                                p--;
                            }
                        }

                        char* tmp = MALLOC(str2+region->beg[0]-p+1);
                        memcpy(tmp, p, str2+region->beg[0]-p);
                        tmp[str2+region->beg[0]-p] = 0;

                        mprintw(tmp);
                        FREE(tmp);
                        

                        /// 変換文字列を表示 ///
                        tmp = MALLOC(region->end[0]-region->beg[0]+1);
                        memcpy(tmp, str2 + region->beg[0], region->end[0]-region->beg[0]);
                        tmp[region->end[0]-region->beg[0]] = 0;

                        mattron(kCAReverse);
                        mprintw(tmp);
                        mattroff();

                        FREE(tmp);

                        /// 変換場所から行後を表示 ///
                        count = 0;
                        p = str2+region->end[0];
                        while(p < str2+strlen(str2)) {
                            if(is_line_field2(gLineField, p)) {
                                if(gLineField == kCRLF) {
                                    count++;
                                    if(count == 6) {
                                        p-=2;
                                        break;
                                    }
                                }
                                else {
                                    count++;
                                    if(count == 6) {
                                        p--;
                                        break;
                                    }
                                }

                                p++;
                            }
                            else {
                                p++;
                            }
                        }

                        tmp = MALLOC(p-str2-region->end[0]+1);
                        memcpy(tmp, str2+region->end[0], p-str2-region->end[0]);
                        tmp[p-str2-region->end[0]] = 0;

                        mprintw(tmp);
                        FREE(tmp);

                        mprintw("\n");
                        mprintw("sub ok? (y)es/(n)o/(a)ll yes/(A)ll no");
                        mrefresh();

                        int meta;
                        int key = mgetch(&meta);
                        
                        mendwin();
                        mrestore_ttysettings();
                        mrestore_screen();

                        if(key == 3) {
                            break;
                        }
                        quetion = key == 'y' || key == 'a';
                        check_all = key == 'a';
                        check_all_no = key == 'A';
                    }
                    else {
                        quetion = TRUE;
                    }

                    /// グループ化したものを
                    /// ローカル変数に代入
                    int i;
                    for (i=1; i<region->num_regs; i++)
                    {
                        char* tmp =
                            MALLOC(region->end[i]
                                -region->beg[i]+1);

                        memcpy(tmp
                         , str2 + region->beg[i]
                         , region->end[i]-region->beg[i]);

                        tmp[region->end[i]
                            - region->beg[i]] = 0;

                        char name[16];
                        snprintf(name, 16, "%d", i);

                        saphire_set_local_var(name, tmp);

                        FREE(tmp);
                    }
                    /// グループ化したものを
                    ///  $1 $2などに変換 ///
                    if(point == 0) {
                        char* p = destination;

                        while(*p) {
                            if(*p == '$' 
                                && *(p+1) == '$')
                            {
                                p++;
                                string_push_back2(
                                    hennkan_mojiretu
                                        , *p++);
                            }
                            else if(*p == '$' 
                                && *(p+1) >= '1'
                                && *(p+1) <= '9')
                            {
                                char tmp[BUFSIZ];
                                int n = *(p+1) - '0';

                                if(n < region->num_regs) 
                                {
                                    p+=2;
                                    memcpy(tmp
                                        , str2 
                                        + region->beg[n]
                                        , region->end[n]
                                        -region->beg[n]);

                                    tmp[region->end[n]
                                      - region->beg[n]] 
                                        = 0;

                                    char* p3 = tmp;
                                    while(*p3) {
                                       string_push_back2(
                                        hennkan_mojiretu
                                         , *p3++);
                                    }
                                }
                                else {
                                    string_push_back2(
                                        hennkan_mojiretu
                                        , *p++);
                                    string_push_back2(
                                        hennkan_mojiretu
                                        , *p++);
                                }
                            }
                            else { 
                                string_push_back2(
                                    hennkan_mojiretu
                                        , *p++);
                            }
                        }
                    }

                    /// 変換開始 ///
                    if(quetion) {
                        /// マッチした文字列の前 ///
                        char* tmp = MALLOC(
                            region->beg[0] - point + 1);

                        memcpy(tmp, str2 + point
                            , region->beg[0] - point);
                        tmp[region->beg[0] - point] = 0;
                        if(!quiet) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                onig_region_free(region, 1);
                                string_delete(str);
                                string_delete(hennkan_mojiretu);
                                return FALSE;
                            }
                        }
                        FREE(tmp);

                        /// 変換文字列を入れる ///
                        if(!quiet) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(hennkan_mojiretu)))
                            {
                                string_delete(str);
                                string_delete(hennkan_mojiretu);
                                onig_region_free(region, 1);
                                return FALSE;
                            }
                        }

                        if(region->beg[0] == region->end[0]) 
                        {
                            char buf[2];
                            buf[0] = str2[region->beg[0]];
                            buf[1] = 0;
                            if(!quiet) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                {
                                    string_delete(str);
                                    onig_region_free(region, 1);
                                    string_delete(hennkan_mojiretu);
                                    return FALSE;
                                }
                            }
                            point = region->end[0] + 1;
                        }
                        else {
                            point = region->end[0];
                        }

                        hennkann_kaisuu++;
                    }
                    else {
                        /// マッチした文字列の前 ///
                        char* tmp = MALLOC(
                            region->beg[0] - point + 1);

                        memcpy(tmp, str2 + point
                            , region->beg[0] - point);
                        tmp[region->beg[0] - point] = 0;
                        if(!quiet) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                string_delete(str);
                                onig_region_free(region, 1);
                                string_delete(hennkan_mojiretu);
                                return FALSE;
                            }
                        }

                        FREE(tmp);

                        if(region->beg[0] == region->end[0]) {
                            char buf[2];
                            buf[0] = str2[region->beg[0]];
                            buf[1] = 0;
                            if(!quiet) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                {
                                    string_delete(str);
                                    string_delete(hennkan_mojiretu);
                                    onig_region_free(region, 1);
                                    return FALSE;
                                }
                            }

                            point = region->end[0] + 1;
                        }
                        else {
                            char* tmp = MALLOC(region->end[0] - region->beg[0] + 1);
                            memcpy(tmp, str2 + region->beg[0], region->end[0]-region->beg[0]);
                            tmp[region->end[0]-region->beg[0]] = 0;
                            if(!quiet) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                                {
                                    onig_region_free(region, 1);
                                    string_delete(str);
                                    string_delete(hennkan_mojiretu);
                                    return FALSE;
                                }
                            }

                            point = region->end[0];

                            FREE(tmp);
                        }
                    }

                    onig_region_free(region, 1);

                    if(str2 + point>=str2 + strlen(str2)) 
                    {
                        break;
                    }
                }
            }

            if(point < strlen(str2)) {
                if(!quiet) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, str2 + point))
                    {
                        string_delete(str);
                        string_delete(hennkan_mojiretu);
                        return FALSE;
                    }
                }
            }

            string_delete(hennkan_mojiretu);

            if(gKitutukiSigInt) {
                err_msg("interrupt", sname, sline);
                gKitutukiSigInt = FALSE;
                string_delete(str);
                return FALSE;
            }

            string_delete(str);
        }
    }

    char buf[128];
    snprintf(buf, 128, "%d", hennkann_kaisuu);
    saphire_set_local_var("SUB_COUNT", buf);

    if(hennkann_kaisuu > 0) {
        *rcode = 0;
    }

    return TRUE;
}

int str_lflen(enum eLineField lf, char* str)
{
    char* p = str;
    int result = 0;

    if(lf == kCR) {
        while(*p) {
            if(*p == '\r') {
                result++;
                p++;
            }
            else {
                p++;
            }
        }
    }
    else if(lf == kLF) {
        while(*p) {
            if(*p == '\n') {
                result++;
                p++;
            }
            else {
                p++;
            }
        }
    }
    else {
        while(*p) {
            if(*p == '\r' && *(p+1) == '\n') {
                result++;
                p+=2;
            }
            else {
                p++;
            }
        }
    }

    return result;
}

// TRUE 正常終了
// FALSE エラー
BOOL statment_tree_internal_commands(sCommand* command, int* rcode, vector_obj* argv, vector_obj* blocks, vector_obj* parent_blocks, sWFd* nextout, sRFd* nextin, int nexterr, int j, vector_obj* cprogs, char* title, BOOL input, BOOL* return_, char* sname, int sline)
{
    /// go ///
    switch(command->mKind) {
        case kSubshell: {
            if(command->mExtra) {
                sSubshell* s = (sSubshell*) command->mExtra;
                *rcode = run(s->contents, title, nextout
                                , nextin, nexterr
                                , FALSE, FALSE
                                , return_, parent_blocks);
                if(*return_) {
                    return TRUE;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
            }
            }
            break;

        case kMSleep: {
            *rcode = 0;

            /// 出力 ///
            char c [] = {'.', 'o', 'O'};
            int n = atoi(string_c_str(vector_item(argv, 1)));
            int i;
            for(i=0; i < n*5; i++) {
                char buf[32];
                snprintf(buf, 32, "%c\033[D", c[i % 3]);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf)) 
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }

                if(gKitutukiSigInt) {
                    err_msg("interrupt", sname, sline);
                    gKitutukiSigInt = FALSE;
                    return FALSE;
                }

                usleep(200000);
            }

            }
            break;

        case kTrue: 
            *rcode = 0;
            break;

        case kFalse:
            break;

        case kTest: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));

                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str;

            /// 標準入力で受け取る場合は第1引数に代入 ///
            if(input) {
                str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str) ;
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    *rcode = 2;
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
            }
            /// 違う場合はstr = NULL
            else {
                str = NULL;
            }

            if(!statment_tree_internal_commands_test(rcode, argv, str, sname, sline)) {
                if(str) string_delete(str);
                return FALSE;
            }

            if(str) string_delete(str);
            }
            break;

        case kIndex: {
            BOOL quiet = FALSE;
            BOOL new_line = TRUE;
            BOOL flg_n = FALSE;
            int n;
            BOOL flg_c = FALSE;
            int c;
            enum eKanjiCode code = gKanjiCode;
            BOOL byte = FALSE;
            BOOL terminal = FALSE;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-nl") == 0) {
                    new_line = FALSE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-t") == 0) {
                    terminal = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0 && l+1 < vector_size(argv)) {
                    flg_c = TRUE;
                    c = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            char* word;
            char* target;
            int start;
            int match_count;
            string_obj* str = STRING_NEW("");

            if(input) {
                if(vector_size(argv) != 2) {
                    err_msg("index -I string", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                /// 文字列 ///
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    string_delete(str);
                    *rcode = 1;
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                target = string_c_str(str);

                /// 検索文字列 ///
                word = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(
                        code, target)) 
                    {
                        err_msg("index -> invalid -n range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = 0;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }
            else {
                if(vector_size(argv) != 3) {
                    err_msg("index -> invalid option number", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                /// 検索文字列 ///
                word = string_c_str(vector_item(argv, 2));

                /// 文字列 ///
                target = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(
                            code, target)) 
                    {
                        err_msg("index -> invalid -n range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = 0;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }

            char* start_byte = str_kanjipos2pointer(code, target, start);
            int count = match_count;
            char* p = start_byte;
            char* result = NULL;
            while(p < start_byte + strlen(start_byte)) {
                result = strstr(p, word);
                if(result) {
                    count--;
                    if(count == 0) {
                        break;
                    }
                    p = result+strlen(word);
                }
                else {
                    break;
                }
            }

            char msg[64];
            if(result == NULL || count !=0) {
                snprintf(msg, 64, "-1");
            }
            else {
                int c;
                if(byte) {
                    c = result - target;
                }
                else if(terminal) {
                    int pos = str_pointer2kanjipos(code, target, result);
                    c = str_termlen2(code, target, pos);
                }
                else {
                    c = str_pointer2kanjipos(code, target, result);
                }
                snprintf(msg, 64, "%d", c);
                *rcode = 0;
            }

            /// 出力 ///
            if(quiet == FALSE) {
                if(!statment_tree_internal_commands_write_nextout(nextout, msg))
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                if(new_line) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
            }

            string_delete(str);
            }
            break;

        case kRIndex: {
            BOOL quiet = FALSE;
            BOOL new_line = TRUE;
            BOOL flg_n = FALSE;
            int n;
            BOOL flg_c = FALSE;
            BOOL byte = FALSE;
            BOOL terminal = FALSE;
            int c;
            enum eKanjiCode code = gKanjiCode;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-nl") == 0) {
                    new_line = FALSE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-t") == 0) {
                    terminal = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0 && l+1 < vector_size(argv)) {
                    flg_c = TRUE;
                    c = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            char* word;
            char* target;
            int start;
            int match_count;
            string_obj* str = STRING_NEW("");

            if(input) {
                if(vector_size(argv) != 2) {
                    err_msg("index -I string", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                /// 文字列 ///
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("intterrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                target = string_c_str(str);

                /// 検索文字列 ///
                word = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(code, target)) 
                    {
                        err_msg("index -> invalid -n range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = str_kanjilen(code, target) -1;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }
            else {
                if(vector_size(argv) != 3) {
                    err_msg("index -> invalid option number", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                /// 文字列 ///
                word = string_c_str(vector_item(argv, 2));

                /// 検索文字列 ///
                target = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(
                                code, target)) 
                    {
                        err_msg("index -> invalid -n range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = str_kanjilen(code, target) -1;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }
            
            int count = match_count;

            char* start_byte = str_kanjipos2pointer(code
                    , target, start+1);
            char* p = start_byte;
            char* result = NULL;
            while(p>=target) {
                result = strstr_back(p, target, word, sname, sline);

                if(gKitutukiSigInt) {
                    err_msg("interrupt", sname, sline);
                    *rcode = 1;
                    string_delete(str);
                    return FALSE;
                }

                if(result != NULL) {
                    count--;
                    if(count == 0) {
                        break;
                    }
                    p = result - 1;
                }
                else {
                    break;
                }
            }

            char msg[64];
            if(result == NULL || count !=0) {
                snprintf(msg, 64, "-1");
            }
            else {
                int c;
                if(byte) {
                    c = result - target;
                }
                else if(terminal) {
                    int pos = str_pointer2kanjipos(code, target, result);
                    c = str_termlen2(code, target, pos);
                }
                else {
                    c = str_pointer2kanjipos(code, target, result);
                }
                snprintf(msg, 64, "%d", c);
                *rcode = 0;
            }

            if(quiet == FALSE) {
                if(!statment_tree_internal_commands_write_nextout(nextout, msg))
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                if(new_line) {
                   if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                   {
                       err_msg("signal interrupt", sname, sline);
                       string_delete(str);
                       return FALSE;
                   }
                }
            }

            string_delete(str);
            }
            break;

        case kLength: {
            int i;
            BOOL line_field = TRUE;
            enum eKanjiCode code = gKanjiCode;
            BOOL terminal = FALSE;
            BOOL byte = FALSE;
            BOOL line_field_count = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nl") == 0) {
                    line_field = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-t") == 0) {
                    terminal = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-L") == 0) {
                    line_field_count = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                            nextin, str);

                /// EOF
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int len;
                    if(terminal)
                        len = str_termlen(code, string_c_str(str));
                    else if(byte)
                        len = strlen(string_c_str(str));
                    else if(line_field_count)
                        len = str_lflen(gLineField, string_c_str(str));
                    else
                        len = str_kanjilen(code, string_c_str(str));

                    char buf[128];
                    if(line_field)
                        snprintf(buf, 128, "%d\n", len);
                    else
                        snprintf(buf, 128, "%d", len);

                    if(!statment_tree_internal_commands_write_nextout(nextout,  buf))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    string_delete(str);

                    *rcode = 0;
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    int len;
                    if(terminal)
                        len = str_termlen(code, string_c_str(vector_item(argv, 1)));
                    else if(byte) 
                        len = strlen(string_c_str(vector_item(argv, 1)));
                    else if(line_field_count)
                        len = str_lflen(gLineField, string_c_str(vector_item(argv, 1)));
                    else
                        len = str_kanjilen(code, string_c_str(vector_item(argv, 1)));
                    char buf[128];

                    if(line_field)
                        snprintf(buf, 128, "%d\n", len);
                    else
                        snprintf(buf, 128, "%d", len);

                    if(!statment_tree_internal_commands_write_nextout(nextout, buf)) 
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    *rcode = 0;
                }
            }
            }
            break;

        case kUc: {
            int i;
            BOOL line_field = FALSE;
            enum eKanjiCode code = gKanjiCode;

            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                                nextin, str);

                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    string_toupper(str, code);

                    /// 出力 ///
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str))) 
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    string_obj* str = STRING_NEW(
                        string_c_str(vector_item(argv, 1)));

                    string_toupper(str, code);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str))) 
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            }
            break;

        case kLc: {
            int i;
            BOOL line_field = FALSE;
            enum eKanjiCode code = gKanjiCode;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                    nextin, str);

                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    string_tolower(str, code);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    string_obj* str = STRING_NEW(
                        string_c_str(vector_item(argv, 1)));

                    string_tolower(str, code);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        string_delete(str);
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            }
            break;

        case kChomp: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
//puts("read_nextin start");
                int ret = statment_tree_internal_commands_read_nextin(
                        nextin, str);

                if(ret == 1) {
                    string_delete(str);
                    *rcode = 1;
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
//puts("read_nextin end");
                    if(string_chomp(str)) {
                        *rcode = 0;
                    }

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    string_delete(str);
                }
            }
            }
            break;

        case kSubstr: {
            int i;
            BOOL line_field = TRUE;
            BOOL byte = FALSE;
            BOOL terminal = FALSE;
            BOOL chomp = FALSE;
            enum eKanjiCode code = gKanjiCode;

            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nl") == 0) {
                    line_field = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0) {
                    chomp = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str = NULL;
            char* p;
            int index;
            int count = 1;
            if(input) {
                str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                        nextin, str) ;
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0) 
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                if(chomp) string_chomp(str);
                p = string_c_str(str);

                if(vector_size(argv) < 2) {
                    err_msg("substr: invalid arguments", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                index = atoi(string_c_str(vector_item(argv, 1)));

                if(vector_size(argv) >= 3) {
                    count = atoi(string_c_str(vector_item(argv, 2)));
                }
            }
            else {
                if(vector_size(argv) < 3) {
                    err_msg("substr: invalid arguments", sname, sline);
                    return FALSE;
                }

                p = string_c_str(vector_item(argv, 1));
                index = atoi(string_c_str(vector_item(argv, 2)));

                if(vector_size(argv) >= 4) {
                    count = atoi(string_c_str(vector_item(argv, 3)));
                }
            }

            if(index < 0) index += str_kanjilen(code, p);
            if(index < 0) index = 0;
            if(index >= str_kanjilen(code, p)) 
                index = str_kanjilen(code, p)-1;

            if(count <= 0) {
                count = 1;
            }
            if(index + count > strlen(p)) {
                count = strlen(p)-index;
            }

            char* p1;
            char* p2;
            if(byte) {
                p1 = p + index;
                p2 = p + index + count;
            }
            else {
                p1 = str_kanjipos2pointer(code, p, index);
                p2 = str_kanjipos2pointer(code, p, index + count);
            }


            char* buf = MALLOC(p2 -p1 + 1);
            memcpy(buf, p1, p2-p1);
            buf[p2-p1] = 0;

            /// 出力 ///
            if(!statment_tree_internal_commands_write_nextout(nextout, buf))
            {
                err_msg("signal interrupt", sname, sline);
                if(str)string_delete(str);
                FREE(buf);
                return FALSE;
            }
            if(line_field) {
                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                {
                    err_msg("signal interrupt", sname, sline);
                    FREE(buf);
                    if(str)string_delete(str);
                    return FALSE;
                }
            }

            FREE(buf);

            if(str) string_delete(str);

            *rcode = 0;
            }
            break;

        case kEval: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                        nextin, str) ;
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0) 
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);
                
                *rcode = saphire_shell(string_c_str(str), fname
                                , nextout, nextin, nexterr);
                string_delete(str);

                FREE(fname);
                if(*rcode == -1) {
                    return FALSE;
                }

                if(*rcode < 0) {
                    return FALSE;
                }
            }
            else if(vector_size(blocks) == 1) {
                sStatments* statments = vector_item(blocks, 0);

                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);
                
                *rcode = run(statments, fname, nextout
                            , nextin, nexterr
                            , FALSE, FALSE
                            , return_, parent_blocks);

                FREE(fname);
                if(*return_) {
                    return TRUE;
                }
                if(*rcode == -1) {
                    return FALSE;
                }

                if(*rcode < 0) {
                    return FALSE;
                }
            }
            }
            break;

        case kFg:
            if(gAppType != kATOptC && gJobControl) {
                if(vector_size(argv) >= 2) {
                    char* argv2 
                       = string_c_str((string_obj*)vector_item(argv, 1));
                    if(argv2[0] >= '1' && argv2[0] <= '9') {
                        int num = argv2[0] - '0';
                        if(!forground_job(num-1)) {
                            return FALSE;
                        }
                    }
                    else {
                        err_msg("invalid job number", sname, sline);
                        return FALSE;
                    }

                    *rcode = 0;
                }
                else {
                    if(!forground_job(0)) {
                        return FALSE;
                    }
                    *rcode = 0;
                }
            }
            break;

        case kBg:
            if(gAppType != kATOptC && gJobControl) {
                if(vector_size(argv) >= 2) {
                    char* argv2 = string_c_str((string_obj*)vector_item(argv, 1));
                    if(argv2[0] >= '1' && argv2[0] <= '9') {
                        int num = argv2[0] - '0';
                        background_job(num-1);
                    }

                    *rcode = 0;
                }
                else {
                    background_job(0);
                    *rcode = 0;
                }
            }
            break;

        case kCpg:
            if(vector_size(argv) == 3) {
                int job_id = atoi(string_c_str(vector_item(argv, 1))) -1;
                pid_t pgid = atoi(string_c_str(vector_item(argv, 2)));

                if(job_id >= 0 && job_id < vector_size(gJobs)) {
                    sJob* job = vector_item(gJobs, job_id);

                    job->mPGroup = pgid;
                    *rcode = 0;
                }
            }
            break;

        case kJobs:
            {
            int i;
            if(!statment_tree_internal_commands_write_nextout(nextout, "number name pgroup\n")) 
            {
                err_msg("signal interrupt", sname, sline);
                return FALSE;
            }
            for(i=0; i<vector_size(gJobs); i++) {
                sJob* job = vector_item(gJobs, i);
                char* str = MALLOC(string_length(job->mName) + 16);
                snprintf(str, string_length(job->mName) + 16, "%d. %s %d\n", i+1, string_c_str(job->mName), job->mPGroup);
                *rcode = 0;
                if(!statment_tree_internal_commands_write_nextout(nextout, str)) 
                {
                    FREE(str);
                    return FALSE;
                }

                FREE(str);
            }
            }
            break;

        case kRehash:
            saphire_rehash();
            break;

        case kKanjiCode: {
            int l;
            BOOL quiet = FALSE;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-s") == 0) {
                    *rcode = 0;
                    gKanjiCode = kSjis;

                    /// エンコーディングが変わるので正規表現をクリア ///
                    hash_it* it = hash_loop_begin(gRegexs);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexs);

                    it = hash_loop_begin(gRegexsI);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsI);

                    it = hash_loop_begin(gRegexsM);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsM);

                    it = hash_loop_begin(gRegexsIM);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsIM);
                    
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    *rcode = 0;
                    gKanjiCode = kEucjp;
                    /// エンコーディングが変わるので正規表現をクリア ///
                    hash_it* it = hash_loop_begin(gRegexs);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexs);

                    it = hash_loop_begin(gRegexsI);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsI);

                    it = hash_loop_begin(gRegexsM);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsM);

                    it = hash_loop_begin(gRegexsIM);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsIM);
                    
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    *rcode = 0;
                    gKanjiCode = kUtf8;
                    /// エンコーディングが変わるので正規表現をクリア ///
                    hash_it* it = hash_loop_begin(gRegexs);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexs);

                    it = hash_loop_begin(gRegexsI);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsI);

                    it = hash_loop_begin(gRegexsM);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsM);

                    it = hash_loop_begin(gRegexsIM);
                    while(it != NULL) {
                        onig_free(hash_loop_item(it));
                        it = hash_loop_next(it);
                    }
                    hash_clear(gRegexsIM);
                    
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(!quiet) {
                if(gKanjiCode == kSjis) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "sjis\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                else if(gKanjiCode == kEucjp) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "eucjp\n")) 
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "utf8\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                *rcode = 0;
            }

            }
            break;

        case kLineField: {
            int l;
            BOOL quiet = FALSE;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-Lw") == 0) {
                    *rcode = 0;
                    gLineField = kCRLF;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-Lm") == 0) {
                    *rcode = 0;
                    gLineField = kCR;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-Lu") == 0) {
                    *rcode = 0;
                    gLineField = kLF;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            /// 出力 ///
            if(!quiet) {
                if(gLineField == kLF) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "LF\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                else if(gLineField == kCRLF) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "CRLF\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "CR\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                *rcode = 0;
            }
            }
            break;

        case kVar:
            {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            BOOL field = FALSE;
            string_obj* field_str = NULL;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1<vector_size(argv))
                {
                    field = TRUE;
                    field_str 
                      = STRING_NEW(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin(
                                        nextin, str);
                    if(ret == 1) {
                        saphire_set_local_var(p, "");
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    if(ret < 0) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    saphire_set_local_var(p, string_c_str(str));
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }

                    string_delete(str);
                }
                else {
                    err_msg("var -a must have one var_name", sname, sline);
                    return FALSE;
                }
            }
            else if(input) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* item = string_c_str(vector_item(argv, k));

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    if(result < 0) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else if(result == 1) {
                        saphire_set_local_var(item, "");
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
//printf("one_line (%s)\n", string_c_str(str));
                    
                    if(chomp) string_chomp(str);

                    saphire_set_local_var(item, string_c_str(str));
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }

                    string_delete(str);
                }
            }
            else {
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* p = string_c_str((string_obj*)vector_item(argv, i));
                    char* p2 = strstr(p, "=");

                    if(p2) {
                        char* var_name = MALLOC(p2 - p + 1);
                        string_obj* value = STRING_NEW(p2 + 1);

                        memcpy(var_name, p, p2 - p);
                        var_name[p2 - p] = 0;

                        saphire_set_local_var(var_name, string_c_str(value));
                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(value))) 
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(var_name);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(var_name);
                                    return FALSE;
                                }
                            }
                        }

                        FREE(var_name);
                        string_delete(value);
                    }
                    else {
                        saphire_set_local_var(p, "");
                        *rcode = 0;

                        if(print) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
            }
            }
            break;

        case kGlobal:
            {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            BOOL field = FALSE;
            string_obj* field_str = NULL;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1<vector_size(argv))
                {
                    field = TRUE;
                    field_str 
                      = STRING_NEW(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin(
                                        nextin, str);
                    if(ret == 1) {
                        saphire_delete_same_name_var(p);
                        hash_put(gGlobals, p, STRING_NEW(""));
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else if(ret < 0)
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    saphire_delete_same_name_var(p);
                    hash_put(gGlobals, p, str);
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }
                }
                else {
                    err_msg("var -a must have one var_name", sname, sline);
                    return FALSE;
                }
            }
            else if(input) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* item = string_c_str(vector_item(argv, k));

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    if(result == 1) {
                        *rcode = 1;
                        saphire_delete_same_name_var(item);
                        hash_put(gGlobals, item, STRING_NEW(""));
                        string_delete(str);
                        break;
                    }
                    else if(result == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    if(chomp) string_chomp(str);

                    saphire_delete_same_name_var(item);
                    hash_put(gGlobals, item, str);
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }
                }
            }
            else {
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* p = string_c_str((string_obj*)vector_item(argv, i));
                    char* p2 = strstr(p, "=");

                    if(p2) {
                        char* var_name = MALLOC(p2 - p + 1);
                        string_obj* value = STRING_NEW(p2 + 1);

                        memcpy(var_name, p, p2 - p);
                        var_name[p2 - p] = 0;

                        /// グローバル変数に変数があるなら ///
                        saphire_delete_same_name_var(p);
                        hash_put(gGlobals, var_name, value);
                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(value))) 
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(var_name);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(var_name);
                                    return FALSE;
                                }
                            }
                        }

                        FREE(var_name);
                    }
                    else {
                        saphire_delete_same_name_var(p);
                        hash_put(gGlobals, p, STRING_NEW(""));
                        *rcode = 0;

                        if(print) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
            }
            }
            break;

        case kExport:
            {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            BOOL field = FALSE;
            string_obj* field_str = NULL;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1<vector_size(argv))
                {
                    field = TRUE;
                    field_str 
                      = STRING_NEW(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin(
                                        nextin, str);
                    if(ret == 1) {
                        saphire_delete_same_name_var(p);
                        setenv(p, "", 1);
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else if(ret < 0)
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    saphire_delete_same_name_var(p);
                    setenv(p, string_c_str(str), 1);
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }
                }
                else {
                    err_msg("export -a must have one var_name", sname, sline);
                    return FALSE;
                }
            }
            else if(input) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* item = string_c_str(vector_item(argv, k));

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    if(result == 1) {
                        saphire_delete_same_name_var(item);
                        setenv(item, "", 1);
                        string_delete(str);
                        break;
                    }
                    else if(result == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    if(chomp) string_chomp(str);

                    saphire_delete_same_name_var(item);
                    setenv(item, string_c_str(str), 1);
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }

                    string_delete(str);
                }
            }
            else {
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* p = string_c_str((string_obj*)vector_item(argv, i));
                    char* p2 = strstr(p, "=");

                    if(p2) {
                        char* var_name = MALLOC(p2 - p + 1);
                        string_obj* value = STRING_NEW(p2 + 1);

                        memcpy(var_name, p, p2 - p);
                        var_name[p2 - p] = 0;
    
                        saphire_delete_same_name_var(var_name);
                        setenv(var_name, string_c_str(value), 1);
                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(value))) 
                            {
                                err_msg("signal interrupt", sname, sline);
                                FREE(var_name);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    FREE(var_name);
                                    return FALSE;
                                }
                            }
                        }

                        string_delete(value);

                        FREE(var_name);
                    }
                    else {
                        saphire_delete_same_name_var(p);
                        setenv(p, "", 1);
                        *rcode = 0;

                        if(print) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
            }
            }
            break;


        case kCalc: {
            int i;
            BOOL new_line=FALSE;
            for(i=0; i<vector_size(argv); i++) {
                if(strcmp(string_c_str(vector_item(argv, i)), "-l") == 0)
                {
                    new_line = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(string_c_str(vector_item(argv, i)), "-I")
                        == 0)
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str = NULL;
            char* p;
            if(input) {
                str = STRING_NEW("");
                int result = statment_tree_internal_commands_read_nextin(
                                    nextin, str);
                if(result < 0)
                {
                    err_msg("calc: interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else if(result == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }

                p = string_c_str(str);
            }
            else {
                if(vector_size(argv) == 1) {
                    err_msg("calc expression", sname, sline);
                    return FALSE;
                }

                p = string_c_str(vector_item(argv, 1));
            }

            double n;
            BOOL result = calculate(&n, &p, sname, sline);
            if(!result) {
                if(str) string_delete(str);
                return FALSE;
            }
            
            char buf2[128];
            snprintf(buf2, 128, "%lf", n);

            /// 小数点以下の末尾の０を除外 ///
            if(strstr(buf2, ".")) {
                char* p3 = buf2 + strlen(buf2) -1;
                while(*p3 == '0') {
                    p3--;
                }
                if(*p3 == '.') p3--;
                *(p3+1) = 0;
            }

            if(strcmp(buf2, "-0") == 0) {
                strcpy(buf2, "0");
            }

            if(new_line) {
                strcat(buf2, "\n");
            }

            if(!statment_tree_internal_commands_write_nextout(nextout, buf2))
            {
                if(str) string_delete(str);
                err_msg("signal interrupt", sname, sline);
                return FALSE;
            }

            *rcode = 0;

            if(str) string_delete(str);
            }
            break;

        case kPrint:
        case kPuts: 
            {
            int i;
            BOOL line_field;
            string_obj* field = STRING_NEW(" ");
            if(command->mKind == kPrint) {
                line_field= FALSE;
            }
            else {
                line_field = TRUE;
            }
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = !line_field;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1 < vector_size(argv)) 
                {
                    string_put(field 
                         , string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                                    nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    string_delete(field);
                    return FALSE;
                }

                *rcode = 0;
                if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)) )
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(str);
                    string_delete(field);
                    return FALSE;
                }

                string_delete(str);
            }
            else if(vector_size(argv) > 1) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* arg = string_c_str((string_obj*)
                                        vector_item(argv, k));
                    if(!statment_tree_internal_commands_write_nextout(nextout,  arg))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(field);
                        return FALSE;
                    }

                    if(k<vector_size(argv)-1) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(field)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(field);
                            return FALSE;
                        }
                    }
                }
                if(line_field) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(field);
                        return FALSE;
                    }
                }
                *rcode = 0;
            }
            string_delete(field);
            }
            break;


        case kCompile: 
            if(vector_size(argv) == 2) {
                char* fname = string_c_str(vector_item(argv, 1));
                char out_fname[PATH_MAX];

                noextname(out_fname, fname);
                strcat(out_fname, ".sao");

                if(!saphire_compile(fname, out_fname)) {
                    return FALSE;
                }

                *rcode = 0;
            }
            break;

        case kRaise: 
            if(vector_size(argv) == 2) {
                char* msg = string_c_str(vector_item(argv, 1));

                err_msg(msg, sname, sline);
                *rcode = 0;

                return FALSE;
            }
            break;

        case kLoad: {
            BOOL compiled = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-sao") == 0) {
                    compiled = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(vector_size(argv) == 2) {
                char* fname = string_c_str(vector_item(argv, 1));
                if(access(fname, R_OK) != 0) {
                    err_msg("can't load file", sname, sline);
                    return FALSE;
                }
                if(compiled) {
                    *rcode = saphire_load_obj(
                        string_c_str(vector_item(argv, 1))
                        , nextout, nextin, nexterr, blocks);
                }
                else {
                    *rcode = saphire_load(
                            string_c_str(vector_item(argv, 1))
                            , nextout, nextin, nexterr);
                }
            }

            if(*rcode < 0) {
                return FALSE;
            }
            }
            break;

        case kDef: {
            sFunction* func = (sFunction*)command->mExtra;

            sFunction* func_before = hash_item(gFuncs, string_c_str(func->name));

            sFunction* new_fun = sFunction_new(string_c_str(func->name), func->statments, string_c_str(func->arg_name));
            func->statments = NULL;
               // gFuncにstatmentsを渡して、実体はそっちで管理する

            if(func_before) {
                sFunction_delete(func_before);
                hash_put(gFuncs, string_c_str(func->name), new_fun);
            }
            else {
                hash_put(gFuncs, string_c_str(func->name), new_fun);
            }

            *rcode = 0;

            }
            break;

        case kWhile: {
            sWhile* w = (sWhile*)command->mExtra;

            while(1) {
                *rcode = run(w->joukensiki, title, nextout
                                , nextin, nexterr
                                , FALSE, TRUE
                                , return_, parent_blocks);

                if(*return_) {
                    return TRUE;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
                else if(*rcode == SIGUSR2) {
                    *rcode = 0; // そのままだと2重whileの時だめ
                    break;
                }
                else if(*rcode == SIGINT) {
                    break;
                }
                else if(*rcode != 0) {
                    break;
                }

                *rcode = run(w->contents, "while command", nextout
                            , nextin, nexterr, FALSE, TRUE
                            , return_, parent_blocks);

                if(*return_) {
                    return TRUE;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
                else if(*rcode == SIGINT) {
                    break;
                }
                else if(*rcode == SIGUSR2) {
                    *rcode = 0;  // そのままだと2重whileの時breakで
                                 // 全部脱出してしまう
                    break;
                }
            }
            }
            break;

        case kBreak:
            *rcode = SIGUSR2;
            return TRUE;

        case kExit:
            if(vector_size(argv) == 2) {
                gKitutukiExit = atoi(string_c_str(vector_item(argv, 1)));
            }
            else {
                gKitutukiExit = 0;
            }
            break;

        case kIf: {
            sIf* if_ = (sIf*)command->mExtra;

            /// if, elif節 ///
            int k;
            for(k=0; k<vector_size(if_->joukensiki_list); k++) {
                sStatments* joukensiki 
                        = vector_item(if_->joukensiki_list, k);
                *rcode = run(joukensiki, title, nextout, nextin, nexterr
                                , FALSE, FALSE
                                , return_, parent_blocks);

                if(*return_) {
                    return TRUE;
                }
                else if(*rcode == 0) {
                    sStatments* contents 
                            = vector_item(if_->contents_list, k);
                    *rcode = run(contents, title, nextout, nextin
                                , nexterr, FALSE, FALSE
                                , return_, parent_blocks);
                    if(*return_) {
                        return TRUE;
                    }
                    else if(*rcode < 0) {
                        return FALSE;
                    }
                    goto if_end;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
            }

            /// else節 ///
            if(vector_size(if_->contents_list) 
                    > vector_size(if_->joukensiki_list)) 
            {
                sStatments* contents = vector_item(if_->contents_list, k);
                *rcode = run(contents, title, nextout, nextin
                            , nexterr, FALSE, FALSE
                            , return_, parent_blocks);
                if(*return_) {
                    return TRUE;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
            }
if_end:
            null_fun();
            }
            break;

        case kSplit: {
            string_obj* field = STRING_NEW("\n");
            BOOL ignore_case = FALSE;
            BOOL multiline = FALSE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-f") == 0 && i+1 < vector_size(argv)) 
                {
                    string_put(field 
                         , string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0)
                {
                    ignore_case = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0)
                {
                    multiline = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                if(vector_size(argv) <= 2) {
                    char* regex;
                    if(vector_size(argv) == 2) {
                        regex = string_c_str(vector_item(argv, 1));
                    }
                    else {
                        regex = "\\s+";
                    }

                    if(multiline) {
                        string_obj* line = STRING_NEW("");

                        int ret = statment_tree_internal_commands_read_nextin(nextin, line);
                        if(ret < 0) {
                            err_msg("interrupt", sname, sline);
                            string_delete(line);
                            string_delete(field);
                            return FALSE;
                        }
                        else if(ret == 1) {
                            *rcode = 1;
                            string_delete(line);
                            string_delete(field);
                            break;
                        }
                        else {
                            if(!statment_tree_internal_commands_split(line, regex, nextout, field, multiline, ignore_case, sname, sline)) {
                                string_delete(line);
                                string_delete(field);
                                return FALSE;
                            }

                            *rcode = 0;
                        }

                        if(!statment_tree_internal_commands_write_nextout(nextout,  "\n"))
                        {
                            err_msg("singal interrupt", sname, sline);
                            string_delete(line);
                            string_delete(field);
                            return FALSE;
                        }

                        string_delete(line);
                    }
                    else {
                        if(!statment_tree_internal_commands_split_oneline(regex, nextin, nextout, field, multiline, ignore_case, rcode, sname, sline)) {
                            string_delete(field);
                            return FALSE;
                        }

                        *rcode = 0;
                    }
                }
            }
            else {
                if(vector_size(argv) >= 2) {
                    char* regex;
                    if(vector_size(argv) == 2) {
                        regex = "\\s+";
                    }
                    else {
                        regex = string_c_str(vector_item(argv, 2));
                    }

                    string_obj* line = vector_item(argv, 1);
                    if(!statment_tree_internal_commands_split(line, regex, nextout, field, multiline, ignore_case, sname, sline)) {
                        string_delete(field);
                        return FALSE;
                    }

                    *rcode = 0;

                    if(!statment_tree_internal_commands_write_nextout(nextout,  "\n"))
                    {
                        err_msg("singal interrupt", sname, sline);
                        string_delete(field);
                        return FALSE;
                    }
                }
            }

            string_delete(field);
            }
            break;

        case kAdd: {
            BOOL flg_n = FALSE;
            int n;
            BOOL flg_l = FALSE;
            int l;
            BOOL byte = FALSE;
            enum eKanjiCode code = gKanjiCode;

            int k;
            for(k=0; k<vector_size(argv); k++) {
                char* arg = string_c_str(vector_item(argv, k));

                if(strcmp(arg, "-n") == 0 && k+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, k+1)));
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    k--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0 && k+1 < vector_size(argv)) {
                    flg_l = TRUE;
                    l = atoi(string_c_str(vector_item(argv, k+1)));
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    k--;
                    continue;
                }
                else if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    k--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    k--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    k--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, k));
                    vector_erase(argv, k);
                    k--;
                    continue;
                }
            }

            if(vector_size(argv) == 2) {
                if(flg_l) {
                    vector_obj* v = VECTOR_NEW(100);

                    while(1) {
                        if(gKitutukiSigInt) {
                            gKitutukiSigInt = FALSE;
                            err_msg("interrupt", sname, sline);
                            *rcode = 1;
                            int j;
                            for(j=0; j<vector_size(v); j++) {
                                string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }
                        string_obj* line = STRING_NEW("");
                        int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);
                        if(ret == -1) {
                            err_msg("interrupt", sname, sline);
                            string_delete(line);
                            int j;
                            for(j=0; j<vector_size(v); j++) {
                                string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }
                        else if(ret == 1) {
                            string_delete(line);
                            break;
                        }

                        vector_add(v, line);
                    }

                    if(l < 0) {
                        l+=vector_size(v)+1;
                    }
                    if(l < 0) {
                        l = 0;
                    }

                    if(l < vector_size(v)) {
                        int k;
                        for(k=0; k<l; k++) {
                            char* str = string_c_str(vector_item(v, k));

                            if(!statment_tree_internal_commands_write_nextout(nextout, str))
                            {
                                err_msg("singal interrupt", sname, sline);
                                int j;
                                for(j=0; j<vector_size(v); j++) {
                                    string_delete(vector_item(v, j));
                                }
                                vector_delete(v);
                                return FALSE;
                            }
                        }

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 1))))
                        {
                            err_msg("singal interrupt", sname, sline);
                            int j;
                            for(j=0; j<vector_size(v); j++) {
                                string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }

                        for(k=l; k<vector_size(v); k++) {
                            char* str = string_c_str(vector_item(v, k));

                            if(!statment_tree_internal_commands_write_nextout(nextout, str))
                            {
                                err_msg("singal interrupt", sname, sline);
                                int j;
                                for(j=0; j<vector_size(v); j++) {
                                    string_delete(vector_item(v, j));
                                }
                                vector_delete(v);
                                return FALSE;
                            }
                        }
                    }
                    else {
                        int k;
                        for(k=0; k<vector_size(v); k++) {
                            char* str = string_c_str(vector_item(v, k));

                            if(!statment_tree_internal_commands_write_nextout(nextout, str))
                            {
                                err_msg("singal interrupt", sname, sline);
                                int j;
                                for(j=0; j<vector_size(v); j++) {
                                    string_delete(vector_item(v, j));
                                }
                                vector_delete(v);
                                return FALSE;
                            }
                        }
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 1))))
                        {
                            err_msg("singal interrupt", sname, sline);
                            int j;
                            for(j=0; j<vector_size(v); j++) {
                                string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }
                    }

                    for(j=0; j<vector_size(v); j++) {
                        string_delete(vector_item(v, j));
                    }
                    vector_delete(v);
                }
                else if(flg_n) {
                    string_obj* str = STRING_NEW("");

                    int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                    if(ret < 0) 
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else
                    {
                        char* str2 = string_c_str(vector_item(argv, 1));
                        if(byte) {
                            if(n < 0) {
                                n = strlen(string_c_str(str)) + n + 1;
                            }
                            if(n < 0) {
                                n = 0;
                            }
                            if(n > strlen(string_c_str(str))) {
                                n = strlen(string_c_str(str)) -1;
                            }

                            string_insert(str, n, str2);
                        }
                        else {
                            if(n < 0) {
                                n = str_kanjilen(code, string_c_str(str)) + n + 1;
                            }

                            if(n < 0) {
                                n = 0;
                            }
                            if(n > str_kanjilen(code, string_c_str(str))) {
                                n = str_kanjilen(code, string_c_str(str))-1;
                            }

                            int m = str_kanjipos2pointer(code, string_c_str(str), n) - string_c_str(str);
                            string_insert(str, m, str2);
                        }

                        if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(str)))
                        {
                            string_delete(str);
                            err_msg("singal interrupt", sname, sline);
                            return FALSE;
                        }
                        *rcode = 0;
                        string_delete(str);
                    }
                }
                else {
                    string_obj* str = STRING_NEW("");

                    int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                    if(ret < 0)
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else {
                        char* str2 = string_c_str(vector_item(argv, 1));

                        string_push_back(str, str2);

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            string_delete(str);
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }

                        *rcode = 0;
                        string_delete(str);
                    }
                }
            }
            }
            break;

        case kDel: {
            BOOL flg_n = FALSE;
            int n;
            BOOL byte = FALSE;
            enum eKanjiCode code = gKanjiCode;
            BOOL flg_l = FALSE;
            int L;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0 && l+1 < vector_size(argv)) {
                    flg_l = TRUE;
                    L = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(flg_l) {
                int N;
                if(vector_size(argv) >= 2) {
                    N = atoi(string_c_str(vector_item(argv, 1)));
                    if(N < 1) {
                        N = 1;
                    }
                }
                else {
                    N = 1;
                }
                vector_obj* v = VECTOR_NEW(100);

                while(1) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        err_msg("interrupt", sname, sline);
                        *rcode = 1;
                        int j;
                        for(j=0; j<vector_size(v); j++) {
                            string_delete(vector_item(v, j));
                        }
                        vector_delete(v);
                        return FALSE;
                    }
                    string_obj* line = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);
                    if(ret == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(line);
                        int j;
                        for(j=0; j<vector_size(v); j++) {
                            string_delete(vector_item(v, j));
                        }
                        vector_delete(v);
                        return FALSE;
                    }
                    else if(ret == 1) {
                        string_delete(line);
                        break;
                    }

                    vector_add(v, line);
                }

                if(L < 0) {
                    L+=vector_size(v)+1;
                }
                if(L < 0) {
                    L = 0;
                }

                if(L < vector_size(v)) {
                    int k;
                    for(k=0; k<L; k++) {
                        char* str = string_c_str(vector_item(v, k));

                        if(!statment_tree_internal_commands_write_nextout(nextout, str))
                        {
                            err_msg("singal interrupt", sname, sline);
                            int j;
                            for(j=0; j<vector_size(v); j++) {
                                string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }
                    }

                    for(k=L + N; k<vector_size(v); k++) {
                        char* str = string_c_str(vector_item(v, k));

                        if(!statment_tree_internal_commands_write_nextout(nextout, str))
                        {
                            err_msg("singal interrupt", sname, sline);
                            int j;
                            for(j=0; j<vector_size(v); j++) {
                                string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }
                    }
                }
            }
            else if(flg_n) {
                string_obj* str = STRING_NEW("");

                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret == -1)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int num;
                    if(vector_size(argv) >= 2) {
                        char* arg1 = string_c_str(vector_item(argv, 1));
                        num = atoi(arg1);
                    }
                    else {
                        num = 1;
                    }

                    if(byte) {
                        if(n < 0) {
                            n = strlen(string_c_str(str)) + n + 1;
                        }

                        if(n < 0 || n > strlen(string_c_str(str))) {
                            err_msg("del: invalid -n range", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }

                        string_erase(str, n, num);
                    }
                    else {
                        if(n < 0) {
                            n = str_kanjilen(code, string_c_str(str)) + n + 1;
                        }

                        if(n < 0 || n > str_kanjilen(code, string_c_str(str))) {
                            err_msg("del: invalid -n range", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }

                        int m = str_kanjipos2pointer(code, string_c_str(str), n) - string_c_str(str);
                        int l = str_kanjipos2pointer(code, string_c_str(str), n+num) - string_c_str(str);
                        string_erase(str, m, l-m);
                    }

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    *rcode = 0;
                    string_delete(str);
                }
            }
            else {
                string_obj* str = STRING_NEW("");

                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1)
                {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int num;
                    if(vector_size(argv) >= 2) {
                        char* arg1 = string_c_str(vector_item(argv, 1));
                        num = atoi(arg1);
                    }
                    else {
                        num = 1;
                    }

                    if(byte) {
                        string_trunc(str, string_length(str)-num+1);
                    }
                    else {
                        int m = str_kanjipos2pointer(code, string_c_str(str), str_kanjilen(code, string_c_str(str))-num) - string_c_str(str);
                        
                        string_trunc(str, m);
                    }

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        string_delete(str);
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }

                    *rcode = 0;
                    string_delete(str);
                }
            }
            }
            break;

        case kX: {
            int i;
            BOOL line_field = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) 
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            int count;
            string_obj* str;
            if(vector_size(argv) == 3) {
                str = STRING_NEW(string_c_str(vector_item(argv, 1)));
                count = atoi(string_c_str(vector_item(argv, 2)));
            }
            else if(vector_size(argv) == 2 && input) {
                count = atoi(string_c_str(vector_item(argv, 1)));
                str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
            }
            else {
                err_msg("invalid argument with x", sname, sline);
                return FALSE;
            }

            string_obj* out = STRING_NEW("");
            for(i=0; i<count; i++) {
                string_push_back(out, string_c_str(str));
            }

            if(line_field) {
                string_push_back2(out, '\n');
            }

            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(out)))
            {
                err_msg("signal interrupt", sname, sline);
                string_delete(out);
                string_delete(str);
                return FALSE;
            }

            string_delete(out);
            string_delete(str);

            *rcode = 0;
            }
            break;

        case kJoin: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str = STRING_NEW("");

            if(input) {
                char* field;
                if(vector_size(argv) == 2) {
                    field = string_c_str(vector_item(argv, 1));
                }
                else {
                    field = " ";
                }

                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    string_chomp(str);
                    char* p = string_c_str(str);
                    string_obj* buf = STRING_NEW("");
                    while(*p) {
                        if(is_line_field2(gLineField, p)) {
                            if(gLineField == kCRLF) {
                                p+=2;
                            }
                            else {
                                p++;
                            }
                            string_push_back(buf, field);

                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(buf)))
                            {
                                err_msg("signal interrupt", sname, sline);
                                string_delete(str);
                                string_delete(buf);
                                return FALSE;
                            }

                            string_put(buf, "");
                        }
                        else {
                            string_push_back2(buf, *p++);
                        }
                    }

                    if(strcmp(string_c_str(buf), "") != 0) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(buf)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            string_delete(buf);
                            return FALSE;
                        }
                    }

                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(str);
                        string_delete(buf);
                        return FALSE;
                    }

                    string_delete(str);
                    string_delete(buf);

                    *rcode = 0;
                }
            }
            else if(vector_size(argv) > 1) {
                char* field;
                if(vector_size(argv) == 2) {
                    field = " ";
                }
                else {
                    field = string_c_str(vector_item(argv, 2));
                }

                char* p = string_c_str(vector_item(argv, 1));
                string_obj* buf = STRING_NEW("");
                while(*p) {
                    if(is_line_field2(gLineField, p)) {
                        if(gLineField == kCRLF) {
                            p+=2;
                        }
                        else {
                            p++;
                        }
                        string_push_back(buf, field);

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(buf)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(buf);
                            return FALSE;
                        }

                        string_put(buf, "");
                    }
                    else {
                        string_push_back2(buf, *p++);
                    }
                }

                if(strcmp(string_c_str(buf), "") != 0) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(buf)))
                    {
                        err_msg("signal interrupt", sname, sline);
                        string_delete(buf);
                        return FALSE;
                    }
                }

                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(buf);
                    return FALSE;
                }

                string_delete(buf);

                *rcode = 0;
            }
            }
            break;

        case kRows: {
            int i;

            sLinesArg* larg = MALLOC(sizeof(sLinesArg)*vector_size(argv));
            enum eKanjiCode code = gKanjiCode;
            int larg_num = 0;
            BOOL byte = FALSE;
            BOOL reverse = FALSE;
            int block_num = 0;

            for(i=1; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                string_obj* next_arg;
                if(i+1 < vector_size(argv)) {
                    next_arg = vector_item(argv, i+1);
                }
                else {
                    next_arg = NULL;
                }

               /// byte ///
               if(strcmp(arg, "-b") == 0) {
                    byte = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-r") == 0) {
                    reverse = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                /// 範囲 ///
                else if(strstr(arg, "..")) {
                    char* first = MALLOC(sizeof(arg) + 1);
                    char* last = MALLOC(sizeof(arg) + 1);
                    char* p = arg;

                    char* p2 = first;

                    if(*p == '-'  || *p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    while(*p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    *p2 = 0;

                    if(*p != '.' || *(p+1) != '.') {
                        err_msg("invalid lines argument", sname, sline);
                        FREE(first);
                        FREE(last);
                        FREE(larg);
                        return FALSE;
                    }
                    p+=2;

                    p2 = last;

                    if(*p == '-'  || *p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    while(*p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    *p2 = 0;

                    /// 次の引数がブロックなら ///
                    if(next_arg && strcmp(string_c_str(next_arg), "-B") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].kind = 2;
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else if(next_arg && strcmp(string_c_str(next_arg), "-b") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].kind = 1;
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else {
                        larg[larg_num].kind = 1;
                        larg[larg_num].statments = NULL;
                    }

                    larg[larg_num].num1 = atoi(first);
                    larg[larg_num].num2 = atoi(last);
                    larg_num++;

                    FREE(first);
                    FREE(last);
                }

                else {
                    larg[larg_num].kind = 0;
                    larg[larg_num].num1 = atoi(arg);

                    /// 次の引数がブロックなら ///
                    if(next_arg && strcmp(string_c_str(next_arg), "-b") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else {
                        larg[larg_num].statments = NULL;
                    }
                    larg_num++;
                }
            }

            string_obj* line = STRING_NEW("");

            int ret = statment_tree_internal_commands_read_nextin(nextin, line);
            if(ret == 1) {
                *rcode = 1;
                break;
            }
            else if(ret == -1) {
                err_msg("interrupt", sname, sline);
                string_delete(line);
                FREE(larg);
                return FALSE;
            }

            if(reverse) {
                if(code == kUtf8) {
                    char* p = string_c_str(line);
                    int len = strlen(p);
                    wchar_t* wcs = MALLOC(len*MB_CUR_MAX+1);
                    if(mbstowcs(wcs, p, len*MB_CUR_MAX+1) < 0) {
                        FREE(wcs);
                        string_delete(line);
                        FREE(larg);
                        break;
                    }
                    wchar_t* wcs2 = MALLOC(len*MB_CUR_MAX+1);
                    int i;
                    int j;
                    for(i=wcslen(wcs)-1, j=0; i>=0; i--, j++) {
                        wcs2[j] = wcs[i];
                    }
                    wcs2[j] = 0;

                    char* mbs = MALLOC(len*MB_CUR_MAX+1);
                    wcstombs(mbs, wcs2, len*MB_CUR_MAX+1);

                    if(!statment_tree_internal_commands_write_nextout(nextout, mbs))
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(line);
                        FREE(larg);
                        FREE(mbs);
                        FREE(wcs);
                        FREE(wcs2);
                        return FALSE;
                    }

                    FREE(mbs);
                    FREE(wcs);
                    FREE(wcs2);

                    *rcode = 0;
                }
                else {
                    char* p = string_c_str(line);
                    int len = strlen(p);
                    char* str = MALLOC(len+1);
                    char* p2 = str + len;
                    *p2=0;
                    p2--;

                    while(*p) {
                        if(is_kanji(code, *p)) {
                            *(p2-1) = *p++;
                            *p2 = *p++;
                            p2-=2;
                        }
                        else {
                            *p2-- = *p++;
                        }
                    }

                    if(!statment_tree_internal_commands_write_nextout(nextout, str))
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(line);
                        FREE(larg);
                        FREE(str);
                        return FALSE;
                    }

                    FREE(str);

                    *rcode = 0;
                }
            }
            else {
                for(i=0; i<larg_num; i++) {
                    switch(larg[i].kind) {
                        case 0: {
                            int n = larg[i].num1;
                            char buf[MB_CUR_MAX + 1];

                            if(byte) {
                                if(n < 0) {
                                    n += string_length(line);
                                }

                                if(n < 0 || n >= string_length(line)) {
                                    break;
                                }

                                buf[0] = string_c_str(line)[n];
                                buf[1] = 0;
                            }
                            else {
                                if(n < 0) {
                                    n += str_kanjilen(code, string_c_str(line));
                                }

                                if(n < 0 || n >= str_kanjilen(code, string_c_str(line))) {
                                    break;
                                }

                                char* c = str_kanjipos2pointer(code, string_c_str(line), n);
                                char* d = str_kanjipos2pointer(code, string_c_str(line), n+1);
                                memcpy(buf, c, d-c);
                                buf[d-c] = 0;
                            }

                            /// ブロックがあるなら ///
                            if(larg[i].statments) {
                                sRFd* nextin2 = sRFd_new2(-1, buf);

                                char* fname = MALLOC(strlen(sname) + 32);
                                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);
                                *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                sRFd_delete(nextin2);
                                FREE(fname);
                                if(*return_) {
                                    FREE(larg);
                                    string_delete(line);
                                    return TRUE;
                                }
                                else if(*rcode < 0) {
                                    FREE(larg);
                                    string_delete(line);
                                    return FALSE;
                                }
                            }
                            else {
                                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                {
                                    FREE(larg);
                                    string_delete(line);
                                    return FALSE;
                                }
                                *rcode = 0;
                            }
                            }
                            break;

                        case 1: {
                            int n1 = larg[i].num1;
                            int n2 = larg[i].num2;

                            if(byte) {
                                if(n1 < 0) {
                                    n1+=string_length(line);
                                }
                                if(n2 < 0) {
                                    n2+=string_length(line);
                                }

                                if(n1 < 0 || n1 >= string_length(line)) {
                                    break;
                                }
                                if(n2 < 0 || n2 >= string_length(line)) {
                                    break;
                                }

                                if(n1 > n2) {
                                    err_msg("invalid range", sname, sline);
                                    FREE(larg);
                                    string_delete(line);

                                    return FALSE;
                                }

                                char buf[MB_CUR_MAX * (n2-n1+1) + 1];

                                int i;
                                for(i=n1; i<=n2; i++) {
                                    buf[i-n1] = string_c_str(line)[i];
                                }
                                buf[i] = 0;

                                if(larg[i].statments) {
                                    char* fname = MALLOC(strlen(sname) + 32);
                                    snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                    sRFd* nextin2 = sRFd_new2(-1, buf);
                                    *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                    sRFd_delete(nextin2);
                                    if(*return_) {
                                        FREE(larg);
                                        string_delete(line);
                                        FREE(fname);
                                        return TRUE;
                                    }
                                    else if(*rcode < 0) {
                                        FREE(larg);
                                        string_delete(line);
                                        FREE(fname);
                                        return FALSE;
                                    }

                                    FREE(fname);
                                }
                                else {
                                    if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                    {
                                        FREE(larg);
                                        string_delete(line);
                                        return FALSE;
                                    }
                                    *rcode = 0;
                                }
                            }
                            else {
                                if(n1 < 0) {
                                    n1 += str_kanjilen(code, string_c_str(line));
                                }

                                if(n1 < 0 || n1 >= str_kanjilen(code, string_c_str(line))) {
                                    break;
                                }
                                if(n2 < 0) {
                                    n2 += str_kanjilen(code, string_c_str(line));
                                }

                                if(n2 < 0 || n2 >= str_kanjilen(code, string_c_str(line))) {
                                    break;
                                }

                                if(n1 > n2) {
                                    err_msg("invalid range", sname, sline);
                                    FREE(larg);
                                    string_delete(line);

                                    return FALSE;
                                }

                                char buf[MB_CUR_MAX * (n2-n1+1) + 1];

                                char* c = str_kanjipos2pointer(code, string_c_str(line), n1);
                                char* d = str_kanjipos2pointer(code, string_c_str(line), n2+1);
                                memcpy(buf, c, d-c);
                                buf[d-c] = 0;

                                if(larg[i].statments) {
                                    char* fname = MALLOC(strlen(sname) + 32);
                                    snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                    sRFd* nextin2 = sRFd_new2(-1, buf);

                                    *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                    sRFd_delete(nextin2);
                                    if(*return_) {
                                        FREE(larg);
                                        string_delete(line);
                                        FREE(fname);
                                        return TRUE;
                                    }
                                    else if(*rcode < 0) {
                                        FREE(larg);
                                        string_delete(line);
                                        FREE(fname);
                                        return FALSE;
                                    }

                                    FREE(fname);
                                }
                                else {
                                    if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                    {
                                        FREE(larg);
                                        string_delete(line);
                                        return FALSE;
                                    }
                                    *rcode = 0;
                                }
                            }
                            }
                            break;

                        case 2: {
                            int n1 = larg[i].num1;
                            int n2 = larg[i].num2;

                            if(n1 > n2) {
                                err_msg("invalid range", sname, sline);
                                FREE(larg);
                                string_delete(line);

                                return FALSE;
                            }

                            if(byte) {
                                if(n1 < 0) {
                                    n1+=string_length(line);
                                }
                                if(n2 < 0) {
                                    n2+=string_length(line);
                                }

                                if(n1 < 0 || n1 >= string_length(line)) {
                                    break;
                                }
                                if(n2 < 0 || n2 >= string_length(line)) {
                                    break;
                                }
                            }
                            else {
                                if(n1 < 0) {
                                    n1 += str_kanjilen(code, string_c_str(line));
                                }

                                if(n1 < 0 || n1 >= str_kanjilen(code, string_c_str(line))) {
                                    break;
                                }

                                if(n2 < 0) {
                                    n2 += str_kanjilen(code, string_c_str(line));
                                }
                                if(n2 < 0 || n2 >= str_kanjilen(code, string_c_str(line))) {
                                    break;
                                }
                            }

                            if(larg[i].statments) {
                                char* fname = MALLOC(strlen(sname) + 32);
                                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                int j;
                                for(j=n1; j<=n2; j++) {
                                    char buf[MB_CUR_MAX * (n2-n1+1) + 1];

                                    if(byte) {
                                        buf[0] = string_c_str(line)[j];
                                        buf[1] = 0;
                                    }
                                    else {
                                        char* c = str_kanjipos2pointer(code, string_c_str(line), j);
                                        char* d = str_kanjipos2pointer(code, string_c_str(line), j+1);
                                        memcpy(buf, c, d-c);
                                        buf[d-c] = 0;
                                    }

                                    sRFd* nextin2 = sRFd_new2(-1, buf);

                                    *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                    sRFd_delete(nextin2);

                                    if(*return_) {
                                        FREE(larg);
                                        string_delete(line);
                                        FREE(fname);
                                        return TRUE;
                                    }
                                    else if(*rcode < 0) {
                                        FREE(larg);
                                        string_delete(line);
                                        FREE(fname);
                                        return FALSE;
                                    }
                                }

                                FREE(fname);
                            }
                            else {
                                char buf[MB_CUR_MAX * (n2-n1+1) + 1];

                                if(byte) {
                                    int i;
                                    for(i=n1; i<=n2; i++) {
                                        buf[i-n1] = string_c_str(line)[i];
                                    }
                                    buf[i] = 0;
                                }
                                else {
                                    char* c = str_kanjipos2pointer(code, string_c_str(line), n1);
                                    char* d = str_kanjipos2pointer(code, string_c_str(line), n2+1);
                                    memcpy(buf, c, d-c);
                                    buf[d-c] = 0;
                                }

                                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                                {
                                    FREE(larg);
                                    string_delete(line);
                                    return FALSE;
                                }
                                *rcode = 0;
                            }
                            }
                            break;
                    }
                }
            }

            FREE(larg);
            string_delete(line);
            }
            break;

        case kLines: {
            int i;
            BOOL reverse = FALSE;

            sLinesArg* larg = MALLOC(sizeof(sLinesArg)*vector_size(argv));
            int larg_num = 0;
            int block_num = 0;

            for(i=1; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                string_obj* next_arg;
                if(i+1 < vector_size(argv)) {
                    next_arg = vector_item(argv, i+1);
                }
                else {
                    next_arg = NULL;
                }

                if(strcmp(arg, "-r") == 0) {
                    reverse = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                /// 倍数 ///
                else if(arg[0] == '*') {
                    char* p = arg + 1;
                    char* start = MALLOC(sizeof(arg) + 1);
                    char* n = MALLOC(sizeof(arg) + 1);

                    char* p2 = n;
                    if(*p == '-' || *p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }
                    while(*p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }
                    *p2 = 0;

                    if(*p != '+') {
                        err_msg("invalid lines argument", sname, sline);
                        FREE(start);
                        FREE(n);
                        FREE(larg);
                        return FALSE;
                    }
                    p++;

                    p2 = start;
                    if(*p == '-' || *p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }
                    while(*p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }
                    *p2 = 0;

                    /// 次の引数がブロックなら ///
                    if(next_arg && strcmp(string_c_str(next_arg), "-B") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].kind = 2;
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else if(next_arg && strcmp(string_c_str(next_arg), "-b") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].kind = 5;
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else {
                        larg[larg_num].kind = 2;
                        larg[larg_num].statments = NULL;
                    }

                    larg[larg_num].num1 = atoi(n);
                    larg[larg_num].num2 = atoi(start);
                    larg_num++;

                    FREE(start);
                    FREE(n);
                }
                else if((arg[0] < '0' || arg[0] > '9') && arg[0] != '-') {
                    larg[larg_num].kind = 3;
                    larg[larg_num].condition = vector_item(argv, i);

                    /// 次の引数がブロックなら ///
                    if(next_arg && strcmp(string_c_str(next_arg), "-b") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else {
                        larg[larg_num].statments = NULL;
                    }
                    larg_num++;
                }
                /// 範囲 ///
                else if(strstr(arg, "..")) {
                    char* first = MALLOC(sizeof(arg) + 1);
                    char* last = MALLOC(sizeof(arg) + 1);
                    char* p = arg;
                    char* p2 = first;

                    if(*p == '-'  || *p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    while(*p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    *p2 = 0;

                    if(*p != '.' || *(p+1) != '.') {
                        err_msg("invalid lines argument", sname, sline);
                        FREE(first);
                        FREE(last);
                        FREE(larg);
                        return FALSE;
                    }
                    p+=2;

                    p2 = last;

                    if(*p == '-'  || *p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    while(*p >= '0' && *p <= '9') {
                        *p2++ = *p++;
                    }

                    *p2 = 0;

                    /// 次の引数がブロックなら ///
                    if(next_arg && strcmp(string_c_str(next_arg), "-B") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].kind = 1;
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else if(next_arg && strcmp(string_c_str(next_arg), "-b") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].kind = 4;
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else {
                        larg[larg_num].kind = 1;
                        larg[larg_num].statments = NULL;
                    }
                    larg[larg_num].num1 = atoi(first);
                    larg[larg_num].num2 = atoi(last);
                    larg_num++;

                    FREE(first);
                    FREE(last);
                }
                else {
                    larg[larg_num].kind = 0;
                    larg[larg_num].num1 = atoi(arg);

                    /// 次の引数がブロックなら ///
                    if(next_arg && strcmp(string_c_str(next_arg), "-b") == 0 && block_num < vector_size(blocks)) 
                    {
                        larg[larg_num].statments = vector_item(blocks, block_num++);
                        i++;
                    }
                    else {
                        larg[larg_num].statments = NULL;
                    }
                    larg_num++;
                }
            }

            vector_obj* v = VECTOR_NEW(100);

            while(1) {
                if(gKitutukiSigInt) {
                    err_msg("interrupt", sname, sline);
                    *rcode = 1;
                    FREE(larg);
                    int j;
                    for(j=0; j<vector_size(v); j++) {
                        string_delete(vector_item(v, j));
                    }
                    vector_delete(v);
                    return FALSE;
                }
                string_obj* line = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);
                //string_chomp(line);
                if(ret == -1) {
                    err_msg("interrupt", sname, sline);
                    string_delete(line);
                    FREE(larg);
                    int j;
                    for(j=0; j<vector_size(v); j++) {
                        string_delete(vector_item(v, j));
                    }
                    vector_delete(v);
                    return FALSE;
                }
                else if(ret == 1) {
                    string_delete(line);
                    break;
                }

                vector_add(v, line);
            }

            /// 反転 ///
            if(reverse) {
                const int size = vector_size(v);
                for(i=0; i<size; i++) {
                    char* str = string_c_str(vector_item(v, size-i-1));
                    string_obj* str2 = STRING_NEW(str);
                    string_chomp(str2);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                    {
                        string_delete(str2);
                        FREE(larg);
                        int j;
                        for(j=0; j<vector_size(v); j++) {
                            string_delete(vector_item(v, j));
                        }
                        vector_delete(v);
                        return FALSE;
                    }
                    string_delete(str2);
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        FREE(larg);
                        int j;
                        for(j=0; j<vector_size(v); j++) {
                            string_delete(vector_item(v, j));
                        }
                        vector_delete(v);
                        return FALSE;
                    }
                }
            }
            else {
                for(i=0; i<larg_num; i++) {
                    switch(larg[i].kind) {
                        case 0: {
                            int n = larg[i].num1;

                            if(n < 0) {
                                n += vector_size(v);
                            }
                            if(n >= 0 && n < vector_size(v)) {
                                string_obj* str = vector_item(v, n);

                                /// ブロックがあるなら ///
                                if(larg[i].statments) {
                                    char* fname = MALLOC(strlen(sname) + 32);
                                    snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);
                                    sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                                    *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                    FREE(fname);
                                    sRFd_delete(nextin2);
                                    if(*return_) {
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);
                                        return TRUE;
                                    }
                                    else if(*rcode < 0) {
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);
                                        return FALSE;
                                    }
                                    //gGPipeStat = save_value;
                                }
                                else {
                                    string_obj* str2 = STRING_NEW(string_c_str(str));
                                    string_chomp(str2);
                                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                    {
                                        string_delete(str2);
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);
                                        return FALSE;
                                    }
                                    string_delete(str2);
                                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                    {
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);
                                        return FALSE;
                                    }
                                }
                            }

                            }
                            break;

                        case 1: {
                            int n1 = larg[i].num1;
                            int n2 = larg[i].num2;

                            if(n1 < 0) {
                                n1+=vector_size(v);
                            }
                            if(n2 < 0) {
                                n2+=vector_size(v);
                            }

                            if(n1 >= 0 && n2>=0 && n1 < vector_size(v)
                                && n2 < vector_size(v)) 
                            {
                                if(larg[i].statments) {
                                    char* fname = MALLOC(strlen(sname) + 32);
                                    snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                    if(n1 < n2) {
                                        int k;
                                        for(k=n1; k<=n2; k++) {
                                            string_obj* str = vector_item(v, k);
                                            sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));

                                            *rcode = run(larg[i].statments
                                                    , fname, nextout
                                                    , nextin2, nexterr
                                                    , FALSE, FALSE
                                                    , return_, parent_blocks);
                                            sRFd_delete(nextin2);
                                            if(*return_) {
                                                FREE(fname);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);
                                                return TRUE;
                                            }
                                            else if(*rcode < 0) {
                                                FREE(fname);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);
                                                return FALSE;
                                            }
                                        }
                                    }
                                    else {
                                        int k;
                                        for(k=n1; k>=n2; k--) {
                                            string_obj* str = vector_item(v, k);
                                            sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                                            *rcode = run(larg[i].statments
                                                    , fname, nextout
                                                    , nextin2, nexterr
                                                    , FALSE, FALSE
                                                    , return_, parent_blocks);
                                            sRFd_delete(nextin2);
                                            if(*return_) {
                                                FREE(fname);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);
                                                return TRUE;
                                            }
                                            else if(*rcode < 0) {
                                                FREE(fname);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);
                                                return FALSE;
                                            }
                                        }
                                    }

                                    FREE(fname);
                                }
                                else {
                                    if(n1 < n2) {
                                        int k;
                                        for(k=n1; k<=n2; k++) {
                                            string_obj* str = vector_item(v, k);
                                            string_obj* str2 = STRING_NEW(string_c_str(str));
                                            string_chomp(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                            {
                                                string_delete(str2);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                            string_delete(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                            {
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                        }
                                    }
                                    else {
                                        int k;
                                        for(k=n1; k>=n2; k--) {
                                            string_obj* str = vector_item(v, k);
                                            string_obj* str2 = STRING_NEW(string_c_str(str));
                                            string_chomp(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                            {
                                                string_delete(str2);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                            string_delete(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                            {
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                        }
                                    }
                                }
                            }
                            }
                            break;

                        case 2: {
                            int baisuu = larg[i].num1;
                            int start = larg[i].num2;

                            /// ブロックがあるなら ///
                            if(larg[i].statments) {
                                char* fname = MALLOC(strlen(sname) + 32);
                                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                int k = 0;
                                while(1) {
                                    int x = k * baisuu + start;

                                    if(x < 0 || x >= vector_size(v)) {
                                        break;
                                    }

                                    string_obj* str = vector_item(v, x);

                                    sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));

                                    *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                    sRFd_delete(nextin2);
                                    if(*return_) {
                                        FREE(fname);
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);

                                        return TRUE;
                                    }
                                    else if(*rcode < 0) {
                                        FREE(fname);
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);

                                        return FALSE;
                                    }


                                    k++;
                                }

                                FREE(fname);
                            }
                            else {
                                int k = 0;
                                while(1) {
                                    int x = k * baisuu + start;

                                    if(x < 0 || x >= vector_size(v)) {
                                        break;
                                    }

                                    string_obj* str = vector_item(v, x);
                                    string_obj* str2 = STRING_NEW(string_c_str(str));
                                    string_chomp(str2);

                                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                    {
                                        string_delete(str2);
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);

                                        return FALSE;
                                    }
                                    string_delete(str2);
                                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                    {
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);

                                        return FALSE;
                                    }

                                    k++;
                                }
                            }
                            }
                            break;

                        case 3: {
                            char* fname = MALLOC(strlen(sname) + 32);
                            snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                            int j;
                            for(j=0; j<vector_size(v); j++)  {
                                string_obj* str = vector_item(v, j);

                                /// 条件式 ///
                                sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));

                                *rcode = saphire_shell(
                                    string_c_str(larg[i].condition)
                                        , fname
                                        , nextout, nextin2, nexterr
                                        );
                                sRFd_delete(nextin2);

                                if(*rcode < 0) {
                                    FREE(larg);
                                    FREE(fname);
                                    int j;
                                    for(j=0; j<vector_size(v); j++) {
                                        string_delete(vector_item(v, j));
                                    }
                                    vector_delete(v);

                                    return FALSE;
                                }

                                /// 文 ///
                                if(*rcode == 0) {
                                    if(larg[i].statments == NULL) {
                                        string_obj* str2 = STRING_NEW(string_c_str(str));
                                        string_chomp(str2);
                                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                        {
                                            string_delete(str2);
                                            FREE(larg);
                                            FREE(fname);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);

                                            return FALSE;
                                        }
                                        string_delete(str2);
                                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                        {
                                            FREE(larg);
                                            FREE(fname);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);

                                            return FALSE;
                                        }
                                    }
                                    else {
                                        sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                                        *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                        sRFd_delete(nextin2);

                                        if(*return_) {
                                            FREE(larg);
                                            FREE(fname);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);

                                            return TRUE;
                                        }
                                        else if(*rcode < 0) {
                                            FREE(larg);
                                            FREE(fname);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);

                                            return FALSE;
                                        }
                                    }
                                }
                            }

                            FREE(fname);

                            }
                            break;

                        case 4: {
                            int n1 = larg[i].num1;
                            int n2 = larg[i].num2;

                            if(n1 < 0) {
                                n1+=vector_size(v);
                            }
                            if(n2 < 0) {
                                n2+=vector_size(v);
                            }

                            if(n1 >= 0 && n2>=0 && n1 < vector_size(v)
                                && n2 < vector_size(v)) 
                            {
                                if(larg[i].statments) {
                                    char* fname = MALLOC(strlen(sname) + 32);
                                    snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                    if(n1 < n2) {
                                        string_obj* str = STRING_NEW("");
                                        int k;
                                        for(k=n1; k<=n2; k++) {
                                            if(gKitutukiSigInt) {
                                                err_msg("interrupt", sname, sline);
                                                FREE(fname);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);
                                                string_delete(str);
                                                return FALSE;
                                            }
                                            string_obj* str2 = vector_item(v, k);
                                            string_push_back(str, string_c_str(str2));
                                        }

                                        sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                                        string_delete(str);

                                        *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                        sRFd_delete(nextin2);
                                        if(*return_) {
                                            FREE(fname);
                                            FREE(larg);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);
                                            return TRUE;
                                        }
                                        else if(*rcode < 0) {
                                            FREE(fname);
                                            FREE(larg);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);
                                            return FALSE;
                                        }
                                    }
                                    else {
                                        string_obj* str = STRING_NEW("");

                                        int k;
                                        for(k=n1; k>=n2; k--) {
                                            if(gKitutukiSigInt) {
                                                err_msg("interrupt", sname, sline);
                                                FREE(fname);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);
                                                string_delete(str);
                                                return FALSE;
                                            }

                                            string_obj* str2 = vector_item(v, k);
                                            string_push_back(str, string_c_str(str2));
                                        }

                                        sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                                        string_delete(str);

                                        *rcode = run(larg[i].statments
                                            , fname, nextout
                                            , nextin2, nexterr
                                            , FALSE, FALSE
                                            , return_, parent_blocks);
                                        sRFd_delete(nextin2);

                                        if(*return_) {
                                            FREE(fname);
                                            FREE(larg);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);
                                            return TRUE;
                                        }
                                        if(*rcode < 0) {
                                            FREE(fname);
                                            FREE(larg);
                                            int j;
                                            for(j=0; j<vector_size(v); j++) {
                                                string_delete(vector_item(v, j));
                                            }
                                            vector_delete(v);
                                            return FALSE;
                                        }
                                    }

                                    FREE(fname);
                                }
                                else {
                                    if(n1 < n2) {
                                        int k;
                                        for(k=n1; k<=n2; k++) {
                                            string_obj* str = vector_item(v, k);
                                            string_obj* str2 = STRING_NEW(string_c_str(str));
                                            string_chomp(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                            {
                                                string_delete(str2);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                            string_delete(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                            {
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                        }
                                    }
                                    else {
                                        int k;
                                        for(k=n1; k>=n2; k--) {
                                            string_obj* str = vector_item(v, k);
                                            string_obj* str2 = STRING_NEW(string_c_str(str));
                                            string_chomp(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                            {
                                                string_delete(str2);
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                            string_delete(str2);
                                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                            {
                                                FREE(larg);
                                                int j;
                                                for(j=0; j<vector_size(v); j++) {
                                                    string_delete(vector_item(v, j));
                                                }
                                                vector_delete(v);

                                                return FALSE;
                                            }
                                        }
                                    }
                                }
                            }
                            }
                            break;

                        case 5: {
                            int baisuu = larg[i].num1;
                            int start = larg[i].num2;

                            /// ブロックがあるなら ///
                            if(larg[i].statments) {
                                char* fname = MALLOC(strlen(sname) + 32);
                                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                                string_obj* str = STRING_NEW("");
                                int k = 0;
                                while(1) {
                                    int x = k * baisuu + start;

                                    if(x < 0 || x >= vector_size(v)) {
                                        break;
                                    }

                                    string_push_back(str, string_c_str(vector_item(v, x)));


                                    k++;
                                }

                                sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));

                                *rcode = run(larg[i].statments
                                        , fname, nextout
                                        , nextin2, nexterr
                                        , FALSE, FALSE
                                        , return_, parent_blocks);
                                sRFd_delete(nextin2);
                                string_delete(str);
                                if(*return_) {
                                    FREE(fname);
                                    FREE(larg);
                                    int j;
                                    for(j=0; j<vector_size(v); j++) {
                                        string_delete(vector_item(v, j));
                                    }
                                    vector_delete(v);

                                    return TRUE;
                                }
                                else if(*rcode < 0) {
                                    FREE(fname);
                                    FREE(larg);
                                    int j;
                                    for(j=0; j<vector_size(v); j++) {
                                        string_delete(vector_item(v, j));
                                    }
                                    vector_delete(v);

                                    return FALSE;
                                }

                                FREE(fname);
                            }
                            else {
                                int k = 0;
                                while(1) {
                                    int x = k * baisuu + start;

                                    if(x < 0 || x >= vector_size(v)) {
                                        break;
                                    }

                                    string_obj* str = vector_item(v, x);
                                    string_obj* str2 = STRING_NEW(string_c_str(str));
                                    string_chomp(str2);

                                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str2)))
                                    {
                                        string_delete(str2);
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);

                                        return FALSE;
                                    }
                                    string_delete(str2);
                                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                    {
                                        FREE(larg);
                                        int j;
                                        for(j=0; j<vector_size(v); j++) {
                                            string_delete(vector_item(v, j));
                                        }
                                        vector_delete(v);

                                        return FALSE;
                                    }

                                    k++;
                                }
                            }
                            }
                            break;
                    }
                }
            }

            FREE(larg);
            int j;
            for(j=0; j<vector_size(v); j++) {
                string_delete(vector_item(v, j));
            }
            vector_delete(v);
            }
            break;

        case kAryNew: {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-a") == 0) 
                {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) 
                {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) 
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) 
                {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) 
                {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) != 2) {
                    err_msg("ary_new -a ary_name", sname, sline);

                    return FALSE;
                }

                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                                    nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
                else {
                    char* key = string_c_str(vector_item(argv, 1));

                    saphire_delete_same_name_var(key);
                    vector_obj* v = VECTOR_NEW(50);
                    hash_put(gArrays, key, v);
                
                    vector_add(v, str);
                
                    update_ary_env(key);
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            string_delete(str);
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                string_delete(str);
                                return FALSE;
                            }
                        }
                    }
                }
            }
            else if(input) {
                if(vector_size(argv) != 2) {
                    err_msg("ary_new -I ary_name", sname, sline);

                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));

                saphire_delete_same_name_var(key);
                vector_obj* v = VECTOR_NEW(50);
                hash_put(gArrays, key, v);

                while(1) {
                    string_obj* line = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);
                    if(ret == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(line);
                        return FALSE;
                    }
                    else if(ret == 1) {
                        string_delete(line);
                        break;
                    }
                    if(chomp) string_chomp(line);

                    vector_add(v, line);

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }
                }
                
                update_ary_env(key);

                *rcode = 0;
            }
            else {
                if(vector_size(argv) < 2) {
                    err_msg("ary_new ary_name value1 value2...", sname, sline);

                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));
                
                saphire_delete_same_name_var(key);
                vector_obj* v = VECTOR_NEW(50);
                hash_put(gArrays, key, v);
                
                for(i=2; i<vector_size(argv); i++) {
                    char* arg = string_c_str(vector_item(argv, i));
                    vector_add(v, STRING_NEW(arg));
                    
                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, arg))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }
                }
                
                update_ary_env(key);
                
                *rcode = 0;
            }
            }
            break;

        case kAryAdd: {
            int i;
            BOOL all = FALSE;
            BOOL flg_n = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            int n;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-n") == 0 
                        && i+1 < vector_size(argv)) 
                {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-a") == 0) 
                {
                    all = TRUE;
                    string_delete(vector_item(argv,i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) 
                {
                    chomp = FALSE;
                    string_delete(vector_item(argv,i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                /// 標準入力から、設定
                if(vector_size(argv) != 2) {
                    err_msg("ary_add: invalid option", sname, sline);
                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v == NULL) {
                    saphire_delete_same_name_var(key);
                    v = VECTOR_NEW(50);
                    hash_put(gArrays, key, v);
                }

                int i=0;
                while(1) {
                    string_obj* line = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);

                    if(ret == -1) {
                        err_msg("intterrupt", sname, sline);
                        string_delete(line);
                        return FALSE;
                    }
                    else if(ret == 1) {      // EOF
                        string_delete(line);
                        break;
                    }

                    if(chomp) string_chomp(line);

                    if(flg_n) {
                        if(n < 0) {
                            n = n + vector_size(v) + 1;
                        }

                        if(n < 0) {
                            err_msg("ary_add: invalid -n range", sname, sline);
                            string_delete(line);
                            return FALSE;
                        }

                        if(n+i < vector_size(v)) {
                            vector_insert(v, n+i, line);
                        }
                        else {
                            vector_add(v, line);
                        }
                    }
                    else {
                        vector_add(v, line);
                    }

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }

                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }
                        }
                    }
                    
                    i++;
                }
               
                update_ary_env(key);

                *rcode = 0;
            }
            else if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin(
                                        nextin, str);
                    if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else if(ret < 0)
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else {
                        vector_obj* v = hash_item(gArrays, p);
                        if(v == NULL) {
                            saphire_delete_same_name_var(p);
                            v = VECTOR_NEW(50);
                            hash_put(gArrays, p, v);
                        }

                        if(flg_n) {
                            if(n < 0) {
                                n = n + vector_size(v) + 1;
                            }

                            if(n < 0) {
                                err_msg("ary_add: invalid -n range", sname, sline);
                                string_delete(str);
                                return FALSE;
                            }

                            if(n < vector_size(v)) {
                                vector_insert(v, n, str);
                            }
                            else {
                                vector_add(v, str);
                            }
                        }
                        else {
                            vector_add(v, str);
                        }

                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }

                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("singal inttrupt", sname, sline);
                                    return FALSE;
                                }
                            }
                        }
                    }

                    update_ary_env(p);
                }
                else {
                    err_msg("ary_add: ary_add -a must have one var_name", sname, sline);
                    return FALSE;
                }
            }
            else {
                if(vector_size(argv) < 3) {
                    err_msg("ary_add: invalid option", sname, sline);
                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v == NULL) {
                    saphire_delete_same_name_var(key);
                    v = VECTOR_NEW(50);
                    hash_put(gArrays, key, v);
                }
                
                if(flg_n) {
                    if(n < 0) {
                        n = n + vector_size(v) + 1;
                    }

                    if(n < 0) {
                        err_msg("ary_add: invalid -n range", sname, sline);
                        return FALSE;
                    }

                    int i;
                    for(i=2; i<vector_size(argv); i++) {
                        string_obj* line = STRING_NEW("");
                        char* str = string_c_str(vector_item(argv, i));
                        string_put(line, str);

                        if(n+i-2 < vector_size(v)) {
                            vector_insert(v, n+i-2, line);
                        }
                        else {
                            vector_add(v, line);
                        }

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }

                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
                else {
                    int i;
                    for(i=2; i<vector_size(argv); i++) {
                        string_obj* line = STRING_NEW("");
                        char* str = string_c_str(vector_item(argv, i));
                        string_put(line, str);

                        vector_add(v, line);

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                            {
                                err_msg("signal interrupt", sname, sline);
                                return FALSE;
                            }

                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal interrupt", sname, sline);
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
                
                update_ary_env(key);

                *rcode = 0;
            }
            }
            break;

        case kAryErase:
            if(vector_size(argv) == 3) {
                char* key = string_c_str(vector_item(argv, 1));
                char* number = string_c_str(vector_item(argv, 2));
                int n = atoi(number);

                vector_obj* v = hash_item(gArrays, key);
                if(v && n >= 0 && n < vector_size(v)) {
                    string_delete(vector_item(v, n));
                    vector_erase(v, n);
                    *rcode = 0;
                }
                update_ary_env(key);
            }
            break;

        case kAryClear:
            if(vector_size(argv) == 2) {
                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v) {
                    int i;
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                    }
                    vector_delete(v);

                    hash_erase(gArrays, key);
                    clear_ary_env(key);

                    *rcode = 0;
                }
            }
            break;

        case kHashNew: {
            BOOL chomp = TRUE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-nc") == 0) 
                {
                    chomp = FALSE;
                    string_delete(vector_item(argv,i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                hash_obj* hash = HASH_NEW(100);
                char* k = string_c_str(vector_item(argv, 1));

                saphire_delete_same_name_var(k);
                hash_put(gHashs, k , hash);

                BOOL key = TRUE;
                string_obj* key_str = STRING_NEW("");

                while(1) {
                    if(gKitutukiSigInt) {
                        err_msg("intterrupt", sname, sline);
                        return FALSE;
                    }
                    string_obj* line = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);

                    if(ret == -1) {
                        err_msg("intterrupt", sname, sline);
                        string_delete(line);
                        string_delete(key_str);
                        return FALSE;
                    }
                    else if(ret == 1) {      // EOF
                        string_delete(line);
                        break;
                    }

                    if(chomp || key) string_chomp(line);

                    if(key) {
                        string_put(key_str, string_c_str(line));
                        string_delete(line);
                    }
                    else {
                        hash_put(hash
                            , string_c_str(key_str), line);
                    }

                    key = !key;
                }
                string_delete(key_str);
            }
            else {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* key = string_c_str(vector_item(argv, k));
                    saphire_delete_same_name_var(key);
                    hash_put(gHashs, key, HASH_NEW(100));
                }
            }
            }
            break;

        case kHashAdd: {
            BOOL chomp = TRUE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-nc") == 0) 
                {
                    chomp = FALSE;
                    string_delete(vector_item(argv,i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                char* k = string_c_str(vector_item(argv, 1));
                hash_obj* hash = hash_item(gHashs, k);

                if(hash) {
                    BOOL key = TRUE;
                    string_obj* key_str = STRING_NEW("");

                    while(1) {
                        if(gKitutukiSigInt) {
                            err_msg("innterrupt", sname, sline);
                            return FALSE;
                        }
                        string_obj* line = STRING_NEW("");
                        int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line);

                        if(ret == -1) {
                            err_msg("intterrupt", sname, sline);
                            string_delete(line);
                            string_delete(key_str);
                            return FALSE;
                        }
                        else if(ret == 1) {      // EOF
                            string_delete(line);
                            break;
                        }

                        if(chomp || key) string_chomp(line);

                        if(key) {
                            string_put(key_str, string_c_str(line));
                            string_delete(line);
                        }
                        else {
                            string_obj* item = hash_item(hash, string_c_str(key_str));
                            if(item) {
                                string_delete(item);
                            }

                            hash_put(hash
                                    , string_c_str(key_str), line);
                        }

                        key = !key;
                    }
                    string_delete(key_str);

                    *rcode = 0;
                }
            }
            }
            break;

        case kHashErase: {
            if(vector_size(argv) == 3) {
                char* var_name = string_c_str(vector_item(argv, 1));
                char* key = string_c_str(vector_item(argv, 2));

                hash_obj* hash = hash_item(gHashs, var_name);
                if(hash) {
                    string_obj* item = hash_item(hash, key);
                    if(item) {
                        string_delete(item);
                        hash_erase(hash, key);
                        *rcode = 0;
                    }
                }
            }
            }
            break;

        case kHashClear: {
            int k;
            for(k=1; k<vector_size(argv); k++) {
                string_obj* str = vector_item(argv, k);

                hash_obj* hash = hash_item(gHashs, string_c_str(str));

                if(hash) {
                    hash_it* it2 = hash_loop_begin(hash);
                    while(it2 != NULL) {
                        string_delete(hash_loop_item(it2));
                        it2 = hash_loop_next(it2);
                    }
                    hash_delete(hash);

                    hash_erase(gHashs, string_c_str(str));

                    *rcode = 0;
                }
            }
            }
            break;

        case kScan: {
            BOOL ignore_case = FALSE;
            BOOL multiline = FALSE;
            string_obj* field = STRING_NEW("\t");
            BOOL quiet = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-i") == 0)
                {
                    ignore_case = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0)
                {
                    quiet = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0)
                {
                    multiline = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0)
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    string_put(field 
                         , string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            saphire_delete_local_var("PREMATCH");
            saphire_delete_local_var("MATCH");
            saphire_delete_local_var("POSTMATCH");
            saphire_delete_local_var("MATCH_COUNT");

            if(multiline) {
                if(!statment_tree_internal_commands_global_match(argv, field, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline, quiet)) {
                    string_delete(field);
                    return FALSE;
                }
            }
            else {
                if(!statment_tree_internal_commands_global_match_oneline(argv, field, input , nextout, nextin, rcode, ignore_case, multiline, sname, sline, quiet))
                {
                    string_delete(field);
                    return FALSE;
                }
            }

            string_delete(field);
            }
            break;

        case kErase: {
            BOOL ignore_case = FALSE;
            BOOL multiline = FALSE;
            BOOL global = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-i") == 0)
                {
                    ignore_case = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-g") == 0)
                {
                    global = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0)
                {
                    multiline = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0)
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(global) {
                if(multiline) {
                    if(!statment_tree_internal_commands_erase_global(argv, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline)) {
                        return FALSE;
                    }
                }
                else {
                    if(input) {
                        if(!statment_tree_internal_commands_erase_global_oneline(argv, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline)) {
                            return FALSE;
                        }
                    }
                    else {
                        if(!statment_tree_internal_commands_erase_global(argv, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline)) {
                            return FALSE;
                        }
                    }
                }
            }
            else {
                if(!statment_tree_internal_commands_erase(argv, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline)) {
                    return FALSE;
                }
            }

            }
            break;

        case kMatch: {
            BOOL quiet = FALSE;
            BOOL line_field = TRUE;
            BOOL all = FALSE;
            BOOL line_num = FALSE;
            BOOL global = FALSE;
            BOOL line_oriented = FALSE;
            BOOL ignore_case = FALSE;
            BOOL multiline = FALSE;
            BOOL restart = FALSE;
            string_obj* field = STRING_NEW("\t");
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-q") == 0)
                {
                    quiet = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-r") == 0) {
                    restart = TRUE;
                    global = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-L") == 0)
                {
                    line_oriented = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0)
                {
                    ignore_case = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0)
                {
                    multiline = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-g") == 0)
                {
                    global = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-a") == 0)
                {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nl") == 0)
                {
                    line_field = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0)
                {
                    line_num = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0)
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    string_put(field 
                         , string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            saphire_delete_local_var("PREMATCH");
            saphire_delete_local_var("MATCH");
            saphire_delete_local_var("POSTMATCH");
            saphire_delete_local_var("MATCH_COUNT");

            if(global) {
                if(multiline) {
                    if(!statment_tree_internal_commands_global_match(argv, field, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline, quiet)) {
                        string_delete(field);
                        return FALSE;
                    }
                }
                else {
                    if(input) {
                        if(!statment_tree_internal_commands_global_match_oneline(argv, field, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline, quiet)) {
                            string_delete(field);
                            return FALSE;
                        }
                    }
                    else {
                        if(!statment_tree_internal_commands_global_match(argv, field, input, nextout, nextin, rcode, ignore_case, multiline, sname, sline, quiet)) {
                            string_delete(field);
                            return FALSE;
                        }
                    }
                }
            }
            else if(input) {
                int match_count = 0;

                if(vector_size(argv) >= 2) {
                    if(line_oriented) {
                        char* regex = string_c_str(vector_item(argv, 1));
                        char* regex2 = regex;

                        int n = 1;
                        for(;;) {
                            string_obj* str = STRING_NEW("");
                            int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                            if(ret == -1) {
                                err_msg("interrupt", sname, sline);
                                string_delete(str);
                                string_delete(field);
                                return FALSE;
                            }
                            else if(ret == 1) 
                            {
                                string_delete(str);
                                break;
                            }

                            string_chomp(str);
                            
                            if(!statment_tree_internal_commands_match(
                                string_c_str(str), regex2, rcode, nextout
                                , !quiet, line_field, line_num, n, field
                                , line_oriented, &match_count
                                , ignore_case
                                , multiline
                                , restart
                                , sname
                                , sline
                                ))
                            {
                                string_delete(str);
                                string_delete(field);
                                return FALSE;
                            }

                            string_delete(str);

                            n++;
                        }
                    }
                    else {
                        char* regex = string_c_str(vector_item(argv, 1));
                        char* regex2 = regex;

                        int n = 0;
                        string_obj* str = STRING_NEW("");
                        BOOL ret = statment_tree_internal_commands_read_nextin(nextin, str);
                        if(ret == 1) {
                            *rcode = 1;
                            string_delete(str);
                            string_delete(field);
                            break;
                        }
                        else if(ret < 0) {
                            err_msg("interrupt", sname, sline);
                            string_delete(str);
                            string_delete(field);
                            return FALSE;
                        }
                        
                        if(!statment_tree_internal_commands_match(
                            string_c_str(str), regex2, rcode, nextout
                            , !quiet, line_field, line_num, n, field
                            , line_oriented, &match_count, ignore_case
                            , multiline
                            , restart
                            , sname
                            , sline
                            ))
                        {
                            string_delete(str);
                            string_delete(field);
                            return FALSE;
                        }

                        string_delete(str);
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", match_count);
                saphire_set_local_var("MATCH_COUNT", buf);
            }
            else {
                int match_count = 0;

                string_obj* str;
                if(vector_size(argv) == 3) {
                    str = STRING_NEW(string_c_str(vector_item(argv, 1)));
                }
                else {
                    string_delete(field);
                    break;
                }

                char* str2 = string_c_str(str);

                char* regex = string_c_str(vector_item(argv, 2));
                char* regex2 = regex;

                if(!statment_tree_internal_commands_match(
                    str2, regex2, rcode, nextout, !quiet, line_field
                    , line_num, 1, field, line_oriented, &match_count
                    , ignore_case
                    , multiline
                    , restart
                    , sname
                    , sline
                    ))
                {
                    string_delete(str);
                    string_delete(field);
                    return FALSE;
                }

                string_delete(str);

                char buf[256];
                snprintf(buf, 256, "%d", match_count);
                saphire_set_local_var("MATCH_COUNT", buf);
            }

            string_delete(field);
            }
            break;

        case kPos: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                if(vector_size(argv) == 1) {
                    string_obj* str = STRING_NEW("");

                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else if(ret < 0) 
                    {
                        err_msg("intterrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    int position = (int)hash_item(gMatchRestart, string_c_str(str));
                    string_delete(str);

                    char buf[1024];
                    snprintf(buf, 1024, "%d\n", position);

                    if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
                else if(vector_size(argv) == 2) {
                    string_obj* str = STRING_NEW("");

                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else if(ret < 0) 
                    {
                        err_msg("intterrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }

                    int position = atoi(string_c_str(vector_item(argv, 1)));
                    if(position < 0) {
                        position += string_length(str);
                    }
                    if(position >= 0 && position < string_length(str)) {
                        hash_put(gMatchRestart, string_c_str(str), (void*)position);
                    }

                    string_delete(str);
                }
            }
            else if(vector_size(argv) == 2){
                char* str = string_c_str(vector_item(argv, 1));
                int position = (int)hash_item(gMatchRestart, str);

                char buf[1024];
                snprintf(buf, 1024, "%d\n", position);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            else if(vector_size(argv) == 3) {
                char* str = string_c_str(vector_item(argv, 1));
                int position = atoi(string_c_str(vector_item(argv, 2)));
                if(position < 0) {
                    position += strlen(str);
                }
                if(position >= 0 && position < strlen(str)) {
                    hash_put(gMatchRestart, str, (void*)position);
                }
            }

            }
            break;

        case kSub: {
            BOOL global = FALSE;
            BOOL quiet = FALSE;
            BOOL multiline = FALSE;
            BOOL check = FALSE;
            BOOL ignore_case = FALSE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-g") == 0) {
                    global = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    ignore_case = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0) {
                    multiline = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0) {
                    check = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }
            
            if(input) {
                if(multiline || !global) {
                    if(vector_size(argv) == 3) {
                        string_obj* str = STRING_NEW("");

                        int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                        if(ret == 1) { // EOF
                            string_delete(str);
                            *rcode = 1;
                            break;
                        }
                        else if(ret == -1) {
                            err_msg("interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }

                        char* regex = string_c_str(vector_item(argv, 1));
                        char* destination = string_c_str(vector_item(argv, 2));

                        if(!statment_tree_internal_commands_sub(str, regex, destination, global, quiet, multiline, check,ignore_case, rcode, nextout, sname, sline))
                        {
                            string_delete(str);
                            return FALSE;
                        }

                        string_delete(str);
                    }
                }
                else {
                    if(vector_size(argv) == 3) {
                        char* regex = string_c_str(vector_item(argv, 1));
                        char* destination = string_c_str(vector_item(argv, 2));
                        if(!statment_tree_internal_commands_sub_global_oneline(regex, destination, global, quiet, multiline, check,ignore_case, rcode, nextout, nextin, sname, sline))
                        {
                            return FALSE;
                        }
                    }
                }
            }
            else if(vector_size(argv) == 4) {
                string_obj* str = vector_item(argv, 1);
                char* regex = string_c_str(vector_item(argv, 2));
                char* destination = string_c_str(vector_item(argv, 3));
                if(!statment_tree_internal_commands_sub(str, regex, destination, global, quiet, multiline, check,ignore_case, rcode, nextout, sname, sline))
                    return FALSE;
            }
            }
            break;

        case kRead: {
            BOOL flg_n = FALSE;
            int n;
            BOOL all = FALSE;
            BOOL preserve = FALSE;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    preserve = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(vector_size(argv) == 1) {
                if(all) {
                    string_obj* str = STRING_NEW("");
                    int ret;
                    if(preserve) {
                        ret = statment_tree_internal_commands_read_nextin_preserve(nextin, str);
                    }
                    else {
                        ret = statment_tree_internal_commands_read_nextin(nextin, str);
                    }

                    if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        break;
                    }
                    else if(ret < 0)
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }

                        *rcode = 0;
                    }

                    string_delete(str);
                }
                else if(flg_n) {
                    if(preserve) {
                        string_obj* out = STRING_NEW("");

                        int result = statment_tree_internal_commands_read_nextin_oneline_preserve_num(nextin, out, n);

                        if(result == -1) {
                            err_msg("interrupt", sname, sline);
                            string_delete(out);

                            return FALSE;
                        }
                        else if(result == 1) {
                            break;
                        }

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(out)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(out);

                            return FALSE;
                        }
                        *rcode = 0;

                        string_delete(out);
                    }
                    else {
                        string_obj* out = STRING_NEW("");
                        int i;
                        for(i=0; i<n; i++) {
                            string_obj* str = STRING_NEW("");
                            int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);

                            if(result == -1) {
                                err_msg("interrupt", sname, sline);
                                string_delete(str);
                                string_delete(out);

                                return FALSE;
                            }
                            else if(result == 1) {
                                string_delete(str);
                                break;
                            }

                            string_push_back(out, string_c_str(str));
                            string_delete(str);
                        }

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(out)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(out);

                            return FALSE;
                        }
                        *rcode = 0;

                        string_delete(out);
                    }
                }
                else {
                    string_obj* str = STRING_NEW("");
                    int result;
                    if(preserve) {
                        result = statment_tree_internal_commands_read_nextin_oneline_preserve(nextin, str);
                    }
                    else {
                        result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    }

                    if(result == 0) { 
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);

                            return FALSE;
                        }
                        *rcode = 0;
                    }
                    else if(result == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);

                        return FALSE;
                    }

                    string_delete(str);
                }
            }
            else if(vector_size(argv) == 2) {
                char* fname = string_c_str(vector_item(argv, 1));

                sRFd* rfd = hash_item(gReadBuffers, fname);
                if(rfd == 0) {
                    int fd = open(fname, O_RDONLY);

                    if(fd < 0) {
                        char buf[64];
                        snprintf(buf, 64, "read: can't open file %s", fname);
                        err_msg(buf, sname, sline);
                        return FALSE;
                    }
                    
                    rfd = sRFd_new(fd);
                    hash_put(gReadBuffers, fname, rfd);
                }

                if(all) {
                    string_obj* str = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin(rfd, str);
                    if(ret == 1) {
                        *rcode = 1;
                        string_delete(str);
                        sRFd_close(rfd);
                        sRFd_delete(rfd);
                        hash_erase(gReadBuffers, fname);
                        break;
                    }
                    else if(ret < 0)
                    {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }

                        sRFd_close(rfd);
                        sRFd_delete(rfd);
                        hash_erase(gReadBuffers, fname);

                        *rcode = 0;
                    }

                    string_delete(str);
                }
                else if(flg_n) {
                    int i;
                    for(i=0; i<n; i++) {
                        string_obj* str = STRING_NEW("");
                        int result = statment_tree_internal_commands_read_nextin_oneline(rfd, str);

                        if(result == 0) { 
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                            {
                                err_msg("signal interrupt", sname, sline);
                                string_delete(str);

                                return FALSE;
                            }
                            *rcode = 0;
                        }
                        else if(result == -1) {
                            err_msg("interrupt", sname, sline);
                            string_delete(str);
                            return FALSE;
                        }
                        else {
                            sRFd_close(rfd);
                            sRFd_delete(rfd);
                            hash_erase(gReadBuffers, fname);
                            string_delete(str);
                            break;
                        }

                        string_delete(str);
                    }
                }
                else {
                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(rfd, str);

                    if(result == 0) { 
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal interrupt", sname, sline);
                            string_delete(str);

                            return FALSE;
                        }
                        *rcode = 0;
                    }
                    else if(result == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else {
                        sRFd_close(rfd);
                        sRFd_delete(rfd);
                        hash_erase(gReadBuffers, fname);
                    }

                    string_delete(str);
                }
            }

            }
            break;

        case kCd:
            if(vector_size(argv) == 1) {
                char* path = getenv("HOME");
                char cwd[PATH_MAX];
                char path2[PATH_MAX];

                mygetcwd(cwd);
                if(correct_path(cwd, path, path2)) {
                    struct stat stat_;
                    if(stat(path2, &stat_) == 0) {
                        if(S_ISDIR(stat_.st_mode)) {
                            setenv("PWD", path2, 1);
                            if(chdir(path2) >= 0) {
                                *rcode = 0;
                            }
                        }
                    }
                }
            }
            else if(vector_size(argv) == 2) {
                char* path = string_c_str(vector_item(argv, 1));
                char cwd[PATH_MAX];
                char path2[PATH_MAX];

                mygetcwd(cwd);
                if(correct_path(cwd, path, path2)) {
                    struct stat stat_;
                    if(stat(path2, &stat_) == 0) {
                        if(S_ISDIR(stat_.st_mode)) {
                            setenv("PWD", path2, 1);
                            if(chdir(path2) >= 0) {
                                *rcode = 0;
                            }
                        }
                    }
                }
            }
            break;

        case kSelector: {
            static int scrolltop = 0;
            static int cursor = 0;
            enum eKanjiCode code = gKanjiCode;
            BOOL multiple = FALSE;
            BOOL option_r_exist = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-r") == 0) {
                    option_r_exist = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    cursor = atoi(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-t") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    scrolltop = atoi(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0) {
                    multiple = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(!option_r_exist) {
                scrolltop = 0;
                cursor = 0;
            }

            string_obj* str = STRING_NEW("");
            int ret = statment_tree_internal_commands_read_nextin(nextin,str) ;
            if(ret == 1) {
                *rcode = 1;
                string_delete(str);
                break;
            }
            else if(ret < 0)
            {
                err_msg("interrupt", sname, sline);
                string_delete(str);
                return FALSE;
            }
            string_obj* result = STRING_NEW("");
            statment_tree_internal_commands_selector(str,result
                    , &scrolltop, &cursor, code, multiple, gLineField);

            *rcode = 0;

            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(result)))
            {
                err_msg("sinal interrupt", sname, sline);

                string_delete(result);
                string_delete(str);

                return FALSE;
            }
            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
            {
                err_msg("signal interrupt", sname, sline);
                string_delete(result);
                string_delete(str);

                return FALSE;
            }

            string_delete(result);
            string_delete(str);
            }
            break;

        case kMax: {
            BOOL new_line = FALSE;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(vector_size(argv) == 3) {
                int left = atoi(string_c_str(vector_item(argv, 1)));
                int right = atoi(string_c_str(vector_item(argv, 2)));

                if(left > right) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 1))))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 2))))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                    }
                }
            }
            }
            break;

        case kMin: {
            BOOL new_line = FALSE;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(vector_size(argv) == 3) {
                int left = atoi(string_c_str(vector_item(argv, 1)));
                int right = atoi(string_c_str(vector_item(argv, 2)));

                if(left < right) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 1))))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 2))))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal interrupt", sname, sline);
                            return FALSE;
                        }
                    }
                }
            }
            }
            break;
        

        case kExtName: {
            BOOL new_line = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                string_obj* item = vector_item(argv, i);
                if(strcmp(string_c_str(item), "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(string_c_str(item), "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 1 && input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                char* target = string_c_str(str);
                char path[PATH_MAX];
                extname(path, target);
                if(new_line) strcat(path, "\n");
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                string_delete(str);

                *rcode = 0;
            }
            else if(vector_size(argv) == 2) {
                char path[PATH_MAX];
                extname(path, string_c_str(vector_item(argv, 1)));
                if(new_line) strcat(path, "\n");
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }

                *rcode = 0;
            }
            }
            break;

        case kParentName: {
            BOOL new_line = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                string_obj* item = vector_item(argv, i);
                if(strcmp(string_c_str(item), "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(string_c_str(item), "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(vector_size(argv) == 1 && input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                char* target = string_c_str(str);
                char path[PATH_MAX];
                parentname(path, target);
                if(new_line) strcat(path, "\n");
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                string_delete(str);

                *rcode = 0;
            }
            else if(vector_size(argv) == 2) {
                char path[PATH_MAX];
                parentname(path, string_c_str(vector_item(argv, 1)));
                if(new_line) strcat(path, "\n");

                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
                *rcode = 0;
            }
            }
            break;

        case kNoExtName: {
            BOOL new_line = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                string_obj* item = vector_item(argv, i);
                if(strcmp(string_c_str(item), "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(string_c_str(item), "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(vector_size(argv) == 1 && input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(
                        nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                char* target = string_c_str(str);
                char path[PATH_MAX];
                noextname(path, target);
                if(new_line) strcat(path, "\n");
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                string_delete(str);

                *rcode = 0;
            }
            else if(vector_size(argv) == 2) {
                char path[PATH_MAX];
                noextname(path, string_c_str(vector_item(argv, 1)));
                if(new_line) strcat(path, "\n");
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }

                *rcode = 0;
            }
            }
            break;

        case kPopd: {
            string_obj* pwd = vector_pop_back(gDirStack);

            if(pwd) {
                char* path = string_c_str(pwd);
                char cwd[PATH_MAX];
                char path2[PATH_MAX];

                mygetcwd(cwd);
                if(correct_path(cwd, path, path2)) {
                    struct stat stat_;
                    if(stat(path2, &stat_) == 0) {
                        if(S_ISDIR(stat_.st_mode)) {
                            setenv("PWD", path2, 1);
                            if(chdir(path2) >= 0) {
                                *rcode = 0;
                            }
                        }
                    }
                }
                string_delete(pwd);
            }
            }
            break;

        case kPushd:
            vector_add(gDirStack, STRING_NEW(getenv("PWD")));
            *rcode = 0;
            break;

        case kSort: {
            char* sort_block;
            if(vector_size(argv) == 1) {
                sort_block = "{ [ $a -slt $b ] }";
            }
            else if(vector_size(argv) == 2) {
                sort_block = string_c_str(vector_item(argv, 1));
            }
            else {
                break;
            }

            vector_obj* v = VECTOR_NEW(100);
            while(1) {
                if(gKitutukiSigInt)  {
                    err_msg("intterrupt", sname, sline);
                    int i;
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                    }
                    vector_delete(v);
                    return FALSE;
                }
                string_obj* str = STRING_NEW("");
                int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);

                if(result == -1) {  // CTRL_C
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    int i;
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                    }
                    vector_delete(v);
                    return FALSE;
                }
                else if(result == 1) { // EOF
                    string_delete(str);
                    break;
                }

                vector_add(v, str);
            }

            char* sort_block2 = MALLOC(strlen(sort_block)-2+1);
            memcpy(sort_block2, sort_block + 1, strlen(sort_block)-2);
            sort_block2[strlen(sort_block)-2] = 0;
            gSortBlock = sort_block2;
            gSortNextOut = nextout;
            gSortNextIn = nextin;
            gSortNextErr = nexterr;
            if(!vector_sort(v, sort_fun)) {
                err_msg("interrupt", sname, sline);
                FREE(sort_block2);
                int i;
                for(i=0; i<vector_size(v); i++) {
                    string_delete(vector_item(v, i));
                }
                vector_delete(v);
                return FALSE;
            }

            FREE(sort_block2);

            /// 出力 ///
            int i;
            for(i=0; i<vector_size(v); i++) {
                if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(v, i))))
                {
                    err_msg("signal interrupt", sname, sline);
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                    }
                    vector_delete(v);
                    return FALSE;
                }
            }

            for(i=0; i<vector_size(v); i++) {
                string_delete(vector_item(v, i));
            }
            vector_delete(v);
            }
            break;

        case kMinusMinus: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) - 1;
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    char* arg = string_c_str(vector_item(argv, 1));

                    char* var = saphire_get_local_var(arg);
                    if(var) {
                        int n = atoi(var) - 1;
                        char buf[128];
                        snprintf(buf, 128, "%d", n);
                        saphire_set_local_var(arg, buf);
                    }
                }
            }
            }
            break;

        case kPlusPlus: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    string_delete(str);
                    *rcode = 1;
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) + 1;
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    char* arg = string_c_str(vector_item(argv, 1));

                    char* var = saphire_get_local_var(arg);
                    if(var) {
                        int n = atoi(var) + 1;
                        char buf[128];
                        snprintf(buf, 128, "%d", n);
                        saphire_set_local_var(arg, buf);
                    }
                }
            }
            }
            break;

        case kPlus: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) + atoi(string_c_str(vector_item(argv, 1)));
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            }
            break;

        case kMinus: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) - atoi(string_c_str(vector_item(argv, 1)));
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            }
            break;

        case kMult: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) * atoi(string_c_str(vector_item(argv, 1)));
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            }
            break;

        case kDiv: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0) 
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) / atoi(string_c_str(vector_item(argv, 1)));
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            }
            break;

        case kMod: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = atoi(string_c_str(str)) % atoi(string_c_str(vector_item(argv, 1)));
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            }
            break;

        case kPow: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str);
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                int n = pow(atoi(string_c_str(str)), atoi(string_c_str(vector_item(argv, 1))));
                char buf[128];
                snprintf(buf, 128, "%d", n);

                string_delete(str);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
            }
            break;

        case kRange: {
            if(vector_size(argv) == 3) {
                int first = atoi(string_c_str(vector_item(argv, 1)));
                int last = atoi(string_c_str(vector_item(argv, 2)));

                int i;
                for(i=first; i<=last; i++) {
                    char buf[BUFSIZ];
                    snprintf(buf, BUFSIZ, "%d\n", i);
                    if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                    {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                }
            }
            }
            break;

        case kPrintf: {
            int i;
            BOOL line_field = FALSE;
            BOOL check = FALSE;
            enum eKanjiCode code = gKanjiCode;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0) {
                    check = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(argv) == 2) {
                /// 引数を取る ///
                vector_obj* v = VECTOR_NEW(30);
                while(1) {
                    if(gKitutukiSigInt) {
                        err_msg("interrupt", sname, sline);
                        for(j=0; j<vector_size(v); j++) {
                           string_delete(vector_item(v, j));
                        }
                        vector_delete(v);
                        return FALSE;
                    }

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);
                    if(result == 0) {
                        string_chomp(str);
                        vector_add(v, str);
                    }
                    else if(result == -1) {
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        for(j=0; j<vector_size(v); j++) {
                           string_delete(vector_item(v, j));
                        }
                        vector_delete(v);
                        return FALSE;
                    }
                    else {
                        string_delete(str);
                        break;
                    }
                }

                /// 本題 ///
                char* format = string_c_str(vector_item(argv, 1));

                if(!statment_tree_internal_commands_printf(format, v, nextout, check, line_field, sname, sline, code)) {
                    int j;
                    for(j=0; j<vector_size(v); j++) {
                        string_delete(vector_item(v, j));
                    }
                    vector_delete(v);

                    return FALSE;
                }

                /// 廃棄 ///
                int j;
                for(j=0; j<vector_size(v); j++) {
                    string_delete(vector_item(v, j));
                }
                vector_delete(v);
            }
            else if(vector_size(argv) > 2) {
                /// 引数を取る ///
                vector_obj* v = VECTOR_NEW(30);
                int i;
                for(i=2; i<vector_size(argv); i++) {
                    vector_add(v, STRING_NEW(string_c_str(vector_item(argv, i))));
                }

                /// 本題 ///
                char* format = string_c_str(vector_item(argv, 1));

                if(!statment_tree_internal_commands_printf(format, v, nextout, check, line_field, sname, sline, code)) {
                    int j;
                    for(j=0; j<vector_size(v); j++) {
                        string_delete(vector_item(v, j));
                    }
                    vector_delete(v);

                    return FALSE;
                }

                /// 廃棄 ///
                int j;
                for(j=0; j<vector_size(v); j++) {
                    string_delete(vector_item(v, j));
                }
                vector_delete(v);
            }
            
            }
            break;

        case kPomch: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str;
            if(input) {
                str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str) ;
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0) {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
            }
            else {
                if(vector_size(argv) >= 2) {
                    str = STRING_NEW(string_c_str(vector_item(argv, 1)));
                }
                else {
                    str = STRING_NEW("");
                }
            }

            string_pomch(str);
            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str))) {
                err_msg("interrupt", sname, sline);
                string_delete(str);
                return FALSE;
            }
            *rcode = 0;
            string_delete(str);
            }
            break;

        case kPTee: {
            string_obj* str = STRING_NEW("");
            int ret = statment_tree_internal_commands_read_nextin(nextin, str) ;
            if(ret == 1) {
                *rcode = 1;
                string_delete(str);
                break;
            }
            else if(ret < 0) {
                err_msg("interrupt", sname, sline);
                string_delete(str);
                return FALSE;
            }

            int i;
            for(i=0; i<vector_size(blocks); i++) {
                sStatments* statments = vector_item(blocks, i);
                sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));

                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: ptee", sname, sline);
                *rcode = run(statments, fname, nextout
                            , nextin2, nexterr
                            , FALSE, FALSE
                            , return_, parent_blocks);
                FREE(fname);

                sRFd_delete(nextin2);

                if(*rcode < 0) {
                    string_delete(str);
                    return FALSE;
                }
                else if(*return_) {
                    string_delete(str);
                    return TRUE;
                }
            }

            string_delete(str);
            }
            break;

        case kPCat: {
            int i;
            for(i=0; i<vector_size(blocks); i++) {
                sStatments* statments = vector_item(blocks, i);

                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: pcat", sname, sline);
                *rcode = run(statments, fname, nextout
                            , nextin, nexterr
                            , FALSE, FALSE
                            , return_, parent_blocks);
                FREE(fname);

                if(*return_) {
                    return TRUE;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
            }
            }
            break;

        case kReturn: {
            if(vector_size(argv) == 1) {
                *return_ = TRUE;
                *rcode = 0;
                return TRUE;
            }
            else {
                *return_ = TRUE;
                *rcode = atoi(string_c_str(vector_item(argv, 1)));
                return TRUE;
            }
            }
            break;

        case kMInitScr:
            minitscr();
            *rcode = 0;
            break;

        case kMEndWin:
            mendwin();
            *rcode = 0;
            break;

        case kMClear:
            mclear();
            *rcode = 0;
            break;

        case kMClearImmediately:
            mclear_immediately();
            *rcode = 0;
            break;

        case kMRefresh:
            mrefresh();
            *rcode = 0;
            break;

        case kMMove:
            if(vector_size(argv) == 3) {
                mmove(atoi(string_c_str(vector_item(argv, 1)))
                    , atoi(string_c_str(vector_item(argv, 2))));
                *rcode = 0;
            }
            break;

        case kMMoveImmediately:
            if(vector_size(argv) == 3) {
                mmove_immediately(atoi(string_c_str(vector_item(argv, 1)))
                    , atoi(string_c_str(vector_item(argv, 2))));
                *rcode = 0;
            }
            break;

        case kMPrintw: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str;
            if(input) {
                str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str) ;
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0) {
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }
            }
            else {
                if(vector_size(argv) >= 2) {
                    str = STRING_NEW(string_c_str(vector_item(argv, 1)));
                }
                else {
                    str = STRING_NEW("");
                }
            }

            *rcode = 0;
            mprintw("%s", string_c_str(str));
            string_delete(str);
            }
            break;

        case kSweep: {
            int i;
            BOOL funcs = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-f") == 0) {
                    funcs = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            int len = vector_size(gGlobalPipes);
            for(i=0; i<len; i++) {
                string_delete(vector_item(gGlobalPipes, i));
            }
            vector_clear(gGlobalPipes);

            len = vector_size(gQGlobalPipes);
            for(i=0; i<len; i++) {
                string_delete(vector_item(gQGlobalPipes, i));
            }
            vector_clear(gQGlobalPipes);

            hash_it* it = hash_loop_begin(gGlobals);
            while(it != NULL) {
                string_delete(hash_loop_item(it));
                it = hash_loop_next(it);
            }
            hash_clear(gGlobals);

            it = hash_loop_begin(gArrays);
            while(it != NULL) {
                vector_obj* v = hash_loop_item(it);

                int i;
                for(i=0; i<vector_size(v); i++) {
                    string_delete(vector_item(v, i));
                }
                vector_delete(v);

                it = hash_loop_next(it);
            }
            hash_clear(gArrays);

            it = hash_loop_begin(gHashs);
            while(it != NULL) {
                hash_obj* h = hash_loop_item(it);

                hash_it* it2 = hash_loop_begin(h);
                while(it2 != NULL) {
                    string_delete(hash_loop_item(it2));
                    it2 = hash_loop_next(it2);
                }
                hash_delete(h);

                it = hash_loop_next(it);
            }
            hash_clear(gHashs);

            it = hash_loop_begin(gRegexs);
            while(it != NULL) {
                onig_free(hash_loop_item(it));
                it = hash_loop_next(it);
            }
            hash_clear(gRegexs);

            it = hash_loop_begin(gRegexsI);
            while(it != NULL) {
                onig_free(hash_loop_item(it));
                it = hash_loop_next(it);
            }
            hash_clear(gRegexsI);

            it = hash_loop_begin(gRegexsM);
            while(it != NULL) {
                onig_free(hash_loop_item(it));
                it = hash_loop_next(it);
            }
            hash_clear(gRegexsM);

            it = hash_loop_begin(gRegexsIM);
            while(it != NULL) {
                onig_free(hash_loop_item(it));
                it = hash_loop_next(it);
            }
            hash_clear(gRegexsIM);

            if(funcs) {
                it = hash_loop_begin(gFuncs);
                while(it != NULL) {
                    sFunction_delete(hash_loop_item(it));
                    it = hash_loop_next(it);
                }
                hash_clear(gFuncs);
            }
            
            *rcode = 0;
            }
            break;

        case kShow: {
            int i;
            BOOL verbose = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-v") == 0) {
                    verbose = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }
            if(!statment_tree_internal_commands_write_nextout(nextout, "+++ Globals +++\n"))
            {
                err_msg("interrupt", sname, sline);
                return FALSE;
            }

            hash_it* it = hash_loop_begin(gGlobals);
            while(it != NULL) {
                if(!statment_tree_internal_commands_write_nextout(nextout,  hash_loop_key(it))) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }
                if(verbose) {
                    if(!statment_tree_internal_commands_write_nextout(nextout,   ":")) 
                    {
                        err_msg("interrupt", sname, sline);
                        return FALSE;
                    }
                    if(!statment_tree_internal_commands_write_nextout(nextout,   string_c_str(hash_loop_item(it)))) 
                    {
                        err_msg("interrupt", sname, sline);
                        return FALSE;
                    }
                }
                if(!statment_tree_internal_commands_write_nextout(nextout,  "\n")) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }

                it = hash_loop_next(it);
            }

            if(!statment_tree_internal_commands_write_nextout(nextout, "+++ Arrays +++\n"))
            {
                err_msg("interrupt", sname, sline);
                return FALSE;
            }

            it = hash_loop_begin(gArrays);
            while(it != NULL) {
                if(!statment_tree_internal_commands_write_nextout(nextout,  hash_loop_key(it))) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }
                if(!statment_tree_internal_commands_write_nextout(nextout,  "\n")) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }

                it = hash_loop_next(it);
            }

            if(!statment_tree_internal_commands_write_nextout(nextout, "+++ Hashs +++\n"))
            {
                err_msg("interrupt", sname, sline);
                return FALSE;
            }

            it = hash_loop_begin(gHashs);
            while(it != NULL) {
                if(!statment_tree_internal_commands_write_nextout(nextout,  hash_loop_key(it))) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }
                if(!statment_tree_internal_commands_write_nextout(nextout,  "\n")) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }

                it = hash_loop_next(it);
            }

            if(!statment_tree_internal_commands_write_nextout(nextout, "+++ Funcs +++\n"))
            {
                err_msg("interrupt", sname, sline);
                return FALSE;
            }

            it = hash_loop_begin(gFuncs);
            while(it != NULL) {
                if(!statment_tree_internal_commands_write_nextout(nextout,  hash_loop_key(it))) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }
                if(!statment_tree_internal_commands_write_nextout(nextout,  "\n")) 
                {
                    err_msg("interrupt", sname, sline);
                    return FALSE;
                }

                it = hash_loop_next(it);
            }
            
            *rcode = 0;
            }
            break;

        case kYeild: {
            int block_num;
            if(vector_size(argv) > 1) {
                block_num = atoi(string_c_str(vector_item(argv, 1)));
            }
            else {
                block_num = 0;
            }

            if(parent_blocks && block_num < vector_size(parent_blocks)) {
                sStatments* statments = vector_item(parent_blocks, block_num);

                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: yeild", sname, sline);

                *rcode = run(statments, fname, nextout
                            , nextin, nexterr
                            , FALSE, FALSE
                            , return_, parent_blocks);

                if(*return_) {
                    FREE(fname);
                    return TRUE;
                }
                if(*rcode < 0) {
                    FREE(fname);
                    return FALSE;
                }
                FREE(fname);
            }
            }
            break;

        case kEach: {
            if(vector_size(blocks) >= 1) {
                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: each", sname, sline);
                sStatments* block = vector_item(blocks, 0);

                while(1) {
                    if(gKitutukiSigInt)  {
                        err_msg("intterrupt", sname, sline);
                        return FALSE;
                    }
                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str);

                    if(result == -1) {  // CTRL_C
                        err_msg("interrupt", sname, sline);
                        string_delete(str);
                        return FALSE;
                    }
                    else if(result == 1) { // EOF
                        string_delete(str);
                        break;
                    }

                    sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                    *rcode = run(block
                            , fname, nextout
                            , nextin2, nexterr
                            , FALSE, FALSE
                            , return_, parent_blocks);
                    sRFd_delete(nextin2);
                    string_delete(str);

                    if(*return_) {
                        FREE(fname);
                        return TRUE;
                    }
                    else if(*rcode < 0) {
                        FREE(fname);
                        return FALSE;
                    }
                }

                FREE(fname);
            }
            }
            break;

        case kForeach: {
            if(vector_size(blocks) >= 1) {
                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: each", sname, sline);
                sStatments* block = vector_item(blocks, 0);

                int i;
                for(i=1; i<vector_size(argv); i++) {
                    if(gKitutukiSigInt)  {
                        err_msg("intterrupt", sname, sline);
                        return FALSE;
                    }
                    string_obj* str = vector_item(argv, i);

                    sRFd* nextin2 = sRFd_new2(-1, string_c_str(str));
                    *rcode = run(block
                            , fname, nextout
                            , nextin2, nexterr
                            , FALSE, FALSE
                            , return_, parent_blocks);
                    sRFd_delete(nextin2);

                    if(*return_) {
                        FREE(fname);
                        return TRUE;
                    }
                    else if(*rcode < 0) {
                        FREE(fname);
                        return FALSE;
                    }
                }

                FREE(fname);
            }
            }
            break;
/*
        case kParse: {
            enum eKanjiCode code = gKanjiCode;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-I") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input && vector_size(blocks) == 1) {
                string_obj* str = STRING_NEW("");
                int ret = statment_tree_internal_commands_read_nextin(nextin, str) ;
                if(ret == 1) {
                    *rcode = 1;
                    string_delete(str);
                    break;
                }
                else if(ret < 0)
                {
                    *rcode = 2;
                    err_msg("interrupt", sname, sline);
                    string_delete(str);
                    return FALSE;
                }

                sStatments* statments = vector_item(blocks, 0);

                char* fname = MALLOC(strlen(sname) + 32);
                snprintf(fname, strlen(sname) + 32, "%s %d: eval", sname, sline);

                int len = str_kanjilen(code, string_c_str(str));
                for(i=0; i<len; i++) {
                    char* p = str_kanjipos2pointer(code, string_c_str(str), i);
                    char* p2 = str_kanjipos2pointer(code, string_c_str(str), i+1);
                    char buf[128];
                    memcpy(buf, p, p2-p);
                    buf[p2-p] =0;

                    sRFd* nextin2 = sRFd_new2(-1, buf);
                    *rcode = run(statments, fname, nextout
                                , nextin2, nexterr
                                , FALSE, FALSE
                                , return_, parent_blocks);
                    sRFd_delete(nextin2);

                    if(*return_) {
                        string_delete(str);
                        FREE(fname);
                        return TRUE;
                    }
                    else if(*rcode < 0) {
                        string_delete(str);
                        FREE(fname);
                        return FALSE;
                    }
                }
                string_delete(str);
                FREE(fname);
            }
            }
            break;
*/

        default: {
            fprintf(stderr, "unexpected err. illegal command(%s)\n", string_c_str(vector_item(argv, 0)));
            exit(1);
        }
    }

    return TRUE;
}

