#include "config.h"

#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <signal.h>
#include <dirent.h>

#include <limits.h>

#include "saphire.h"
#include "saphire_inner.h"

// 内部コマンド用のリダイレクト処理
static BOOL statment_tree_redirect(sCommand* command, sRFd ** nextin, sWFd** nextout, int* nexterr, sRFd* pipein, sWFd* pipeout, int pipeerr, char* sname, int sline)
{
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);
        if(redirect->mType == kRedirectErrAndOutput) {
            err_msg("inner command can't take error and output redirect", sname, sline);
            return FALSE;
        }
        else {
            mode_t mode;
            switch(redirect->mType) {
                case kRedirectInput:
                    mode = O_RDONLY;
                    break;
                
                case kRedirectOverwrite:
                    mode = O_WRONLY | O_CREAT | O_TRUNC;
                    break;
                    
                case kRedirectAppend:
                    mode = O_WRONLY | O_CREAT | O_APPEND;
                    break;
            }
            
            int openfd = open(string_c_str(redirect->mFName), mode, 0666);
            if(openfd < 0) {
                char buf[PATH_MAX];
                snprintf(buf, PATH_MAX, "open %s is err1\n"
                            , string_c_str(redirect->mFName));
                err_msg(buf, sname, sline);
                return FALSE;
            }

            switch(redirect->mType) {
                case kRedirectInput:
                    if(*nextin == pipein) {
                        *nextin = sRFd_new(openfd);
                    }
                    else {
                        if(!sRFd_close(*nextin)) {
                            return FALSE;
                        }
                        *nextin = sRFd_new(openfd);
                    }
                    break;
                
                case kRedirectOverwrite:
                    if(redirect->mFd == 1) {
                        if(*nextout == pipeout) {
                            *nextout = sWFd_new(openfd);
                        }
                        else {
                            if((*nextout)->mFd != -1 && (*nextout)->mFd != 1) {
                                (void)close((*nextout)->mFd);
                            }
                            (*nextout)->mFd = openfd;
                        }
                    }
                    else {
                        if(*nexterr != 2 && *nexterr != pipeerr) (void)close(*nexterr);
                        
                        *nexterr = openfd;
                    }
                    break;
                    
                case kRedirectAppend:
                    if(redirect->mFd == 1) {
                        if(*nextout == pipeout) {
                            *nextout = sWFd_new(openfd);
                        }
                        else {
                            if((*nextout)->mFd != -1 && (*nextout)->mFd != 1) {
                                (void)close((*nextout)->mFd);
                            }
                            (*nextout)->mFd = openfd;
                        }
                    }
                    else {
                        if(*nexterr != 2 && *nexterr != pipeerr) (void)close(*nexterr);
                        
                        *nexterr = openfd;
                    }
                    break;
            }
        }
    }

    return TRUE;
}

// forkした子供(外部コマンド)をexecさせたい関数
static BOOL statment_tree_exec_cprog(sCommand* command, int nextin, int nextout, int nexterr, vector_obj* argv, char* sname, int sline)
{
//puts("statment_tree_exec_cprog");
    /// リダイレクト ///
    int err_and_output = FALSE;
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);

        if(redirect->mType == kRedirectErrAndOutput) {
            err_and_output = TRUE;
        }
        else {
            mode_t mode;
            switch(redirect->mType) {
                case kRedirectInput:
                    mode = O_RDONLY;
                    break;
                
                case kRedirectOverwrite:
                    mode = O_RDWR | O_CREAT | O_TRUNC;
                    break;
                    
                case kRedirectAppend:
                    mode = O_RDWR | O_CREAT | O_APPEND;
                    break;
            }

            int openfd = open(string_c_str(redirect->mFName), mode, 0666);
            if(openfd < 0) {
                char buf[PATH_MAX];
                snprintf(buf, PATH_MAX, "open %s is err2\n"
                                 , string_c_str(redirect->mFName));
                err_msg(buf, sname, sline);
                return FALSE; 
            }
            
            switch(redirect->mFd) {
                case 0:
                    if(nextin != 0 && nextin != -1) {
                        (void)close(nextin);
                    }
                    nextin = openfd;
                    break;
                    
                case 1:
                    if(nextout != 1 && nextout != -1) {
                        (void)close(nextout);
                    }
                    nextout = openfd;
                    break;
                    
                case 2:
                    if(nexterr != 2) {
                        (void)close(nexterr);
                    }
                    nexterr = openfd;
                    break;
            }
        }
    }

    /// パイプ ///
    if(err_and_output) {
        if(dup2(nextout, 2) < 0) {
            perror("dup2 1");
            exit(1);
        }
    }
    
    if(nextin != 0) {
        if(dup2(nextin, 0) < 0) {
            char buf[32];
            snprintf(buf, 32, "dup2 3 nextin %d", nextin);
            perror(buf);
            exit(1);
        }
        if(close(nextin) < 0) { return FALSE; }
    }
    
    if(nextout != 1) {
        if(dup2(nextout, 1) < 0) {
            char buf[128];
            snprintf(buf, 128, "dup2 nextout (%d)", nextout);
            perror(buf);
            exit(1);
        }
        
        if(close(nextout) < 0) { return FALSE; }
    }
    
    if(nexterr != 2) {
        if(dup2(nexterr, 2) < 0) {
            perror("dup2 5");
            exit(1);
        }
        
        if(close(nexterr) < 0) { return FALSE; }
    }

    /// char** に変換 ///
    char** argv2 = (char**)malloc(sizeof(char*)*(vector_size(argv)+1));

    int i;
    for(i=0; i<vector_size(argv); i++) {
        string_obj* item = (string_obj*)vector_item(argv, i);
        argv2[i] = string_c_str(item);
    }
    argv2[i] = NULL;

    /// exec ///
    execvp(argv2[0], argv2);
    fprintf(stderr, "exec('%s') error\n", argv2[0]);
    kill(getppid(), SIGUSR1);
    exit(1);
}

// 外部コマンドの処理
static BOOL statment_tree_external_command(sStatment* statment, vector_obj* argv, sRFd* nextin, sWFd* nextout, int nexterr, sWFd* pipeout, sCommand* command, BOOL last_program, int* last_program_pid, char* sname, int sline, vector_obj* envs)
{
//printf("nextin %d nextout %d\n", nextin->mFd, nextout->mFd);
    char* argv0 = string_c_str(vector_item(argv, 0));

    /// メモリーパイプをパイプに変換 ///
    int mpipefds[2] = { -1, -1 };
    int mpipefds2[2] = { -1, -1 };
    int nextin2 = nextin->mFd;
    int nextout2 = nextout->mFd;
    if(nextin2 == -1) {
        if(pipe(mpipefds) < 0) {
            perror("pipe");
            exit(1);
        }
        nextin2 = mpipefds[0];
        mpipefds[0] = -1;
    }
    if(nextout2 == -1) {
        if(pipe(mpipefds2) < 0) {
            perror("pipe");
            exit(1);
        }
        
        nextout2 = mpipefds2[1];
        mpipefds2[1] = -1;
    }
 
    /// fork ///
    pid_t pid = fork();
    if(pid < 0) {
        perror("fork");
        exit(1);
    }

    /// 子プロセスの処理 ///
    if(pid == 0) {
        /// 環境変数 ///
        int i;
        for(i=0; i<vector_size(envs); i+=2) {
            char* name = string_c_str(vector_item(envs, i));
            char* value = string_c_str(vector_item(envs, i+1));

            setenv(name, value, 1);
        }
        
        if(mpipefds[0] != -1) (void)close(mpipefds[0]);
        if(mpipefds[1] != -1) (void)close(mpipefds[1]);
        if(mpipefds2[0] != -1) (void)close(mpipefds2[0]);
        if(mpipefds2[1] != -1) (void)close(mpipefds2[1]);

        pid = getpid();

        // プログラムは固有のプロセスグループに属する
        if(gAppType != kATOptC) {
            (void)setpgid(pid, pid);
        }
        /// 前に持ってくる
        if((gAppType == kATCursesApp 
                || gAppType == kATConsoleApp)
            && !statment->mBackground 
            && pipeout->mFd == STDOUT_FILENO) 
        {
            if(tcsetpgrp(0, pid) < 0) {
                perror("tcsetpgrp(child)");
                exit(1);
            }
        }

        saphire_restore_signal_default();
        sigchld_block(0);

        if(!statment_tree_exec_cprog(command, nextin2, nextout2, nexterr , argv ,sname, sline))
        {
            return FALSE;
        }

        return TRUE;
    }
    /// 親プロセスの処理 ///
    else {
        if(last_program) *last_program_pid = pid;

        // プログラムはすべて固有のプロセスグループに属する
        if(gAppType != kATOptC) {
            (void)setpgid(pid, pid);
        }

        /// 子へのメモリパイプのバッファ渡しがあって
        /// 子がTSTP食らうとsaphire本体のパイプの書き込みがブロックし
        /// waitpidもできないためデッドロックする。
        /// そのためバッファを書き込むプロセスを用意してsaphire
        /// 本体はバッファの書き込みをしない
        if(mpipefds[1] != -1) {
            int pid2 = fork();
            if(pid2 < 0) {
                perror("fork2");
                exit(1);
            }
            /// writer process ///
            if(pid2 == 0) {             /// 子プロセス
                pid2 = getpid();

                if(gAppType != kATOptC) {
                    (void)setpgid(pid2, pid);
                }

                if(mpipefds[1] != -1) {
                    (void)close(nextin2);
                    BOOL epipe = FALSE;
                    if(!gSigUser && write(mpipefds[1], string_c_str(nextin->mBuffer), string_length(nextin->mBuffer)) < 0)
                    {
                        if(errno != EPIPE ) {
                            perror("write memory pipe");
                            exit(1);
                        }
                    }
                    (void)close(mpipefds[1]);
                }
                exit(0);
            }
            /// 親プロセス ///
            else {
                if(gAppType != kATOptC) {
                    (void)setpgid(pid2, pid);
                }

                if(mpipefds[1] != -1) {
                    (void)close(nextin2);
                    (void)close(mpipefds[1]);
                }

                if(mpipefds2[0] != -1) {
                    (void)close(nextout2);

                    if(!gSigUser) {
                        char buf[BUFSIZ+1];
                        while(1) {
                            if(gKitutukiSigInt) {
                                err_msg("signal interrupt", sname, sline);
                                gKitutukiSigInt = FALSE;
                                return FALSE;
                            }
                            int r = read(mpipefds2[0], buf, BUFSIZ);
                            if(r < 0) {
                                if(errno == EINTR) {
                                    buf[0] = 0;
                                    sWFd_push_back(nextout, buf);
                                    break;
                                }
                                else {
                                    perror("read mpipe");
                                    exit(1);
                                }
                            }
                            buf[r] = 0;

                            sWFd_push_back(nextout, buf);

                            if(r == 0) {
                                break;
                            }
                        }
                    }

                    (void)close(mpipefds2[0]);
                }
            }
            if(gAppType == kATOptC) {
                while(1) {
                    int status;
                    pid2 = waitpid(pid2, &status, WUNTRACED);

                    if(pid2 < 0 && errno == EINTR) {
                    }
                    else if(WIFSTOPPED(status)) {
                        kill(pid2, SIGCONT);
                    }
                    else if(WIFSIGNALED(status)) {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    else {
                        break;
                    }
                }
            }
            else {
                int status;
                pid2 = waitpid(pid2, &status, WUNTRACED);

                if(WIFSIGNALED(status) || WIFSTOPPED(status))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
        }
        else {
            if(mpipefds2[0] != -1) {
                (void)close(nextout2);

                if(!gSigUser) {
                    char buf[BUFSIZ+1];
                    while(1) {
                        if(gKitutukiSigInt) {
                            err_msg("signal interrupt", sname, sline);
                            gKitutukiSigInt = FALSE;
                            return FALSE;
                        }
                        int r = read(mpipefds2[0], buf, BUFSIZ);
                        if(r < 0) {
                            if(errno == EINTR) {
                                buf[0] = 0;
                                sWFd_push_back(nextout, buf);
                                break;
                            }
                            else {
                                perror("read mpipe");
                                exit(1);
                            }
                        }
                        buf[r] = 0;

                        sWFd_push_back(nextout, buf);

                        if(r == 0) {
                            break;
                        }
                    }
                }

                (void)close(mpipefds2[0]);
            }
        }

        // 毎回waitする
        if(!last_program) {
            if(gAppType == kATOptC) {
                while(1) {
                    int status;
                    pid = waitpid(pid, &status, WUNTRACED);

                    if(pid < 0 && errno == EINTR) {
                    }
                    else if(WIFSTOPPED(status)) {
                        kill(pid, SIGCONT);
                    }
                    else if(WIFSIGNALED(status)) {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    else {
                        break;
                    }
                }
            }
            else {
                int status;
                pid = waitpid(pid, &status, WUNTRACED);

                if(WIFSIGNALED(status) || WIFSTOPPED(status)) {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
        }

        return TRUE;
    }
}

/// argvを作る
static BOOL statment_tree_make_argv(sCommand* command, vector_obj* argv, vector_obj* blocks, vector_obj* parent_blocks, BOOL* return_, BOOL enable_return, BOOL* break_, BOOL enable_break, sRFd* nextin, sRFd* pipein, char* sname, int sline, vector_obj* envs, vector_obj* psubs, sObject* object, sFunction* running_fun, sClass* running_class)
{
    int ii;
    for(ii=0; ii<vector_size(command->mArgs); ii++) {
        sArg* arg = vector_item(command->mArgs, ii);
        switch(arg->mKind) {
            /// string ///
            case 0:
                vector_add(argv, STRING_NEW(string_c_str(arg->mBody)));
                break;

            /// @ のコマンド展開　///
            case 2: {
                sStatments* statments = arg->mBody;
                sWFd* pipeout = sWFd_new(-1);
                int rcode = run(statments, "quick command expansion"
                    , pipeout, pipein, 2, return_, enable_return, break_, enable_break, parent_blocks, FALSE
                    , envs, object, running_fun, running_class);

                if(rcode == -1) {
                    return FALSE;
                }

                vector_add(argv, STRING_NEW(pipeout->mBuffer));
                }
                break;

            /// @@のコマンド展開 ///
            case 4: {
                sStatments* statments = arg->mBody;
                sWFd* pipeout = sWFd_new(-1);
                int rcode = run(statments, "quick command expansion"
                    , pipeout, pipein, 2, return_, enable_return, break_, enable_break, parent_blocks, FALSE
                    , envs, object, running_fun, running_class);

                if(rcode == -1) {
                    return FALSE;
                }

                sRFd* tmp = sRFd_new2(-1, pipeout->mBuffer);
                while(1) {
                    string_obj* str = STRING_NEW("");
                    int ret = sRFd_read_oneline(tmp, str, gLineField);

                    if(ret < -1) {
                        return FALSE;
                    }
                    else if(ret == 1) {
                        break;
                    }

                    string_chomp(str);
                    vector_add(argv, str);
                }

                }
                break;

            /// ブロック ///
            case 5: {
                vector_add(blocks, arg->mBody);
                }
                break;

            /// プロセス置換 ///
            case 6: {
                sStatments* statments = arg->mBody;
                sWFd* pipeout = sWFd_new(-1);
                int rcode = run(statments, "process substitution"
                    , pipeout, pipein, 2, return_, enable_return, break_, enable_break, parent_blocks, FALSE
                    , envs, object, running_fun, running_class);

                if(rcode == -1) {
                    return FALSE;
                }

                /// 一時ファイルを作る ///
                char fname[PATH_MAX];
                while(1) {
                    int n = random() % 1000;
                    snprintf(fname, PATH_MAX, "%s/psub2%d", gTmpDir, n);
                    int flg = 0;
                    int i;
                    for(i=0; i<vector_size(gTmpFiles); i++) {
                        char* fname2 = string_c_str(vector_item(gTmpFiles, i));
                        if(strcmp(fname, fname2) == 0) {
                            flg++;
                        }
                    }

                    if(flg == 0) break;
                }
                vector_add(gTmpFiles, STRING_NEW(fname));
                
                int fd = open(fname, O_WRONLY|O_TRUNC|O_CREAT, 0600);
                if(write(fd, pipeout->mBuffer, strlen(pipeout->mBuffer)) < 0)
                {
                    perror("write");
                    exit(1);
                }

                (void)close(fd);

                vector_add(argv, STRING_NEW(fname));
                }
                break;

            /// ヘアドキュメント ///
            case 7: {
                string_obj* text = arg->mBody;

                /// 一時ファイルを作る ///
                char fname[PATH_MAX];
                while(1) {
                    int n = random() % 1000;
                    snprintf(fname, PATH_MAX, "%s/here_doc%d", gTmpDir, n);
                    int flg = 0;
                    int i;
                    for(i=0; i<vector_size(gTmpFiles); i++) {
                        char* fname2 = string_c_str(vector_item(gTmpFiles, i));
                        if(strcmp(fname, fname2) == 0) {
                            flg++;
                        }
                    }

                    if(flg == 0) break;
                }

                vector_add(gTmpFiles, STRING_NEW(fname));

                int f = open(fname, O_WRONLY|O_TRUNC|O_CREAT, 0644);
                if(write(f, string_c_str(text), string_length(text)) < 0) {
                    perror("write on here document");
                    exit(1);
                }
                (void)close(f);

                vector_add(argv, STRING_NEW(fname));
                }
                break;

            /// プロセス置換2 ///
            case 8: {
                sStatments* statments = arg->mBody;

                /// 一時ファイルを作る ///
                char fname[PATH_MAX];
                while(1) {
                    int n = random() % 1000;
                    snprintf(fname, PATH_MAX, "%s/psub%d", gTmpDir, n);
                    int flg = 0;
                    int i;
                    for(i=0; i<vector_size(gTmpFiles); i++) {
                        char* fname2 = string_c_str(vector_item(gTmpFiles, i));
                        if(strcmp(fname, fname2) == 0) {
                            flg++;
                        }
                    }

                    if(flg == 0) break;
                }
                vector_add(gTmpFiles, STRING_NEW(fname));
                vector_add(argv, STRING_NEW(fname));
                vector_add(psubs, sPSub_new(fname, statments));
                }
                break;

        }
    }

    return TRUE;
}

// 文を処理したい関数。本体。
// 文を回す
// TRUE 正常終了
// FALSE エラー
static BOOL statment_tree(sStatment* statment, int* last_program_pid, sWFd* pipeout, sRFd* pipein, int pipeerr, char* title, int* rcode, BOOL* return_, BOOL enable_return, BOOL* break_, BOOL enable_break, vector_obj* parent_blocks, char* sname, int sline, vector_obj* envs, vector_obj* psubs, sObject* object, sFunction* running_fun, sClass* running_class) 
{
    sRFd* nextin;
    int nexterr = pipeerr;

    /// パイプ準備 ///
    int exist_globalin = statment->mGlobalPipeIn & 0x0F;
    BOOL exist_stdinpipe = statment->mSTDINPipe;

    if(exist_globalin == 1) {
        nextin = sRFd_new(-1);
        string_put(nextin->mBuffer, string_c_str(gGlobalPipe));
    }
    else if(exist_globalin == 4) {
        int num = statment->mGlobalPipeInNum;
        nextin = sRFd_new(-1);
        string_put(nextin->mBuffer, string_c_str(gGlobalPipeNum[num]));
    }
    else if(exist_stdinpipe) {
        if(pipein->mFd == -1) {
            nextin = sRFd_new(-1);
            string_put(nextin->mBuffer, string_c_str(pipein->mBuffer));
        }
        else {
            sRFd_read_all_to_buffer(pipein);
            nextin = sRFd_new2(pipein->mFd, string_c_str(pipein->mBuffer));
        }
    }
    else {
        nextin = pipein;
    }

    int exist_globalout = 0;
    int exist_globalout_append = 0;

    /// 回す ///
    int j;
    for(j=0; j<vector_size(statment->mCommands); j++) {
        sCommand* command = vector_item(statment->mCommands, j);

        const BOOL first_program = j == 0;
        const BOOL last_program = j == (vector_size(statment->mCommands)-1);

        /// パイプ ///
        sWFd* nextout;
        if(last_program) {
            exist_globalout = statment->mGlobalPipeOut & 0x0F;
            exist_globalout_append = statment->mGlobalPipeOut & 0xF0;

            if(exist_globalout) {
                nextout = sWFd_new(-1);
            }
            else {
                nextout = pipeout;
            }
        }
        else {
            nextout = sWFd_new(-1);
        }

        // argvを作成 sArgから
        vector_obj* argv = VECTOR_NEW(30);
        vector_obj* blocks = VECTOR_NEW(30);
        if(!statment_tree_make_argv(command, argv, blocks, parent_blocks, return_, enable_return, break_, enable_break, nextin, pipein, sname, sline, envs, psubs, object, running_fun, running_class)) {
            return FALSE;
        }
        if(*return_) {
            return TRUE;
        }

        int ii;

        char* arg0 = string_c_str(vector_item(argv, 0));
        sObject* object2;
        sFunction* method;
        sFunction* fun;
        sClass* klass;
        sInnerCommand* inner_fun;
        char* tmp = strstr(arg0, "->");
        BOOL method_class = FALSE;
        if(tmp) {
            char* tmp2 = MALLOC(tmp - arg0 + 1);
            memcpy(tmp2, arg0, tmp-arg0);
            tmp2[tmp - arg0] = 0;

            if(object) {
                if(strcmp(tmp2, "self") == 0) {
                    object2 = object;
                }
                else {
                    object2 = hash_item(object->mObjects, tmp2);
                }
            }
            else {
                object2 = hash_item(gObjects, tmp2);
            }

            if(object2 == NULL) {
                char str[1024];
                snprintf(str, 1024, "object(%s) is not found", tmp2);
                err_msg(str, sname, sline);
                return FALSE;
            }

            tmp += 2;
            char* tmp3 = MALLOC(arg0 + strlen(arg0) - tmp + 1);
            strcpy(tmp3, tmp);

            if(strcmp(tmp3, "class") == 0) {
                method_class = TRUE;
            }
            else {
                method = hash_item(object2->mMethods, tmp3);

                if(method == NULL) {
                    char str[1024];
                    snprintf(str, 1024, "method(%s) is not found", tmp3);
                    err_msg(str, sname, sline);
                    return FALSE;
                }
            }
            fun = NULL;
            inner_fun = NULL;
            klass = NULL;
        }
        else {
            object2 = NULL;
            method = NULL;
            fun = hash_item(gFuncs, arg0);
            inner_fun = hash_item(gInnerCommands, arg0);
            if(object) {
                klass = hash_item(object->mClasses, arg0);
                if(klass == NULL) {
                    sObject* parent = object->mParent;
                    while(parent) {
                        klass = hash_item(parent->mClasses, arg0);
                        if(klass) break;
                        parent = parent->mParent;
                    }
                }
                if(klass == NULL) {
                    klass = hash_item(gClasses, arg0);
                }
            }
            else {
                klass = hash_item(gClasses, arg0);
            }
        }

        if(fun || inner_fun) {
            command->mKind = kInnerCommand;
        }

        /// クラス ///
        if(klass) {
            if(vector_size(argv) >= 2) {
                char* objname = string_c_str(vector_item(argv, 1));

                /// スタックフレームの記録を付ける
                vector_add(gStackTraceFName, STRING_NEW(sname));
                vector_add(gStackTraceLineNum, (void*)sline);

                // スタックを掘る
                vector_add(gStackFrame, HASH_NEW(30));
                vector_add(gStackFramePointers, HASH_NEW(30));
                vector_add(gStackFrameHashs, HASH_NEW(30));
                vector_add(gStackFrameArrays, HASH_NEW(30));

                sObject* object2 = sObject_new(objname, object);
                if(object) {
                    hash_put(object->mObjects, objname, object2);
                }
                else {
                    hash_put(gObjects, objname, object2);
                }

                *rcode = run(klass->statments, title, nextout
                                , nextin, nexterr
                                , return_, TRUE, break_, FALSE, parent_blocks, FALSE, envs, object2, NULL, klass);
                *return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
                *break_ = FALSE;

                // スタックを埋める
                vector_pop_back(gStackFrame);
                vector_pop_back(gStackFrameArrays);
                vector_pop_back(gStackFrameHashs);
                vector_pop_back(gStackFramePointers);

                /// スタックフレームの記録を消す
                string_obj* sname_strace = vector_pop_back(gStackTraceFName);
                int line_strace = (int)vector_pop_back(gStackTraceLineNum);

                /// 終了コード ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) 
                        *rcode = 1;
                    else
                        *rcode = 0;
                }

                char buf2[256];
                snprintf(buf2, 256, "%d", *rcode);
                saphire_set_global_var("RCODE", buf2);

                if(*rcode < 0) {
                    string_push_back(gErrMsg, string_c_str(sname_strace));
                    char tmp[1024];
                    snprintf(tmp, 1024, ":%d\n", line_strace);
                    string_push_back(gErrMsg, tmp);

                    return FALSE;
                }
            }
            else {
                err_msg("reuire object name", sname, sline);
                return FALSE;
            }
        }
        /// 継承 ///
        else if(method_class) {
            if(vector_size(argv) >= 2) {
                char* klassname = string_c_str(vector_item(argv, 1));

                /// スタックフレームの記録を付ける
                vector_add(gStackTraceFName, STRING_NEW(sname));
                vector_add(gStackTraceLineNum, (void*)sline);

                // スタックを掘る
                vector_add(gStackFrame, HASH_NEW(30));
                vector_add(gStackFramePointers, HASH_NEW(30));
                vector_add(gStackFrameHashs, HASH_NEW(30));
                vector_add(gStackFrameArrays, HASH_NEW(30));

                if(object) {
                    klass = hash_item(object->mClasses, klassname);
                    if(klass == NULL) {
                        sObject* parent = object->mParent;
                        while(parent) {
                            klass = hash_item(parent->mClasses, klassname);
                            if(klass) break;
                            parent = parent->mParent;
                        }
                    }
                    if(klass == NULL) {
                        klass = hash_item(gClasses, klassname);
                    }
                }
                else {
                    klass = hash_item(gClasses, klassname);
                }

                if(klass == NULL) {
                    err_msg("class name is not found", sname, sline);
                    return FALSE;
                }
                
                *rcode = run(klass->statments, title, nextout
                                , nextin, nexterr
                                , return_, TRUE, break_, FALSE, parent_blocks, FALSE, envs, object2, NULL, klass);
                *return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
                *break_ = FALSE;

                // スタックを埋める
                vector_pop_back(gStackFrame);
                vector_pop_back(gStackFrameArrays);
                vector_pop_back(gStackFrameHashs);
                vector_pop_back(gStackFramePointers);

                /// スタックフレームの記録を消す
                string_obj* sname_strace = vector_pop_back(gStackTraceFName);
                int line_strace = (int)vector_pop_back(gStackTraceLineNum);

                /// 終了コード ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) 
                        *rcode = 1;
                    else
                        *rcode = 0;
                }

                char buf2[256];
                snprintf(buf2, 256, "%d", *rcode);
                saphire_set_global_var("RCODE", buf2);

                if(*rcode < 0) {
                    string_push_back(gErrMsg, string_c_str(sname_strace));
                    char tmp[1024];
                    snprintf(tmp, 1024, ":%d\n", line_strace);
                    string_push_back(gErrMsg, tmp);

                    return FALSE;
                }
            }
            else if(vector_size(blocks) == 1) {
                /// スタックフレームの記録を付ける
                vector_add(gStackTraceFName, STRING_NEW(sname));
                vector_add(gStackTraceLineNum, (void*)sline);

                // スタックを掘る
                vector_add(gStackFrame, HASH_NEW(30));
                vector_add(gStackFramePointers, HASH_NEW(30));
                vector_add(gStackFrameHashs, HASH_NEW(30));
                vector_add(gStackFrameArrays, HASH_NEW(30));

                sBlock* block = vector_item(blocks, 0);
                sStatments* statments = block->mStatments;
                
                *rcode = run(statments, title, nextout
                                , nextin, nexterr
                                , return_, TRUE, break_, FALSE, parent_blocks, FALSE, envs, object2, NULL, NULL);
                *return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
                *break_ = FALSE;

                // スタックを埋める
                vector_pop_back(gStackFrame);
                vector_pop_back(gStackFrameArrays);
                vector_pop_back(gStackFrameHashs);
                vector_pop_back(gStackFramePointers);

                /// スタックフレームの記録を消す
                string_obj* sname_strace = vector_pop_back(gStackTraceFName);
                int line_strace = (int)vector_pop_back(gStackTraceLineNum);

                /// 終了コード ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) 
                        *rcode = 1;
                    else
                        *rcode = 0;
                }

                char buf2[256];
                snprintf(buf2, 256, "%d", *rcode);
                saphire_set_global_var("RCODE", buf2);

                if(*rcode < 0) {
                    string_push_back(gErrMsg, string_c_str(sname_strace));
                    char tmp[1024];
                    snprintf(tmp, 1024, ":%d\n", line_strace);
                    string_push_back(gErrMsg, tmp);

                    return FALSE;
                }
            }
            else {
                err_msg("reuire class name or a block", sname, sline);
                return FALSE;
            }
        }
        /// メソッド ///
        else if(method) {
            if(!statment_tree_redirect(command, &nextin, &nextout
                            , &nexterr, pipein, pipeout, pipeerr, sname, sline))
            {
                return FALSE;
            }

            /// 引数名が指定されていなければスタックフレームを増やす ///
            vector_obj* fun_argv_before;
            vector_obj* fun_argv;
            if(strcmp(string_c_str(method->arg_name), "") == 0) {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(object2->mArrays, "ARGV");

                /// 引数を渡す ///
                int k;
                fun_argv = VECTOR_NEW(30);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(object2->mArrays, "ARGV", fun_argv);

                vector_add(gStackFrame, HASH_NEW(30));
                vector_add(gStackFramePointers, HASH_NEW(30));
                vector_add(gStackFrameHashs, HASH_NEW(30));
                vector_add(gStackFrameArrays, HASH_NEW(30));
            }
            /// 引数名が指定されていなければスタックフレームは増やさない
            else {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(object2->mArrays
                                        , string_c_str(method->arg_name));

                /// 引数を渡す ///
                int k;
                fun_argv = VECTOR_NEW(30);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(object2->mArrays, string_c_str(method->arg_name), fun_argv);
            }

            /// スタックフレームの記録を付ける
            vector_add(gStackTraceFName, STRING_NEW(sname));
            vector_add(gStackTraceLineNum, (void*)sline);

            /// 実行 ///
            *rcode = run(method->statments, title
                , nextout, nextin, nexterr, return_, TRUE, break_, FALSE, blocks, FALSE, envs, object2, method, running_class);

            *break_ = FALSE;
            if(strcmp(string_c_str(method->arg_name), "") == 0) {
                *return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
            }
            else {
                if(enable_return && *return_) {
                    return TRUE;
                }
            }

            /// スタックフレームの記録を消す
            string_obj* sname_strace = vector_pop_back(gStackTraceFName);
            int line_strace = (int)vector_pop_back(gStackTraceLineNum);


            if(strcmp(string_c_str(method->arg_name), "") == 0) {
                /// スタックを消す ///
                vector_pop_back(gStackFrame);
                vector_pop_back(gStackFrameArrays);
                vector_pop_back(gStackFrameHashs);
                vector_pop_back(gStackFramePointers);

                /// 引数を消す ///
                hash_erase(object2->mArrays, "ARGV");

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(object2->mArrays, "ARGV", fun_argv_before);
                }
            }
            else {
                /// 引数を消す ///
                hash_erase(object2->mArrays, string_c_str(method->arg_name));

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(object2->mArrays, string_c_str(method->arg_name)
                                , fun_argv_before);
                }
            }

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf2[256];
            snprintf(buf2, 256, "%d", *rcode);
            saphire_set_global_var("RCODE", buf2);

            if(*rcode == -1) {
                string_push_back(gErrMsg, string_c_str(sname_strace));
                char tmp[1024];
                snprintf(tmp, 1024, ":%d\n", line_strace);
                string_push_back(gErrMsg, tmp);
                
                return FALSE;
            }
        }

        /// ユーザーコマンド ///
        else if(fun) {
            if(!statment_tree_redirect(command, &nextin, &nextout
                            , &nexterr, pipein, pipeout, pipeerr, sname, sline))
            {
                return FALSE;
            }

            /// 引数名が指定されていなければスタックフレームを増やす ///
            vector_obj* fun_argv_before;
            vector_obj* fun_argv;
            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(gArrays, "ARGV");

                /// 引数を渡す ///
                int k;
                fun_argv = VECTOR_NEW(30);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, "ARGV", fun_argv);

                vector_add(gStackFrame, HASH_NEW(30));
                vector_add(gStackFramePointers, HASH_NEW(30));
                vector_add(gStackFrameHashs, HASH_NEW(30));
                vector_add(gStackFrameArrays, HASH_NEW(30));
            }
            /// 引数名が指定されていなければスタックフレームは増やさない
            else {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(gArrays
                                        , string_c_str(fun->arg_name));

                /// 引数を渡す ///
                int k;
                fun_argv = VECTOR_NEW(30);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, string_c_str(fun->arg_name), fun_argv);
            }

            /// スタックフレームの記録を付ける
            vector_add(gStackTraceFName, STRING_NEW(sname));
            vector_add(gStackTraceLineNum, (void*)sline);

            /// 実行 ///
            *rcode = run(fun->statments, title
                , nextout, nextin, nexterr, return_, TRUE, break_, FALSE, blocks, FALSE, envs, object, fun, running_class);
            *break_ = FALSE;
            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                *return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
            }
            else {
                if(enable_return && *return_) {
                    return TRUE;
                }
            }

            /// スタックフレームの記録を消す
            string_obj* sname_strace = vector_pop_back(gStackTraceFName);
            int line_strace = (int)vector_pop_back(gStackTraceLineNum);


            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                /// スタックを消す ///
                vector_pop_back(gStackFrame);
                vector_pop_back(gStackFrameArrays);
                vector_pop_back(gStackFrameHashs);
                vector_pop_back(gStackFramePointers);

                /// 引数を消す ///
                hash_erase(gArrays, "ARGV");

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(gArrays, "ARGV", fun_argv_before);
                }
            }
            else {
                /// 引数を消す ///
                hash_erase(gArrays, string_c_str(fun->arg_name));

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(gArrays, string_c_str(fun->arg_name)
                                , fun_argv_before);
                }
            }

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf2[256];
            snprintf(buf2, 256, "%d", *rcode);
            saphire_set_global_var("RCODE", buf2);

            if(*rcode == -1) {
                string_push_back(gErrMsg, string_c_str(sname_strace));
                char tmp[1024];
                snprintf(tmp, 1024, ":%d\n", line_strace);
                string_push_back(gErrMsg, tmp);
                
                return FALSE;
            }
        }

        /// 外部登録の内部コマンド ///
        else if(inner_fun) {
            if(!statment_tree_redirect(command, &nextin, &nextout
                                    , &nexterr
                                    , pipein, pipeout, pipeerr, sname, sline))
            {
                return FALSE;
            }

            *rcode = 1; // 初期値は1。コマンドが成功したら0を入れる

            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp))
            {
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp(inner command)");
                    exit(1);
                }
            }

            BOOL r = inner_fun->mFun(rcode, argv, blocks, parent_blocks
                        , nextout, nextin, nexterr
                        , title, !first_program || exist_globalin || exist_stdinpipe);

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf[256];
            snprintf(buf, 256, "%d", *rcode);
            saphire_set_global_var("RCODE", buf);
            if(!r) {
                return FALSE;
            }
        }

        /// 内部コマンド ///
        else if(command->mKind != kCommand) {
//printf("command->mKind %d %s\n", command->mKind, gStatmentKindStrs[command->mKind]);
            if(!statment_tree_redirect(command, &nextin, &nextout
                                    , &nexterr
                                    , pipein, pipeout, pipeerr, sname, sline))
            {
                return FALSE;
            }

            *rcode = 1; // 初期値は1。コマンドが成功したら0を入れる

            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp))
            {
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp inner command");
                    exit(1);
                }
            }

            BOOL r = statment_tree_internal_commands(command, rcode
                , argv, blocks, parent_blocks, nextout, nextin, nexterr, j, title, !first_program || exist_globalin || exist_stdinpipe, return_, enable_return, break_, enable_break, sname, sline, envs, object, running_fun, running_class);

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf[256];
            snprintf(buf, 256, "%d", *rcode);
            saphire_set_global_var("RCODE", buf);
            if(!r) {
                return FALSE;
            }
        }

        /// 外部コマンド ///
        else {
//puts("外部コマンド");
            /// バッファーをフラッシュしておく
            if(nextout == pipeout) {
                if(!sWFd_flash(nextout)) {
                    return FALSE;
                }
            }

            /// リードバッファが空じゃないなら
            if(nextin->mFd != -1 && string_c_str(nextin->mBuffer)[0] != 0)
            {
                int ret = sRFd_read_all_to_buffer(nextin);

                if(ret < 0) {
                    return FALSE;
                }

                nextin->mFd = -1;

                if(!statment_tree_external_command(statment, argv, nextin, nextout, nexterr, pipeout, command, last_program, last_program_pid, sname, sline, envs))
                {
                    return FALSE;
                }
            }
            else {
                if(!statment_tree_external_command(statment, argv, nextin, nextout, nexterr, pipeout, command, last_program, last_program_pid,  sname, sline, envs))
                {
                    return FALSE;
                }
            }
        }

        /// パイプを閉める ///
        if(nextin->mFd != -1 && nextin != pipein) {
            if(!sRFd_close(nextin)) {
                return FALSE;
            }
        }
        if(nextout->mFd != -1 && nextout != pipeout) {
            if(!sWFd_close(nextout)) {
                return FALSE;
            }
        }
        if(nexterr != -1 && nexterr != pipeerr && nexterr != 2 && nexterr != nextout->mFd) {
            if(close(nexterr) < 0) {
                return FALSE;
            }
        }

        /// 出力処理
        if(last_program) {
            /// グローバルパイプへの出力 ///
            if(exist_globalout == 1) {
                if(exist_globalout_append) {
                    string_push_back(gGlobalPipe, nextout->mBuffer);
                }
                else {
                    string_put(gGlobalPipe, nextout->mBuffer);
                }
                *nextout->mBuffer = 0;
            }
            else if(exist_globalout == 4) {
                int num = statment->mGlobalPipeOutNum;
                if(exist_globalout_append) {
                    string_push_back(gGlobalPipeNum[num], nextout->mBuffer);
                }
                else {
                    string_put(gGlobalPipeNum[num], nextout->mBuffer);
                }
                *nextout->mBuffer = 0;
            }
        }
        else {
            /// 次の入力先を決める
            if(nextout->mFd == -1) {
                nextin = sRFd_new(-1);
                string_put(nextin->mBuffer, nextout->mBuffer);
                *nextout->mBuffer = 0;
            }
            else {
                nextin = sRFd_new(-1);
            }
        }

        /// 大域脱出
        if(enable_return && *return_) {
            return TRUE;
        }
        if(enable_break && *break_) {
            return TRUE;
        }
    }

    return TRUE;
}

// 消えた子供を回収したい関数
void run_wait_cprogs(sStatment* statment, int* rcode, sWFd* pipeout, int last_program_pid)
{
    /// - バックグラウンド - ///
    if((gAppType == kATCursesApp || gAppType == kATConsoleApp) 
        && statment->mBackground) 
    {
        sJob* job = sJob_new();

        job->mPGroup = last_program_pid;
        string_put(job->mName, string_c_str(statment->mTitle));

        vector_add(gJobs, job);
    }

    /// - フォアグラウンド - ///
    else {
        sigchld_block(0);       // ブロック解除
        int status = 0;

        if(gAppType == kATOptC) {
            while(1) {
                pid_t pid = waitpid(last_program_pid, &status, WUNTRACED);
                if(pid < 0 && errno == EINTR) {
                }
                else if(WIFSTOPPED(status)) {
                    kill(pid, SIGCONT);
                }
                else {
                    break;
                }
            }
        }
        else {
            pid_t pid = waitpid(last_program_pid, &status, WUNTRACED);

            /// シグナルの配送により終了した場合 ///
            if(WIFSIGNALED(status)) {
                *rcode = WTERMSIG(status) + 128;

                /// 終了コードの反転 ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) {
                        *rcode = 1;
                    }
                    else {
                        *rcode = 0;
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", *rcode);
                saphire_set_global_var("RCODE", buf);
            }
            /// 終了した場合 ///
            else if(WIFEXITED(status)) {
                *rcode = WEXITSTATUS(status);

                /// 終了コードの反転 ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) {
                        *rcode = 1;
                    }
                    else {
                        *rcode = 0;
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", *rcode);
                saphire_set_global_var("RCODE", buf);
            }
            /// シグナルの配送により停止した場合 ///
            else if(WIFSTOPPED(status)) {
                int sig = WSTOPSIG(status) + 128;

                *rcode = sig;

                /// 終了コードの反転 ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) {
                        *rcode = 1;
                    }
                    else {
                        *rcode = 0;
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", *rcode);
                saphire_set_global_var("RCODE", buf);

                sJob* job = sJob_new();

                job->mPGroup = last_program_pid; 
                string_put(job->mName, string_c_str(statment->mTitle));
                vector_add(gJobs, job);

                if(gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp)
                {
                    job->mTty = MALLOC(sizeof(struct termios));
                    tcgetattr(STDIN_FILENO, job->mTty);
                }
            }

            if((gAppType == kATCursesApp || gAppType == kATConsoleApp)
                && pipeout->mFd == STDOUT_FILENO) 
            {
                /// saphireを端末の前面に出す ///
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp(saphire)");
                    exit(1);
                }
            }
        }
    }
}

// 複文を実行したい関数。本体。
int run(sStatments* statments, char* title, sWFd* pipeout, sRFd* pipein, int pipeerr, BOOL* return_, BOOL enable_return, BOOL* break_, BOOL enable_break, vector_obj* parent_blocks, BOOL try_, vector_obj* envs, sObject* object, sFunction* running_fun, sClass* running_class)
{
    int rcode = 0;

    int i;
    for(i=0; i<vector_size(statments->mStatments); i++) {
        sStatment* statment = vector_item(statments->mStatments, i);

        int sline = statment->mLine;
        char* sname = string_c_str(statment->mFName);

        /// $によるコマンド展開
        if(statment->mNotEvaled) {
            string_obj* cmdline = STRING_NEW("");
            if(expand_env(string_c_str(statment->mNotEvaled), cmdline, pipein, sname, &sline, object)) {
                sStatments* estatments = STATMENTS_NEW();

                if(parse(string_c_str(cmdline), sname, &sline, estatments)) 
                {
                    rcode = run(estatments, title, pipeout, pipein
                        , pipeerr, return_, enable_return, break_, enable_break, parent_blocks, FALSE, envs, object, running_fun, running_class);

                    if(rcode < 0) {
                        return rcode;
                    }
                }
                else {
                    return -1;
                }
            }
            else {
                return -1;
            }
        }
        /// パースされている文を実行
        else {
            sigchld_block(1);

            vector_obj* psubs = VECTOR_NEW(30);
            int last_program_pid = -1;
            BOOL r = statment_tree(statment
                          , &last_program_pid
                          , pipeout, pipein, pipeerr
                          , title, &rcode
                          , return_, enable_return, break_, enable_break, parent_blocks , sname, sline
                          , envs, psubs, object, running_fun, running_class);

            /// プロセス回収 ///
            if(last_program_pid != -1) {
                run_wait_cprogs(statment, &rcode, pipeout, last_program_pid);
            }

            /// ランタイムエラー ///
            if(r == FALSE) {
                return -1;
            }
            // reutrn
            if(*return_) {
                return 0;
            }

            /// プロセス置換を実行する ///
            int l;
            for(l=0; l<vector_size(psubs); l++) {
                sPSub* psub = vector_item(psubs, l);

                int f = open(string_c_str(psub->mFName), O_RDONLY);

                sRFd* nextin2 = sRFd_new(f);
                int rcode = run(psub->mStatments, "process substitution2"
                    , pipeout, nextin2, 2, return_, enable_return, break_, enable_break, parent_blocks, FALSE
                    , envs, object, running_fun, running_class);

                (void)close(f);

                if(rcode < 0) {
                    return rcode;
                }
            }
        }

        /// 行末の処理 ///
        if(gSigUser) {
            gSigUser = FALSE;
            err_msg("command not found", sname, sline);
            return -1;
        }
        else if(rcode == 128+SIGINT || gKitutukiSigInt) {
            err_msg("signal interrupt!", sname, sline);
            gKitutukiSigInt = FALSE;
            return -1;
        }
        else if(statment->mTerminated == kTOrOr && rcode == 0) {
            while(i<vector_size(statments->mStatments) 
                    && statment->mTerminated != kTNormal) 
            {
                i++;
                statment = vector_item(statments->mStatments, i);
            }
        }
        else if(statment->mTerminated == kTAndAnd && rcode != 0) {
            while(i<vector_size(statments->mStatments) 
                    && statment->mTerminated != kTNormal) 
            {
                i++;
                statment = vector_item(statments->mStatments, i);
            }
        }
        else if(try_ && rcode != 0) {
            return -1;
        }

        /// 大域脱出
        if(enable_return && *return_) {
            return rcode;
        }
        if(enable_break && *break_) {
            return rcode;
        }
    }

    /// 空文の場合 ///
    if(vector_size(statments->mStatments) == 0) {
        /// CTRL-Cが押された ///
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return SIGINT + 128;
        }
    }

    return rcode;
}

///////////////////////////////////////////////////
// ファイル読み込み
///////////////////////////////////////////////////
// -1ならエラーメッセージがgErrMsgに入っているので出力してください

int saphire_load(char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr)
{
    string_obj* str = STRING_NEW("");

    int f = open(fname, O_RDONLY);
    if(f < 0) {
        return -1;
    }

    char buf[BUFSIZ];
    while(1) {
        int size = read(f, buf, BUFSIZ-1);
        if(size == 0) {
            break;
        }
        if(size < 0) {
            (void)close(f);
            return -1;
        }

        buf[size] = 0;

        string_push_back(str, buf);
    }
    (void)close(f);

    return saphire_shell(string_c_str(str), fname, pipeout, pipein, pipeerr);

}

///////////////////////////////////////////////////
// シェル実行
///////////////////////////////////////////////////

// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell(char* command, char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr)
{
    gKitutukiSigInt = FALSE;

    string_obj* fname2;
    if(fname == NULL) {
        fname2 = STRING_NEW(command);
    }
    else {
        fname2 = STRING_NEW(fname);
    }

    int sline = 1;
    string_obj* str = STRING_NEW("");
    if(!delete_comments(command, str, " ", string_c_str(fname2), &sline)) {
        return -1;
    }

    sline = 1;

    sStatments* statments = STATMENTS_NEW();
    if(!parse(string_c_str(str), string_c_str(fname2), &sline, statments)) {
        return -1;
    }
    else {
        /// 実行 ///
        int return_ = FALSE;
        int break_ = FALSE;
        vector_obj* envs = VECTOR_NEW(10);
        int rcode = run(statments, string_c_str(fname2)
                    , pipeout, pipein, pipeerr
                    , &return_, TRUE, &break_, FALSE, NULL, FALSE, envs, NULL, NULL, NULL);

        /// 解放 ///
        return rcode;
    }
}

// コマンド（文字列）を実行して結果をresultに入れたい関数
// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell3(string_obj* result, char* command, char* title, sRFd* pipein)
{
    sWFd* pipeout = sWFd_new(-1);
    int rcode = saphire_shell(command, title, pipeout, pipein, STDERR_FILENO);
    if(!sWFd_flash(pipeout)) {
        return -1;
    }

    if(rcode < 0) {
        return -1;
    }

    string_put(result, pipeout->mBuffer);

    return rcode;
}

// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell4(char* command, char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr, sObject* object)
{
    gKitutukiSigInt = FALSE;

    string_obj* fname2;
    if(fname == NULL) {
        fname2 = STRING_NEW(command);
    }
    else {
        fname2 = STRING_NEW(fname);
    }

    int sline = 1;
    string_obj* str = STRING_NEW("");
    if(!delete_comments(command, str, " ", string_c_str(fname2), &sline)) {
        return -1;
    }

    sline = 1;

    sStatments* statments = STATMENTS_NEW();
    if(!parse(string_c_str(str), string_c_str(fname2), &sline, statments)) {
        return -1;
    }
    else {
        /// 実行 ///
        int return_ = FALSE;
        int break_ = FALSE;
        vector_obj* envs = VECTOR_NEW(10);
        int rcode = run(statments, string_c_str(fname2)
                    , pipeout, pipein, pipeerr
                    , &return_, TRUE, &break_, FALSE, NULL, FALSE, envs, object, NULL, NULL);

        /// 解放 ///
        return rcode;
    }
}

// コマンド（文字列）を実行して結果をresultに入れたい関数
// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell5(string_obj* result, char* command, char* title, sRFd* pipein, sObject* object)
{
    sWFd* pipeout = sWFd_new(-1);
    int rcode = saphire_shell4(command, title, pipeout, pipein, STDERR_FILENO, object);
    if(!sWFd_flash(pipeout)) {
        return -1;
    }

    if(rcode < 0) {
        return -1;
    }

    string_put(result, pipeout->mBuffer);

    return rcode;
}

///////////////////////////////////////////////////
// ジョブの数
///////////////////////////////////////////////////
int saphire_job_num()
{
    return vector_size(gJobs);
}

///////////////////////////////////////////////////
// ジョブのタイトルを返す
///////////////////////////////////////////////////
char* saphire_job_title(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);
        return string_c_str(job->mName);
    }
    else {
        return NULL;
    }
}

///////////////////////////////////////////////////
// 全てのジョブをkillする
///////////////////////////////////////////////////
void saphire_kill_all_jobs()
{
    int i;
    for(i=0; i<vector_size(gJobs); i++) {
        sJob* job = (sJob*)vector_item(gJobs, i);

        int j;
        for(j=0; j<vector_size(job->mPIDs); j++) {
            int pid = (int)vector_item(job->mPIDs, j);
            kill(pid, SIGKILL);
        }
    }

    vector_clear(gJobs);
}

///////////////////////////////////////////////////
// ジョブを消す
///////////////////////////////////////////////////
void saphire_kill_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);

        int i;
        for(i=0; i<vector_size(job->mPIDs); i++) {
            int pid = (int)vector_item(job->mPIDs, i);
            kill(pid, SIGKILL);
        }
        vector_erase(gJobs, num);
    }
}

///////////////////////////////////////////////////
// ジョブを前面に出す
///////////////////////////////////////////////////
void (*saphire_job_done)(int job_num, char* job_title) = NULL;

BOOL forground_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);

        if(job->mTty) tcsetattr(STDIN_FILENO, TCSANOW, job->mTty);
        if(tcsetpgrp(0, job->mPGroup) < 0) {
            perror("tcsetpgrp(fg)");
            return FALSE;
        }
        
        if(kill(job->mPGroup, SIGCONT) < 0) {
            perror("kill(fg)");
            return FALSE;
        }

        /// wait ///
        int status = 0;
        pid_t pid = waitpid(job->mPGroup, &status, WUNTRACED);

        if(pid < 0) {
            perror("waitpid(run_wait_cprogs)");
            return FALSE;
        }

        /// 終了した場合 ///
        if(WIFEXITED(status)) {
            if(saphire_job_done) 
                saphire_job_done(num, string_c_str(job->mName));

            vector_erase(gJobs, num);

            /// saphireを前面に出す ///
            //mreset_tty();
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp");
                return FALSE;
            }

            int return_code = WEXITSTATUS(status);
            char buf[256];
            snprintf(buf, 256, "%d", return_code);
            saphire_set_global_var("RCODE", buf);
        }
        /// シグナルの配送により終了した場合 ///
        else if(WIFSIGNALED(status)) {
            if(saphire_job_done) 
                saphire_job_done(num, string_c_str(job->mName));

            vector_erase(gJobs, num);

            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp");
                return FALSE;
            }

            int return_code = WTERMSIG(status) + 128;
            char buf[256];
            snprintf(buf, 256, "%d", return_code);
            saphire_set_global_var("RCODE", buf);
        }
        /// シグナルの配送によりフォワグランドジョブが停止した場合 ///
        else if(WIFSTOPPED(status)) {
            if(!job->mTty) {
                job->mTty = MALLOC(sizeof(struct termios));
                tcgetattr(STDIN_FILENO, job->mTty);
            }

            /// saphire を前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp");
                return FALSE;
            }
        }

        return TRUE;
    }

    err_msg("invalid job number", "fg", 1);
    return FALSE;
}

///////////////////////////////////////////////////
// ジョブにSIGCONTする
///////////////////////////////////////////////////
void background_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);
        
        if(kill(job->mPGroup, SIGCONT) < 0) {
            perror("kill(bg)");
            return;
        }
    }
}

///////////////////////////////////////////////////
// バックグラウンドジョブが終了していれば後処理をする
///////////////////////////////////////////////////
void saphire_wait_background_job()
{
    int status;
    pid_t pid = waitpid(-1, &status, WNOHANG);

    if(pid >= 0) {
        int i;
        for(i=0; i<vector_size(gJobs); i++) {
            sJob* job = (sJob*)vector_item(gJobs, i);

            if(pid == job->mPGroup) {
                if(saphire_job_done) 
                    saphire_job_done(i, string_c_str(job->mName));

                vector_erase(gJobs, i);
                break;
            }
        }
    }
}

///////////////////////////////////////////////////////
// 内部コマンドの登録
///////////////////////////////////////////////////////
void saphire_add_inner_command(char* name, fInnerCommand fun)
{
    hash_put(gInnerCommands, name, sInnerCommand_new(name, fun));
}

// コマンド（文字列）をコンパイルしてstatmentsに格納したい関数
// エラー時はgErrMsgを出力する
BOOL saphire_compile2(char* cmdline, char* fname, sStatments* statments)
{
    int sline = 1;
    if(!parse(cmdline, fname, &sline, statments)) {
        return FALSE;
    }
    else {
        return TRUE;
    }
}

// compile2で格納されたstatmentsを実行したい関数
int saphire_run(sStatments* statments, char* title, sWFd* pipeout, sRFd* pipein, int pipeerr)
{
    int return_ = FALSE;
    int break_ = FALSE;
    vector_obj* envs = VECTOR_NEW(10);
    return run(statments, title, pipeout, pipein, pipeerr
            , &return_, TRUE, &break_, FALSE, NULL, FALSE, envs, NULL, NULL, NULL);
}

void saphire_sweep()
{
    tmpfile_sweep();
}

/// ごみ掃除(GCはないけど)したい関数
void tmpfile_sweep()
{
    /// プロセス置換で使った一時ファイルを掃除 ///
    int k;
    for(k=0; k<vector_size(gTmpFiles); k++) {
        string_obj* fname = vector_item(gTmpFiles, k);

        (void)unlink(string_c_str(fname));
    }
    vector_clear(gTmpFiles);
}

/// ごみ掃除(GCはないけど)したい関数
void tmpfile_all_sweep()
{
    DIR* dir = opendir(gTmpDir);
    if(dir) {
        struct dirent* entry;
        while(entry = readdir(dir)) {
            char fname[PATH_MAX];
            snprintf(fname, PATH_MAX, "%s/%s", gTmpDir, entry->d_name);

            (void)unlink(fname);
        }
    }
    closedir(dir);
}
